// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

//
// This file contains the concrete values which can be obtained from generic reflected variables.
// Each one corresponds to a hkReflect::Var class with get and set methods.
//

template<typename T> class hkArrayView;

namespace hkReflect
{
    /// Qualified Type - a type with an optional const qualifier.
    template<typename TYPE>
    class QualifiedType
    {
    public:

        HK_DECLARE_CLASS(QualifiedType, New, Reflect);
        HK_REFLECT_AS_POINTER(&hkReflect::Detail::RawPointerImpl::s_instance, Type);

        HK_ALWAYS_INLINE QualifiedType() : m_type(nullptr) {}
        /*explicit*/ HK_ALWAYS_INLINE QualifiedType(_In_ const TYPE* t) : m_type(t) {} 
        HK_ALWAYS_INLINE QualifiedType(_In_ const TYPE* t, bool _isConst) : m_type(t) {}
        template<typename T>
        HK_ALWAYS_INLINE QualifiedType(QualifiedType<T> t) : m_type(t.m_type) { }

        HK_ALWAYS_INLINE void clear() { m_type = nullptr; }
        HK_ALWAYS_INLINE bool isConst() const { return false; }
        HK_ALWAYS_INLINE bool isNull() const { return hkUlong(m_type) <= 1; }
        HK_ALWAYS_INLINE bool isValid() const { return hkUlong(m_type) > 1; }
        HK_ALWAYS_INLINE bool operator==(QualifiedType t) const { return m_type == t.m_type; }

        HK_ALWAYS_INLINE QualifiedType withConst() const { return QualifiedType(m_type, true); }
        HK_ALWAYS_INLINE QualifiedType withoutConst() const { return QualifiedType(m_type, false); }

        HK_ALWAYS_INLINE void set(_In_ const TYPE* type, bool _isConst)
        {
            //HK_ASSERT_NO_MSG(0x212550e9, ((hkUlong(type))&1) == 0);
            m_type = type;
            //m_isConst = _isConst;//hkAddByteOffsetConst(m_type, _isConst);
        }

        HK_ALWAYS_INLINE _Ret_maybenull_ const TYPE* get() const { return m_type; }//hkClearBits(m_type,1); }
        HK_ALWAYS_INLINE _Ret_maybenull_ operator const TYPE*() const { return m_type; } //hkClearBits(m_type,1); }

        HK_ALWAYS_INLINE const TYPE& operator*() const { return *m_type; }//*hkClearBits(m_type,1); }
        HK_ALWAYS_INLINE _Ret_maybenull_ const TYPE* operator->() const { return m_type; }//hkClearBits(m_type,1); }

        template<typename T>
        HK_ALWAYS_INLINE QualifiedType<T> reinterpret() const { return QualifiedType<T>(static_cast<const T*>(m_type)); }

    private:

        template<typename U> friend class QualifiedType;
        const TYPE* m_type; 
        //bool m_isConst;
    };


    /// Boolean value
    struct HK_EXPORT_COMMON BoolValue
    {
        HK_DECLARE_PLACEMENT_ALLOCATOR();

        BoolValue(bool b = false) : m_value(b) {}
        operator bool() const { return m_value; }
        int compare(BoolValue other) const { return m_value - other.m_value; }
        bool m_value;
    };

    /// Floating point value.
    struct HK_EXPORT_COMMON FloatValue
    {
        HK_DECLARE_PLACEMENT_ALLOCATOR();

        FloatValue(double d = 0.0) : m_value(d) {}
        operator double() const { return m_value; }
        int compare(FloatValue other) const { return (other.m_value < m_value) - (m_value < other.m_value); }
        double m_value;
    };

    /// Null terminated string.
    struct HK_EXPORT_COMMON StringValue
    {
        HK_DECLARE_PLACEMENT_ALLOCATOR();

        StringValue(_In_opt_z_ const char* s = nullptr) : m_value(s) {}
        _Ret_opt_z_ operator const char*() const { return m_value; }
        const char* m_value; 
        bool operator==(StringValue other) const;
        bool operator==(_In_opt_z_ const char* other) const;
        bool operator!=(StringValue other) const;
        bool operator!=(_In_opt_z_ const char* other) const;
        int compare(StringValue other) const;
    };

    /// Signed or unsigned integer
    
    struct HK_EXPORT_COMMON IntValue
    {
    private:

        template<typename INT_TY>
        HK_INLINE void set2(INT_TY t, hkTrait::TrueType issigned)
        {
            m_value = t >= 0 ? t : hkUint64(-(t + 1)) + 1;
            m_negative = t < 0;
        }
        template<typename INT_TY>
        HK_INLINE void set2(INT_TY t, hkTrait::FalseType issigned)
        {
            m_value = t;
            m_negative = false;
        }

        template<typename INT_TY>
        INT_TY convertTo(hkTrait::TrueType isSigned) const
        {
            HK_ON_DEBUG(hkUint64 maxVal = hkUint64(hkTrait::NumericLimits<typename hkTrait::MatchingIntType<sizeof(INT_TY), true>::IntType >::maxValue()) + (m_negative ? 1 : 0));
            HK_ASSERT_NO_MSG(0x7c7f31cb, m_value <= maxVal); // max signed value
            INT_TY t = static_cast<INT_TY>(m_value);
            if (m_negative) t = INT_TY(-t);
            return t;
        }

        template<typename INT_TY>
        INT_TY convertTo(hkTrait::FalseType isSigned) const
        {
            HK_ASSERT_NO_MSG(0x7fc78b28, m_negative == false); // if negative, INT_TY must be signed
            HK_ASSERT_NO_MSG(0x6d831b60, static_cast<INT_TY>(m_value) == m_value); // value fits?
            return static_cast<INT_TY>(m_value);
        }

        template<typename INT_TY>
        bool get(_Out_ INT_TY* out, hkTrait::TrueType isSigned) const
        {
            if (m_value <= hkUint64(hkTrait::NumericLimits<typename hkTrait::MatchingIntType<sizeof(INT_TY), true>::IntType >::maxValue()) + (m_negative ? 1 : 0)) // max signed value
            {
                INT_TY t = static_cast<INT_TY>(m_value);
                if (m_negative) t = INT_TY(-t);
                *out = t;
                return true;
            }
            return false;
        }

        template<typename INT_TY>
        bool get(_Out_ INT_TY* out, hkTrait::FalseType isSigned) const
        {
            if (m_negative == false && static_cast<INT_TY>(m_value) == m_value) // value fits?
            {
                *out = static_cast<INT_TY>(m_value);
                return true;
            }
            return false;
        }

        template<typename INT_TY>
        inline bool equals(INT_TY t, hkTrait::TrueType isSigned) const
        {
            if (t < 0) return m_negative && (-t == m_value);
            else return m_negative == false && (t == m_value);
        }

        template<typename INT_TY>
        inline bool equals(INT_TY t, hkTrait::FalseType isSigned) const
        {
            return m_negative == false && (t == m_value);
        }

    public:

        HK_DECLARE_PLACEMENT_ALLOCATOR();

        IntValue() : m_value(0), m_negative(false) {}

        IntValue(hkUint64 _absValue, bool isNeg) : m_value(_absValue), m_negative(isNeg) {}

        bool operator==(IntValue i) const { return i.m_value == m_value && i.m_negative == m_negative; }
        bool operator<(IntValue i) const { return m_negative != i.m_negative ? m_negative : (m_negative ? m_value > i.m_value : m_value < i.m_value); }
        int compare(IntValue i) const
        {
            const int signDiff = m_negative - i.m_negative;
            return signDiff ? signDiff : ((1 - 2 * m_negative) * ((i.m_value < m_value) - (m_value < i.m_value)));
        }

        template<typename INT_TY>
        IntValue(INT_TY t) { set(t); }

        template<typename INT_TY>
        void set(INT_TY  t)
        {
            set2(t, typename hkTrait::IsSigned<INT_TY>::Type());
        }

        void set(hkUint64 _absValue, bool isNeg)
        {
            m_value = _absValue;
            m_negative = isNeg;
        }

        /// Returns false if the value cannot be represented
        template<typename INT_TY>
        bool get(_Out_ INT_TY* t) const
        {
            return get(t, typename hkTrait::IsSigned<INT_TY>::Type());
        }

        /// Asserts on failure
        template<typename INT_TY>
        INT_TY convertTo() const
        {
            return convertTo<INT_TY>(typename hkTrait::IsSigned<INT_TY>::Type());
        }

        template<typename INT_TY>
        bool equals(INT_TY t) const
        {
            return equals(t, typename hkTrait::IsSigned<INT_TY>::Type());
        }

        hkUint64 absValue() const { return m_value; }
        bool isNegative() const { return m_negative; }
        bool isZero() const { return m_value == 0; }

        friend IntValue operator+(const IntValue& lhs, const IntValue& rhs);

    private:

        hkUint64 m_value;
        bool m_negative;
    };

    struct ContainerValue
    {
        ContainerValue() : m_numElem(0) {}
        ContainerValue(int nelem, QualType etype)
            : m_elemType(etype), m_numElem(nelem) { }

        int getCount() const { return m_numElem; }
        QualType getSubType() const { return m_elemType; }

    private:
        QualType m_elemType;
        int m_numElem;
    };

    namespace Detail
    {
        enum ImplFlags
        {
            IMPL_NATIVE = 0x0,
            IMPL_DYNAMIC = 0x1
        };
        static const ImplFlags IMPL_MASK = IMPL_DYNAMIC;

        struct ImplPtr : private hkPtrAndInt<Detail::Impl, ImplFlags, IMPL_MASK>
        {
            typedef hkPtrAndInt<Detail::Impl, ImplFlags, IMPL_MASK> Base;
            ImplPtr();
            ImplPtr(_In_ const Detail::Impl* impl, bool isDynamic = false);
            ImplPtr(const ImplPtr& rhs);
            void operator=(const ImplPtr& rhs);
            void reset() { Base::reset(); }

#ifdef HK_MOVE_SEMANTICS
            ImplPtr(ImplPtr&& rhs);
            void operator=(ImplPtr&& rhs);
#endif

            ~ImplPtr();

            HK_ALWAYS_INLINE _Ret_maybenull_ Detail::Impl* val() const { return getPtr(); }
            HK_ALWAYS_INLINE _Ret_maybenull_ Detail::Impl* operator->() const { return getPtr(); }
            HK_ALWAYS_INLINE _Ret_maybenull_ Detail::Impl* operator*() const { return getPtr(); }
            HK_ALWAYS_INLINE _Ret_maybenull_ operator Detail::Impl*() const { return getPtr(); }

            HK_ALWAYS_INLINE bool isDynamic() const { return getInt() & IMPL_DYNAMIC; }

        protected:
            void addRefIfDynamic() const;
            void removeRefIfDynamic() const;
        };
    }

    struct ArrayValue
    {
        template<typename T> ArrayValue(_In_ T* elem, int nelem)
            : m_elemPtr(elem),
            m_elemType(hkReflect::getType<T>()),
            m_numElem(nelem),
            m_elemSize(hkReflect::getType<T>()->getSizeOf()),   // not leveraging 'm_elemType' here because using 'm_elemType' would require the full 'hkReflect::Type' declaration,
            m_elemImpl(hkReflect::getType<T>()->getImpl()) {}   // so instead we use 'hkReflect::getType<T>()' which is dependant on T and so isn't resolved until the template is instantiated

        HK_EXPORT_COMMON ArrayValue(_In_ const void* elem, int nelem, QualType etype, _In_ const Detail::Impl* elemImpl = nullptr, bool isDynamic = false);

        HK_ALWAYS_INLINE ArrayValue()
            : m_elemPtr(nullptr), m_elemType(nullptr), m_numElem(0), m_elemSize(0), m_elemImpl(nullptr) {}

        HK_EXPORT_COMMON Var operator[](int i) const;
        HK_ALWAYS_INLINE int getCount() const { return m_numElem; }
        HK_ALWAYS_INLINE QualType getSubType() const { return m_elemType; }
        ArrayValue slice(int startIdx, int endIdx);

        template <typename T>
        hkArrayView<T> isArrayOf() const;

        HK_ALWAYS_INLINE _Ret_maybenull_ void* getAddress() const { return m_elemPtr; }
        HK_ALWAYS_INLINE int getStride() const { return m_elemSize; }
        HK_ALWAYS_INLINE _Ret_maybenull_ const Detail::Impl* getElemImpl() const { return m_elemImpl; }
        _Notvalid_ void* getEndAddress() const;

    private:
        void* m_elemPtr;
        QualType m_elemType;
        int m_numElem;
        int m_elemSize;
        Detail::ImplPtr m_elemImpl;
    };

    
    enum CallableTypeBits
    {
        CALLABLE_METHODS = 0x01,
        CALLABLE_CONSTRUCTORS = 0x02,
        CALLABLE_STATIC_FUNCTIONS = 0x04,

        CALLABLE_ALL = CALLABLE_METHODS | CALLABLE_CONSTRUCTORS | CALLABLE_STATIC_FUNCTIONS
    };
    typedef hkFlags<CallableTypeBits, hkUint8> CallableTypes;
}

// hkReflectValue.inl is included in hkReflect.h (needs definition of Var).

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
