// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

class hkStringBuf;
class hkMemoryAllocator;
class hkStringView;
template<typename T> class hkArrayView;
struct hkContainerHeapAllocator;
template<typename T, typename Allocator> class hkArray;

namespace hkLog
{
    class RegisteredOrigin;
}

namespace hkReflect
{
    /// Base class for all types. Provides access to all the generic information in a reflected type.
    /// A Type object can be a "concrete" type or a "decorator".
    /// A "concrete" type usually corresponds to a C++ type. Every concrete type has a Kind (Bool, Integer, Pointer,
    /// etc.). The Kind identifies what data is contained and what operations are supported by instances of the Type.
    /// The parent of a concrete type (it it has one) is its C++ base class.
    /// A "decorator" is just a wrapper for the parent Type containing additional information such as a typedef name or
    /// attributes. Decorators inherit all the data (including Kind) from the decorated Type. Both decorator and
    /// decorated correspond to the same C++ type.
    /// Type has a subclass for every Kind. You can dynamically cast a generic Type to a specific kind using the
    /// asBool, asInteger, asPointer, etc. methods.
    /// Type objects are variable-sized so they are generally referenced and passed around by pointer.
    class HK_EXPORT_COMMON Type
    {
    public:

        HK_DECLARE_CLASS(Type, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        HK_RECORD_ATTR(hk::ExactTypeFunction(&hkReflect::exactObj));
        typedef void ParentType;

        friend struct TypeDetail;

        enum FlagBits
        {
            TYPE_NOT_SERIALIZABLE = 1 << 0, ///< This type is not serializable
            TYPE_ABSTRACT = 1 << 1, ///< This type cannot be directly instantiated
            TYPE_DECORATOR_HACK = 1 << 2, 
            TYPE_PROPERTY = 1 << 3, ///< This type is a property (uses get/set methods)
            TYPE_INTERFACE = 1 << 4, ///< This type may be used as an interface (it is a record with no fields)
            TYPE_DYNAMIC = 1 << 5, ///< Type has been dynamically generated, it may not be registered
            TYPE_POINTER_WEAK = 1 << 6, ///< The pointer is weak.
            TYPE_OWN_VTABLE = 1 << 7, ///< This type has its own vtable. I.e. it introduces a pointer sized value into the type layout.
            TYPE_IS_ATTRIBUTE = 1 << 8, ///< This type defines an attribute
            TYPE_NOT_RETARGETABLE = 1 << 9, ///< This type layout cannot be computed (when targeting other platforms).
            TYPE_SYNTHETIC = 1 << 10, ///< Dynamic type which has a SyntheticUtil
            TYPE_FOREIGN = 1 << 11, ///< Dynamic type which has a ForeignUtil
            TYPE_TEMPORARY = 1 << 12, ///< Type only exists temporarily and should not be cached
        };
        typedef hkFlags<FlagBits, hkUint32> Flags;

        /// Get the size of the described type
        int getSizeOf() const;
        /// Get the alignment of the described type
        int getAlignOf() const;

        /// Returns the default value of this type.
        hkReflect::Var getDefault() const; 

            /// What kind of thing does this type describe.
        Kind getKind() const;

        /// Returns true if this type inherits from or implements the provided interface.
        bool implements(_In_ const hkReflect::Type* interfaceType) const;

        /// Returns all implemented interfaces.
        void getImplemented(hkArray<const hkReflect::Type*, hkContainerHeapAllocator>& typesOut) const;

        // Returns the most-derived type of this type and the given type.
        // Note that this will only consider the direct parent hierarchy (not considering multi-inheritance), see hkReflect::Type::getParent().
        static _Ret_maybenull_ const hkReflect::Type* getCommonAncestor(_In_opt_ const hkReflect::Type* type0, _In_ const hkReflect::Type* type1);

        /// Returns the name of this type, not including template parameters.
        _Ret_z_ const char* getName() const;

        /// Prints the full name of this type, including the arguments if the type is a template.
        /// nameCallback: appends the string name for a type to the buffer when called.
        /// If 'nameCallback' is not provided, getFullName will use something standard like hkReflect::Type::getName() to determine the string name for a type
        /// fullNameCallback: appends the full string name for a type (including template args) to the buffer when called.
        /// If 'fullNameCallback' is not provided, getFullName will use something standard like hkReflect::Type::getFullName() to determine the full string name for a type
        typedef void(*AppendNameForTypeCallback)(const hkReflect::Type*, hkStringBuf& buf);
        typedef void(*AppendFullNameForTypeCallback)(const hkReflect::Type*, hkStringBuf& buf);
        const char* getFullName(hkStringBuf& nameBuf, AppendNameForTypeCallback nameCallback = HK_NULL, AppendFullNameForTypeCallback fullNameCallback = HK_NULL) const;

        /// Get the name of this type.
        TypeName getFullName() const;

        /// Returns -1,0,+1 if typeA is lexicographically less,equal or greater than typeB. This includes template parameters.
        static int HK_CALL compareName(_In_ const hkReflect::Type* typeA, _In_ const hkReflect::Type* typeB);

        /// Returns the version of this type. Will skip aliases.
        int getVersion() const;

        /// For fields, the enclosing declaration.
        _Ret_maybenull_ const Type* getDeclContext() const;

        /// Find the named decl. See also DeclIter<T>.
        Decl findDecl(_In_opt_z_ const char* name, bool parents = true) const;

        _Ret_maybenull_ const class Callable* findCallable(_In_z_ const char* name, CallableTypes types = CALLABLE_ALL) const;

        /// Find an attribute of the given type t attached to this type or to a type in its parent chain.
        Var findAttribute(_In_ const Type* t) const;

        /// Find an attribute of the given template type T attached to this type or to a type in its parent chain.
        template<typename T>
        _Ret_maybenull_ const T* findAttribute() const { return static_cast<const T*>(findAttribute(hkReflect::getType<T>()).getAddress()); }

        /// Shortcut to extract an attributes value or use a default one if not present.
        template<typename ValueAttribute>
        typename ValueAttribute::AttributeType getValueAttribute(typename ValueAttribute::AttributeType ifNotFound) const;

        /// Get the types parent.
        /// See the user guide for subtleties.
        _Ret_maybenull_ const Type* getParent() const;

        /// Get the template parameters or null if not templated.
        _Ret_maybenull_ const Template* getTemplate() const;

        /// Returns the "world" of this type. This value is a simple marker to prevent
        /// accidentally mixing native types with temporary types created during serialization, for example.
        /// Native types have a type world of HK_NULL.
        _Ret_maybenull_ const void* getTypeWorld() const;

        /// Does this type support downcasting
        bool isPolymorphic() const;

        _Ret_maybenull_ const VoidType* asVoid() const;
        _Ret_maybenull_ VoidType* asVoid();
        _Ret_maybenull_ const StringType* asString() const;
        _Ret_maybenull_ StringType* asString();
        _Ret_maybenull_ const PointerType* asPointer() const;
        _Ret_maybenull_ PointerType* asPointer();
        _Ret_maybenull_ const CompoundType* asCompound() const;
        _Ret_maybenull_ CompoundType* asCompound();
        _Ret_maybenull_ const ArrayType* asArray() const;
        _Ret_maybenull_ ArrayType* asArray();
        _Ret_maybenull_ const RecordType* asRecord() const;
        _Ret_maybenull_ RecordType* asRecord();
        _Ret_maybenull_ const ValueType* asValue() const;
        _Ret_maybenull_ ValueType* asValue();
        _Ret_maybenull_ const BoolType* asBool() const;
        _Ret_maybenull_ BoolType* asBool();
        _Ret_maybenull_ const IntType* asInteger() const;
        _Ret_maybenull_ IntType* asInteger();
        _Ret_maybenull_ const FloatType* asFloat() const;
        _Ret_maybenull_ FloatType* asFloat();
        _Ret_maybenull_ const OpaqueType* asOpaque() const;
        _Ret_maybenull_ OpaqueType* asOpaque();

        /// Does this type refer to a field.
        FieldDecl isField() const;
        /// Has this type been created dynamically
        bool isDynamicType() const;
        /// Does this class add a new vtable to the layout
        bool hasOwnVtable() const;
        /// Is this type abstract.
        bool isAbstract() const;
        /// Is this type a typedef?
        /// Use getParent() to retrieve the backing type of the typedef.
        bool isTypedef() const;

        /// Equality test.
        /// Use equals instead of raw pointer comparison because template instantiation and use of DLLs
        /// can lead to multiple copies of the same type. Note, this uses the "inheritanceinfo" values
        /// internally which means that the underlying types are compared, ignoring whether either of the
        /// types is a typedef or field for instance.
        bool equals(_In_ const hkReflect::Type* t) const;
        /// Forwards to equals(Type*)
        template<typename T>
        bool equals() const { return equals(getType<T>()); }

        /// Tests if this type is equal to, or derives from "base".
        bool extendsOrEquals(_In_ const hkReflect::Type* base) const;
        /// Forwards to extendsOrEquals(Type*)
        template<typename T>
        bool extendsOrEquals() const { return extendsOrEquals(hkReflect::getType<T>()); }

        /// Returns true if an instance of this type can be created through reflection.
        bool canBeInstantiated() const;
        /// Create a new instance of this type.
        hkReflect::Var newInstance() const;

        inline bool isDecorator() const { return (m_optional & Opt::FORMAT) == 0; }
        inline bool isProperty() const { return getTypeFlags().anyIsSet(TYPE_PROPERTY) != 0; }
        inline bool isSerializable() const { return getTypeFlags().noneIsSet(TYPE_NOT_SERIALIZABLE); }
        inline bool isRetargetable() const { return getTypeFlags().noneIsSet(TYPE_NOT_RETARGETABLE); }
        inline bool isInterface() const { return getTypeFlags().anyIsSet(TYPE_INTERFACE); }
        inline bool isAttribute() const { return getTypeFlags().anyIsSet(TYPE_IS_ATTRIBUTE); }
        inline bool isSynthetic() const { return getTypeFlags().anyIsSet(TYPE_SYNTHETIC); }
        inline bool isForeign() const { return getTypeFlags().anyIsSet(TYPE_FOREIGN); }
        inline bool isTemporary() const { return getTypeFlags().anyIsSet(TYPE_TEMPORARY); }
        bool hasCustomSetter() const;
        bool hasCallbacks() const;

        /// Returns the format of the Type. The format contains Kind-specific details about the Type (e.g.
        /// size and signedness for integers, precision for floats, etc.).
        hkReflect::Format::Value getFormat() const { return hkReflect::Format::Value::create(hkLosslessCast<hkUint32>(valueDecorator(Opt::FORMAT))); }

        /// Access the fields of this type.
        hkArrayView<const FieldDecl> getFields() const;

        /// Access the non-property fields of this type.
        hkArrayView<const DataFieldDecl> getDataFields() const;

        /// Get the number of field declared in this record (not including ancestor record fields).
        int getNumFields() const;

        /// Get the number of fields that are simple data (and not properties)
        int getNumDataFields() const;
        /// Get the number of fields that are properties
        int getNumPropertyFields() const;
        /// Get the index'th field declared in this record (not including ancestor record fields).
        FieldDecl getField(int index) const;

        /// Find a field with the given name or null if not found.
        /// Use methods on the return value to find the index, offset and containing type.
        /// If the field is not found and includeParents is true, search continues in the parent.
        FieldDecl findField(_In_opt_z_ const char* name, bool includeParents) const;

        FieldDecl findFieldNoCase(_In_opt_z_ const char* name, bool includeParents) const;

        /// Get the virtual implementation underlying this type
        _Ret_maybenull_ const Detail::Impl* getImpl() const;

    protected:

        hkUlong m_optional;
        const hkReflect::Type* m_parent;

        Type() {}

        Flags getTypeFlags() const;

        // Internal: when searching for components of a type, there are generally three scopes to consider.
        // local - only search this type, not parents. This is fairly rare, and mostly used by
        // internal code which really needs to care about the internal structure.
        // global - search this type, continuing with parents until found. This is used for looking for
        // e.g. the type Impl or the types downcast function.
        // decorator - the most common type of search. The search stops when the opt is found or after a type
        // with an Opt::FORMAT is searched. Types without Opt::FORMAT are named decorators and can only change and add
        // some superficial properties. Types with an Opt::FORMAT differ in a significant way to their parents. For instance
        // for records, it indicates a derived class and we'd expect a different name, size, constructor etc from its parent.

        HK_ALWAYS_INLINE hkUlong derefOrZero(_In_opt_ hkUlong* p) const { return p ? *p : 0; }
        // Check all parents
        _Ret_maybenull_ hkUlong* addressGlobal(Optional opt) const;
        inline hkUlong valueGlobal(Optional opt) const { return derefOrZero(addressGlobal(opt)); }
        template<typename T> T valueGlobal(Optional opt) const { return hkBitCast<T>(valueGlobal(opt)); }
        // Stop search at first non-decorator
        _Ret_maybenull_ hkUlong* addressDecorator(Optional opt) const;
        inline hkUlong valueDecorator(Optional opt) const { return derefOrZero(addressDecorator(opt)); }
        template<typename T> T valueDecorator(Optional opt) const { return hkBitCast<T>(valueDecorator(opt)); }
        // Search only this type
        hkUlong hasLocal(Optional opt) const { return m_optional & opt; }
        _Ret_notnull_ hkUlong* addressLocalUnchecked(Optional opt) const;
        inline _Ret_maybenull_ hkUlong* addressLocal(Optional opt) const { return (m_optional&opt) ? addressLocalUnchecked(opt) : HK_NULL; }
        inline hkUlong valueLocal(Optional opt) const { return derefOrZero(addressLocal(opt)); }
        template<typename T> T valueLocal(Optional opt) const { return hkBitCast<T>(valueLocal(opt)); }
        // Like addressLocal but hard error if not found
        _Ret_notnull_ hkUlong* accessLocal(Optional opt) const;

        inline _Ret_notnull_ hkUlong* optionalsArray() const
        {
            return reinterpret_cast<hkUlong*>(const_cast<Type*>(this) + 1);
        }
    };
    HK_COMPILE_TIME_ASSERT(sizeof(Type) % sizeof(hkUlong) == 0);

    /// Base class for value types (bools, numbers, strings).
    class HK_EXPORT_COMMON ValueType : public Type
    {
    public:

        HK_DECLARE_CLASS(ValueType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef Type ParentType;

        _Ret_maybenull_ const Detail::ValueImpl* getImpl() const { return static_cast<const Detail::ValueImpl*>(Type::getImpl()); }

    protected:
        ValueType() {}
    };

    /// Base class for compound types.
    /// A compound type is whatever supports iteration on its contents (pointers, records, arrays).
    class HK_EXPORT_COMMON CompoundType : public Type
    {
    public:

        HK_DECLARE_CLASS(CompoundType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef Type ParentType;

        _Ret_maybenull_ const Detail::CompoundImpl* getImpl() const { return static_cast<const Detail::CompoundImpl*>(Type::getImpl()); }

    protected:
        CompoundType() {}
    };

    /// Boolean types.
    class HK_EXPORT_COMMON BoolType : public ValueType
    {
    public:
        HK_DECLARE_CLASS(BoolType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef ValueType ParentType;

        _Ret_maybenull_ const Detail::BoolImpl* getImpl() const { return static_cast<const Detail::BoolImpl*>(Type::getImpl()); }

    protected:
        BoolType() {}
    };

    /// Integer types.
    class HK_EXPORT_COMMON IntType : public ValueType
    {
    public:
        HK_DECLARE_CLASS(IntType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef ValueType ParentType;

        _Ret_maybenull_ const Detail::IntImpl* getImpl() const { return static_cast<const Detail::IntImpl*>(Type::getImpl()); }

        /// See also getFormat and Format::OfInt
        bool isSigned() const;

    protected:
        IntType() {}
    };

    /// Floating point types.
    class HK_EXPORT_COMMON FloatType : public ValueType
    {
    public:
        HK_DECLARE_CLASS(FloatType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef ValueType ParentType;

        _Ret_maybenull_ const Detail::FloatImpl* getImpl() const { return static_cast<const Detail::FloatImpl*>(Type::getImpl()); }

    protected:
        FloatType() {}
    };

    /// Null terminated octets.
    class HK_EXPORT_COMMON StringType : public ValueType
    {
    public:

        HK_DECLARE_CLASS(StringType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef ValueType ParentType;

        _Ret_maybenull_ const Detail::StringImpl* getImpl() const { return static_cast<const Detail::StringImpl*>(Type::getImpl()); }
    protected:
        StringType() {}
    };

    /// Pointer types.
    class HK_EXPORT_COMMON PointerType : public CompoundType
    {
    public:
        HK_DECLARE_CLASS(PointerType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef CompoundType ParentType;

        /// Get the static type of the pointee or null if there isn't one.
        /// See also PointerVar.getSubType which returns the dynamic pointee type.
        QualType getSubType() const { return valueDecorator<QualType>(Opt::SUBTYPE); }

        void setSubType(QualType n) { *accessLocal(Opt::SUBTYPE) = hkBitCast<hkUlong>(n); }
        _Ret_maybenull_ const Detail::PointerImpl* getImpl() const { return static_cast<const Detail::PointerImpl*>(Type::getImpl()); }
        bool isWeak() const { return getTypeFlags().anyIsSet(TYPE_POINTER_WEAK); }
        bool canBeNull() const { return getImpl()->canBeNull(); }
    protected:
        PointerType() {}
    };

    /// Type for generic containers. Do not use directly, non-array containers are not supported currently.
    class HK_EXPORT_COMMON ContainerType : public CompoundType
    {
    public:
        HK_DECLARE_CLASS(ContainerType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef CompoundType ParentType;

        /// Get the static type of the contents or null if there isn't one.
        /// See also ContainerVar.getSubType which returns the dynamic contents type.
        QualType getSubType() const { return valueDecorator<QualType>(Opt::SUBTYPE); }
        void setSubType(QualType eltType) { *accessLocal(Opt::SUBTYPE) = hkBitCast<hkUlong>(eltType); }
        _Ret_maybenull_ const Detail::ContainerImpl* getImpl() const { return static_cast<const Detail::ContainerImpl*>(Type::getImpl()); }

    protected:
        ContainerType() {}
    };

    /// Abstract array type - this covers a wide set of implementations, from variable arrays
    /// to relarrays to fixed-size carrays.
    class HK_EXPORT_COMMON ArrayType : public ContainerType
    {
    public:
        HK_DECLARE_CLASS(ArrayType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef ContainerType ParentType;
        /// Dereference the array to get access to the flat storage.
        int getFixedCount() const { return int(valueDecorator(Opt::FORMAT) >> 8); }
        _Ret_maybenull_ const Detail::ArrayImpl* getImpl() const { return static_cast<const Detail::ArrayImpl*>(Type::getImpl()); }

    protected:
        ArrayType() {}
    };
    /// Record types, correspond to C++ structs/classes.
    class HK_EXPORT_COMMON RecordType : public CompoundType
    {
    public:
        HK_DECLARE_CLASS(RecordType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        typedef CompoundType ParentType;

        /// Return the parent record to this type, or null if we are a toplevel record.
        _Ret_maybenull_ const RecordType* getParentRecord() const;

        /// Get the record impl.
        _Ret_maybenull_ const Detail::RecordImpl* getImpl() const { return static_cast<const Detail::RecordImpl*>(Type::getImpl()); }

        friend class hkReflect::TypeBuilder;

    protected:

        RecordType() {}
        RecordType(const RecordType&); // not implemented
    };

    /// Kind of "void" type.
    class HK_EXPORT_COMMON VoidType : public Type
    {
    public:
        HK_DECLARE_CLASS(VoidType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        HK_RECORD_ATTR(hk::IncludeInMgd(false));
        typedef Type ParentType;

        _Ret_maybenull_ const Detail::VoidImpl* getImpl() const { return static_cast<const Detail::VoidImpl*>(Type::getImpl()); }
    };

    /// Types opaque to reflection.
    class HK_EXPORT_COMMON OpaqueType : public Type
    {
    public:
        HK_DECLARE_CLASS(OpaqueType, NoNew, Reflect);
        HK_REFLECT_AS_OPAQUE();
        HK_RECORD_ATTR(hk::IncludeInMgd(false));
        typedef Type ParentType;

        _Ret_maybenull_ const Detail::OpaqueImpl* getImpl() const { return static_cast<const Detail::OpaqueImpl*>(Type::getImpl()); }
    };

    struct InvalidTag;

    ///
    template <hkReflect::Kind kind> struct TypeFromKind { typedef InvalidTag Type; };
    template <hkReflect::Kind kind> struct VarFromKind { typedef InvalidTag Type; };
    template <typename Type> struct VarFromType;
    template <typename Type> struct ImplFromType;
    template <typename Type> struct ValueFromType;

    ///
    template <typename T> struct KindFromType;

#define HK_LINK_KIND_TO_TYPE(KIND, TYPE) \
        template<> struct TypeFromKind<KIND> { typedef TYPE##Type Type; }; \
        template<> struct KindFromType<TYPE##Type> {static const Kind kind = KIND; }; \
        template<> struct VarFromKind<KIND> { typedef TYPE##Var Type; }; \
        template<> struct VarFromType<TYPE##Type> { typedef TYPE##Var Type; }; \
        template<> struct ImplFromType<TYPE##Type> { typedef Detail::TYPE##Impl Type; }

    HK_LINK_KIND_TO_TYPE(KIND_ARRAY, Array);
    HK_LINK_KIND_TO_TYPE(KIND_BOOL, Bool);
    HK_LINK_KIND_TO_TYPE(KIND_STRING, String);
    HK_LINK_KIND_TO_TYPE(KIND_FLOAT, Float);
    HK_LINK_KIND_TO_TYPE(KIND_OPAQUE, Opaque);
    HK_LINK_KIND_TO_TYPE(KIND_INT, Int);
    HK_LINK_KIND_TO_TYPE(KIND_POINTER, Pointer);
    HK_LINK_KIND_TO_TYPE(KIND_RECORD, Record);
    //HK_LINK_KIND_TO_TYPE(KIND_CONTAINER, Container);
    HK_LINK_KIND_TO_TYPE(KIND_VOID, Void);

    template<> struct ImplFromType < hkReflect::Type > { typedef Detail::Impl Type; };
    template<> struct ImplFromType < hkReflect::CompoundType > { typedef Detail::CompoundImpl Type; };
    template<> struct ImplFromType < hkReflect::ValueType > { typedef Detail::ValueImpl Type; };

    template<> struct VarFromType < hkReflect::Type > { typedef hkReflect::Var Type; };
    template<> struct VarFromType < hkReflect::CompoundType > { typedef CompoundVar Type; };
    template<> struct VarFromType < hkReflect::ValueType > { typedef ValueVar Type; };

    template<> struct ValueFromType < BoolType > { typedef BoolValue Type; };
    template<> struct ValueFromType < IntType > { typedef IntValue Type; };
    template<> struct ValueFromType < StringType > { typedef StringValue Type; };
    template<> struct ValueFromType < FloatType > { typedef FloatValue Type; };
    template<> struct ValueFromType < ArrayType > { typedef ArrayValue Type; };
    template<> struct ValueFromType < PointerType > { typedef Var Type; };
    template<> struct ValueFromType < RecordType > { typedef RecordVar Type; };

#undef HK_LINK_KIND_TO_TYPE

    /// Get the type which reflects the type corresponding to the kind (i.e. KIND_INT -> hkReflect::getType<IntegerType>()).
    HK_EXPORT_COMMON _Ret_maybenull_ const Type* HK_CALL typeFromKind(Kind kind);

    /// Represents a types "full name" which including the class name and template parameters.
    class HK_EXPORT_COMMON TypeName
    {
    public:
        HK_DECLARE_CLASS(TypeName, Reflect);
        TypeName() : m_name(nullptr), m_template(nullptr) { }
        explicit TypeName(_In_ const Type* t);
        TypeName(_In_opt_z_ const char* n, const Template* t) : m_name(n), m_template(t) {}

        _Ret_opt_z_ const char* getName() const { return m_name; }
        _Ret_maybenull_ const Template* getTemplate() const { return m_template; }

        /// Returns -1,0,+1 if nameA is lexicographically less,equal or greater than nameB.
        static int HK_CALL compare(hkReflect::TypeName nameA, hkReflect::TypeName nameB);

        bool operator!=(TypeName other) const { return compare(*this, other) != 0; }
        bool operator==(TypeName other) const { return compare(*this, other) == 0; }
        bool operator< (TypeName other) const { return compare(*this, other) < 0; }
        bool operator!=(const char* n) const { return compare(*this, TypeName(n, nullptr)) != 0; }
        bool operator==(const char* n) const { return compare(*this, TypeName(n, nullptr)) == 0; }
        bool operator< (const char* n) const { return compare(*this, TypeName(n, nullptr)) < 0; }
        /// Convert to a string
        _Ret_z_ const char* toString(hkStringBuf& sb) const;
        // Reflection based tostring
        HK_RECORD_ATTR(hk::ToString(&hkReflect::TypeName::toString));
        static void HK_CALL toString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra);

        public HK_ATTR(hk::Reflect(false)) :
            HK_INLINE void* HK_CALL operator new(hk_size_t, void* p) { return p; }
        HK_INLINE void HK_CALL operator delete(void*, void*) {}
    private:
        const char* m_name;
        const Template* m_template HK_ATTR(hk::OpaqueType);
    };

    /// Declaration in a type (field, property).
    class HK_EXPORT_COMMON Decl
    {
    public:

        HK_DECLARE_CLASS(Decl, New, Reflect);
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck, hk::ToString(&toString));

        Decl() {}
        Decl(QualType t);
        template <typename T>
        Decl(QualifiedType<T> t);

        void init(QualType t);

        enum DeclFlagBits
        {
            
            DECL_NONE = 0,
            DECL_NOT_SERIALIZABLE = 0x01, // this is not serializable
            DECL_PROTECTED = 0x02, ///< Declared as protected
            DECL_PRIVATE = 0x04, ///< Declared as private
            DECL_HIDDEN = 0x08, ///< Normally skipped when iterating
            DECL_PROPERTY_FIELD = 0x10, ///< This field is a property (get/set)
            DECL_DATA_FIELD = 0x20, ///< This field has an address
            DECL_CUSTOM_SETTER = 0x40, ///< This field has a custom setter
        };

        // Keep separate not to break VS enum debug visualizer.
        enum
        {
            /// Either kind of field.
            DECL_FIELD = DECL_PROPERTY_FIELD | DECL_DATA_FIELD
        };

        struct DeclFlags : public hkFlags<DeclFlagBits, int>
        {
            HK_DECLARE_CLASS(DeclFlags, New, Reflect);
            HK_REFLECT_AS_TYPEDEF(hkFlags<hkReflect::Decl::DeclFlagBits, int>);

            DeclFlags(int f = DECL_NONE) : hkFlags<DeclFlagBits, int>(f) {}

            bool isField() const { return anyIsSet(DECL_FIELD); }
            bool isDataField() const { return anyIsSet(DECL_DATA_FIELD); }
            bool isPropertyField() const { return anyIsSet(DECL_PROPERTY_FIELD); }

            
            bool isSerializable() const { return anyIsSet(DECL_NOT_SERIALIZABLE) ? false : true; }
            bool isPublic() const { return anyIsSet(DECL_PROTECTED | DECL_PRIVATE) ? false : true; }
            bool isHidden() const { return anyIsSet(DECL_HIDDEN); }
        };

        /// Returns true if this field is valid.
        operator hkBoolOperator::BoolType() const { return hkBoolOperator::cast(m_type.isNull() == false); }

        bool operator==(const Decl& rhs) const;
        HK_ALWAYS_INLINE bool operator!=(const Decl& rhs) const { return !(*this == rhs); }

        FieldDecl asField() const;
        DataFieldDecl asDataField() const;
        PropertyFieldDecl asPropertyField() const;

        bool isSerializable() const { return getFlags().anyIsSet(DECL_NOT_SERIALIZABLE) ? false : true; }
        bool isPublic() const { return getFlags().anyIsSet(DECL_PROTECTED | DECL_PRIVATE) ? false : true; }
        bool isHidden() const { return getFlags().anyIsSet(DECL_HIDDEN); }
        bool hasCustomSetter() const { return getFlags().anyIsSet(DECL_CUSTOM_SETTER); }
        bool hasCallbacks() const { return getFlags().anyIsSet(DECL_PROPERTY_FIELD | DECL_CUSTOM_SETTER); }

        /// Returns the name of the decl
        _Ret_opt_z_ const char* getName() const;

        /// Returns the type of the decl
        hkReflect::QualType getType() const;

        /// Return the type where this field is declared. e.g. the containing RecordType
        _Ret_maybenull_ const Type* getDeclContext() const;

        /// Invalidate the field.
        void clear() { m_type.clear(); }

        DeclFlags getFlags() const;

        static void toString(const hkReflect::Var& var, hkStringBuf& buf, const hkStringView& extra);

    protected:

        QualType m_type;
    };

    /// Field declaration. Includes data fields and properties.
    class HK_EXPORT_COMMON FieldDecl : public Decl
    {
    public:

        HK_DECLARE_CLASS(FieldDecl, New, Reflect);
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

        enum { MASK = Decl::DECL_FIELD };

        FieldDecl() {}
        FieldDecl(QualType t);
        template <typename T>
        FieldDecl(QualifiedType<T> t);
        FieldDecl(const FieldDecl& r) : Decl(r.m_type) {}

        void init(QualType t);

        /// Returns the offset of the field. Note: undefined if this is a property.
        int getOffset() const;

        bool operator==(const FieldDecl& rhs) const;
        HK_ALWAYS_INLINE bool operator!=(const FieldDecl& rhs) const { return !(*this == rhs); }
    };

    /// Declaration corresponding to a C++ field.
    class HK_EXPORT_COMMON DataFieldDecl : public FieldDecl
    {
    public:
        HK_DECLARE_CLASS(DataFieldDecl, New, Reflect);
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

        enum { MASK = Decl::DECL_DATA_FIELD };
        DataFieldDecl() {}
        DataFieldDecl(QualType t);
        template <typename T>
        DataFieldDecl(QualifiedType<T> t);
        DataFieldDecl(const DataFieldDecl& r) : FieldDecl(r) {}

        void init(QualType t);

        /// Creates the field declaration matching a member pointer.
        template<typename Type, typename FieldType>
        DataFieldDecl(_In_ FieldType Type::*memberPointer);

        bool operator==(const DataFieldDecl& rhs) const;
        HK_ALWAYS_INLINE bool operator!=(const DataFieldDecl& rhs) const { return !(*this == rhs); }

    private:
        void createFromField(_In_ const hkReflect::Type* type, _In_ const hkReflect::Type* fieldType, hkLong memberOffset);
    };

    /// Experimental. Declaration corresponding to a property declared with HK_PROPERTY.
    class HK_EXPORT_COMMON PropertyFieldDecl : public FieldDecl
    {
    public:
        HK_DECLARE_CLASS(PropertyFieldDecl, New, Reflect);
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

        enum { MASK = Decl::DECL_PROPERTY_FIELD };
        PropertyFieldDecl() {}
        PropertyFieldDecl(QualType t);
        template <typename T>
        PropertyFieldDecl(QualifiedType<T> t);
        PropertyFieldDecl(const PropertyFieldDecl& r) : FieldDecl(r) {}

        void init(QualType t);
    };

    /// Iterator for all declarations on a type.
    /// The order of declarations is from the root, downwards.
    /// If 'startAfter' is not null, iteration starts at the first type derived from it.
    class HK_EXPORT_COMMON DeclIterAll
    {
    public:
        DeclIterAll(_In_ const hkReflect::Type* t, _In_opt_ const Type* startAfter = HK_NULL);
        bool advance();
        const Decl& current() const { HK_ASSERT(0x31e23ad0, m_declCur, "Call advance() first"); return *m_declCur; }
    private:
        bool advance2();
        const Decl* m_declCur;
        const Decl* m_declEnd;
        const Type* m_curType;
        const Type* m_endType;
    };

    /// Iterator for specific type of declarations in a type.
    /// See classes inherited from hkReflect::Decl.
    /// Most common uses are FieldDecl or DataFieldDecl.
    template<typename DECL>
    class DeclIter
    {
    public:
        DeclIter(_In_ const Type* t, _In_opt_ const Type* startAfter = HK_NULL) : m_it(t, startAfter) {}
        bool advance();
        const DECL& current() const { return static_cast<const DECL&>(m_it.current()); }
    private:
        DeclIterAll m_it;
    };

    /// Like DeclIter but with a filter. Only decls for which FilterFunc returns true are returned.
    template<typename DECL>
    class FilteredDeclIter : public DeclIter<DECL>
    {
    public:
        typedef bool(*FilterFunc)(const DECL& decl);
        FilteredDeclIter(_In_ const Type* t, FilterFunc f, _In_opt_ const Type* startAfter = HK_NULL) : DeclIter<DECL>(t, startAfter), m_filter(f) {}
        bool advance();
    private:
        FilterFunc m_filter;
    };

    /// Array of template arguments for the Type corresponding to a template instantiation.
    class HK_EXPORT_COMMON Template
    {
    public:
        enum ParamKind
        {
            KIND_TYPE = 't',
            KIND_VALUE = 'v'
        };

        /// An individual template parameter.
        struct Parameter
        {
        public:
            int getKind() const { HK_ASSERT_NO_MSG(0x504aae26, isValidKind(m_kindAndName[0])); return m_kindAndName[0]; }
            bool isType() const { HK_ASSERT_NO_MSG(0x10b3ba73, isValidKind(m_kindAndName[0])); return m_kindAndName[0] == KIND_TYPE; }
            bool isValue() const { HK_ASSERT_NO_MSG(0x8404c9c, isValidKind(m_kindAndName[0])); return m_kindAndName[0] == KIND_VALUE; }

            _Ret_notnull_ const hkReflect::Type* getAsType() const { HK_ASSERT_NO_MSG(0x460cc91, isType()); return reinterpret_cast<const hkReflect::Type*>(m_storage); }
            hkUlong getAsValue() const { HK_ASSERT_NO_MSG(0x2bc90c48, isValue()); return m_storage; }
            _Ret_z_ const char* getName() const { HK_ASSERT_NO_MSG(0x1fe379d0, isValidKind(m_kindAndName[0])); return m_kindAndName + 1; }

            static bool isValidKind(char kind)
            {
                return(kind == KIND_TYPE || kind == KIND_VALUE);
            }

            hkUlong m_storage;
            const char* m_kindAndName; //first character is ParamKind
        };


        int getNumParams() const;
        _Ret_notnull_ const Parameter* getParam(int idx) const;
        _Ret_notnull_ const Parameter* getParams() const;

        _Ret_notnull_ static Template* create(int numParams, hkMemoryAllocator& a);
        static int getSizeOf(int numParams);

        void destroy(hkMemoryAllocator& a);

        /// Advanced use.
        _Ret_maybenull_ const hkReflect::Type* getUndecoratedType() const;

    private:
        Template(); //Not implemented, use create

        
        const hkReflect::Type* m_undecoratedType;

        hkLong m_numParams;
    };

    /// Type used in the visualizers. The size of the optional array doesn't really matter. For some reason if this is moved to other
    /// headers like hkPodPunsDetail.h the visualizers break...
    namespace Detail
    {
        struct VisualizerType
        {
            hkUlong m_bits;
            const hkReflect::Type* m_parent;
            hkUlong m_optional[1];
        };

        HK_DECLARE_HAS_METHOD_TRAIT(bool, reflectValidate);

        /// If the _Auto contains bool isValid(const T& t)
        template<typename T>
        hkResult _isObjectValid(const T& t, hkTrait::TrueType&)
        {
            bool res = t.reflectValidate();
            return res ? HK_SUCCESS : HK_FAILURE;
        }

        template<typename T>
        hkResult _isEnumObjectValid(const T& t, hkTrait::TrueType&)
        {
            bool res = hkReflect::Detail::isValidForPresets(hkReflect::Detail::PresetsOf<T>::presets, t);
            return res ? HK_SUCCESS : HK_FAILURE;
        }

        template<typename T>
        hkResult _isEnumObjectValid(const T& t, hkTrait::FalseType&)
        {
            // Not an enum
            return HK_SUCCESS;
        }
        /// If no isValid method is defined, there is nothing to do and the object is valid
        template<typename T>
        hkResult _isObjectValid(const T& t, hkTrait::FalseType&)
        {
            typename hkTrait::IsReflectedEnum<T>::Type isReflectedEnum;

            return _isEnumObjectValid(t, isReflectedEnum);
        }

        /// General purpose Reflection logging information
        HK_EXPORT_COMMON hkLog::RegisteredOrigin& getLog();
    }

    // Validate object based on the type attributes visible at compile-time. The type must be
    // known statically and the reflection must be visible
    template<typename T>
    hkResult isObjectValid(const T& t)
    {
        static_assert(Detail::HasMethod_reflectValidate<T>::NonConst == 0, "Method must be const");
        typename hkTrait::TraitBool<Detail::HasMethod_reflectValidate<T>::Const> x;
        return Detail::_isObjectValid(t, x);
    }

    // Validate object based on the type reflection
    hkResult HK_EXPORT_COMMON validateByType(hkReflect::Var value);

    // Validate object against a different type. The types must compare equal
    hkResult HK_EXPORT_COMMON validateByType(hkReflect::Var value, _In_ const hkReflect::Type* validationType);

}

// Validate object based on the type attributes visible at compile-time. The type must be
// known statically and the reflection must be visible
#define HK_REFLECT_VALIDATE_FIELD(TYPE, FIELD, VALUE) hkResult(TYPE::_Auto::isValid_ ## FIELD(&VALUE) ? HK_SUCCESS : HK_FAILURE)
#define HK_REFLECT_VALIDATE_OBJECT(TYPE, VALUE) (hkReflect::validateByType(hkReflect::Var(&VALUE), hkReflect::getType<TYPE>()))

namespace hkHash
{
    hkUint32 hkHashValue(const hkReflect::TypeName& typeName);
}

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkReflect::Type::FlagBits);
HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkReflect::Decl::DeclFlagBits);
HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkReflect::Optional);

#include <Common/Base/Reflect/Core/hkReflectType.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
