// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

namespace hkReflect
{
    class FieldDecl;

namespace Detail
{
        /// Abstract interfaces for the overridable behaviors of each type.
    struct HK_EXPORT_COMMON Impl
    {
        public:
            HK_DECLARE_CLASS(Impl, New);
            Impl();
            virtual ~Impl();

            /// Check for a specific implementation.
            virtual _Ret_maybenull_ void* queryInterfaceImpl(_In_ const Type* type, const Var& self) const;

            /// Helper for above
            template<typename T>
            T* queryInterface(const Var& self) const { return static_cast<T*>(queryInterfaceImpl(getType<T>(), self)); }

            virtual hkResult inplaceFixup(_Inout_ void* self, _In_ const Type* selfType, _Inout_ void* target, _In_ const Type* targetType, int count) const;

            
            typedef bool(*SetterType)(void*, const void*);
            virtual SetterType getSetter() const;
            _Ret_maybenull_ virtual const Impl* getFieldImpl(const Var& parent, FieldDecl field) const;

            
            virtual bool isProperty() const { return false; }

        
        protected:
            void addReference() const;
            void removeReference() const;

            mutable hkUint32 m_refCount;

        friend struct ImplPtr;
    };

        /// Base class for value types
    struct HK_EXPORT_COMMON ValueImpl : public Impl
    {
    };

        /// Boolean value
    struct HK_EXPORT_COMMON BoolImpl : public ValueImpl
    {
        virtual hkResult setValue(_Inout_ void* addr, _In_ const BoolType* type, BoolValue val) const = 0;
        virtual hkResult getValue(_In_ const void* addr, _In_ const BoolType* type, _Out_ BoolValue* val) const = 0;
    };

        /// Integer value
    struct HK_EXPORT_COMMON IntImpl : public ValueImpl
    {
        virtual hkResult setValue(_Inout_ void* addr, _In_ const IntType* type, const IntValue& val) const = 0;
        virtual hkResult getValue(_In_ const void* addr, _In_ const IntType* type, _Out_ IntValue* val) const = 0;
    };

        /// Floating point value
    struct HK_EXPORT_COMMON FloatImpl : public ValueImpl
    {
        virtual hkResult setValue(_Inout_ void* addr, _In_ const FloatType* type, FloatValue val) const = 0;
        virtual hkResult getValue(_In_ const void* addr, _In_ const FloatType* type, _Out_ FloatValue* val) const = 0;
    };

        /// Null terminated string
    struct HK_EXPORT_COMMON StringImpl : public ValueImpl
    {
        virtual hkResult setValue(_Inout_ void* string, _In_ const StringType* type, StringValue newValue) const = 0;
        virtual hkResult getValue(_In_ const void* string, _In_ const StringType* type, _Out_ StringValue* val) const = 0;

            /// Free internal allocations. Called in place of the destructor when cleaning up partially cloned objects.
        virtual void clearAllocs(_Inout_ void* string, _In_ const StringType* type) const {}
    };

        /// Null terminated string
    struct HK_EXPORT_COMMON CompoundImpl : public Impl
    {
            /// Find the given var in the collection. Depending on the underlying type, "var"
            /// could either be a key or key/value pair. If "prev" is given, the find continues
            /// from that point.
            /// The default implementation simply steps through each element with iterBegin/iterNext.
        virtual VarIter find(const VarIter& prev, Var var) const;
            /// Get the first item in the compound.
        virtual VarIter iterBegin(_In_ const void* comAddr, _In_ const CompoundType* comType) const = 0;
            /// Get item after "prev" in the collection.
        virtual void iterNext(VarIter& prev) const = 0;
    };

        /// Abstract interface to containers.
        /// Calling insert or remove will in general invalidate all existing iterators.
    struct HK_EXPORT_COMMON ContainerImpl : public CompoundImpl
    {
        virtual ContainerValue getContainerValue(_In_ const void* conAddr, _In_ const ContainerType* conType) const = 0;
            /// Insert an item into the collection.
            /// The type of "var" must match the type of the collection.
            /// An insert hint may be given. 
        virtual bool insert( Var var, const VarIter& hint ) const = 0;
            /// Remove an item from the collection.
        virtual bool remove( const VarIter& var ) const = 0; 
    };

        /// Pointer
    struct HK_EXPORT_COMMON PointerImpl : public CompoundImpl
    {
            /// Sets the pointer value from the given var.
            /// Returns false on failure (usually type incompatibility)
        virtual hkResult setValue(_Inout_ void* self, _In_ const PointerType* type, const Var& var) const = 0;
            /// Gets the pointed to object and type.
        virtual hkResult getValue(_In_ const void* addr, _In_ const PointerType* type, _Out_ Var* val) const = 0;
            /// True for regular pointers, false for references.
            
        virtual bool canBeNull() const { return true; }
            // CompoundImpl methods
        void iterNext( VarIter& prev ) const HK_OVERRIDE;
        VarIter iterBegin(_In_ const void* comAddr, _In_ const CompoundType* comType) const HK_OVERRIDE;
    };

        /// Interface for array implementations.
        /// Arrays are a contiguous sequence of elements.
        /// The element type need not be fixed, but must be the same for all elements (like vertex buffers).
    struct HK_EXPORT_COMMON ArrayImpl : public ContainerImpl
    {
            /// Default implementation calls spliceInto replacing all contents with val.
        virtual hkResult setValue(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, const ArrayValue& val) const;
            /// Get an ArrayValue pointing to the array buffer.
        virtual hkResult getValue(_In_ const void* addr, _In_ const ArrayType* type, _Out_ ArrayValue* val) const = 0;
            /// Resize the array. Destruct removed elements, default-construct new element.
        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int nelem) const = 0;
            /// Default implementation replaces the elements in arrInOut with the elements in source if
            /// numToDel and numToInserts are equal, otherwise it returns HK_FAILURE.
            /// Implementations should remove all elements if numToDel is -1.
            /// Implementations should append to the array if index is -1.
        virtual hkResult spliceInto(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int index, int numToDel, const ArrayValue& val) const;
            /// Return type for allocateElements.
        enum AllocResult { ALLOC_SUCCESS, ALLOC_FAILURE, ALLOC_NOT_EMPTY };
            /// Initialize an array for the cloning/deserialization of n elements.
            /// Should only be called by hkReflect::Cloner during cloning/deserialization on newly constructed arrays
            /// before elements values are written. Should usually allocate a buffer and reflect-construct the elements
            /// (and set the element type if the array is variant).
            /// Arrays should not contain elements/allocations before being passed to this function (since they will be
            /// discarded/overwritten by cloning). If the passed array contains elements/allocation, this function
            /// should return ALLOC_NOT_EMPTY. This will cause hkReflect::Cloner to print a warning and fall back to
            /// setNumElements. Otherwise, this function should return ALLOC_SUCCESS or ALLOC_FAILURE.
        virtual AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, QualType elemType, int n) const { return ALLOC_FAILURE; }
            /// Free internal allocations. Called in place of the destructor when cleaning up partially cloned objects.
            /// Should *not* destruct/recurse on the elements.
        virtual void clearAllocs(_Inout_ void* arrAddr, _In_ const ArrayType* type) const {}

            // These CompoundImpl methods can be implemented generically here
        VarIter iterBegin(_In_ const void* comAddr, _In_ const CompoundType* comType) const HK_OVERRIDE;
        void iterNext(VarIter& prev) const HK_OVERRIDE;
            // Also spliceInto can be used for ContainerImpl
        virtual bool insert(Var var, const VarIter& hint) const HK_OVERRIDE;
        virtual bool remove(const VarIter& var) const HK_OVERRIDE;
            ///
        virtual ContainerValue getContainerValue(_In_ const void* addr, _In_ const ContainerType* type) const HK_OVERRIDE;
    };

        /// Interface for record implementations.
    struct HK_EXPORT_COMMON RecordImpl : public CompoundImpl
    {
        virtual hkResult setValue(_Inout_ void* addr, _In_ const RecordType* type, const RecordVar& other) const = 0;
        virtual hkResult getValue(_In_ const void* addr, _In_ const RecordType* type, _Out_ RecordVar* val) const = 0;

        VarIter find(const VarIter& prev, Var var) const HK_OVERRIDE;
        VarIter iterBegin(_In_ const void* comAddr, _In_ const CompoundType* comType) const HK_OVERRIDE;
        void iterNext(VarIter& prev) const HK_OVERRIDE;
    };

    struct HK_EXPORT_COMMON OpaqueImpl : public Impl
    {
        // construct
        // destruct
        // copy
        // serialize
        // deserialize
        OpaqueImpl() {}
    };

    struct HK_EXPORT_COMMON VoidImpl : public Impl
    {
        VoidImpl() {}
    };

    class HK_EXPORT_COMMON AllocationImpl
    {
    public:
        /// Allocate an object of the given type.
        virtual Var allocate(_In_ const Type* type) const = 0;

        /// Allocate an object of the given type which is suitable for holding a copy of "src".
        /// This method is used to construct variable sized objects, such as objects containing
        /// hkRelArrays.
        virtual Var allocateForClone(const Var& src, _In_ const Type* type) const = 0;

        /// Deallocate the given object.
        virtual void deallocate(_Inout_ void* target, _In_ const Type* type) const = 0;

        /// If true, an instance of a type using this impl can be constructed in any existing buffer
        /// which satisfies type size and alignment; otherwise allocate() must be used to obtain a valid buffer.
        
        virtual bool isInPlaceConstructible(_In_ const hkReflect::Type* type) const { return true; }

        virtual ~AllocationImpl() {}
    };
}
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
