// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

namespace hkReflect
{
namespace Detail
{
    // Helper struct which can be constructed from a typed pointer.
    struct PointerToT
    {
        // Extracts the exact type on construction. Uses SFINAE on T to know whether it can call getExactType()
        // or not.
        template <typename T>
        HK_ALWAYS_INLINE PointerToT(_In_opt_ T* t) : m_addrAndType(Helper<T>::getExact(t)) {}

        // SFINAE check helper.
        template <typename T>
        struct HasVirtualGetExactType
        {
            template <typename C> static char func(typename C::_GetExactTypeIsVirtual*);
            template <typename C> static int func(...);
            enum { result = sizeof(func<T>(0)) == sizeof(char) };
        };

        template <typename T, bool HAS_EXACT_TYPE_METHOD = HasVirtualGetExactType<T>::result> struct Helper;

        // If C has a virtual getExactType, call it.
        template <typename T>
        struct Helper<T, true>
        {
            static HK_INLINE AddrAndType getExact(_In_opt_ T* obj) { return obj ? obj->getExactType() : AddrAndType(HK_NULL, HK_NULL); }
        };


        // Otherwise fallback on the 2-args exactObj.
        template <typename T>
        struct Helper<T, false>
        {
            static HK_INLINE AddrAndType getExact(_In_opt_ T* obj) { return obj ? exactObj(obj, hkReflect::getType<T>()) : AddrAndType(HK_NULL, HK_NULL); }
        };

        AddrAndType m_addrAndType;
    };

    /// Helper struct storing the exact type of a T* through a user-conversion (for overload resolution ordering).
    /// Also provides an implicit cast operator.
    class AutoCast
    {
    public:

        AutoCast() : m_addrAndType(HK_NULL, HK_NULL) {}

        template<typename From>
        AutoCast(_In_opt_ From* from);

        AutoCast(_In_opt_ const void* obj, hkReflect::QualType type);

        template<typename To>
        _Ret_maybenull_ operator To*() const;

    private:

        AddrAndType m_addrAndType;
    };
}
}

template<typename T>
HK_ALWAYS_INLINE hkReflect::QualType hkReflect::getType()
{
    static_assert(hkReflect::IsReflected<T>::result, "Type T must be reflected");

    return QualType(HK_REFLECT_GET_TYPE(T), hkTrait::IsConstType<T>::result);
}

template <typename To, typename From>
_Ret_maybenull_ To* hkReflect::exactMatchDynCast(_In_opt_ From* from)
{
    using hkReflect::exactTypeOf;
    return hkReflect::getType<To>()->equals(exactTypeOf(from)) ? static_cast<To*>(from) : HK_NULL;
}

template <typename To, typename From>
_Ret_maybenull_ const To* hkReflect::exactMatchDynCast(_In_opt_ const From* from)
{
    using hkReflect::exactTypeOf;
    return hkReflect::getType<const To>()->equals(exactTypeOf(from)) ? static_cast<const To*>(from) : HK_NULL;
}

template <typename To, typename VoidOrConstVoid>
typename hkTrait::ConstIfConst<VoidOrConstVoid, To>::type* hkReflect::upCast(_In_ VoidOrConstVoid* object, _In_opt_ const hkReflect::Type* srcType)
{
    return static_cast<typename hkTrait::ConstIfConst<VoidOrConstVoid, To>::type*>(upCast(object, srcType, hkReflect::getType<To>()));
}

template <typename VoidOrConstVoid>
VoidOrConstVoid* hkReflect::upCast(_In_ VoidOrConstVoid* object, _In_opt_ const hkReflect::Type* srcType, _In_ const hkReflect::Type* toType)
{
    return srcType && srcType->extendsOrEquals(toType) ? object : HK_NULL;
}

template <typename KindType>
_Ret_maybenull_ const KindType* hkReflect::exactMatchDynCast(_In_ const hkReflect::Type* b)
{
    return b->getKind() == hkReflect::KindFromType<KindType>::kind
        ? static_cast<const KindType*>(b)
        : HK_NULL;
}

template <typename To, typename From>
_Ret_maybenull_ typename hkTrait::ConstIfConst<From, To>::type* hkDynCast(_In_opt_ From* from)
{
    // Forwards to the 2 args version.
    auto res = hkDynCast(from, hkReflect::getType<To>());

    // Cast the result.
    return static_cast<typename hkTrait::ConstIfConst<From, To>::type*>(res);
}

template <typename From>
_Ret_maybenull_ typename hkTrait::ConstIfConst<From, void>::type* hkDynCast(_In_opt_ From* from, _In_ const hkReflect::Type* toType)
{
    if (!from) { return HK_NULL; }

    // Downcast to exact type.
    using hkReflect::exactObj;
    hkReflect::Detail::AddrAndType res = exactObj(from);

    // Upcast to toType.
    return hkReflect::upCast(res.m_addr, res.m_type, toType);
}

template <typename KindType>
_Ret_maybenull_ const KindType* hkDynCast(_In_ const hkReflect::Type* b)
{
    return hkReflect::Detail::kindMatches<KindType>(b->getKind())
        ? static_cast<const KindType*>(b)
        : HK_NULL;
}

// Use an AutoCast as argument instead of a templated From* pointer for overload ordering purposes
// (this should be the last overload selected).
HK_ALWAYS_INLINE hkReflect::Detail::AutoCast hkDynCast(hkReflect::Detail::AutoCast from)
{
    return from;
}

template<typename T>
inline _Ret_z_ const char* hkReflect::getName()
{
    const hkReflect::Type* type = getType<T>();
    return type->getName();
}

template<typename From>
HK_ALWAYS_INLINE hkReflect::Detail::AutoCast::AutoCast(_In_opt_ From* from)
    : m_addrAndType(hkReflect::exactObj(from))
{
}

HK_ALWAYS_INLINE hkReflect::Detail::AutoCast::AutoCast(_In_opt_ const void* obj, hkReflect::QualType type)
    : m_addrAndType(obj, type)
{
}


template<typename To>
_Ret_maybenull_ HK_ALWAYS_INLINE hkReflect::Detail::AutoCast::operator To*() const
{
    return m_addrAndType.m_addr ? hkReflect::upCast<To>(m_addrAndType.m_addr, m_addrAndType.m_type) : HK_NULL;
}

template<typename T>
HK_ALWAYS_INLINE const hk::Presets& hkReflect::getPresetsOf()
{
    return hkReflect::Detail::PresetsOf<T>::presets;
}

HK_ALWAYS_INLINE hkReflect::Detail::AddrAndType hkReflect::exactObj(const hkReflect::Detail::PointerToT& ptr)
{
    // The type in the PointerToT is already exact.
    return ptr.m_addrAndType;
}

HK_ALWAYS_INLINE hkReflect::Detail::AddrAndType hkReflect::exactObj(_In_ const hkReflect::Type* t)
{
    // Since Type is not a polymorphic class, calling Type::getExactType would not work. Therefore we call typeFromKind
    // which returns the type which reflects the type corresponding to the given kind. This is faster than calling
    // exactTypeOf(t, hkReflect::getType<hkReflect::Type>()) since it doesn't have to do the attribute lookup.
    return Detail::AddrAndType(t, t ? hkReflect::typeFromKind(t->getKind()) : HK_NULL);
}

template<typename T>
inline _Ret_notnull_ const hkReflect::Type* hkReflect::exactTypeOf(_In_ T* ptr)
{
    using hkReflect::exactObj;
    Detail::AddrAndType res = exactObj(ptr);
    HK_ASSERT(0x66bf8ee5, res.m_addr == ptr, "Downcast from interface pointer. Use exactObj to get the correct downcast object pointer");
    return res.m_type;
}

template<typename T>
inline _Ret_notnull_ const hkReflect::Type* hkReflect::exactTypeOfUnchecked(_In_ T* ptr)
{
    using hkReflect::exactObj;
    Detail::AddrAndType res = exactObj(ptr);
    return res.m_type;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
