// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Reflect/Util/hkReflectAny.h>

namespace hkReflect
{

class ArgsBuffer;

namespace Detail
{
    
    class SingleInheritanceClass { };
    typedef void (SingleInheritanceClass::*SingleInheritanceMethodPointer)();

    typedef hkUlong CallableId;
    struct CallableSignature;
}

typedef Detail::SingleInheritanceMethodPointer SiMethodPointer;

    /// Experimental. Abstraction for things which can be called.
    /// Callables contain logical parts
    /// * some information about the input and return types
    /// * a low-level invoker which forwards to the implementation.
    /// The invoker needs the call arguments packed in a very particular way.
    
    
    
class HK_EXPORT_COMMON Callable
{
    public:

        HK_DECLARE_CLASS(Callable, New);

        enum { MAX_ARGS = 6 };

        _Ret_z_ const char* getName() const { return m_name; }

            /// Fills name with a name that uniquely identifies that callable
        _Ret_z_ const char* getFullName(hkStringBuf& nameBuf) const;

            /// If this callable corresponds to a method, return the type of the associated object.
            /// Otherwise, return null.
        inline _Ret_maybenull_ const Type* getObjectType() const;

            /// Get the return type of the callable.
        inline _Ret_maybenull_ const Type* getReturnType() const;

            /// Does not include the implicit "this" for method calls, see getObjectType()
        inline int getNumParams() const;

            /// Does not include the implicit "this" for method calls, see getObjectType().
        inline _Ret_notnull_ const Type* getParamType(int i) const;
        inline _Ret_z_ const char* getParamName(int i) const;
        inline hkReflect::Var getParamDefaultArg(int i) const;

            /// Get the required offset of the i'th parameter in the argument buffer.
        inline int getParamOffset(int i) const;

            /// Find an attribute of the given type t attached to this callable.
        inline Var findAttribute(_In_ const Type* t) const;

            /// Find an attribute of the given type T attached to this callable.
        template <typename T> inline _Ret_maybenull_ const T* findAttribute() const;

            //
            // Optimizations when the callable is known to be a method call
            //

            /// Call a method. It is an error if this callable is not a method (i.e. has no "this" parameter)
            
        HK_INLINE void callMethod(void* thisPtr, void* packedArgs, void* returnBuffer) const;

            //
            // Function calls
            //

            /// Call a function. It is an error if this callable is not a method (i.e. has no "this" parameter)
            
        HK_INLINE void callFunction(void* args, void* returnBuffer) const;

            
        inline void callGeneric(void* object, void* params, void* retbuf) const;

            //
            // Advanced usage.
            //

            // Advanced usage.
            
        inline SiMethodPointer getMethodPointer() const;

            // Advanced usage.
            /// Returns a compile time accessible id for a method
        inline Detail::CallableId getCallableId() const;

            // Advanced usage.
        HK_INLINE Detail::CallInvoker getCallInvoker() const;

            /// Advanced usage - the required size the parameters buffer including padding
        inline int getParamBufSize() const;

            /// Advanced usage - the required alignment of the parameter buffer
        inline int getParamBufAlignment() const;

    private:

            /// The low-level dispatch mechanism.
        Detail::CallInvoker m_invoker;

            /// The signature, or the tail of the signature in the case of methods.
        Detail::CallableSignature* m_paramInfo;

            /// If this represents a method, the type of the object type.
        const Type* m_objectType;

            /// Array of the parameter names.
        const char* const * m_paramNames;

        const char* m_name;

        const hkReflect::Type* m_attributes;

        const void* const* m_defaultArgs;
};

/// Get the function signature for a function. The first parameter is the return type, subsequent parameters are argument types.
#define HK_REFLECT_CALLABLE_SIGNATURE_OF(...) \
    reinterpret_cast<const hkReflect::Detail::CallableSignature*>( &hkReflect::Detail::FunctionSignatureOf<__VA_ARGS__>::pass )


typedef Callable NamedCallable;

/// Utility class to invoke a Callable.
/// E.g. hkReflect::Var result = hkReflect::Call(myCallable).onObj(myObj).into(someAny).withArg(myArg).
class HK_EXPORT_COMMON Call
{
public:
    HK_DECLARE_CLASS( Call, New );

    Call(_In_ const Callable* callable);

    Call& onObj( const hkReflect::Var& obj );
    Call& onObjRaw( hkArrayView<void> buffer );

    Call& into( const Var& returnBuf );
    Call& into( Any& returnBuf );
    Call& intoRaw( hkArrayView<void> buffer );

    Var withArg( const Var& arg = Var() ) const;
    
    Var withArgsRaw( hkArrayView<void> buffer ) const;

private:

    Var call( void* argsBuf ) const;

    const Callable* m_callable;
    void* m_obj;
    void* m_returnBuf;
};

// ---------------------------- Function Iterators ------------------------------- //

class HK_EXPORT_COMMON CallableIterator
{
    public:
        CallableIterator(_In_ const Type* bt, CallableTypes types);

        bool advance();
        bool advanceTo(int index);
        _Ret_notnull_ const NamedCallable* current() const;

    private:
        bool nextArray();

    private:
        hkArrayView<const NamedCallable> m_currentArray;
        int m_currentIndex;

        const hkReflect::Type* m_currentType;

        CallableTypes m_callableTypes;
        hkUint8 m_currentCallableType;
};

namespace Detail
{
    // Type information about a callable function.
    // An array of types, representing a function, constructor or method signature.
    // In the case of methods, the object type is not one of the types.
    struct HK_EXPORT_COMMON CallableSignature
    {
        HK_DECLARE_CLASS(CallableSignature, New);

        struct Param
        {
            HK_DECLARE_CLASS(Param, New);

            const Type* m_type;
            hkUint16 m_offset;
        };

        // Number of types in the signature.
        hkUint8 m_numParams;

        hkUint8 m_align;

        hkUint16 m_size;

        const Type* m_returnType;

        // params directly follow the signature
        _Ret_notnull_ const Param* getParams() const { return reinterpret_cast<const Param*>(this + 1); }
        const Param& getParam(int i) const { HK_ASSERT_NO_MSG(0x629e4d1f, unsigned(i) < m_numParams); return getParams()[i]; }
    };

    /// Helper return type for methods marked with a hk::StringOut parameter.
    struct HK_EXPORT_COMMON StringOut
    {
        HK_DECLARE_CLASS(StringOut, New, Reflect);
        HK_REFLECT_AS_STRING_UTF8_MUTABLE(&hkReflect::Detail::StringOut::s_impl);
        HK_RECORD_ATTR(hk::Serialize(false), hk::IncludeInMgd(false));

        StringOut() : m_string(m_buf.cString()) {}

        hkStringBuf m_buf;
        const char* m_string;

        struct StringOutImpl;
        static const StringOutImpl s_impl;
    };
}

}

#include <Common/Base/Reflect/Core/hkReflectCallable.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
