// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Reflect/Core/hkReflectCallable.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>
#include <Common/Base/Object/hkSingleton.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Reflect/Detail/hkTypeHasher.h>
#include <Common/Base/Reflect/TypeReg/Detail/hkTypeRegDetail.h>

HK_COMPILE_TIME_ASSERT(sizeof(hkReflect::Detail::SizeAlign) == sizeof(hkUlong));

int hkReflect::Detail::SizeAlign::retargetAlignment(int sizeofReal) const
{
    switch(int l2 = m_reqAlignEncoded)
    {
        case ALIGN_REQ_NATURAL://0
            return m_alignOf;
        case ALIGN_REQ_REAL4:
            return hkLosslessCast<int>(hkMath::max2(m_alignOf, sizeofReal*4));
        default:
            return hkLosslessCast<int>(hkMath::max2( m_alignOf, 1 << l2 ));
    }
}

void hkReflect::Detail::SizeAlign::toString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra)
{
    const hkReflect::Detail::SizeAlign* sa = (const hkReflect::Detail::SizeAlign*)var.getAddress();
    sb.format("{}{{ size={} align={} }}", var.getType()->getName(), sa->m_sizeOf, sa->m_alignOf);
}

// -------------------------- Functions -------------------------------- //

_Ret_notnull_ const hkReflect::NamedCallable* hkReflect::Detail::Functions::getUnboundMethod(int i) const
{
    HK_ASSERT_NO_MSG(0x2d9321a1, i < m_numMethods);
    return &m_callableInfo[i];
}

_Ret_notnull_ const hkReflect::NamedCallable* hkReflect::Detail::Functions::getConstructor(int i) const
{
    HK_ASSERT_NO_MSG(0x6f801224, i < m_numConstructors);
    const int index = i + m_numMethods;
    return &m_callableInfo[index];
}

_Ret_notnull_ const hkReflect::NamedCallable* hkReflect::Detail::Functions::getFunction(int i) const
{
    HK_ASSERT_NO_MSG(0x26644911, i < m_numFunctions);
    const int index = i + m_numMethods + m_numConstructors;
    return &m_callableInfo[index];
}

void hkReflect::TypeDetail::localSetSizeAlignPreserveReq(_Inout_ Type* t, int size, int align)
{
    Detail::SizeAlign cur = TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(t);
    cur.m_sizeOf = size;
    cur.m_alignOf = align;
    TypeDetail::localSetOptionalUlong(t, Opt::SIZE_ALIGN, cur.asUlong());
}

void hkReflect::TypeDetail::setFieldType(_Inout_ Type* rec, _In_z_ const char* name, _In_ const Type* type)
{
    for( Type* cur = rec; cur; cur = const_cast<Type*>(cur->getParent()) )
    {
        auto decls = TypeDetail::decoratorGetOptional<Opt::DECLS>(cur);
        auto fields = decls->getFields();
        for( int i = 0; i < fields.getSize(); ++i )
        {
            FieldDecl f = fields[i];
            if (f.getName())
            {
                if (hkString::strCmp(f.getName(), name) == 0)
                {
                    const_cast<hkReflect::Detail::DeclsArray*>(decls)->setField(i, type);
                    return;
                }
            }
        }
    }
    HK_ASSERT_NO_MSG(0x366a133b,0);
}


hkReflect::Var hkReflect::TypeDetail::allocate(_In_ const hkReflect::Type* type)
{
    if (const Detail::AllocationImpl* alloc = globalGetOptional<Opt::ALLOC_IMPL>(type))
    {
        return alloc->allocate(type);
    }
    return hkReflect::Var();
}

hkReflect::Var hkReflect::TypeDetail::allocateForClone(const Var& src, _In_ const hkReflect::Type* type)
{
    if (const Detail::AllocationImpl* alloc = globalGetOptional<Opt::ALLOC_IMPL>(type))
    {
        return alloc->allocateForClone(src, type);
    }
    return hkReflect::Var();
}

void hkReflect::TypeDetail::deallocate(_Inout_ void* target, _In_ const hkReflect::Type* type)
{
    const Detail::AllocationImpl* alloc = globalGetOptional<Opt::ALLOC_IMPL>(type);
    HK_ASSERTV(0x51955944, alloc, "Attempting to deallocate type {} that does not have ALLOC_IMPL. Ensure the type is allocatable (not ReflectIdentity).", (type->getName() ? type->getName() : "(unnamed)"));
    alloc->deallocate(target, type);
}

bool hkReflect::TypeDetail::isInPlaceConstructible(_In_ const hkReflect::Type* type)
{
    if (const Detail::AllocationImpl* alloc = globalGetOptional<Opt::ALLOC_IMPL>(type))
    {
        return alloc->isInPlaceConstructible(type);
    }
    return false;
}

hkReflect::Detail::UnaryFunction hkReflect::TypeDetail::getDefaultConstructionFunction(_In_ const hkReflect::Type* type)
{
    return decoratorGetOptional<Opt::DEF_CONSTRUCTOR>(type);
}

hkReflect::Detail::BinaryFunction hkReflect::TypeDetail::getCopyConstructionFunction(_In_ const hkReflect::Type* type)
{
    return decoratorGetOptional<Opt::COPY_CONSTRUCTOR>(type);
}

hkReflect::Detail::BinaryFunction hkReflect::TypeDetail::getCopyAssignmentFunction(_In_ const hkReflect::Type* type)
{
    return decoratorGetOptional<Opt::COPY_ASSIGNMENT>(type);
}

hkReflect::Detail::UnaryFunction hkReflect::TypeDetail::getDestructionFunction(_In_ const hkReflect::Type* type)
{
    return decoratorGetOptional<Opt::DESTRUCTOR>(type);
}

hkReflect::Detail::AfterReflectNewFunction hkReflect::TypeDetail::getAfterReflectNewFunction(_In_ const hkReflect::Type* type)
{
    return decoratorGetOptional<Opt::AFTER_REFLECT_NEW>(type);
}

hkReflect::Detail::UnaryFunction hkReflect::TypeDetail::getReflectConstructionFunction(_In_ const hkReflect::Type* type)
{
    if (Detail::UnaryFunction func = decoratorGetOptional<Opt::REFLECT_CONSTRUCTOR>(type))
    {
        return func;
    }
    return decoratorGetOptional<Opt::DEF_CONSTRUCTOR>(type);
}

hkResult hkReflect::TypeDetail::reflectConstruct(_Inout_updates_bytes_(_Inexpressible_(count * size))  void* addr, _In_ const hkReflect::Type* type, int num /* = 1 */)
{
    HK_ASSERT_NO_MSG(0x79af8c1c, !type->isProperty());

    if (Detail::UnaryFunction func = getReflectConstructionFunction(type))
    {
        if (type->extendsOrEquals<hkReferencedObject>())
        {
            // ReferencedObjects expect the m_memSizeAndFlags to remain untouched, even over a constructor call
            // Mostly this is used by relarray these days.
            int stride = type->getSizeOf();
            hkReferencedObject* r = static_cast<hkReferencedObject*>(addr);
            for (int i = 0; i < num; ++i)
            {
                int memSize = r->getMemorySizeAndFlags();
                func(r, type, 1);
                // Initialize reference count to 1 so that the object stays alive even if cloning adds/removes references.
                // Will be removed at the end of cloning.
                r->setMemorySizeFlagsAndReferenceCount(memSize, 1);
                r = hkAddByteOffset(r, stride);
            }
        }
        else
        {
            func(addr, type, num);
        }
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

hkResult hkReflect::TypeDetail::defaultConstruct(_Inout_updates_bytes_(_Inexpressible_(count * size)) void* addr, _In_ const hkReflect::Type* type, int num /* = 1 */)
{
    HK_ASSERT_NO_MSG(0x54db9527, !type->isProperty());
    if (Detail::UnaryFunction func = getDefaultConstructionFunction(type))
    {
        func(addr, type, num);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

hkResult hkReflect::TypeDetail::copyConstruct(_Inout_updates_bytes_(_Inexpressible_(count * size)) void* tgt, _In_reads_bytes_(_Inexpressible_(count * size)) const void* src, _In_ const hkReflect::Type* type, int num /* = 1 */)
{
    HK_ASSERT_NO_MSG(0x22191c63, !type->isProperty());
    if (Detail::BinaryFunction func = getCopyConstructionFunction(type))
    {
        func(tgt, src, type, num);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

hkResult hkReflect::TypeDetail::copyAssign(_Inout_updates_bytes_(_Inexpressible_(count * size))  void* tgt, _In_reads_bytes_(_Inexpressible_(count * size)) const void* src, _In_ const hkReflect::Type* type, int num /* = 1 */)
{
    HK_ASSERT_NO_MSG(0x337f012c, !type->isProperty());
    if (Detail::BinaryFunction func = getCopyAssignmentFunction(type))
    {
        func(tgt, src, type, num);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}


void hkReflect::TypeDetail::destruct(_Inout_updates_bytes_(_Inexpressible_(count * size))  void* addr, _In_ const hkReflect::Type* type, int num)
{
    HK_ASSERT_NO_MSG(0x15bb5204, !type->isProperty());
    if (Detail::UnaryFunction func = getDestructionFunction(type))
    {
        func(addr, type, num);
    }
}

_Ret_maybenull_ const hkReflect::Detail::PointerImpl* hkReflect::TypeDetail::getPointerImpl(_In_ const hkReflect::Type* type)
{
    if (const hkReflect::Detail::PointerImpl* impl = globalGetOptional<Opt::POINTER_IMPL>(type))
    {
        return impl;
    }
    else
    {
        return &hkReflect::Detail::RawPointerImpl::s_instance; 
    }
}

_Ret_maybenull_ const hkReflect::Detail::InheritanceInfo* hkReflect::TypeDetail::getInheritance(_In_ const hkReflect::Type* type)
{
    return decoratorAddressOptional<Opt::INHERITANCE>(type);
}

_Ret_maybenull_ const hkReflect::Detail::Functions* hkReflect::TypeDetail::getDeclaredFunctions(_In_ const hkReflect::Type* type)
{
    return decoratorGetOptional<Opt::FUNCTIONS>(type);
}

namespace hkReflect { namespace Detail
{
    struct VtableCache
    {
        struct VtableInfo
        {
            void set(hkUlong o, void* v) { offset = o, value = v; }
            hkUlong offset;
            const void* value;
        };

        void clear()
        {
            m_vtableInfoFromType.clearAndDeallocate();
        }

        typedef hkHashMap<const hkReflect::Type*, hkArray<VtableInfo> > VtableInfoFromTypeMap;
        VtableInfoFromTypeMap m_vtableInfoFromType;

    protected:

        void computeVtableOffsets(_In_ const hkReflect::Type* topType, hkArray<VtableInfo>& vtables, hkLong offset = 0)
        {
            using namespace hkReflect;
            for(const Type* type = topType; type; type = type->getParent())
            {
                while(type->isDecorator())
                {
                    type = type->getParent();
                }
                if(const hkReflect::ArrayType* arr = type->asArray())
                {
                    for(int i = arr->getFixedCount() - 1; i >= 0; --i)
                    {
                        computeVtableOffsets(arr->getSubType(), vtables, offset + (i * arr->getSubType()->getSizeOf()));
                    }
                }
                if(auto decls = TypeDetail::localGetOptional<Opt::DECLS>(type) )
                {
                    auto fields = decls->getDataFields();
                    for (auto cur = fields.rbegin(), end = fields.rend(); cur != end; ++cur)
                    {
                        DataFieldDecl f = *cur;
                        computeVtableOffsets(f.getType(), vtables, offset + f.getOffset());
                    }
                }
                if(const InterfaceArray* iptr = TypeDetail::localGetOptional<Opt::INTERFACES>(type))
                {
                    hkArrayView<const Interface> ifaces = iptr->items();
                    for(int ni = ifaces.getSize() - 1; ni >= 0; --ni)
                    {
                        computeVtableOffsets(ifaces[ni].m_interfaceType, vtables, ifaces[ni].m_offset + offset);
                    }
                }
                if(type->hasOwnVtable())
                {
                    vtables.expandOne().set(offset, HK_NULL);
                }
            }
        }


        hkResult populateVtableValues(_In_ const hkReflect::Type* type, hkArray<VtableInfo>& vtables)
        {
            hkArray<char>::Temp buf; buf.setSize(type->getSizeOf(), 0);
            if(hkReflect::TypeDetail::reflectConstruct(buf.begin(), type).isSuccess())
            {
                for(int i = 0; i < vtables.getSize(); ++i)
                {
                    VtableInfo& v = vtables[i];
                    v.value = *(void**)hkAddByteOffset(buf.begin(), v.offset);
                }
                hkReflect::TypeDetail::destruct(buf.begin(), type, 1);
                return HK_SUCCESS;
            }
            else
            {
                HK_WARN_ALWAYS(0xabbac0ae, "Attempted to get the vtable from a non-constructible type, it must have either an accessible default constructor or a bypass constructor");
                return HK_FAILURE;
            }
        }

    public:

        VtableInitializer::PreCalc vtablesPreCalc(_In_ const hkReflect::Type* type)
        {
            typedef VtableInitializer::PreCalc PreCalc;
            if(type->isAbstract())
            {
                HK_ASSERT(0x443329da, false, "Vtable init of abstract class requested");
                return PreCalc(PreCalc::EMPTY, 0);
            }
            VtableInfoFromTypeMap::Iterator it = m_vtableInfoFromType.findKey(type);
            if(m_vtableInfoFromType.isValid(it) == false)
            {
                // This is two stages
                // 1: walk to find the offsets
                // 2: use the offsets to copy the vtable values from an instance
                hkArray<VtableInfo> info;
                computeVtableOffsets(type, info);
                if(info.getSize())
                {
                    // If we have no callable constructor this will fail
                    if(populateVtableValues(type, info).isFailure())
                    {
                        HK_ASSERT(0x6feebbfc, false, "Unable to extract vtable info");
                        info.clear(); // store the empty array to prevent retrying
                    }
                }
                // this trickiness is to insert the array without copying it
                it = m_vtableInfoFromType.findOrInsertKey(type, hkArray<VtableInfo>());
                m_vtableInfoFromType.getValue(it).swap(info);
            }
            int idx = m_vtableInfoFromType.getValue(it).getSize() ? it.m_i : PreCalc::EMPTY;
            return PreCalc(idx, type->getSizeOf());
        }

        hkResult vtablesInit(const VtableInitializer::PreCalc& preCalc, _Inout_updates_bytes_(_Inexpressible_(count * size))  void* addr, int count)
        {
            HK_ASSERT(0x57b6cc37, preCalc.m_idx >= 0, "Invalid precalc used");
            const hkArray<VtableInfo>& vtables = m_vtableInfoFromType.getValue(VtableInfoFromTypeMap::Iterator(preCalc.m_idx, 0));
            void* current = addr;
            for(int ci = 0; ci < count; ++ci)
            {
                for(int vi = 0; vi < vtables.getSize(); ++vi)
                {
                    const VtableInfo& v = vtables[vi];
                    *(const void**)hkAddByteOffset(current, v.offset) = v.value;
                }
                current = hkAddByteOffset(current, preCalc.m_stride);
            }
            return HK_SUCCESS;
        }
    };

    struct AfterCache
    {
        struct AfterInfo
        {
            void set(hkUlong o, AfterReflectNewFunction f) { offset = o, func = f; }
            hkUlong offset;
            AfterReflectNewFunction func;
        };

        void clear()
        {
            m_afterInfoFromType.clearAndDeallocate();
        }

        typedef hkHashMap<const hkReflect::Type*, hkArray<AfterInfo> > AfterInfoFromTypeMap;
        AfterInfoFromTypeMap m_afterInfoFromType;

    protected:

        void computeAfterOffsets(_In_ const hkReflect::Type* topType, hkArray<AfterInfo>& afters, hkLong offset = 0)
        {
            using namespace hkReflect;
            for(const Type* type = topType; type; type = type->getParent())
            {
                while(type->isDecorator())
                {
                    type = type->getParent();
                }
                if(AfterReflectNewFunction f = TypeDetail::getAfterReflectNewFunction(type))
                {
                    afters.expandOne().set(offset, f);
                }
                if(const hkReflect::ArrayType* arr = type->asArray())
                {
                    for(int i = arr->getFixedCount() - 1; i >= 0; --i)
                    {
                        computeAfterOffsets(arr->getSubType(), afters, offset + (i * arr->getSubType()->getSizeOf()));
                    }
                }
                if(auto decls = TypeDetail::localGetOptional<Opt::DECLS>(type))
                {
                    auto fields = decls->getDataFields();
                    for (auto cur = fields.rbegin(), end = fields.rend(); cur != end; ++cur)
                    {
                        DataFieldDecl f = *cur;
                        computeAfterOffsets(f.getType(), afters, offset + f.getOffset());
                    }
                }
            }
        }

        AfterInfoFromTypeMap::Iterator findAfterOffsets(_In_ const hkReflect::Type* type)
        {
            AfterInfoFromTypeMap::Iterator it = m_afterInfoFromType.findKey(type);
            if(m_afterInfoFromType.isValid(it) == false)
            {
                hkArray<AfterInfo> info;
                computeAfterOffsets(type, info);
                // this trickiness is to insert the array without copying it
                it = m_afterInfoFromType.findOrInsertKey(type, hkArray<AfterInfo>());
                m_afterInfoFromType.getValue(it).swap(info);
            }
            return it;
        }

    public:

        AfterInitializer::PreCalc aftersPreCalc(_In_ const hkReflect::Type* type)
        {
            typedef AfterInitializer::PreCalc PreCalc;
            AfterInfoFromTypeMap::Iterator it = findAfterOffsets(type);
            int idx = m_afterInfoFromType.getValue(it).getSize() ? it.m_i : PreCalc::EMPTY;
            return PreCalc(idx, type->getSizeOf());
        }

        hkResult aftersInit(const AfterInitializer::PreCalc& preCalc, _Inout_updates_bytes_(_Inexpressible_(count * size))  void* addr, _In_ const hkReflect::Type* type, int count)
        {
            HK_ASSERT(0x131ede5f, preCalc.m_idx >= 0, "Invalid precalc used");
            hkArrayView<AfterInfo> afters = m_afterInfoFromType.getValue(AfterInfoFromTypeMap::Iterator(preCalc.m_idx, 0));
            if(afters.getSize())
            {
                void* current = addr;
                for(int ci = 0; ci < count; ++ci)
                {
                    for(int vi = afters.getSize() - 1; vi >= 0; --vi)
                    {
                        const AfterInfo& v = afters[vi];
                        (*v.func)(hkAddByteOffset(current, v.offset));
                    }
                    current = hkAddByteOffset(current, preCalc.m_stride);
                }
            }
            return HK_SUCCESS;
        }
    };


        // Stores some reusable data about types. e.g.
        // vtable offsets & values
        // afterreflect offsets & functions
        // hashes of a type
    struct DerivedDataCache : public hkReferencedObject
    {
        HK_DECLARE_CLASS(DerivedDataCache, New, Singleton, NonCopyable);

        DerivedDataCache() {}
        VtableCache m_vtableCache;
        AfterCache m_afterCache;
        TypeHasher m_hashCache;
        PropertyHasher m_propHashCache;
        hkHashMap<hkUint32, hkArray<const hkReflect::Type*> > m_typesFromHash;

        hkCriticalSection m_lock; 
    };
} }

HK_SINGLETON_IMPLEMENTATION(hkReflect::Detail::DerivedDataCache);


hkReflect::Detail::LockedDerivedDataCache::LockedDerivedDataCache()
    : m_cache(DerivedDataCache::getInstance())
{
    m_cache.m_lock.enter();
}


hkReflect::Detail::LockedDerivedDataCache::~LockedDerivedDataCache()
{
    m_cache.m_lock.leave();
}


hkReflect::Detail::VtableInitializer::PreCalc hkReflect::Detail::VtableInitializer::preCalc(_In_ const Type* type)
{
    //ASSERT type is native
    return m_cache.m_vtableCache.vtablesPreCalc(type);
}

hkResult hkReflect::Detail::VtableInitializer::init(const PreCalc& pre, _Inout_updates_bytes_(_Inexpressible_(count * size)) void* addr, int count)
{
    HK_ASSERT_NO_MSG(0x57fcd4a1, pre.isInitialized());
    if( pre.hasSome())
    {
        return m_cache.m_vtableCache.vtablesInit(pre, addr, count);
    }
    return HK_SUCCESS;
}


hkReflect::Detail::AfterInitializer::PreCalc hkReflect::Detail::AfterInitializer::preCalc(_In_ const Type* type)
{
    //ASSERT type is native
    return m_cache.m_afterCache.aftersPreCalc(type);
}


hkResult hkReflect::Detail::AfterInitializer::init(const PreCalc& pre, _Inout_updates_bytes_(_Inexpressible_(count * size)) void* addr, _In_ const Type* type, int count)
{
    HK_ASSERT_NO_MSG(0x5b21332d, pre.isInitialized());
    if(pre.hasSome())
    {
        return m_cache.m_afterCache.aftersInit(pre, addr, type, count);
    }
    return HK_SUCCESS;
}


hkUint32 hkReflect::Detail::TypeHashCache::calc(_In_ const Type* type)
{
    return m_cache.m_hashCache.calc(type);
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::Detail::TypeHashCache::findNative(_In_ const Type* type, hkUint32 hash)
{
    typedef hkHashMap<hkUint32, hkArray<const Type*> > Map;
    Map& map = m_cache.m_typesFromHash;
    Map::Iterator it = map.find(hash);
    if(map.isValid(it))
    {
        const hkArray<const Type*>& types = map.getValue(it);
        for(int i = 0; i < types.getSize(); ++i)
        {
            if( Type::compareName(types[i], type) == 0 )
            {
                return types[i]; // cache hit!
            }
        }
    }
    if(const Type* native = typeFromType(type))
    {
        if(m_cache.m_hashCache.calc(native) == hash)
        {
            it = map.findOrInsertKey(hash, hkArray<const Type*>());
            map.getValue(it).pushBack(native);
            return native;
        }
        // else type found but hash mismatches
    }
    // else type not found at all
    return HK_NULL;
}

hkUint32 hkReflect::Detail::PropertyHashCache::calc( const Type* type )
{
    return m_cache.m_propHashCache.calc( type );
}

void hkReflect::Detail::clearDerivedDataCache()
{
    DerivedDataCache& cache = DerivedDataCache::getInstance();
    cache.m_lock.enter();
    cache.m_vtableCache.clear();
    cache.m_afterCache.clear();
    cache.m_hashCache.clear();
    cache.m_propHashCache.clear();
    cache.m_typesFromHash.clearAndDeallocate();
    cache.m_lock.leave();
}



hkResult hkReflect::Detail::initializeVtables(_Inout_updates_(_Inexpressible_(count * size)) void* addr, _In_ const Type* type, int count)
{
    VtableInitializer vtable;
    VtableInitializer::PreCalc pre = vtable.preCalc(type);
    return vtable.init(pre, addr, count);
}


void hkReflect::Detail::callAfterReflectNew(_Inout_updates_(_Inexpressible_(count * size)) void* addr, _In_ const Type* type, int count)
{
    AfterInitializer after;
    AfterInitializer::PreCalc pre = after.preCalc(type);
    after.init(pre, addr, type, count);
}

bool hkReflect::Detail::hasAfterReflectNew(_In_ const Type* type)
{
    AfterInitializer after;
    AfterInitializer::PreCalc pre = after.preCalc(type);
    return pre.hasSome();
}


hkReflect::Var hkReflect::TypeDetail::localFindAttribute(_In_ const Type* t, _In_ const Type* attributeType)
{
    if( const Detail::AttributeArray* attrs = t->valueLocal<const Detail::AttributeArray*>(Opt::ATTRIBUTES) )
    {
        hkArrayView<const Detail::AttributeItem> items = attrs->items();
        for(int i = 0; i < items.getSize(); ++i)
        {
            if( items[i].type->extendsOrEquals(attributeType) )
            {
                return Var( items[i].data, items[i].type );
            }
        }
    }
    return Var();
}

hkReflect::Var hkReflect::TypeDetail::decoratorFindAttribute(_In_ const Type* type, _In_ const Type* attributeType)
{
    while(true)
    {
        if(Var attr = hkReflect::TypeDetail::localFindAttribute(type, attributeType))
        {
            return attr;
        }
        if(type->isDecorator() && type->getParent())
        {
            type = type->getParent();
        }
        else
        {
            return Var();
        }
    }
}

_Ret_opt_z_ const char* hkReflect::TypeDetail::getAttributeString(_In_ const Type* t)
{
    return t->valueDecorator<const char*>(Opt::ATTRIBUTE_STRING);
}

namespace
{
    struct CopyCallback : public hkReflect::Detail::CloneOnHeap
    {
        hkReflect::Var m_orig;
        CopyCallback(const hkReflect::Var& orig) : m_orig(orig) {}
        virtual hkResult beginVar(hkReflect::Var& dst, hkReflect::Var& src) HK_OVERRIDE
        {
            HK_ASSERT_NO_MSG(0x4891b741, dst.isValid());
            HK_ASSERT_NO_MSG(0x20a08594, src == m_orig || src.getType()->asArray() || src.getType()->asString());
            return HK_SUCCESS;
        }
        virtual hkResult atPointer(hkReflect::PointerVar& dst, hkReflect::PointerVar& src, _Out_ hkReflect::Cloner::PointerAction::Enum* action) HK_OVERRIDE
        {
            // Warn if the object contains pointers (but not if it *is* a pointer).
            if (!src.isNull() && src != m_orig)
            {
                HK_WARN_ONCE_ON_DEBUG_IF(1, 0xa759f032, "Copying instances of " << m_orig.getType()->getName()
                    << " using cloner since the class is not copy-constructible/assignable."
                    << " Objects pointed from its instances will not be copied.");
            }

            // Do only a shallow copy.
            *action = hkReflect::Cloner::PointerAction::SKIP;
            return hkReflect::Detail::getPlain(dst).assign(src);
        }
    };
}

hkResult hkReflect::Detail::copyObject(const hkReflect::Var& target, const hkReflect::Var& source)
{
    HK_ASSERT_NO_MSG(0x576a169d, !target.getType()->isProperty());

    if ( target.getType()->equals( source.getType() ) )
    {
        if ( !source.getType()->isProperty() )
        {
            // Try copy-ctor.
            if ( TypeDetail::copyConstruct( target.getAddress(), source.getAddress(), target.getType() ).isSuccess() )
            {
                return HK_SUCCESS;
            }
        }

        // If the source is a property or cannot be copy-constructed, fall back to cloner.
        hkReflect::Var dstVar( target );
        hkReflect::Var srcVar( source );
        CopyCallback cb( srcVar );
        if ( hkReflect::Detail::cloneVarInto( dstVar, srcVar, cb ) )
        {
            if ( const hkReferencedObject* refObj = hkDynCast<hkReferencedObject>( dstVar ) )
            {
                // The copy ctor initializes the refcount to 1, while the cloner sets it to 0.
                refObj->addReference();
            }
            return HK_SUCCESS;
        }

    }

    // Target type is different from source. Try to emulate a conversion constructor by default-construct + assign.
    hkResult constructed = TypeDetail::defaultConstruct( target.getAddress(), target.getType() );

    if ( constructed.isFailure() && ( target.getType()->asValue() || target.getType()->asPointer() ) )
    {
        // Assume that simple objects can be fully initialized even by bypass construction.
        // This handles simple types which cannot be default-constructed like references.
        constructed = TypeDetail::reflectConstruct( target.getAddress(), target.getType() );
    }

    if ( constructed.isSuccess() )
    {
        if ( target.assign( source ).isSuccess() )
        {
            return HK_SUCCESS;
        }
        // Assignment failed, destruct target.
        TypeDetail::destruct( target.getAddress(), target.getType() );
    }

    return HK_FAILURE;
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::TypeDetail::skipDecorators(_In_ const Type* type)
{
    while (type && type->isDecorator())
    {
        type = type->getParent();
    }

    return type;
}

hkArrayView<const hkReflect::FieldDecl> hkReflect::TypeDetail::localGetDataFields(_In_ const hkReflect::Type* type)
{
    // Still needed??
    hkReflect::Detail::DeclsArray* fa = type->valueLocal<hkReflect::Detail::DeclsArray*>(Opt::DECLS);
    if(fa)
    {
        return fa->getDataFields();
    }
    return hkArrayView<const hkReflect::FieldDecl>();
}

hkArrayView<const hkReflect::FieldDecl> hkReflect::TypeDetail::localGetFields(_In_ const hkReflect::Type* type)
{
    hkReflect::Detail::DeclsArray* fa = type->valueLocal<hkReflect::Detail::DeclsArray*>(Opt::DECLS);
    if (fa)
    {
        return fa->getFields();
    }
    return hkArrayView<const hkReflect::FieldDecl>();
}

namespace
{
    // Helper class, keeps the pointers to the unknown special methods optionals of a type
    struct SpecialMethods
    {
        SpecialMethods(_In_ hkReflect::Type* type )
        {
            init<hkReflect::Opt::DEF_CONSTRUCTOR>( type, m_defCtor );
            init<hkReflect::Opt::COPY_CONSTRUCTOR>( type, m_copyCtor );
            init<hkReflect::Opt::COPY_ASSIGNMENT>( type, m_copyAssign );
        }

        bool anyUnknown() const { return m_defCtor || m_copyCtor || m_copyAssign; }

        void update(_In_ const hkReflect::Type* dependency ) const
        {
            update<hkReflect::Opt::DEF_CONSTRUCTOR>( dependency, m_defCtor );
            update<hkReflect::Opt::COPY_CONSTRUCTOR>( dependency, m_copyCtor );
            update<hkReflect::Opt::COPY_ASSIGNMENT>( dependency, m_copyAssign );
        }

        void setToInvalid() const
        {
            setToInvalid( m_defCtor );
            setToInvalid( m_copyCtor );
            setToInvalid( m_copyAssign );
        }

        template <hkReflect::Optional OPT>
        void init(_In_ const hkReflect::Type* type, typename hkReflect::Detail::OptionStorage<OPT>::Storage*& optValue)
        {
            optValue = hkReflect::TypeDetail::localAddressOptional<OPT>( type );
            if ( optValue && *optValue == &hkReflect::Detail::Unknown<OPT>::func )
            {
                // The method is unknown and must be resolved, initialize as trivial.
                *optValue = &hkReflect::Detail::Trivial<OPT>::func;
            }
            else
            {
                optValue = HK_NULL;
            }
        }

        template <hkReflect::Optional OPT>
        void update(_In_ const hkReflect::Type* dependency, typename hkReflect::Detail::OptionStorage<OPT>::Storage* const & optValue ) const
        {
            // If unknown and not invalid
            if ( optValue != HK_NULL && *optValue != HK_NULL )
            {
                auto depMethod = hkReflect::TypeDetail::decoratorGetOptional<OPT>( dependency );
                if ( depMethod == HK_NULL )
                {
                    // The method will be invalid.
                    *optValue = HK_NULL;
                }
                else if ( depMethod != *optValue ) // not both trivial
                {
                    // The method will be implicit.
                    *optValue = &hkReflect::Detail::Implicit<OPT>::func;
                }
            }
        }

        template <typename FUNC>
        void setToInvalid( FUNC* const & optValue ) const
        {
            if ( optValue != HK_NULL ) { *optValue = HK_NULL; }
        }

        hkReflect::Detail::UnaryFunction* m_defCtor;
        hkReflect::Detail::BinaryFunction* m_copyCtor;
        hkReflect::Detail::BinaryFunction* m_copyAssign;
    };

    static void s_fixupUnknownRecursive(_Inout_ hkReflect::Type* type, hkArray<hkReflect::Type*>::Temp& visited)
    {
        SpecialMethods sm( type );

        if ( sm.anyUnknown() )
        {
            if ( visited.lastIndexOf( type ) == -1 )
            {
                visited.pushBack( type );

                if ( const hk::FixupReflectedMethods* attr = type->findAttribute<hk::FixupReflectedMethods>() )
                {
                    // Fixup the argument types.
                    const hkReflect::Template* params = type->getTemplate();
                    for ( int i = 0; i < params->getNumParams(); ++i )
                    {
                        if ( params->getParam( i )->isType() )
                        {
                            s_fixupUnknownRecursive( const_cast<hkReflect::Type*>(params->getParam( i )->getAsType()), visited );
                        }
                    }
                    // Use the manual fixup function.
                    attr->m_func( type );
                }
                else
                {
                    if ( hkReflect::RecordType* rec = type->asRecord() )
                    {
                        // Check parent and fields and update the methods accordingly.
                        if ( const hkReflect::Type* parent = rec->getParent() )
                        {
                            s_fixupUnknownRecursive( const_cast<hkReflect::Type*>(parent), visited );
                            sm.update( parent );
                        }
                        hkArrayView<const hkReflect::FieldDecl> fields = rec->getDataFields();
                        for ( int i = 0; i < fields.getSize(); ++i )
                        {
                            s_fixupUnknownRecursive( const_cast<hkReflect::Type*>(fields[i].getType().get()), visited );
                            sm.update( fields[i].getType() );
                        }
                    }
                    else
                    {
                        sm.setToInvalid();
                    }
                }

                visited.popBack();
            }
        }
    }
}

void hkReflect::TypeDetail::fixupUnknownSpecialMethods(_Inout_ hkReflect::Type* type, bool fixupDtor )
{
    hkArray<hkReflect::Type*>::Temp visited;
    s_fixupUnknownRecursive( type, visited );

    // Destructor only needs to be fixed up in dynamic types so we do that as an additional step.
    if ( fixupDtor )
    {
        bool needsDtor = false;
        if ( type->getParent() && TypeDetail::getDestructionFunction( type->getParent() ) )
        {
            needsDtor = true;
        }
        else if ( auto decls = TypeDetail::getDeclsArray( type ) )
        {
            for ( auto decl : decls->getDataFields() )
            {
                if ( TypeDetail::getDestructionFunction( decl.getType() ) )
                {
                    needsDtor = true;
                    break;
                }
            }
        }
        TypeDetail::localSetOptional<Opt::DESTRUCTOR>( type, needsDtor ? &Detail::Implicit<Opt::DESTRUCTOR>::func : HK_NULL);
    }
}

_Ret_notnull_ const hkReflect::Type* hkReflect::TypeDetail::getUndecorated(_In_ const Type* type )
{
    type = skipDecorators( type );
    if ( auto tpl = type->getTemplate() )
    {
        if ( auto resolved = tpl->getUndecoratedType() )
        {
            return resolved;
        }
    }
    return type;
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::Detail::builtinFromFormat( hkReflect::Format::Value format )
{
    return BuiltinTypeReg::getInstance().builtinFromFormat( format.get() );
}

hkResult hkReflect::Detail::coerceArg(const hkReflect::Var& dst, const hkReflect::Var& src)
{
    if (hkReflect::Detail::copyObject(dst, src).isSuccess())
    {
        return HK_SUCCESS;
    }
    else if (auto asPtr = dst.getType()->asPointer())
    {
        if (isPointerTo(dst.getType(), src.getType()))
        {
            // Use reflect construction rather than default since T& is not default constructible.
            if (hkReflect::TypeDetail::reflectConstruct(dst.getAddress(), dst.getType()).isSuccess())
            {
                return hkReflect::PointerVar(dst).setValue(src);
            }
        }
        else if (auto* varRef = dst.dynCast< hkReflect::Detail::Reference< const hkReflect::Var > >())
        {
            // "const hkReflect::Var&". Set the var pointer
            *varRef = src;
            return HK_SUCCESS;
        }
        
    }
    else
    {
        hkReflect::ArrayVar dstArr = dst;
        hkReflect::ArrayVar srcArr = src;
        if (dstArr && srcArr && dstArr.getSubType() && srcArr.getSubType() &&
            dstArr.getSubType()->equals(srcArr.getSubType()) &&
            hkString::strCmp(dstArr.getType()->getName(), "hkArrayView") == 0) 
        {
            auto view = static_cast< hkArrayView<void>* >(dst.getAddress());
            hkReflect::ArrayValue val = srcArr.getValue();
            *view = hkArrayView<void>(val.getAddress(), val.getEndAddress());
            return HK_SUCCESS;
        }
    }
    return HK_FAILURE;
}

hkResult hkReflect::Detail::coerceArgIfNecessary(_In_ const hkReflect::Type* dstType, const hkReflect::Var& src, hkReflect::Var& dst, hkReflect::Any& buf)
{
    if (src.getType()->equals(dstType))
    {
        dst = src;
        return HK_SUCCESS;
    }
    else
    {
        buf.allocateOnly(dstType);
        if (coerceArg(buf.var(), src).isSuccess())
        {
            dst = buf.var();
            return HK_SUCCESS;
        }
        else
        {
            buf.deallocateOnly();
            return HK_FAILURE;
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
