// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// This file contains POD structures which match non-POD reflection structures.
// They are used to enable us to cheaply statically initialize lots of static reflection data.

namespace hkReflect
{
    namespace Detail
    {
        struct CallableSignature;
        class InheritanceInfo;
        class Functions;

        struct TypePod
        {
            hkUlong m_bits;
            TypePod* m_parent;
            hkUlong m_optional[1];
        };

            /// This is used for the debug visualizer
        struct NameOfTypePod
        {
            hkUlong m_bits;
            TypePod* m_parent;
            hkUlong m_optional[1];
        };

        /// Very low-level function-call interface. hkReflect::Callable is the higher level interface.
        struct HK_EXPORT_COMMON CallInvoker
        {
            /// A pointer to a function that can call a method with a particular signature.
            typedef void(*InvokeFunc)(void* thisPtr, void* packedArgs, void* returnBuffer, const void* invokerdata);

            HK_INLINE void invoke(void* thisPtr, void* packedArgs, void* returnBuffer) const
            {
                m_func(thisPtr, packedArgs, returnBuffer, m_data);
            }
            /// The function which knows how to call certain functions of the kind described by the signature.
            InvokeFunc m_func HK_ATTR(hk::Reflect(false));
            /// In the case of native methods, this is the address of the method pointer.
            /// In the case of native functions, this is the function pointer itself.
            /// In the case of native constructors, it isn't used.
            const void* m_data;

            bool operator==(const CallInvoker& c) const { return m_func == c.m_func && m_data == c.m_data; }
        };

        struct NamedCallablePod
        {
            CallInvoker m_invoker;
            const hkReflect::Detail::CallableSignature* m_signature;
            const hkReflect::Type* m_objectType;
            const char* const * m_paramNames;
            const char* m_name;
            const hkUlong* m_attributes;
            const void* const* m_defaultArgs;

            const hkReflect::Callable* cast() const { return reinterpret_cast<const hkReflect::Callable*>(this); }
        };

        struct FunctionsPod
        {
            HK_DECLARE_PLACEMENT_ALLOCATOR();
            const NamedCallablePod* m_methodInfo;
            hkUint16 m_numMethods;
            hkUint16 m_numConstructors;
            hkUint16 m_numFunctions;

            const hkReflect::Detail::Functions* cast() const { return reinterpret_cast<const hkReflect::Detail::Functions*>(this); }
            hkReflect::Detail::Functions* cast() { return reinterpret_cast<hkReflect::Detail::Functions*>(this); }
        };

            /// A field inside the record.
            /// This structure should naturally be defined inside hkRecord, but that is not possible
            /// as all reflected classes need a static array of these structures in order to
            /// access protected member offsets while building these hkRecordFields. Some of these
            /// classes are in hkBases.h and we cannot include this file before hkBases.h.
            /// In C++ it's not possible to forward declare nested structures.
        struct RecordFieldPod
        {
            HK_DECLARE_CLASS(RecordFieldPod, New, Pod );
            const hkReflect::Type* m_type;
        };

        typedef hkUlong TypeDataMax[32+2];


        template<int NUM_OPT>
        struct TypeDataN
        {
            hkUlong m_bits;
            const hkReflect::Type* m_parent;
            hkUlong m_opts[NUM_OPT];
        };


        // Utility class for the common case of an inplace array
        // which has a size N, directly followed by N elements
        template<typename HEAD, typename ITEM, int NUM>
        struct FixedArrayStorage
        {
            typedef ITEM Item;
            HEAD m_head; // ulong not int, to keep alignment
            ITEM m_items[NUM]; // variable length array
        };
        template<typename HEAD, typename ITEM>
        struct FixedArrayStorage<HEAD, ITEM,0>
        {
            typedef ITEM Item;
            HEAD m_head;
        };
        // This is the "inteface" class, the previous two are for storage
        template<typename HEAD, typename TYPE, typename ITEM>
        struct FixedArray
        {
            static int sizeOfArrayN(int num) { return hkLosslessCast<int>( sizeof(FixedArrayStorage<HEAD,ITEM,0>) + num * sizeof(ITEM) ); }
            static TYPE* createInPlace(void* addr, int num) { FixedArrayStorage<HEAD,ITEM,0>* a = reinterpret_cast<FixedArrayStorage<HEAD,ITEM,0>*>(addr); a->m_head = num; return reinterpret_cast<TYPE*>(a); }

            hkArrayView<const ITEM> items() const   { return hkArrayView<const ITEM>(m_store.m_items, m_store.m_head); }
            hkArrayView<ITEM> items()               { return hkArrayView<ITEM>(m_store.m_items, m_store.m_head); }
            const ITEM& item(int i) const           { return m_store.m_items[i]; }
            ITEM& item(int i)                       { return m_store.m_items[i]; }
        protected:
            FixedArrayStorage<HEAD,ITEM,1> m_store;
        };



        /// A set of attributes on an hkRecord or hkRecordField
        /// This structure is an aggregate used in we have attributes declared on an hkRecord
        /// or hkRecordField which result in some output being generated inside the reflection
        /// .cpp file. A pointer to an object of this structure is saved in the hkRecord
        /// or hkRecordField in case the entity has attributes producing C++ output.
        struct AttributeItem
        {
            template<typename T> void init(const T* t) { data = t; type = hkReflect::getType<T>(); }
            const void* data;
            const hkReflect::Type* type;
        };

        struct AttributeArray : public FixedArray<hkUlong, AttributeArray, AttributeItem>
        {
        };


        //
        struct HK_EXPORT_COMMON DeclsArray
        {
            struct Head
            {
                hkInt16 m_numDataFields;
                hkInt16 m_numPropertyFields;
                hkInt16 m_numTotalDecls;
                hkInt16 m_padding;
            };
            static _Ret_notnull_ DeclsArray* create(_In_reads_opt_((numFields + numProps) * sizeof(void*)) const void* fieldTypes, int numFields, int numProps, hkMemoryAllocator& alloc);
            void deallocate(hkMemoryAllocator& m);
            int getNumDataFields() const { return m_store.m_head.m_numDataFields; }
            int getNumPropertyFields() const { return m_store.m_head.m_numPropertyFields; }
            int getNumFields() const { return m_store.m_head.m_numDataFields + m_store.m_head.m_numPropertyFields; }
            int getNumDecls() const { return m_store.m_head.m_numTotalDecls; }
            const Decl& getDecl(int i) const { return getAllDecls()[i]; }
            const FieldDecl& getField(int i) const { return getFields()[i]; }
            void setField(int i, const hkReflect::Type* t) { m_store.m_items[i] = t; }
            hkArrayView<const FieldDecl> getFields() const { return hkArrayViewT::make( reinterpret_cast<const FieldDecl*>(m_store.m_items), m_store.m_head.m_numDataFields + m_store.m_head.m_numPropertyFields); }
            hkArrayView<const DataFieldDecl> getDataFields() const { return hkArrayViewT::make(reinterpret_cast<const DataFieldDecl*>(m_store.m_items), m_store.m_head.m_numDataFields); }
            hkArrayView<const Decl> getAllDecls() const { return hkArrayViewT::make(reinterpret_cast<const Decl*>(m_store.m_items), m_store.m_head.m_numTotalDecls); }
            void reset(int numFields, int numProps);
        private:
            FixedArrayStorage<Head, const Type*,1> m_store;
        };

        struct TemplateParameter
        {
            bool isType() const { HK_ASSERT_NO_MSG(0x2c12a5ab, m_kindAndName[0] == 't' || m_kindAndName[0] == 'v'); return m_kindAndName[0] == Template::KIND_TYPE; }
            bool isValue() const { HK_ASSERT_NO_MSG(0xa6050ff, m_kindAndName[0] == 't' || m_kindAndName[0] == 'v'); return m_kindAndName[0] == Template::KIND_VALUE; }

            _Ret_notnull_ const hkReflect::Type* getAsType() const { HK_ASSERT_NO_MSG(0x34487fe7, isType()); return reinterpret_cast<const hkReflect::Type*>(m_storage); }
            hkUlong getAsValue() const { HK_ASSERT_NO_MSG(0x73727836, isValue()); return m_storage; }
            _Ret_z_ const char* getName() const { HK_ASSERT_NO_MSG(0x2e448dfe, m_kindAndName[0] == 't' || m_kindAndName[0] == 'v'); return m_kindAndName + 1; }

            void setInternal(_In_z_ const char* k, hkUlong u) { m_kindAndName = k; m_storage = u; }
            void setTypeParam(_In_ const hkReflect::Type* t) { m_storage = hkUlong(t); }

            hkUlong m_storage;
            const char* m_kindAndName; // The name of the parameter in the template (T, TYPE etc). m_kindAndName[0] == t or v, m_kindAndName+1 == name
        };

        struct TemplateParameterArray
        {
            
            static int sizeOfArrayN( int num ) { return sizeof( const hkReflect::Type* ) + ArrayType::sizeOfArrayN( num ); }
            _Ret_notnull_ static TemplateParameterArray* createInPlace(_In_ void* addr, int num)
            {
                auto res = reinterpret_cast<TemplateParameterArray*>(addr);
                // Keep null in dynamic types.
                res->m_undecoratedType = HK_NULL;
                ArrayType::createInPlace( &res->m_array, num );
                return res;
            }
            const TemplateParameter& item( int i ) const { return m_array.item( i ); }
            TemplateParameter& item( int i ) { return m_array.item( i ); }

            
            const hkReflect::Type* m_undecoratedType;

            typedef FixedArray<hkUlong, TemplateParameterArray, TemplateParameter> ArrayType;
            ArrayType m_array;
        };

        template<int NUM>
        struct TemplatePod
        {
            
            const hkReflect::Type* m_undecoratedType;

            FixedArrayStorage<hkUlong, TemplateParameter, NUM> m_array;
        };

        // Used to store information about interfaces in case of multiple inheritance.
        class HK_EXPORT_COMMON Interface
        {
        public:
            const Type* m_interfaceType;
            hkLong m_offset;
        };

        struct InterfaceArray : public FixedArray < hkUlong, InterfaceArray, Interface >
        {
        };

        // Wrapper functions for types special methods.
        typedef void (HK_CALL *UnaryFunction)(void* addr, const hkReflect::Type* type, int num);
        typedef void (HK_CALL *BinaryFunction)(void* target, const void* source, const hkReflect::Type* type, int num);
        typedef void (HK_CALL *AfterReflectNewFunction)(void* addr);
        typedef bool (HK_CALL *ValidateFunction)(const void* addr);

        /// Function type to dynamically get the type from an object.
        typedef AddrAndType (HK_CALL *ExactTypeFunction)(const void*);

        /// Storage types for optionals, needed for basic type checking in generated reflection.
        /// Impls have void* rather than the correct type because some of them are out-of-line.
        template<hkReflect::Optional OPTION> struct OptionStorage {};

        template<> struct OptionStorage < Opt::FORMAT > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::FORMAT }; };
        template<> struct OptionStorage < Opt::SUBTYPE > { typedef const Type* Storage; enum { BitIndex = Opt::Bits::SUBTYPE }; };
        template<> struct OptionStorage < Opt::IMPL > { typedef const hkReflect::Detail::Impl* Storage; enum { BitIndex = Opt::Bits::IMPL }; };
        template<> struct OptionStorage < Opt::NAME > { typedef const char* Storage; enum { BitIndex = Opt::Bits::NAME }; };
        template<> struct OptionStorage < Opt::VERSION > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::VERSION }; };
        template<> struct OptionStorage < Opt::ATTRIBUTES > { typedef const hkReflect::Detail::AttributeArray* Storage; enum { BitIndex = Opt::Bits::ATTRIBUTES }; };
        template<> struct OptionStorage < Opt::DEF_CONSTRUCTOR > { typedef UnaryFunction Storage; enum { BitIndex = Opt::Bits::DEF_CONSTRUCTOR }; };
        template<> struct OptionStorage < Opt::COPY_CONSTRUCTOR > { typedef BinaryFunction Storage; enum { BitIndex = Opt::Bits::COPY_CONSTRUCTOR }; };
        template<> struct OptionStorage < Opt::DESTRUCTOR > { typedef UnaryFunction Storage; enum { BitIndex = Opt::Bits::DESTRUCTOR }; };
        template<> struct OptionStorage < Opt::COPY_ASSIGNMENT > { typedef BinaryFunction Storage; enum { BitIndex = Opt::Bits::COPY_ASSIGNMENT }; };
        template<> struct OptionStorage < Opt::FUNCTIONS > { typedef const hkReflect::Detail::Functions* Storage; enum { BitIndex = Opt::Bits::FUNCTIONS }; };
        template<> struct OptionStorage < Opt::EXACT_TYPE > { typedef ExactTypeFunction Storage; enum { BitIndex = Opt::Bits::EXACT_TYPE }; };
        template<> struct OptionStorage < Opt::MESSAGING > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::MESSAGING }; };
        template<> struct OptionStorage < Opt::DEFAULT > { typedef const void* Storage; enum { BitIndex = Opt::Bits::DEFAULT }; };
        template<> struct OptionStorage < Opt::INHERITANCE > { typedef hkReflect::Detail::InheritanceInfo Storage; enum { BitIndex = Opt::Bits::INHERITANCE }; };
        template<> struct OptionStorage < Opt::POINTER_IMPL > { typedef const hkReflect::Detail::PointerImpl* Storage; enum { BitIndex = Opt::Bits::POINTER_IMPL }; };
        template<> struct OptionStorage < Opt::INTERFACES > { typedef const InterfaceArray* Storage; enum { BitIndex = Opt::Bits::INTERFACES }; };
        template<> struct OptionStorage < Opt::TEMPLATE > { typedef const hkReflect::Template* Storage; enum { BitIndex = Opt::Bits::TEMPLATE }; };
        template<> struct OptionStorage < Opt::DECL_NAME > { typedef const char* Storage; enum { BitIndex = Opt::Bits::DECL_NAME }; };
        template<> struct OptionStorage < Opt::DECL_FORMAT > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::DECL_FORMAT }; };
        template<> struct OptionStorage < Opt::DECL_CONTEXT > { typedef const hkReflect::Type* Storage; enum { BitIndex = Opt::Bits::DECL_CONTEXT }; };
        template<> struct OptionStorage < Opt::ALLOC_IMPL > { typedef const hkReflect::Detail::AllocationImpl* Storage; enum { BitIndex = Opt::Bits::ALLOC_IMPL }; };
        template<> struct OptionStorage < Opt::SIZE_ALIGN > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::SIZE_ALIGN }; };
        template<> struct OptionStorage < Opt::FLAGS > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::FLAGS }; };
        template<> struct OptionStorage < Opt::REFLECT_CONSTRUCTOR > { typedef UnaryFunction Storage; enum { BitIndex = Opt::Bits::REFLECT_CONSTRUCTOR }; };
        template<> struct OptionStorage < Opt::DECLS > { typedef const hkReflect::Detail::DeclsArray* Storage; enum { BitIndex = Opt::Bits::DECLS }; };
        template<> struct OptionStorage < Opt::AFTER_REFLECT_NEW > { typedef AfterReflectNewFunction Storage; enum { BitIndex = Opt::Bits::AFTER_REFLECT_NEW }; };
        template<> struct OptionStorage < Opt::ATTRIBUTE_STRING > { typedef const char* Storage; enum { BitIndex = Opt::Bits::ATTRIBUTE_STRING }; };
        template<> struct OptionStorage < Opt::TYPE_WORLD > { typedef const void* Storage; enum { BitIndex = Opt::Bits::TYPE_WORLD }; };
        template<> struct OptionStorage < Opt::VALIDATE > { typedef ValidateFunction Storage; enum { BitIndex = Opt::Bits::VALIDATE }; };

        // Compiled-in types are initialized statically and the types can be a bit different to the runtime types.
        template<hkReflect::Optional OPTION> struct OptionStaticInitializer : public OptionStorage<OPTION> {};
        template<> struct OptionStaticInitializer < Opt::IMPL > { typedef const void* Storage; enum { BitIndex = Opt::Bits::IMPL }; };
        template<> struct OptionStaticInitializer < Opt::FUNCTIONS > { typedef const hkReflect::Detail::FunctionsPod* Storage; enum { BitIndex = Opt::Bits::FUNCTIONS }; };
        template<> struct OptionStaticInitializer < Opt::INHERITANCE > { typedef hkUlong Storage; enum { BitIndex = Opt::Bits::INHERITANCE }; };
        template<> struct OptionStaticInitializer < Opt::POINTER_IMPL > { typedef const void* Storage; enum { BitIndex = Opt::Bits::POINTER_IMPL }; };
        template<> struct OptionStaticInitializer < Opt::TEMPLATE > { typedef const TemplateParameterArray* Storage; enum { BitIndex = Opt::Bits::TEMPLATE }; };
        template<> struct OptionStaticInitializer < Opt::DECL_CONTEXT > { typedef const void* Storage; enum { BitIndex = Opt::Bits::DECL_CONTEXT }; };
        template<> struct OptionStaticInitializer < Opt::ALLOC_IMPL > { typedef const void* Storage; enum { BitIndex = Opt::Bits::ALLOC_IMPL }; };

#define HK_REFLECT_TYPE_OPTIONAL(OPT, ...) (hkUlong)static_cast< hkReflect::Detail::OptionStaticInitializer<OPT>::Storage >(__VA_ARGS__)
// Create the value for Opt::SIZE_ALIGN. Note that REQALIGN is a log2 value.
#define HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(SIZE, ALIGN, REQALIGN) (unsigned(SIZE)|(unsigned(ALIGN)<<16)|(unsigned(REQALIGN)<<28))
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
