// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Builder/hkTypeCopier.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Memory/Allocator/Transient/hkTransientAllocator.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Container/String/hkStringPtr.h>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Container/PointerMap/hkPointerMap.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <limits.h>

#define DEBUG_LOG_IDENTIFIER "s11n.TypeCopier"
#include <Common/Base/System/Log/hkLog.hxx>

// Retargeting types happens at two levels
// 1. we set the size/align/endian/etc properties of the basic types.
// 2. we recompute the layout of structs which are composed of the basic types from 1

namespace
{
    struct CheckAlignDouble
    {
        char c;
        double d;
    };
    struct CheckPadBase
    {
        CheckPadBase() {}
        int i;
        char c;
    };
    struct CheckPad : public CheckPadBase
    {
        char c2;
    };

    HK_CLASSALIGN(struct,16) CheckAlign_Base
    {
        CheckAlign_Base() {}
        int ibase;
    };
    struct CheckAlign_0 : public CheckAlign_Base { CheckAlign_0(){}  };
    struct CheckAlign_1 : public CheckAlign_Base { CheckAlign_1() {} int i1; };
    struct CheckAlign_1_1 : public CheckAlign_1 { CheckAlign_1_1() {} int i11; };
    struct CheckAlign_0_1 : public CheckAlign_0 { CheckAlign_0_1() {} int i01; };

#if 0 // MSVC test
    HK_COMPILE_TIME_ASSERT(offsetof(CheckAlign_Base, ibase) == 0);
    HK_COMPILE_TIME_ASSERT(offsetof(CheckAlign_1, i1) == 4);
    HK_COMPILE_TIME_ASSERT(offsetof(CheckAlign_0_1, i01) == 16);
    HK_COMPILE_TIME_ASSERT(offsetof(CheckAlign_1_1, i11) == 16);
#endif

    static const bool hostLongIsPointerSize = sizeof(void*) == sizeof(long);
    static const bool hostAlignedDoubles = HK_OFFSET_OF(CheckAlignDouble, d) == sizeof(double);
        // true:fields start after the last field in the parent
        // false:fields start after sizeof(Parent)
    static const bool hostReusesParentPadding = sizeof(CheckPad) == sizeof(CheckPadBase);
        // Like a restricted version of reusesParentPadding which only kicks in for
        // types with explicit alignment on the struct (not the fields)
        // This only has effect if reusesParentPadding is false.
    static const bool hostReusesClassAlignPadding = HK_OFFSET_OF(CheckAlign_1,i1) == sizeof(int);

    // If you change hostAlignedDoubles, also change hkReflectType.cpp
    #if HK_POINTER_SIZE == 4
        HK_COMPILE_TIME_ASSERT(sizeof(void*) == 4);
        HK_COMPILE_TIME_ASSERT(hostLongIsPointerSize);
        #if defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_NX_WIN32)
            #define TARGET_ID "m"
            HK_COMPILE_TIME_ASSERT(hostAlignedDoubles);
            HK_COMPILE_TIME_ASSERT(hostReusesParentPadding == false);
            HK_COMPILE_TIME_ASSERT(hostReusesClassAlignPadding);
        #elif (defined(HK_PLATFORM_ANDROID) && defined(HK_ARCH_ARM)) || defined(HK_PLATFORM_PSVITA) || ( defined(HK_PLATFORM_NX) && !defined(HK_PLATFORM_NX_WIN32) && !defined(HK_PLATFORM_NX_X64) )
            #define TARGET_ID "a"
            HK_COMPILE_TIME_ASSERT(hostAlignedDoubles);
            HK_COMPILE_TIME_ASSERT(hostReusesParentPadding);
            HK_COMPILE_TIME_ASSERT(hostReusesClassAlignPadding);
        #else // everything else
            #define TARGET_ID "g"
            HK_COMPILE_TIME_ASSERT(hostAlignedDoubles==false);
            HK_COMPILE_TIME_ASSERT(hostReusesParentPadding);
            HK_COMPILE_TIME_ASSERT(hostReusesClassAlignPadding);
        #endif
    #else
        HK_COMPILE_TIME_ASSERT(sizeof(void*) == 8);
        HK_COMPILE_TIME_ASSERT(hostAlignedDoubles);
        #if defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_NX_X64)
            #define TARGET_ID "m"
            HK_COMPILE_TIME_ASSERT(hostLongIsPointerSize==false);
            HK_COMPILE_TIME_ASSERT(hostReusesParentPadding==false);
            HK_COMPILE_TIME_ASSERT(hostReusesClassAlignPadding);
        #else // everything else
            #define TARGET_ID "g"
            HK_COMPILE_TIME_ASSERT(hostLongIsPointerSize);
            HK_COMPILE_TIME_ASSERT(hostReusesParentPadding);
            HK_COMPILE_TIME_ASSERT(hostReusesClassAlignPadding);
        #endif
    #endif
}

// Ensure the list in hkReflectType.cpp POD_INTEGER_TYPE matches
#define FOREACH_BUILTIN_TYPE(ACTION) \
    ACTION(bool, bool) \
    ACTION(char, char) \
    ACTION(signed char, signed_char) \
    ACTION(unsigned char, unsigned_char) \
    ACTION(short, short) \
    ACTION(unsigned short, unsigned_short) \
    ACTION(int, int) \
    ACTION(unsigned int, unsigned_int) \
    ACTION(long, long) \
    ACTION(unsigned long, unsigned_long) \
    ACTION(long long, long_long) \
    ACTION(unsigned long long, unsigned_long_long) \
    ACTION(float, float) \
    ACTION(double, double)
#define FOREACH_TYPEDEF_TYPE(ACTION) \
    ACTION(hkReal, hkReal) \
    ACTION(hkUlong, hkUlong) \
    ACTION(hkLong, hkLong) \
    ACTION(hkUintReal, hkUintReal)

namespace
{
    struct BuiltinValueType : public hkReflect::ValueType
    {
        BuiltinValueType()
            : m_sizeAlign(0)
        {
            m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::TYPE_WORLD;
            m_parent = HK_NULL;
            m_format = hkReflect::Format::OfVoid::Value;
            m_impl = HK_NULL;
            m_name = HK_NULL;
            m_typeWorld = HK_NULL;
        }
        void setParent(_In_opt_ hkReflect::Type* t) { m_parent = t; }
        void init(int f, _In_ const hkReflect::Detail::ValueImpl* impl, int s, int a, _In_ const void* typeWorld)
        {
            m_format = f;
            m_impl = impl;
            m_sizeAlign = hkReflect::Detail::SizeAlign(s, a, hkReflect::Detail::ALIGN_REQ_NATURAL).asUlong();
            m_typeWorld = typeWorld;
        }

        hkUlong m_format;
        const hkReflect::Detail::ValueImpl* m_impl;
        const char* m_name;
        hkUlong m_sizeAlign;
        const void* m_typeWorld;
    };

    struct BuiltinTypedefType : public hkReflect::ValueType
    {
        BuiltinTypedefType()
        {
            m_optional = hkReflect::Opt::NAME | hkReflect::Opt::TYPE_WORLD;
            m_parent = HK_NULL;
            m_name = HK_NULL;
            m_typeWorld = HK_NULL;
        }
        void setParent(_In_ BuiltinValueType* t)
        {
            m_parent = t;
            m_typeWorld = t->m_typeWorld;
        }
        const char* m_name;
        const void* m_typeWorld;
    };
}

struct hkReflect::TypeCopier::Pimpl
{
    HK_DECLARE_CLASS(Pimpl, New);

    typedef hkTuple<Type*, const Type*> TypePair;

    Pimpl(const Options& opts)
        : m_typesAllocator(hkMemHeapAllocator())
    {
        HK_ASSERT_NO_MSG(0x54e13ad6, opts.m_pointerSize == 4 || opts.m_pointerSize == 8);
        HK_ASSERT_NO_MSG(0x11cab6b7, opts.m_minifyFormat != -1);
        HK_ASSERT_NO_MSG(0x375fa63b, opts.m_littleEndian != -1);

        HK_ASSERT_NO_MSG(0x62560a5d, opts.m_longIsPointerSize != -1);
        HK_ASSERT_NO_MSG(0x2f15c0a5, opts.m_alignedDoubles != -1);
        HK_ASSERT_NO_MSG(0x6578137d, opts.m_reusesParentPadding != -1);

        HK_ASSERT_NO_MSG(0x60ec20cf, opts.m_includeProperties != -1);
        HK_ASSERT_NO_MSG(0x5f0dd137, opts.m_pruneNonSerializableFields != -1);
        HK_ASSERT_NO_MSG(0x7ab265e0, opts.m_realIsDouble != -1);

        m_status = HK_SUCCESS;

        m_pruneNonSerializableFields = opts.m_pruneNonSerializableFields;
        m_includeProperties = opts.m_includeProperties;
        m_minifyFormat = opts.m_minifyFormat;

        m_record.m_emptyBaseClassOptimization = true;
        m_record.m_pointerSize = opts.m_pointerSize;
        m_record.m_sizeofReal = opts.m_realIsDouble ? 8 : 4;
        m_record.m_reuseParentPadding = opts.m_reusesParentPadding;
        m_record.m_reuseClassAlignPadding = opts.m_reusesClassAlignPadding;

        // set names
        {
            #define SWITCH(A,B) m_builtins.m_##B.m_name = #A;
            FOREACH_BUILTIN_TYPE(SWITCH)
            FOREACH_TYPEDEF_TYPE(SWITCH)
            #undef SWITCH
        }

        // initialize format and parent
        const int alignOfDouble = opts.m_alignedDoubles ? 8 : 4;
        const int bigEndianFormatBit = (1 << 8);
        // 1 byte values have no endianness
        HK_COMPILE_TIME_ASSERT( int(Format::OfBoolGeneric<0, 8>::Value) == int(Format::OfBoolGeneric<1, 8>::Value) );
        HK_COMPILE_TIME_ASSERT( int(Format::OfIntGeneric<0, 0, 8>::Value) == int(Format::OfIntGeneric<1, 0, 8>::Value) );
        // >1 byte values differ only by a bit
        HK_COMPILE_TIME_ASSERT((Format::OfBoolGeneric<0, 16>::Value ^ Format::OfBoolGeneric<1, 16>::Value) == bigEndianFormatBit);
        HK_COMPILE_TIME_ASSERT((Format::OfIntGeneric<0, 0, 16>::Value ^ Format::OfIntGeneric<1, 0, 16>::Value) == bigEndianFormatBit);
        HK_COMPILE_TIME_ASSERT((Format::OfFloatGeneric<0, 1, 1, 8, 23>::Value ^ Format::OfFloatGeneric<1, 1, 1, 8, 23>::Value) == bigEndianFormatBit);
        const int endianBit = opts.m_littleEndian ? 0 : bigEndianFormatBit;
        {
            m_builtins.m_bool.init(Format::OfBoolGeneric<0, 8>::Value, &hkReflect::Detail::BoolImplN<bool>::s_instance, 1, 1, this);

            m_builtins.m_char.init(Format::OfIntGeneric< 0, 0, 8>::Value, &hkReflect::Detail::IntImplN<char>::s_instance, 1, 1, this); // Reflected "char" is unsigned
            m_builtins.m_signed_char.init(Format::OfIntGeneric< 0, 1, 8>::Value, &hkReflect::Detail::IntImplN<char>::s_instance, 1, 1, this);
            m_builtins.m_unsigned_char.init(Format::OfIntGeneric<0, 0, 8>::Value, &hkReflect::Detail::IntImplN<unsigned char>::s_instance, 1, 1, this);
            m_builtins.m_short.init(endianBit | Format::OfIntGeneric<0, 1, 16>::Value, &hkReflect::Detail::IntImplN<short>::s_instance, 2, 2, this);
            m_builtins.m_unsigned_short.init(endianBit | Format::OfIntGeneric<0, 0, 16>::Value, &hkReflect::Detail::IntImplN<unsigned short>::s_instance, 2, 2, this);
            m_builtins.m_int.init(endianBit | Format::OfIntGeneric<0, 1, 32>::Value, &hkReflect::Detail::IntImplN<int>::s_instance, 4, 4, this);
            m_builtins.m_unsigned_int.init(endianBit | Format::OfIntGeneric<0, 0, 32>::Value, &hkReflect::Detail::IntImplN<unsigned int>::s_instance, 4, 4, this);
            m_builtins.m_long_long.init(endianBit | Format::OfIntGeneric<0, 1, 64>::Value, &hkReflect::Detail::IntImplN<long long>::s_instance, 8, alignOfDouble, this);
            m_builtins.m_unsigned_long_long.init(endianBit | Format::OfIntGeneric<0, 0, 64>::Value, &hkReflect::Detail::IntImplN<unsigned long long>::s_instance, 8, alignOfDouble, this);

            const int sizeOfLong = (opts.m_minifyFormat || (opts.m_pointerSize == 8 && opts.m_longIsPointerSize)) ? 8 : 4;
            if(sizeOfLong == 4)
            {
                m_builtins.m_long.init(endianBit | Format::OfIntGeneric<0, 1, 32>::Value, &hkReflect::Detail::IntImplN<hkInt32>::s_instance, 4, 4, this);
                m_builtins.m_unsigned_long.init(endianBit | Format::OfIntGeneric<0, 0, 32>::Value, &hkReflect::Detail::IntImplN<hkUint32>::s_instance, 4, 4, this);
            }
            else // 8
            {
                m_builtins.m_long.init(endianBit | Format::OfIntGeneric<0, 1, 64>::Value, &hkReflect::Detail::IntImplN<hkInt64>::s_instance, 8, alignOfDouble, this);
                m_builtins.m_unsigned_long.init(endianBit | Format::OfIntGeneric<0, 0, 64>::Value, &hkReflect::Detail::IntImplN<hkUint64>::s_instance, 8, alignOfDouble, this);
            }

            m_builtins.m_float.init(endianBit | Format::OfFloatGeneric<0, 1, 1, 8, 23>::Value, &hkReflect::Detail::FloatImplN<float>::s_instance, 4, 4, this);
            m_builtins.m_double.init(endianBit | Format::OfFloatGeneric<0, 1, 1, 11, 52>::Value, &hkReflect::Detail::FloatImplN<double>::s_instance,8, alignOfDouble, this);

            m_builtins.m_hkReal.setParent(opts.m_realIsDouble ? &m_builtins.m_double : &m_builtins.m_float);
            m_builtins.m_hkUintReal.setParent(opts.m_realIsDouble ? &m_builtins.m_unsigned_long_long : &m_builtins.m_unsigned_int);

            m_builtins.m_hkLong.setParent(((opts.m_pointerSize == 8) || opts.m_minifyFormat) ? &m_builtins.m_long_long : &m_builtins.m_long);
            m_builtins.m_hkUlong.setParent(((opts.m_pointerSize == 8) || opts.m_minifyFormat) ? &m_builtins.m_unsigned_long_long : &m_builtins.m_unsigned_long);
        }

        // register diversions
        {
            #define SWITCH(A,B) m_diversions.insert(#A, &m_builtins.m_##B);
            FOREACH_BUILTIN_TYPE(SWITCH)
            FOREACH_TYPEDEF_TYPE(SWITCH)
            #undef SWITCH
        }
    }

        // Only call this one from outside - internal calls copy()
    _Ret_maybenull_ hkReflect::Type* copyEntryPoint(_In_ const hkReflect::Type* origType, _Inout_opt_ hkArray<TypePair>* copiesOut)
    {
        hkArray<TypePair> copies;
        hkReflect::Type* c = copy(origType, copies);

        if(m_status.isFailure())
        {
            m_status = HK_SUCCESS;
            return HK_NULL;
        }

        for(int i = 0; i < copies.getSize(); ++i)
        {
            RecordLayout::recompute(copies[i].m_0, m_record);
        }
        if(copiesOut)
        {
            copiesOut->append(copies);
        }
        return c;
    }

    inline int getPointerSize() const
    {
        return m_record.m_pointerSize;
    }

    inline int getSizeofReal() const
    {
        return m_record.m_sizeofReal;
    }

protected:

    _Ret_maybenull_ hkReflect::Type* copy(_In_opt_ const hkReflect::Type* origType, hkArray<TypePair>& copiesOut)
    {
        if(origType == HK_NULL || m_status.isFailure())
        {
            return HK_NULL;
        }
        if(m_minifyFormat == false && origType->isRetargetable()==false) //packfile && !targetable
        {
            Log_Warning("Cannot copy {}@{}, it is not retargetable", origType->getFullName(), origType);
            m_status = HK_FAILURE;
            return HK_NULL;
        }

        if(hkReflect::Type* t = m_copyFromOrig.getWithDefault(origType, HK_NULL))
        {
            return t;
        }
        else if (hkReflect::Detail::isOpaqueWrapper(origType))
        {
            // Map all tracker Opaque wrappers on hkReflect::Detail::Opaque so that they are never serialized.
            
            return copy(hkReflect::getType<hkReflect::Detail::Opaque>(), copiesOut);
        }
        else
        {
            
            const hkReflect::Type* alias = HK_NULL;
            if(hkReflect::TypeDetail::localHasOptional(origType, hkReflect::Opt::NAME))
            {
                if(hkReflect::TypeDetail::getInheritance(origType) && !origType->isDynamicType())
                {
                    const hkReflect::Type* canon = hkReflect::typeFromType(origType);
                    if(canon && canon != origType)
                    {
                        if(hkReflect::Type* tc = m_copyFromOrig.getWithDefault(canon, HK_NULL))
                        {
                            m_copyFromOrig.insert(origType, tc);
                            return tc;
                        }
                        else
                        {
                            alias = origType;
                            origType = canon;
                        }
                    }
                }

                if(hkReflect::Type* divert = m_diversions.getWithDefault(origType->getName(), HK_NULL))
                {
                    m_copyFromOrig.insert(origType, divert); // cache it for next time
                    copiesOut.pushBack(TypePair(divert, origType));
                    return divert;
                }
            }

            hkReflect::Type* c = makeNewCopy(origType, alias, copiesOut);
            copiesOut.pushBack(TypePair(c, origType));
            return c;
        }
    }


    _Ret_notnull_ hkReflect::Type* allocateBody(_In_ const hkReflect::Type* origType)
    {
        // We need to be careful of recursion. Consider struct X { X* m_x; };
        // Thus we do it in passes.
        // * Allocate a copy of X and put it in the map so recursive lookups work.
        // * Then recurse for referenced types such as parent, pointer/array element type etc.
        // * Finally compute the layout.

        using namespace hkReflect;
        // size/align and impl are stripped from all the decorators
        const int cloneMask = (Opt::FORMAT|Opt::SUBTYPE|Opt::NAME|Opt::VERSION|Opt::TEMPLATE|Opt::FLAGS|Opt::SIZE_ALIGN|Opt::DECLS|Opt::ATTRIBUTE_STRING)
            | (m_minifyFormat == false ? Opt::INTERFACES : 0)
            | (origType->getKind() != KIND_VOID && origType->getKind() != KIND_OPAQUE ? Opt::IMPL : 0);

        TypeBuilder builder;
        builder.beginShallowClone(origType, cloneMask);
        builder.setTypeWorld(this);
        // Prevents type world assert, will fixup later
        builder.overrideParent(HK_NULL);

        // Why is this here and not in fillBody with the rest of the extra stuff?
        // It is because the builder is destroyed at the end of this block and it's easier to do this now.
        // The type needs to be allocated and put in the map for circular types to work correctly.
        // e.g. consider struct Foo { Foo* m_ptr; }
        if( const char* name = TypeDetail::localGetOptional<Opt::NAME>(origType) )
        {
            const char* block = hkString::strDup(name, m_typesAllocator);
            builder.setItem<Opt::NAME>( block ); // Need to preserve this string
        }

        if( TypeDetail::localHasOptional(origType, Opt::TEMPLATE) )
        {
            const Template* origParams = origType->getTemplate();
            for(int i=0; i < origParams->getNumParams(); i++)
            {
                const Template::Parameter* origParam = origParams->getParam(i);
                if(origParam->isType())
                {
                    // Copying the type now might lead to infinite recursion, so add a placeholder.
                    builder.addTypeTemplateParam( HK_NULL, origParam->getName() );
                }
                else
                {
                    builder.addValueTemplateParam( origParam->getAsValue(), origParam->getName() );
                }
            }
        }
        if( m_minifyFormat == false) // interfaces are only relevant for packfile generation
        {
            if(const hkReflect::Detail::InterfaceArray* interfaceArray = TypeDetail::localGetOptional<Opt::INTERFACES>(origType))
            {
                hkArrayView<const hkReflect::Detail::Interface> ifaces = interfaceArray->items();
                for(int i = 0; i < ifaces.getSize(); ++i)
                {
                    builder.addInterface(HK_NULL, 0); // placeholder, copy later
                }
            }
        }
        if(m_pruneNonSerializableFields) // vtables shouldn't show up in the layout if we're pruning
        {
            if(TypeDetail::localHasOptional(origType, Opt::FLAGS))
            {
                builder.setItem<Opt::FLAGS>(TypeDetail::localGetOptional<Opt::FLAGS>(origType) & ~hkReflect::Type::TYPE_OWN_VTABLE);
            }
        }
        hkReflect::Type* ret = builder.allocate(&m_typesAllocator/*, true*/);
        DLOG("Copy is {}@{}", ret->getFullName(), ret);
        DLOG("Copy copyOpt=0x{:x} from origOpt=0x{:x}",
            hkReflect::TypeDetail::localGetOptionalMask(ret),
            hkReflect::TypeDetail::localGetOptionalMask(origType));
        return ret;
    }

    void fillBody(_In_ hkReflect::Type* copyType, _In_ const hkReflect::Type* origType, hkArray<TypePair>& copiesOut)
    {
        using namespace hkReflect;

        if( const Type* parent = origType->getParent() )
        {
            TypeDetail::setParent( copyType, copy(parent,copiesOut) );
        }

        hkUlong copyOptional = TypeDetail::localGetOptionalMask(copyType);
        if(copyOptional & Opt::IMPL)
        {
            const Detail::Impl* impl = HK_NULL;
            hkUlong format = hkReflect::TypeDetail::localGetOptional<Opt::FORMAT>(origType);
            switch( origType->getKind() )
            {
                case KIND_VOID:
                case KIND_OPAQUE:
                {
                    HK_UNREACHABLE(0x1abfe0c, "Void and Opaque types should not have an Impl");
                }
                case KIND_BOOL:
                {
                    if ( m_minifyFormat )
                    {
                        // Force bool to 1 byte.
                        hkReflect::TypeDetail::localSetOptional<Opt::FORMAT>( copyType, Format::OfBool<hkBool>::Value );
                        impl = &Detail::BoolImplN<hkUint8>::s_instance;
                        break;
                    }
                    // else fall through
                }
                case KIND_INT:
                case KIND_FLOAT:
                {
                    // Use impl correspondint to original format.
                    const hkReflect::Type* t = hkReflect::Detail::builtinFromFormat(hkReflect::Format::Value::create(hkUint32(format)));
                    HK_ASSERT_NO_MSG(0x1dd071ea,t);
                    impl = t->getImpl();
                    break;
                }
                case KIND_STRING:
                case KIND_POINTER:
                case KIND_RECORD:
                case KIND_ARRAY:
                {
                    // These are handled specially during serialization and don't need an Impl.
                    break;
                }
                default:
                    break;
            }
            TypeDetail::localSetOptional<Opt::IMPL>(copyType, impl );
        }

        if (copyOptional & Opt::TEMPLATE)
        {
            const Template* origParams = origType->getTemplate();
            if (int numParams = origParams->getNumParams())
            {
                // fixup template type arguments
                const hkReflect::Template* copyParams = TypeDetail::localGetOptional<Opt::TEMPLATE>(copyType);
                for (int i = 0; i < numParams; ++i)
                {
                    const hkReflect::Template::Parameter* origParam = origParams->getParam(i);
                    if(origParam->isType())
                    {
                        hkReflect::Type* typeCopy = copy(origParam->getAsType(), copiesOut);
                        HK_ASSERT_NO_MSG(0x70361bff, typeCopy);
                        ((Detail::TemplateParameter*)copyParams->getParam(i))->setTypeParam(typeCopy);

                        //copyParams->m_params[i].m_kindAndName =  hkString::strDup(origParam->m_kindAndName, m_typesAllocator);
                    }
                    else
                    {
                        //copyParams->m_params[i].m_storage = origParam->getAsValue();
                        //copyParams->m_params[i].m_kindAndName =  hkString::strDup(origParam->m_kindAndName, m_typesAllocator);
                    }
                }
            }
        }

        if(copyOptional & Opt::INTERFACES)
        {
            hkArrayView<const Detail::Interface> origInterfaces = TypeDetail::localGetOptional<Opt::INTERFACES>(origType)->items();
            if(int numIface = origInterfaces.getSize())
            {
                hkArrayView<Detail::Interface> copyInterfaces = const_cast<Detail::InterfaceArray*>(TypeDetail::localGetOptional<Opt::INTERFACES>(copyType))->items();
                for(int i = 0; i < numIface; ++i)
                {
                    copyInterfaces[i].m_interfaceType = copy(origInterfaces[i].m_interfaceType, copiesOut);
                }
            }
        }

        if(copyOptional & Opt::DECLS)
        {
            const Detail::DeclsArray* origFields = TypeDetail::getDeclsArray(origType);
            Detail::DeclsArray* fieldsArray = Detail::DeclsArray::create(
                HK_NULL, origFields->getNumDataFields(), origFields->getNumPropertyFields(), m_typesAllocator );

            int numSerializable = 0;
            for(int i = 0; i < origFields->getNumDecls(); i++)
            {
                const hkReflect::FieldDecl srcField = origFields->getField(i);
                // size/align and offsets are computed after
                if(srcField.getName())
                {
                    const hkReflect::Type* baseType = copy(srcField.getType()->getParent(), copiesOut);
                    const char* newName = hkString::strDup(srcField.getName(), m_typesAllocator);
                    TypeBuilder fieldBuilder;
                    fieldBuilder.setTypeWorld(this);
                    fieldBuilder.beginDerived(baseType);
                    if(hkUlong sal = TypeDetail::localGetOptionalUlong(srcField.getType(), Opt::SIZE_ALIGN))
                    {
                        fieldBuilder.addItem<Opt::SIZE_ALIGN>( Detail::SizeAlign::reqAlignOnly(sal) );
                    }

                    Decl::DeclFlags flags = srcField.getFlags();
                    if (m_pruneNonSerializableFields)
                    {
                        // Non-serializable fields can be excluded from the layout, while serializable properties have
                        // to be included so that their data can be saved. Transform all serializable decls into data
                        // fields and all non-serializable ones into properties.
                        if (srcField.isSerializable())
                        {
                            ++numSerializable;
                            flags.andNotWith(Decl::DECL_PROPERTY_FIELD);
                            flags.orWith(Decl::DECL_DATA_FIELD);
                        }
                        else
                        {
                            flags.andNotWith(Decl::DECL_DATA_FIELD);
                            flags.orWith(Decl::DECL_PROPERTY_FIELD);
                        }
                    }
                    // Else keep the decls array unchanged.

                    fieldBuilder.setField(newName, 0, flags);
                    hkReflect::Type* fieldType = fieldBuilder.allocate(&m_typesAllocator);
                    fieldsArray->setField(i, fieldType);
                }
            }

            if (m_pruneNonSerializableFields)
            {
                // Move all data fields to the beginning, all the properties to the end.
                hkAlgorithm::insertionSort(const_cast<Decl*>(fieldsArray->getAllDecls().begin()), fieldsArray->getNumDecls(),
                    [] (const Decl& a, const Decl& b)
                {
                    return a.asDataField() && b.asPropertyField();
                });
                fieldsArray->reset(numSerializable, origFields->getNumDecls() - numSerializable);
            }

            TypeDetail::localSetOptional<Opt::DECLS>(copyType, fieldsArray);
        }

        if(TypeDetail::localHasOptional(copyType, Opt::SUBTYPE))
        {
            const Type* st = TypeDetail::localGetOptional<Opt::SUBTYPE>(copyType);
            TypeDetail::localSetOptional<Opt::SUBTYPE>(copyType, copy(st, copiesOut));
        }

        if( !copyType->isDecorator() )
        {
            TypeDetail::localSetOptionalUlong(copyType, Opt::SIZE_ALIGN, Detail::SizeAlign::reqAlignOnly( TypeDetail::localGetOptionalUlong(origType, Opt::SIZE_ALIGN) ) );
            if(const hk::TypeLayout* layout = origType->findAttribute<hk::TypeLayout>())
            {
                HK_ASSERT_NO_MSG(0x2809db63, TypeDetail::localHasOptional(copyType, Opt::DECLS)==false);
                layout->compute(copyType, origType, m_record);
            }
            else if(origType->getKind() != KIND_RECORD && (m_pruneNonSerializableFields || TypeDetail::localHasOptional(copyType, Opt::DECLS) == false))
            {
                switch(origType->getKind())
                {
                    case KIND_VOID:
                    case KIND_OPAQUE:
                    {
                        HK_ASSERT_NO_MSG(0x256134d6, (copyOptional&Opt::IMPL) == 0);
                        break;
                    }
                    case KIND_STRING:
                    {
                        Format::StringValue format = Format::Value::create( hkLosslessCast<hkUint32>(TypeDetail::localGetOptional<Opt::FORMAT>(origType)) );
                        if(int count = format.getFixedCount())
                        {
                            HK_ASSERT_NO_MSG(0x2e537c93, TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(origType));
                            TypeDetail::localSetOptionalUlong(copyType, Opt::SIZE_ALIGN, TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(origType));
                        }
                        else
                        {
                            TypeDetail::localSetSizeAlignPreserveReq(copyType, m_record.m_pointerSize, m_record.m_pointerSize); 
                        }
                        break;
                    }
                    case KIND_POINTER:
                    {
                        TypeDetail::localSetSizeAlignPreserveReq(copyType, m_record.m_pointerSize, m_record.m_pointerSize); 
                        break;
                    }
                    case KIND_BOOL:
                    {
                        if ( m_minifyFormat )
                        {
                            // Force bool to 1 byte.
                            TypeDetail::localSetSizeAlignPreserveReq( copyType, 1, 1 );
                            break;
                        }
                        // else fall through
                    }
                    case KIND_FLOAT:
                    case KIND_INT:
                    {
                        TypeDetail::localSetSizeAlignPreserveReq(copyType, origType->getSizeOf(), origType->getAlignOf()); 
                        break;
                    }
                    case KIND_RECORD:
                    {
                        HK_ASSERT_NO_MSG(0x6bc33b01, TypeDetail::localHasOptional(copyType, Opt::FORMAT) == TypeDetail::localHasOptional(copyType, Opt::SIZE_ALIGN) );
                        break;
                    }
                    case KIND_ARRAY:
                    //case KIND_CONTAINER:
                    {
                        HK_ASSERT_NO_MSG(0x72d00fbc, TypeDetail::localHasOptional(origType, Opt::FORMAT) == TypeDetail::localHasOptional(origType, Opt::SIZE_ALIGN));

                        if(hkUlong format = TypeDetail::localGetOptional<Opt::FORMAT>(origType))
                        {
                            if(hkUlong count = format >> 8) // inplace 
                            {
                                HK_ASSERT_NO_MSG(0x4e177d20, origType->getKind() == KIND_ARRAY);
                                HK_ASSERT_NO_MSG(0x2d5bc300, TypeDetail::localGetOptional<Opt::FORMAT>(origType) != 0);
                            }
                            else // normal array
                            {
                                if(m_pruneNonSerializableFields)
                                {
                                    TypeDetail::localSetSizeAlignPreserveReq(copyType, sizeof(hkUint32), sizeof(hkUint32));
                                }
                                else
                                {
                                    HK_ASSERT_NO_MSG(0x3281b2d8, copyOptional & Opt::DECLS);
                                    //HK_BREAKPOINT(0);
                                }
                            }
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x143e1af2, 0);
                        }
                        break;
                    }
                    default:
                        HK_ASSERT_NO_MSG(0x1f327bc9, 0);
                }
            }
            else
            {
                // leave it to recordlayout
                HK_ASSERT_NO_MSG(0x6bc33b01, TypeDetail::localHasOptional(copyType, Opt::FORMAT) == TypeDetail::localHasOptional(copyType, Opt::SIZE_ALIGN));
            }
        }
        else
        {
            HK_ASSERT_NO_MSG(0x4dc4e8c7, (copyOptional & Opt::SIZE_ALIGN) == 0);
        }

        if (copyOptional & Opt::ATTRIBUTE_STRING)
        {
            const char* attributes = TypeDetail::localGetOptional<Opt::ATTRIBUTE_STRING>(origType);
            TypeDetail::localSetOptional<Opt::ATTRIBUTE_STRING>(copyType, hkString::strDup(attributes, m_typesAllocator));
        }
    }

    _Ret_notnull_ hkReflect::Type* makeNewCopy(_In_ const hkReflect::Type* origType, _In_opt_ const hkReflect::Type* alias, hkArray<TypePair>& copiesOut)
    {
        HK_ASSERT_NO_MSG(0x108bf6c, !origType->isField());

        DLOG_SCOPE("Copying {}@{}", origType->getFullName(), origType);
        using namespace hkReflect;
        // We need to be careful of recursion. Consider struct X { X* m_x; };
        // Thus we it in passes.
        // * Allocate a copy of X and put it in the map so recursive lookups work.
        // * Then recurse for referenced types such as parent, pointer/array element type etc.
        // * Finally compute the layout.

        Type* copyType = allocateBody(origType);
        m_copyFromOrig.insert(origType, copyType);
        if(alias)
        {
            m_copyFromOrig.insert(alias, copyType);
        }

        fillBody(copyType, origType, copiesOut);

        return copyType;
    }


    
    hkResult m_status;
    bool m_pruneNonSerializableFields;
    bool m_includeProperties;
    bool m_minifyFormat;
    hkReflect::RecordLayout::Options m_record;
    hkPointerMap<const hkReflect::Type*, hkReflect::Type*> m_copyFromOrig;
    hkHashMap<const char*, hkReflect::Type*> m_diversions;
    hkTransientAllocator m_typesAllocator;

    struct Builtins
    {
        #define SWITCH(A,B) BuiltinValueType m_##B;
        FOREACH_BUILTIN_TYPE(SWITCH)
        #undef SWITCH
        #define SWITCH(A,B) BuiltinTypedefType m_##B;
        FOREACH_TYPEDEF_TYPE(SWITCH)
        #undef SWITCH
    };
    Builtins m_builtins;
};


hkReflect::TypeCopier::~TypeCopier()
{
    delete m_pimpl;
}


hkReflect::TypeCopier::Options::Options()
{
    hkMemUtil::memSet(this, -1, sizeof(*this));
    setPortableTarget();
}


hkReflect::TypeCopier::Options& hkReflect::TypeCopier::Options::setPortableTarget()
{
    m_pointerSize = 4;
    m_minifyFormat = true;
    m_littleEndian = true;

    m_longIsPointerSize = false; // long is 64 for portable
    m_alignedDoubles = false;
    m_reusesParentPadding = true;
    m_reusesClassAlignPadding = true;

    m_includeProperties = true;
    m_pruneNonSerializableFields = true;
    m_realIsDouble = (sizeof(hkReal) == sizeof(double));

    m_isOk = true;
    return *this;
}


hkReflect::TypeCopier::Options& hkReflect::TypeCopier::Options::setHostTarget()
{
    m_pointerSize = sizeof(void*);
    m_minifyFormat = false;
    m_littleEndian = HK_ENDIAN_LITTLE;

    m_longIsPointerSize = hostLongIsPointerSize;
    m_alignedDoubles = hostAlignedDoubles;
    m_reusesParentPadding = hostReusesParentPadding;
    m_reusesClassAlignPadding = hostReusesClassAlignPadding;

    m_includeProperties = false;
    m_pruneNonSerializableFields = false;
    m_realIsDouble = (sizeof(hkReal) == sizeof(double));

    m_isOk = true;
    return *this;
}


hkReflect::TypeCopier::Options& hkReflect::TypeCopier::Options::setTarget(int ptrBits, char target, _In_opt_z_ const char* extra)
{
    m_isOk = true;
    HK_ASSERT_NO_MSG(0x3d0da5e1, ptrBits == 32 || ptrBits == 64);
    m_pointerSize = ptrBits / 8;
    m_minifyFormat = false;
    m_littleEndian = true;

    m_longIsPointerSize = true;
    m_alignedDoubles = true;
    m_reusesParentPadding = true;
    m_reusesClassAlignPadding = true;

    m_includeProperties = false;
    m_pruneNonSerializableFields = false;
    m_realIsDouble = false;

#define PAIR(P,C) (((P)<<8)|(C))
    switch( PAIR(ptrBits, target) )
    {
        case PAIR(32, 'm'):
            m_reusesParentPadding = false;
            break;
        case PAIR(32, 'a'):
            break;
        case PAIR(32, 'g'):
            m_alignedDoubles = false;
            break;

        case PAIR(64, 'm'):
            m_longIsPointerSize = false;
            m_reusesParentPadding = false;
            break;
        case PAIR(64, 'g'):
            break;
        default:
            m_isOk = false;
            HK_ASSERT(0x6eb31f52, 0, "Unknown target option");
            break;
#undef PAIR
    }

    if(extra)
    {
        for(const char* p = extra; *p; p++)
        {
            switch(*p)
            {
                case 'd':
                    m_realIsDouble = true;
                    break;
                default:
                    m_isOk = false;
                    HK_ASSERT(0x4d47496, 0, "Unknown suffix");
            }
        }
    }
    return *this;
}

hkReflect::TypeCopier::Options& hkReflect::TypeCopier::Options::parseTarget(_In_opt_z_ const char* platform)
{
    if(platform == HK_NULL)
    {
        setHostTarget();
    }
    else if(hkString::strLen(platform) >= 3)
    {
        const char* p = platform;
        bool is32 = (p[0] == '3' && p[1] == '2');
        bool is64 = (p[0] == '6' && p[1] == '4');
        if(is32 || is64)
        {
            setTarget(is32 ? 32 : 64, p[2], p + 3);
        }
    }
    return *this;
}


hkReflect::TypeCopier::Options& hkReflect::TypeCopier::Options::setRealIsDouble(bool b)
{
    m_realIsDouble = b;
    return *this;
}

hkReflect::TypeCopier::Options& hkReflect::TypeCopier::Options::setIncludeProperties(bool b)
{
    m_includeProperties = b;
    return *this;
}

hkReflect::TypeCopier::TypeCopier(const Options& opts)
{
    m_pimpl = new Pimpl(opts);
}


_Ret_maybenull_ const hkReflect::Type* hkReflect::TypeCopier::copy(_In_ const hkReflect::Type* origType, _Inout_opt_ hkArray<TypePair>* typesOut)
{
    HK_ASSERT_NO_MSG(0x63dee8d7, m_pimpl);

    DLOG_SCOPE("hkReflect::TypeCopier::copy {}@{}", origType->getFullName(), origType);
    hkReflect::Type* copyType = m_pimpl->copyEntryPoint(origType, typesOut);

    return copyType;
}

int hkReflect::TypeCopier::getPointerSize() const
{
    return m_pimpl->getPointerSize();
}

int hkReflect::TypeCopier::getSizeofReal() const
{
    return m_pimpl->getSizeofReal();
}

_Ret_z_ const char* hkReflect::TypeCopier::getHostString()
{
    #if HK_POINTER_SIZE == 4
    #define TARGET_PTR "32"
    #else
    #define TARGET_PTR "64"
    #endif

    #if defined(HK_REAL_IS_DOUBLE)
    #define TARGET_SUFFIX "d"
    #else
    #define TARGET_SUFFIX ""
    #endif

    // preprocessor joins these strings
    return TARGET_PTR TARGET_ID TARGET_SUFFIX;
}


namespace
{
    static const hkReflect::TypeCopier::Target TargetStrings[] =
    {
        { "32m", "MSVC x86, Switch32_win" },
        { "32a", "Arm32, Switch32" },
        { "32g", "Generic x86" },
        { "64m", "MSVC x64, XBoxOne, Switch64_win" },
        { "64g", "Generic x64, Arm64, Playstation(R)4, Switch64" },
    };
    static const char* TargetExtra[] =
    {
        "d",
    };
}

hkArrayView<const hkReflect::TypeCopier::Target> hkReflect::TypeCopier::getTargetStrings()
{
    return hkArrayViewT::make(TargetStrings);
}

hkArrayView<const char*> hkReflect::TypeCopier::getTargetExtra()
{
    return hkArrayViewT::make(TargetExtra);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
