// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Memory/Allocator/Transient/hkTransientAllocator.h>

class hkLinearBuffer;
namespace hkReflect
{
    class HK_EXPORT_COMMON TypeBuilder
    {
        public:

            HK_DECLARE_CLASS(TypeBuilder, New);

                /// Initializes an empty TypeBuilder.
                /// Call one of the begin* methods before calling any other methods.
            TypeBuilder();
            virtual ~TypeBuilder();

                /// Sets the type world token to attach to the types. This value is used to prevent
                /// accidentally mixing types from completely separate type hierarchies; e.g. deriving
                /// a temporary type created during deserialization from a built-in one.
                ///
                /// Use HK_NULL (default) to identify types that should be considered as belonging to the same hierarchy
                /// as built-in types.
            void setTypeWorld(_In_ const void* typeWorld);

                /// Begin a record.
                /// See also beginDerived.
            void beginRecord(_In_opt_z_ const char* name, _In_ const Type* parent);

                /// Begin a pointer type
            void beginPointer(_In_ const Type* pointedType);

                /// Begin an array type. Variable-size arrays use fixedSize=0.
            void beginArray(_In_ const Type* elemType, int fixedSize);

                /// Begin an extension of the given parent.
            void beginDerived(_In_opt_ const Type* parent);

                /// Shallow copy the parts of orig with an optional mask.
            void beginShallowClone(_In_ const Type* orig, hkUint32 mask = hkUint32(-1));

                /// Or with the given flags.
            void addFlags(hkReflect::Type::FlagBits flag)
            {
                m_flags.orWith(flag);
                setItem<Opt::FLAGS>(m_flags.get());
            }

            template<hkReflect::Optional OPT>
            typename hkReflect::Detail::OptionStorage<OPT>::Storage getItem() const
            {
                const int BIT = Detail::OptionStorage<OPT>::BitIndex;
                HK_COMPILE_TIME_ASSERT( ( 1 << BIT ) == OPT );
                return reinterpret_cast<typename hkReflect::Detail::OptionStorage<OPT>::Storage>( m_optData[BIT] );
            }

            template<hkReflect::Optional OPT>
            void setItem( typename hkReflect::Detail::OptionStorage<OPT>::Storage s )
            {
                const int BIT = Detail::OptionStorage<OPT>::BitIndex;
                HK_COMPILE_TIME_ASSERT( (1<<BIT)==OPT );
                m_optMask |= OPT;
                m_optData[BIT] = hkUlong(s);
            }

            template<hkReflect::Optional OPT>
            void addItem( typename hkReflect::Detail::OptionStorage<OPT>::Storage s )
            {
#if defined(HK_DEBUG_SLOW)
                if ((m_optMask & OPT) != 0)
                {
                    hkReflect::Optional opt = OPT;
                    const char* optionalName = hkReflect::getPresetsOf<hkReflect::Optional>().getNameByPreset(&opt);
                    HK_ASSERTV(0x29905cfc, false, "Item {} is already present.", optionalName);
                }
#endif

                const int BIT = Detail::OptionStorage<OPT>::BitIndex;
                HK_COMPILE_TIME_ASSERT( (1<<BIT)==OPT );
                m_optMask |= OPT;
                m_optData[BIT] = hkBitCast<hkUlong>(s);
            }

            /// Adds an attribute. The Var contents are not copied.
            /// If the default type deallocation function is used, type deletion will trigger the deletion of the attribute.
            void addAttribute(const hkReflect::Var& attr);

            void addMethod(_In_z_ const char* name,
                const hkReflect::Detail::CallInvoker& invoker,
                _In_ const hkReflect::Type* returnType,
                const hkArrayBase<const hkReflect::Type*>& paramTypes)
            {
                addMethodInternal(name, invoker, returnType, paramTypes, METHOD);
            }

            void addConstructor(_In_z_ const char* name,
                const hkReflect::Detail::CallInvoker& invoker,
                const hkArrayBase<const hkReflect::Type*>& paramTypes)
            {
                addMethodInternal(name, invoker, HK_NULL, paramTypes, CONSTRUCTOR);
            }

            void addFunction(_In_z_ const char* name,
                const hkReflect::Detail::CallInvoker& invoker,
                _In_ const hkReflect::Type* returnType,
                const hkArrayBase<const hkReflect::Type*>& paramTypes)
            {
                addMethodInternal(name, invoker, returnType, paramTypes, STATIC_FUNCTION);
            }

            void addTypeTemplateParam(_In_ const hkReflect::Type* type, _In_z_ const char* name);
            void addValueTemplateParam(hkUlong value, _In_z_ const char* name);
            void addInterface(_In_ const hkReflect::Type* type, int offset);
            void setField(_In_z_ const char* fieldName, int offset, hkReflect::Decl::DeclFlags flags);

            hkReflect::TypeBuilder& addProperty(_In_z_ const char* name, _In_ const hkReflect::Type* type);

            TypeBuilder& addMember(_In_z_ const char* name, hkReflect::Decl::DeclFlags flags, _In_ const hkReflect::Type* type);
            TypeBuilder& addMember(_In_z_ const char* name, int offset, hkReflect::Decl::DeclFlags flags, _In_ const hkReflect::Type* type);

            /// Adds special methods using the Trivial methods in hkBuiltinImpl.h.
            void addTrivialSpecialMethods();

            /// Adds special methods using the Implicit methods in hkBuiltinImpl.h. If a method is trivial in the
            /// parent and all fields, the method will be trivial. If a method is deleted in the parent or any of the
            /// fields, the method will be deleted.
            void addImplicitSpecialMethods();

            /// Adds 'unknown' placeholder optionals for special methods.
            /// Use hkReflect::TypeDetail::fixupSpecialMethods to fixup the optionals after creation.
            void addUnknownSpecialMethods();

            /// Add a hk::DeleteTypeInfo attribute. If func is null, TypeBuilder::deallocateType will be used.
            /// If data is not specified, the size of the type will be used.
            void addDeleteInfo(hk::DeleteTypeInfo::Func func = HK_NULL);
            void addDeleteInfo(hk::DeleteTypeInfo::Func func, hkUlong data);

            /// By default the type builder doesn't recompute the layout of types. You can set the flag to true
            /// to use the automatic layout computation.
            void setRecomputeLayout(bool recomputeLayout) { m_recomputeLayout = recomputeLayout; }

            /// Allocates the type.
            /// The produced Type might keep a reference to the allocator. Avoid passing the Temp allocator, since it is
            /// already used for internal allocations and doing that will violate the LIFO property.
            hkResult allocate(hkMemoryAllocator* allocator, _Outptr_ hkReflect::Type** type);
            _Ret_notnull_ hkReflect::Type* allocate(_Inout_ hkMemoryAllocator* allocator);

            /// Returns the total memory needed to allocate the type. Called in allocate() before the creation of the type buffer.
            virtual int getTotalNeededMemory() const;
            virtual int getHeadNeededMemory() const;
            virtual int getBodyNeededMemory() const;

            /// Writes the type data on the provided buffer. Called in allocate() on the type buffer to be returned to the client.
            virtual _Ret_notnull_ Type* writeOnBuffer(_Out_writes_bytes_(bufSize) void* buffer, int bufSize);
            virtual _Ret_notnull_ Type* writeOnBuffer(hkLinearBuffer& buf);
            virtual _Ret_notnull_ Type* writeHeadOnBuffer(hkLinearBuffer& buf);
            virtual _Ret_notnull_ Type* writeBodyOnBuffer(_Inout_ Type* head, hkLinearBuffer& buf);

            /// Destructs and deallocates a type created by a Type Builder and all the contained attributes.
            static void HK_CALL deallocateType(_Inout_ hkReflect::Type* type, _Inout_ hkMemoryAllocator* allocator, hkUlong data);
            /// Destructs (without deallocating) a type created by a Type Builder and all the contained attributes.
            static void HK_CALL destructType(_Inout_ hkReflect::Type* type);

                // Advanced usage, easy to break!
            void overrideFormat(hkUlong format);
            void overrideParent(_In_opt_ const hkReflect::Type* parent) { m_parent = parent; }

            struct CallableInfo;

        protected:

            enum CallableTypeValue
            {
                METHOD,
                CONSTRUCTOR,
                STATIC_FUNCTION
            };
            typedef hkEnum<CallableTypeValue, hkUint8> CallableType;

        protected:

            void addMethodInternal(_In_z_ const char* name,
                const hkReflect::Detail::CallInvoker& invoker,
                _In_ const hkReflect::Type* returnType,
                const hkArrayBase<const hkReflect::Type*>& paramTypes,
                CallableType callableType);

            void doAddDeleteInfo(hk::DeleteTypeInfo::Func func, hkUlong data);

            bool isSameTypeWorld(_In_opt_ const Type* type) const;

        protected:

            hkTransientAllocator m_tempAllocator;
            hkReflect::Type::Flags m_flags;
            const hkReflect::Type* m_parent;
            hkUlong m_optMask;
            hkUlong m_optData[32];
            const void* m_typeWorld;

            hkArray<hkReflect::Var>::Temp m_attributes;
            hkArray<CallableInfo>::Temp m_methods HK_ATTR(hk::MemoryTracker(opaque=true));
            hkArray<hkReflect::Detail::TemplateParameter>::Temp m_templateParams;
            hkArray<hkReflect::Detail::Interface>::Temp m_interfaces;
            hkArray<TypeBuilder>::Temp m_fields;
            int m_nextField; //m_members is all fields first, then properties

            hk::DeleteTypeInfo* m_deleteAttr;
            bool m_useTypeSizeInDeleteInfo;
            bool m_recomputeLayout;
            bool m_fixupSpecialMethods;
    };

        /// Convenience class to avoid using the type builder for simple cases
    struct HK_EXPORT_COMMON BasicValueType : public hkReflect::ValueType
    {
        HK_DECLARE_CLASS(BasicValueType, New);

        // Constructs an empty value type
        BasicValueType()
            : m_sizeAlign(0)
        {
            m_optional = hkReflect::Opt::FORMAT|hkReflect::Opt::IMPL|hkReflect::Opt::SIZE_ALIGN;
            m_parent = HK_NULL;
            m_format = hkReflect::Format::OfVoid::Value;
            m_impl = HK_NULL;
        }
        hkUlong m_format;
        const Detail::ValueImpl* m_impl;
        hkUlong m_sizeAlign;
    };

        /// Convenience class to avoid using the type builder for simple cases
    struct HK_EXPORT_COMMON BasicPointerType : public hkReflect::PointerType
    {
        HK_DECLARE_CLASS(BasicPointerType, New);

            // Constructs a "raw pointer" type
        BasicPointerType(QualType ptdType, _In_z_ const char* name, Type::Flags extraFlags = 0);
        BasicPointerType(QualType ptdType, _In_z_ const char* name, _Inout_ hkReflect::Detail::TemplateParameterArray* templ, Type::Flags extraFlags = 0);

        hkUlong m_format;
        QualType m_next; //subtype
        const Detail::PointerImpl* m_impl;
        const char* m_name;
        hkReflect::Detail::TemplateParameterArray* m_template;
        hkUlong m_sizeAlign;
        hkUlong m_flags;

        hkUlong m_pad;

        hkReflect::Detail::TemplatePod<1> m_templateStorage;
    };

        /// Convenience class to avoid using the type builder for simple cases
    struct HK_EXPORT_COMMON BasicArrayType : public hkReflect::ArrayType
    {
        HK_DECLARE_CLASS(BasicArrayType, New);

        BasicArrayType(QualType elemType, int sizeArr, int alignArr, _In_ const Detail::ArrayImpl* impl, _In_z_ const char* name, Type::Flags extraFlags = 0);

        hkUlong m_format;
        QualType m_elemType; //subtype
        const Detail::ArrayImpl* m_impl;
        const char* m_name;
        hkUlong m_sizeAlign;
        hkUlong m_flags;
    };

    /// Convenience class to avoid using the type builder for simple cases
    struct HK_EXPORT_COMMON BasicArrayTypeWithParams : public hkReflect::ArrayType
    {
        HK_DECLARE_CLASS(BasicArrayTypeWithParams, New);

        BasicArrayTypeWithParams(QualType elemType, int sizeArr, int alignArr, _In_ const Detail::ArrayImpl* impl, _In_z_ const char* name, Type::Flags extraFlags = 0);

        hkUlong m_format;
        QualType m_elemType; //subtype
        const Detail::ArrayImpl* m_impl;
        const char* m_name;
        hkReflect::Detail::TemplateParameterArray* m_template;
        hkUlong m_sizeAlign;
        hkUlong m_flags;

        hkUlong m_pad;

        hkReflect::Detail::FixedArrayStorage<hkUlong, hkReflect::Detail::TemplateParameter, 2> m_templateStorage;
    };

        /// Convenience class to avoid using the type builder for simple cases
    struct HK_EXPORT_COMMON RepeatArrayType : public hkReflect::ArrayType
    {
        HK_DECLARE_CLASS(RepeatArrayType, New);

        RepeatArrayType(QualType elemType, int repeatCount, int sizeArr, int alignArr, _In_ const Detail::ArrayImpl* impl, _In_z_ const char* name, Type::Flags extraFlags = 0);

        hkUlong m_format;
        QualType m_elemType; //m_subtype
        const Detail::ArrayImpl* m_impl;
        const char* m_name;
        hkReflect::Detail::TemplateParameterArray* m_template;
        hkUlong m_sizeAlign;
        hkUlong m_flags;

        hkUlong m_pad;

        hkReflect::Detail::FixedArrayStorage<hkUlong, hkReflect::Detail::TemplateParameter, 1> m_templateStorage;
    };

        /// Convenience class to avoid using the type builder for simple cases
    struct HK_EXPORT_COMMON BasicStringType : public hkReflect::StringType
    {
        HK_DECLARE_CLASS(BasicStringType, New);

        BasicStringType(int sizeStr, int alignStr, _In_ const Detail::StringImpl* impl, _In_z_ const char* name, bool immutable, Type::Flags extraFlags = 0);

        hkUlong m_format;
        const Detail::StringImpl* m_impl;
        const char* m_name;
        hkUlong m_sizeAlign;
        hkUlong m_flags;
    };
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
