// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Core/hkReflectCallable.h>
#include <Common/Base/Memory/Allocator/Linear/hkLinearBuffer.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>

#define DEBUG_LOG_IDENTIFIER "s11n.TypeBuilder"
#include <Common/Base/System/Log/hkLog.hxx>

#define FAIL_WITH_ID_IF( ID, COND, ... ) \
    if ( (COND) ) \
    { \
        Log_Error( __VA_ARGS__ ).setId( ID ); \
        return HK_FAILURE; \
    }

struct hkReflect::TypeBuilder::CallableInfo
{
    const char* m_name;
    hkReflect::Detail::CallInvoker m_invoker;
    const hkReflect::Type* m_returnType;
    hkArray<const hkReflect::Type*>::Temp m_params;
    CallableType m_callableType;
};

namespace
{
    void writeCallable(
        const hkReflect::TypeBuilder::CallableInfo& info,
        _In_ const hkReflect::Type* retType,
        _In_ const hkReflect::Type* objectType,
        hkReflect::Detail::NamedCallablePod& currCallable,
        hkLinearBuffer& buf)
    {
        using namespace hkReflect;

        Detail::CallableSignature* signature = buf.objAlloc<Detail::CallableSignature>();
        Detail::CallableSignature::Param* dstParams = buf.objAlloc<Detail::CallableSignature::Param>(info.m_params.getSize()); // params directly follow signature

        // write the signature
        signature->m_numParams = hkLosslessCast<hkUint8>(info.m_params.getSize());
        signature->m_returnType = retType;
        signature->m_align = 0;
        signature->m_size = 0;

        
        for (int paramIndex = 0; paramIndex < info.m_params.getSize(); ++paramIndex)
        {
            const hkReflect::Type* srcParam = info.m_params[paramIndex];
            if (srcParam->getAlignOf() > signature->m_align)
            {
                signature->m_align = hkLosslessCast<hkUint8>(srcParam->getAlignOf());
            }
            dstParams[paramIndex].m_offset =
                hkLosslessCast<hkUint16>(HK_NEXT_MULTIPLE_OF(srcParam->getAlignOf(), signature->m_size));
            dstParams[paramIndex].m_type = srcParam;
            signature->m_size =
                hkLosslessCast<hkUint16>(dstParams[paramIndex].m_offset + srcParam->getSizeOf());
        }

        
        currCallable.m_invoker = info.m_invoker;
        currCallable.m_signature = signature;
        currCallable.m_objectType = objectType;
        currCallable.m_name = info.m_name;
    }
}

hkReflect::TypeBuilder::TypeBuilder()
    : m_tempAllocator(&hkMemoryRouter::getInstance().temp())
    , m_flags(0)
    , m_parent(HK_NULL)
    , m_optMask(0)
    , m_typeWorld(HK_NULL)
    , m_nextField(0)
    , m_deleteAttr(HK_NULL)
    , m_useTypeSizeInDeleteInfo(true)
    , m_recomputeLayout(false)
    , m_fixupSpecialMethods(false)
{
    hkMemUtil::memSet(m_optData, 0, sizeof(m_optData));
}

hkReflect::TypeBuilder::~TypeBuilder()
{
}

void hkReflect::TypeBuilder::setTypeWorld(_In_ const void* typeWorld)
{
    m_typeWorld = typeWorld;

    if (m_typeWorld == HK_NULL)
    {
        // HK_NULL is the implied default, so we don't explicitly write it
        // Clear the opt mask to allow clearing the type world
        m_optMask &= ~Opt::TYPE_WORLD;
    }
    else
    {
        setItem<Opt::TYPE_WORLD>(m_typeWorld);
    }
}

void hkReflect::TypeBuilder::beginDerived(_In_opt_ const hkReflect::Type* parent)
{
    m_parent = parent;
}

void hkReflect::TypeBuilder::beginRecord(_In_opt_z_ const char* name, _In_ const hkReflect::Type* parent)
{
    HK_ASSERT_NO_MSG(0x2487550c, parent == HK_NULL || parent->asRecord());
    m_parent = parent;
    addItem<Opt::FORMAT>(hkReflect::Format::OfRecord::Value);
    addItem<Opt::DECLS>(HK_NULL);   // m_fields
    if (name)
    {
        addItem<Opt::NAME>(name);
    }
    addItem<Opt::SIZE_ALIGN>(0);
}

void hkReflect::TypeBuilder::beginPointer(_In_ const Type* pointedType)
{
    HK_ASSERT(0x683bdd1f, isSameTypeWorld(pointedType), "Pointed type must belong to the same world as the pointer type being built");
    addItem<Opt::FORMAT>(hkReflect::Format::OfPointer::Value);
    addItem<Opt::SUBTYPE>(pointedType); // pointed type
    addItem<Opt::SIZE_ALIGN>(0); // HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(void*), alignof(void*)) ??
}

void hkReflect::TypeBuilder::beginArray(_In_ const Type* elementType, int fixed)
{
    HK_ASSERT(0x474abcf1, isSameTypeWorld(elementType), "Element type must belong to the same world as the array type being built");
    addItem<Opt::FORMAT>(hkReflect::Format::ofArray(fixed));
    addItem<Opt::SUBTYPE>(elementType);
    addItem<Opt::SIZE_ALIGN>(0);
}

void hkReflect::TypeBuilder::beginShallowClone(_In_ const Type* orig, hkUint32 mask)
{
    m_parent = orig->getParent();
    m_optMask = TypeDetail::getOptMask(orig) & mask;

    // copy optionals
    int origIndex = 0;
    for (int i = 0; i < 32; ++i)
    {
        Optional opt = Optional(1 << i);
        if (m_optMask & opt)
        {
            m_optData[i] = TypeDetail::localGetOptionalUlong(orig, opt);
            ++origIndex;
        }
        else
        {
            m_optData[i] = 0;
        }
    }

    // Copy the original flags so that they can be edited.
    m_flags = TypeDetail::getFlags(orig);
}


void hkReflect::TypeBuilder::addAttribute(const hkReflect::Var& attr)
{
    if (attr.dynCast<hk::Default>())
    {
        setItem<Opt::DEFAULT>(attr.getAddress());
    }
    else
    {
        setItem<Opt::ATTRIBUTES>(0);
        m_attributes.pushBack(attr);
    }
}

void hkReflect::TypeBuilder::addMethodInternal(_In_z_ const char* name,
    const hkReflect::Detail::CallInvoker& invoker,
    _In_ const hkReflect::Type* returnType,
    const hkArrayBase<const hkReflect::Type*>& paramTypes,
    CallableType callableType)
{
    HK_ASSERT(0x47460267, isSameTypeWorld(returnType), "Function return type must belong to the same world as the type being build");

    for (int i = 0; i < paramTypes.getSize(); ++i)
    {
        HK_ASSERT(0x52896ac2, isSameTypeWorld(paramTypes[i]), "Function return type must belong to the same world as the type being build");
    }

    setItem<Opt::FUNCTIONS>(HK_NULL);

    CallableInfo& info = m_methods.expandOne();
    info.m_name = name;
    info.m_invoker = invoker;
    info.m_returnType = returnType;
    info.m_params.append(paramTypes.begin(), paramTypes.getSize());
    info.m_callableType = callableType;
}

void hkReflect::TypeBuilder::addTypeTemplateParam(_In_ const hkReflect::Type* type, _In_z_ const char* name)
{
    HK_ASSERT(0x471476d8, isSameTypeWorld(type), "Template param type must belong to the same world as the type being build");

    setItem<Opt::TEMPLATE>(HK_NULL);
    HK_COMPILE_TIME_ASSERT('t' == hkReflect::Template::KIND_TYPE);
    hkStringBuf sb("t", name);
    hkReflect::Detail::TemplateParameter* tpm = m_templateParams.expandBy(1);
    tpm->m_kindAndName = hkString::strDup(sb.cString(), m_tempAllocator);
    tpm->m_storage = hkUlong(type);
}

void hkReflect::TypeBuilder::addInterface(_In_ const hkReflect::Type* type, int offset)
{
    HK_ASSERT(0x27574e7b, isSameTypeWorld(type), "Interface type must belong to the same world as the type being build");

    setItem<Opt::INTERFACES>(HK_NULL);
    hkReflect::Detail::Interface* iface = m_interfaces.expandBy(1);
    iface->m_interfaceType = type;
    iface->m_offset = offset;
}

void hkReflect::TypeBuilder::addValueTemplateParam(hkUlong value, _In_z_ const char* name)
{
    setItem<Opt::TEMPLATE>(HK_NULL);
    HK_COMPILE_TIME_ASSERT('v' == hkReflect::Template::KIND_VALUE);
    hkStringBuf sb("v", name);
    hkReflect::Detail::TemplateParameter* tpm = m_templateParams.expandBy(1);
    tpm->m_kindAndName = hkString::strDup(sb.cString(), m_tempAllocator);
    tpm->m_storage = value;
}

int hkReflect::TypeBuilder::getTotalNeededMemory() const
{
    return getHeadNeededMemory() + getBodyNeededMemory();
}

int hkReflect::TypeBuilder::getHeadNeededMemory() const
{
    int size = sizeof(hkReflect::Type);             // base data
    size += hkMath::countBitsSet(hkLosslessCast<hkUint32>(m_optMask)) * sizeof(hkUlong);    // optionals

    // We don't write the type world for decorators, so subtract it again
    if (m_typeWorld != HK_NULL && (m_optMask & Opt::FORMAT) == 0)
    {
        size -= sizeof(hkUlong);
    }
    return size;
}

int hkReflect::TypeBuilder::getBodyNeededMemory() const
{
    int size = 0;
    if (!m_attributes.isEmpty())
    {
        // attr array
        size += Detail::AttributeArray::sizeOfArrayN(m_attributes.getSize());
    }

    if (!m_methods.isEmpty())
    {
        size += sizeof(hkReflect::Detail::FunctionsPod) + m_methods.getSize() * sizeof(hkReflect::Detail::NamedCallablePod);
        for (int i = 0; i < m_methods.getSize(); ++i)
        {
            size += sizeof(Detail::CallableSignature) + m_methods[i].m_params.getSize() * sizeof(Detail::CallableSignature::Param);
        }
    }

    // If the bit is set we will generate the struct, even if there are no params
    if (m_optMask & Opt::TEMPLATE)
    {
        const hkUlong bit = Detail::OptionStorage<Opt::TEMPLATE>::BitIndex;
        if (m_optData[bit]) // we already have the data
        {
            HK_ASSERT_NO_MSG(0x17897715, m_templateParams.getSize() == 0);
        }
        else // we must build them
        {
            size += Template::getSizeOf(m_templateParams.getSize());
            for (int i = 0; i < m_templateParams.getSize(); ++i)
            {
                int len = hkString::strLen(m_templateParams[i].m_kindAndName) + 1;
                size += HK_NEXT_MULTIPLE_OF(sizeof(hkUlong), len);
            }
        }
    }

    if (m_optMask & Opt::INTERFACES)
    {
        HK_ASSERT_NO_MSG(0x41720a40, m_optData[Detail::OptionStorage<Opt::INTERFACES>::BitIndex] == 0);
        size += Detail::InterfaceArray::sizeOfArrayN(m_interfaces.getSize());
    }

    // Opt::DECLS == null is used as a marker that a DeclsArray is required (not beginShallowClone).
    if ((m_optMask & Opt::DECLS) && (getItem<Opt::DECLS>() == nullptr))
    {
        size += sizeof(void*); // numFields
        size += sizeof(Detail::RecordFieldPod) * m_fields.getSize();

        for (hkArray<TypeBuilder>::const_iterator it = m_fields.begin(), end = m_fields.end(); it != end; ++it)
        {
            size += it->getTotalNeededMemory();
        }
        // We allocate this in the decl array
        size += sizeof(Detail::DeclsArray::Head);
    }

    return size;
}


_Ret_notnull_ hkReflect::Type* hkReflect::TypeBuilder::writeOnBuffer(_Out_writes_bytes_(bufferSize) void* bufferStart, int bufferSize)
{
    hkLinearBuffer buf(bufferStart, bufferSize);
    return writeOnBuffer(buf);
}

_Ret_notnull_ hkReflect::Type* hkReflect::TypeBuilder::writeHeadOnBuffer(hkLinearBuffer& buf)
{
    HK_COMPILE_TIME_ASSERT(HK_ALIGN_OF(Type) == HK_ALIGN_OF(Detail::FunctionsPod));
    HK_COMPILE_TIME_ASSERT(HK_ALIGN_OF(Type) == HK_ALIGN_OF(Detail::NamedCallablePod));
    HK_COMPILE_TIME_ASSERT(HK_ALIGN_OF(Type) == HK_ALIGN_OF(Detail::CallableSignature));
    HK_COMPILE_TIME_ASSERT(HK_ALIGN_OF(Type) == HK_ALIGN_OF(Detail::TemplateParameterArray));

    if ((m_optMask & Opt::FORMAT) == 0)
    {
        // We don't write the type world for decorators
        if (m_typeWorld != HK_NULL)
        {
            m_optMask &= ~Opt::TYPE_WORLD;
        }
    }
    else
    {
        HK_ASSERT(0x371dd9c1, isSameTypeWorld(m_parent), "Parent type must belong to the same world as the type being built");
    }

    if (m_optMask & Opt::FORMAT && m_optData[Opt::FORMAT] == hkReflect::KIND_RECORD)
    {
        HK_ASSERT_NO_MSG(0xab7eb7, m_optMask & Opt::DECLS);
    }
    Type* type = buf.objAlloc<Type>();
    TypeDetail::setOptMask(type, m_optMask);
    TypeDetail::setParent(type, m_parent);

    for (int i = 0; i < 32; ++i)
    {
        int mask = 1 << i;
        if (m_optMask & mask)
        {
            *buf.objAlloc<hkUlong>() = m_optData[i];
        }
    }
    return type;
}

_Ret_notnull_ hkReflect::Type* hkReflect::TypeBuilder::writeBodyOnBuffer(_Inout_ hkReflect::Type* type, hkLinearBuffer& buf)
{
    // write attributes
    if (!m_attributes.isEmpty())
    {
        void* addr = buf.blockAlloc(Detail::AttributeArray::sizeOfArrayN(m_attributes.getSize()));
        Detail::AttributeArray* attrArray = Detail::AttributeArray::createInPlace(addr, m_attributes.getSize());
        hkArrayView<Detail::AttributeItem> items = attrArray->items();

        // copy the attribute pointers
        for (int i = 0; i < m_attributes.getSize(); ++i)
        {
            items[i].data = m_attributes[i].getAddress();
            items[i].type = m_attributes[i].getType();
        }

        TypeDetail::localSetOptional<Opt::ATTRIBUTES>(type, attrArray);
    }

    // write methods
    if (!m_methods.isEmpty())
    {
        hkReflect::Detail::FunctionsPod* functions = buf.objAlloc<hkReflect::Detail::FunctionsPod>();
        TypeDetail::localSetOptional<Opt::FUNCTIONS>(type, functions->cast() );

        hkReflect::Detail::NamedCallablePod* currCallable = buf.objAlloc<hkReflect::Detail::NamedCallablePod>(m_methods.getSize());

        functions->m_methodInfo = currCallable;
        functions->m_numMethods = 0;
        functions->m_numConstructors = 0;
        functions->m_numFunctions = 0;

        // write methods, then constructors, then static functions
        for (int i = 0; i < m_methods.getSize(); ++i)
        {
            if (m_methods[i].m_callableType == METHOD)
            {
                writeCallable(m_methods[i], m_methods[i].m_returnType, type, *currCallable, buf);
                functions->m_numMethods += 1;
                currCallable += 1;
            }
        }
        for (int i = 0; i < m_methods.getSize(); ++i)
        {
            if (m_methods[i].m_callableType == CONSTRUCTOR)
            {
                writeCallable(m_methods[i], type, HK_NULL, *currCallable, buf);
                functions->m_numConstructors += 1;
                currCallable += 1;
            }
        }
        for (int i = 0; i < m_methods.getSize(); ++i)
        {
            if (m_methods[i].m_callableType == STATIC_FUNCTION)
            {
                writeCallable(m_methods[i], m_methods[i].m_returnType, HK_NULL, *currCallable, buf);
                functions->m_numFunctions += 1;
                currCallable += 1;
            }
        }
    }

    // write template parameters
    if (m_optMask & Opt::TEMPLATE)
    {
        const hkUlong bit = Detail::OptionStorage<Opt::TEMPLATE>::BitIndex;
        if (m_optData[bit]) // we reference existing data
        {
            HK_ASSERT_NO_MSG(0x7dd9b3c6, m_templateParams.getSize() == 0);
            TypeDetail::localSetOptionalUlong(type, Opt::TEMPLATE, m_optData[bit]);
        }
        else // we build new data
        {
            HK_ASSERT_NO_MSG(0x2aed6daa, m_templateParams.getSize() < 0xffff);
            int numParams = m_templateParams.getSize();
            int sizeNeeded = Detail::TemplateParameterArray::sizeOfArrayN(numParams);
            void* rawMem = buf.blockAlloc(sizeNeeded);
            Detail::TemplateParameterArray* params = Detail::TemplateParameterArray::createInPlace(rawMem, numParams);

            TypeDetail::localSetOptionalUlong(type, Opt::TEMPLATE, reinterpret_cast<hkUlong>(params));

            for (int i = 0; i < numParams; ++i)
            {
                // template kindAndName is is a temporary buffer, duplicate it.
                int len = hkString::strLen(m_templateParams[i].m_kindAndName) + 1;
                char* p = buf.objAlloc<char>(HK_NEXT_MULTIPLE_OF(sizeof(hkUlong), len));
                hkString::memCpy(p, m_templateParams[i].m_kindAndName, len);
                params->item(i).setInternal(p, m_templateParams[i].m_storage);
            }
        }
    }
    else
    {
        HK_ASSERT_NO_MSG(0x5ae6287d, m_templateParams.isEmpty());
    }

    if (m_optMask & Opt::INTERFACES)
    {
        HK_ASSERT_NO_MSG(0x6ab47ba1, m_optData[Detail::OptionStorage<Opt::INTERFACES>::BitIndex] == 0);

        int numInterfaces = m_interfaces.getSize();
        int sizeNeeded = Detail::InterfaceArray::sizeOfArrayN(numInterfaces);
        void* rawMem = buf.blockAlloc(sizeNeeded);
        Detail::InterfaceArray* interfaces = Detail::InterfaceArray::createInPlace(rawMem, numInterfaces);
        for (int i = 0; i < numInterfaces; ++i)
        {
            interfaces->item(i) = m_interfaces[i];
        }
        TypeDetail::localSetOptionalUlong(type, Opt::INTERFACES, reinterpret_cast<hkUlong>(interfaces));
    }

    // write recordfields
    if (m_optMask & Opt::DECLS)
    {
        HK_COMPILE_TIME_ASSERT(HK_ALIGN_OF(Type) == HK_ALIGN_OF(hkReflect::Detail::RecordFieldPod));

        // Opt::DECLS == null is used as a marker that a DeclsArray is required (not beginShallowClone).
        if (getItem<Opt::DECLS>() == nullptr)
        {
            hkReflect::RecordType* rec = static_cast<hkReflect::RecordType*>(type);
            struct Alloc : public hkMemoryAllocator
            {
                hkLinearBuffer& m_buf;
                Alloc(hkLinearBuffer& buf) : m_buf(buf) {}
                virtual _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* blockAlloc(int numBytes) HK_OVERRIDE { return m_buf.blockAlloc(numBytes); }
                virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE { HK_ASSERT_NO_MSG(0x709a4d3d, 0); }
                virtual void getMemoryStatistics(MemoryStatistics& u) const HK_OVERRIDE { HK_ASSERT_NO_MSG(0x3e54b379, 0); }
                virtual int getAllocatedSize(_In_bytecount_(nbytes) const void* obj, int nbytes) const HK_OVERRIDE { HK_ASSERT_NO_MSG(0x55c1aa1, 0); return nbytes; }
            };
            Alloc alloc(buf);

            // stores the Fields just after the Type data
            Detail::DeclsArray* fieldArray = Detail::DeclsArray::create(HK_NULL, m_nextField, m_fields.getSize() - m_nextField, alloc);

            for (int i = 0; i < m_fields.getSize(); ++i)
            {
                
                TypeBuilder& alias = m_fields[i];
                Type* f = alias.writeOnBuffer(buf);
                TypeDetail::localSetOptional<Opt::DECL_CONTEXT>(f, type);
                fieldArray->setField(i, f);
            }

            // set the members
            TypeDetail::localSetOptional<Opt::DECLS>(rec, fieldArray);
        }

        if (m_recomputeLayout)
        {
            hkReflect::RecordLayout::recomputeNative(type);
        }
    }

    // fixup special methods
    if (m_fixupSpecialMethods)
    {
        HK_ASSERT(0x7d7d0eb4, type->asRecord(), "Only records can have implicit special methods" );
        TypeDetail::fixupUnknownSpecialMethods(type, true);
    }
    return type;
}

_Ret_notnull_ hkReflect::Type* hkReflect::TypeBuilder::writeOnBuffer(hkLinearBuffer& buf)
{
    hkReflect::Type* head = writeHeadOnBuffer(buf);
    writeBodyOnBuffer(head, buf);
    return head;
}

void hkReflect::TypeBuilder::addDeleteInfo(hk::DeleteTypeInfo::Func func)
{
    m_useTypeSizeInDeleteInfo = true;
    doAddDeleteInfo(func, 0);
}

void hkReflect::TypeBuilder::addDeleteInfo(hk::DeleteTypeInfo::Func func, hkUlong data)
{
    m_useTypeSizeInDeleteInfo = false;
    doAddDeleteInfo(func, data);
}

void hkReflect::TypeBuilder::doAddDeleteInfo(hk::DeleteTypeInfo::Func func, hkUlong data)
{
    HK_ASSERT_NO_MSG(0x5c97f885, m_deleteAttr == HK_NULL);

    // check that the builder does not contain a DeleteTypeInfo attribute already
    for (int i = 0; i < m_attributes.getSize(); ++i)
    {
        HK_ASSERT_NO_MSG(0x4f32f71a, !hkDynCast<hk::DeleteTypeInfo>(m_attributes[i]));
    }

    if (!func)
    {
        func = &hkReflect::TypeBuilder::deallocateType;
    }

    m_deleteAttr = new hk::DeleteTypeInfo;
    m_deleteAttr->m_func = func;
    m_deleteAttr->m_data = data;
    addAttribute(hkReflect::Var(m_deleteAttr));
}

bool hkReflect::TypeBuilder::isSameTypeWorld(_In_opt_ const Type* type) const
{
    return type == HK_NULL || m_typeWorld == type->getTypeWorld();
}

_Ret_notnull_ hkReflect::Type* hkReflect::TypeBuilder::allocate(_Inout_ hkMemoryAllocator* allocator)
{
    hkReflect::Type* type;
    HK_VERIFY(0x3be40800, allocate(allocator, &type).isSuccess(), "Failed to allocate type from builder");
    return type;
}
hkResult hkReflect::TypeBuilder::allocate(hkMemoryAllocator* allocator, _Outptr_ hkReflect::Type** type)
{
    *type = nullptr;
    FAIL_WITH_ID_IF(0x18bd024f, (m_optMask & Opt::FORMAT) && !(m_optMask & Opt::SIZE_ALIGN), "You must have an Opt::SIZE_ALIGN if you have an Opt::FORMAT");

    int size = getTotalNeededMemory();
    if (m_deleteAttr)
    {
        m_deleteAttr->m_allocator = allocator;
        if (m_useTypeSizeInDeleteInfo)
        {
            m_deleteAttr->m_data = size;
        }
    }

    // saves the allocator pointer at the end of the buffer
    void* buffer = allocator->blockAlloc(size);
    hkMemUtil::memSet(buffer, 0, size);

    *type = writeOnBuffer(buffer, size);
    return HK_SUCCESS;
}

void hkReflect::TypeBuilder::destructType(_Inout_ hkReflect::Type* type)
{
    // delete all the type attributes
    if (auto* attrArray = const_cast<hkReflect::Detail::AttributeArray*>( hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::ATTRIBUTES>(type)) )
    {
        hkArrayView<Detail::AttributeItem> items = attrArray->items();
        for (int i = 0; i < items.getSize(); ++i)
        {
            if (items[i].type->extendsOrEquals<hk::AutoType>())
            {
                hkReflect::Var(items[i].data, type).destroy();
            }
            else
            {
                hkReflect::Var(items[i].data, items[i].type).destroy();
            }
            // clear the deleted attribute to prevent destructors from accessing it again
            items[i].data = HK_NULL;
            items[i].type = hkReflect::getType<Detail::Opaque>();
        }
    }

    if (const void* defVal = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::DEFAULT>(type))
    {
        hkReflect::Var(defVal, type).destroy();
    }
}

void hkReflect::TypeBuilder::deallocateType(_Inout_ hkReflect::Type* type, _Inout_ hkMemoryAllocator* allocator, hkUlong data)
{
    destructType(type);
    // delete the type buffer
    allocator->blockFree(type, (int)data);
}

hkReflect::TypeBuilder& hkReflect::TypeBuilder::addMember(_In_z_ const char* name, hkReflect::Decl::DeclFlags flags, _In_ const hkReflect::Type* type)
{
    HK_ASSERT(0x25df27e7, getItem<hkReflect::Opt::DECLS>() == nullptr, "Opt::DECLS must be reset before using addMember");
    TypeBuilder& member = flags.anyIsSet(hkReflect::Decl::DECL_PROPERTY_FIELD) ?
        m_fields.expandOne() : *m_fields.expandAt(m_nextField++, 1);

    // Set the Opt bit in case it wasn't.
    // Opt::DECLS == null is used as a marker that a DeclsArray is required (not beginShallowClone).
    // Here this means we ignore any member from the cloned type, and recreate a DECLS array from scratch.
    setItem<hkReflect::Opt::DECLS>(nullptr);

    member.setTypeWorld(m_typeWorld);
    member.beginDerived(type);
    member.setField(name, flags.anyIsSet(hkReflect::Decl::DECL_PROPERTY_FIELD) ? 0 : 0xffff, flags);

    return member;
}

hkReflect::TypeBuilder& hkReflect::TypeBuilder::addMember(_In_z_ const char* name, int offset, hkReflect::Decl::DeclFlags flags, _In_ const hkReflect::Type* type)
{
    HK_ASSERT(0x6332d2d9, getItem<hkReflect::Opt::DECLS>() == nullptr, "Opt::DECLS must be reset before using addMember");
    TypeBuilder& member = flags.anyIsSet(hkReflect::Decl::DECL_PROPERTY_FIELD) ?
        m_fields.expandOne() : *m_fields.expandAt(m_nextField++, 1);

    // Set the Opt bit in case it wasn't.
    // Opt::DECLS == null is used as a marker that a DeclsArray is required (not beginShallowClone).
    // Here this means we ignore any member from the cloned type, and recreate a DECLS array from scratch.
    setItem<hkReflect::Opt::DECLS>(nullptr);

    member.setTypeWorld(m_typeWorld);
    member.beginDerived(type);
    member.setField(name, offset, flags);

    return member;
}

hkReflect::TypeBuilder& hkReflect::TypeBuilder::addProperty(_In_z_ const char* name, _In_ const hkReflect::Type* type)
{
    return addMember(name, hkReflect::Decl::DECL_PROPERTY_FIELD, type);
}

void hkReflect::TypeBuilder::setField(_In_z_ const char* fieldName, int offset, hkReflect::Decl::DeclFlags flags)
{
    addItem<hkReflect::Opt::DECL_NAME>(fieldName);
    if (flags.noneIsSet(hkReflect::Decl::DECL_FIELD))
    {
        flags.orWith(hkReflect::Decl::DECL_DATA_FIELD);
    }
    setItem<hkReflect::Opt::DECL_FORMAT>(hkReflect::Detail::FieldDeclFormat(offset, flags.get()));
    setItem<hkReflect::Opt::DECL_CONTEXT>(HK_NULL);
}

void hkReflect::TypeBuilder::addTrivialSpecialMethods()
{
    addItem<Opt::DEF_CONSTRUCTOR>(hkReflect::Detail::Trivial<hkReflect::Opt::DEF_CONSTRUCTOR>::func);
    addItem<Opt::COPY_CONSTRUCTOR>(hkReflect::Detail::Trivial<hkReflect::Opt::COPY_CONSTRUCTOR>::func);
    addItem<Opt::COPY_ASSIGNMENT>(hkReflect::Detail::Trivial<hkReflect::Opt::COPY_ASSIGNMENT>::func);
}

void hkReflect::TypeBuilder::addUnknownSpecialMethods()
{
    addItem<Opt::DEF_CONSTRUCTOR>(hkReflect::Detail::Unknown<Opt::DEF_CONSTRUCTOR>::func);
    addItem<Opt::COPY_CONSTRUCTOR>(hkReflect::Detail::Unknown<Opt::COPY_CONSTRUCTOR>::func);
    addItem<Opt::COPY_ASSIGNMENT>(hkReflect::Detail::Unknown<Opt::COPY_ASSIGNMENT>::func);
    addItem<Opt::DESTRUCTOR>(hkReflect::Detail::assertOnInvalidDestruction);
}

void hkReflect::TypeBuilder::addImplicitSpecialMethods()
{
    addUnknownSpecialMethods();
    m_fixupSpecialMethods = true;
}

void hkReflect::TypeBuilder::overrideFormat(hkUlong format)
{
    setItem<hkReflect::Opt::FORMAT>(format);
}

hkReflect::BasicPointerType::BasicPointerType(QualType ptdType, _In_z_ const char* name, Type::Flags extraFlags)
    : m_sizeAlign(hkReflect::Detail::SizeAlign(sizeof(void*), HK_ALIGN_OF(void*)).asUlong())
{
    m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::TEMPLATE | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS;
    m_parent = HK_NULL;
    m_format = hkReflect::Format::OfPointer::Value;
    m_next = ptdType;
    m_name = name;
    m_flags = extraFlags.get();

    m_impl = &hkReflect::Detail::RawPointerImpl::s_instance;

    m_template = hkReflect::Detail::TemplateParameterArray::createInPlace(&m_templateStorage, 1);
    m_template->item(0).m_storage = hkUlong(ptdType.get());
    m_template->item(0).m_kindAndName = "tT";
}

hkReflect::BasicPointerType::BasicPointerType(QualType ptdType, _In_z_ const char* name, _Inout_ hkReflect::Detail::TemplateParameterArray* templ, Type::Flags extraFlags)
    : m_sizeAlign(hkReflect::Detail::SizeAlign(sizeof(void*), HK_ALIGN_OF(void*)).asUlong())
{
    m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::TEMPLATE | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS;
    m_parent = HK_NULL;
    m_format = hkReflect::Format::OfPointer::Value;
    m_next = ptdType;
    m_name = name;
    m_impl = &hkReflect::Detail::RawPointerImpl::s_instance;
    m_template = templ;
    m_flags = extraFlags.get();
}


hkReflect::BasicArrayType::BasicArrayType(QualType elemType, int sizeArr, int alignArr, _In_ const Detail::ArrayImpl* impl, _In_z_ const char* name, Type::Flags extraFlags)
    : m_name( name ), m_sizeAlign( hkReflect::Detail::SizeAlign( sizeArr, alignArr ).asUlong() )
{
    m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS;
    m_parent = HK_NULL;
    m_format = hkReflect::Format::OfArray<0>::Value;
    m_elemType = elemType;
    m_impl = impl;
    m_flags = extraFlags.get();
}

hkReflect::BasicArrayTypeWithParams::BasicArrayTypeWithParams(QualType elemType, int sizeArr, int alignArr, _In_ const Detail::ArrayImpl* impl, _In_z_ const char* name, Type::Flags extraFlags)
    : m_name(name), m_sizeAlign(hkReflect::Detail::SizeAlign(sizeArr, alignArr).asUlong())
{
    m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::TEMPLATE | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS;
    m_parent = HK_NULL;
    m_format = hkReflect::Format::OfArray<0>::Value;
    m_elemType = elemType;
    m_impl = impl;
    m_flags = extraFlags.get();

    m_template = hkReflect::Detail::TemplateParameterArray::createInPlace(&m_templateStorage, 2);
    m_template->item(0).m_storage = hkUlong(elemType.get());
    m_template->item(0).m_kindAndName = "tT";
    // Assume everything is a heap array
    m_template->item(1).m_storage = hkUlong(hkReflect::getType<hkContainerHeapAllocator>().get());
    m_template->item(1).m_kindAndName = "tAllocator";
}

hkReflect::RepeatArrayType::RepeatArrayType(QualType elemType, int repeatCount, int sizeArr, int alignArr, _In_ const Detail::ArrayImpl* impl, _In_z_ const char* name, Type::Flags extraFlags)
    : m_name(name), m_sizeAlign(hkReflect::Detail::SizeAlign(sizeArr, alignArr).asUlong())
{
    m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::TEMPLATE | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS;
    m_parent = HK_NULL;
    m_format = hkReflect::Format::ofArray(repeatCount);
    m_elemType = elemType;
    m_impl = impl;
    m_flags = extraFlags.get();

    m_template = hkReflect::Detail::TemplateParameterArray::createInPlace(&m_templateStorage, 1);
    m_template->item(0).m_storage = hkUlong(elemType.get());
    m_template->item(0).m_kindAndName = "tT";
}

hkReflect::BasicStringType::BasicStringType(int sizeStr, int alignStr, _In_ const Detail::StringImpl* impl, _In_z_ const char* name, bool immutable, Type::Flags extraFlags)
    : m_sizeAlign(hkReflect::Detail::SizeAlign(sizeStr, alignStr).asUlong())
{
    m_optional = hkReflect::Opt::FORMAT | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS;
    m_format = immutable ? (hkUlong)hkReflect::Format::OfStringUtf8<0, true>::Value : (hkUlong)hkReflect::Format::OfStringUtf8<0, false>::Value;
    m_parent = HK_NULL;
    m_impl = impl;
    m_name = name;
    m_flags = extraFlags.get();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
