// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

#define DEBUG_LOG_IDENTIFIER "s11n.RecordLayout"
#include <Common/Base/System/Log/hkLog.hxx>

// Notes: an alignment of zero means we haven't visited/laid out the type yet
namespace
{
        // Get the amount of space actually used by type which may be less than sizeof.
    int getUsedSizeOf(_In_ const hkReflect::Type* type, int sizeofPointer)
    {
        using namespace hkReflect;
        for(; type != HK_NULL; type = type->getParent())
        {
            // Some fields? We go after the last field
            if(const Detail::DeclsArray* decls = TypeDetail::localGetOptional<Opt::DECLS>(type))
            {
                if(int nf = decls->getNumDataFields())
                {
                    DataFieldDecl fd = decls->getDataFields()[nf - 1];
                    return fd.getOffset() + fd.getType()->getSizeOf();
                }
            }
            // No fields. Maybe some vtable slots? we go after the last one
            if(const Detail::InterfaceArray* iptr = TypeDetail::localGetOptional<Opt::INTERFACES>(type))
            {
                if( int numInterfaces = iptr->items().getSize() )
                {
                    int size = type->getParent() ? getUsedSizeOf(type->getParent(), sizeofPointer) : 0;
                    size = HK_NEXT_MULTIPLE_OF(sizeofPointer, size);
                    if( type->hasOwnVtable() )
                    {
                        size += sizeofPointer;
                    }
                    size += sizeofPointer * numInterfaces;
                    return size;
                }
            }

            // No interfaces, perhaps just a single vtable?
            if( type->hasOwnVtable() )
            {
                int size = type->getParent() ? getUsedSizeOf(type->getParent(), sizeofPointer) : 0;
                size = HK_NEXT_MULTIPLE_OF(sizeofPointer, size);
                return size + sizeofPointer;
            }
        }
        return 0;
    }

    static int computeEffectiveAlignment(_In_ const hkReflect::Type* type, int curAlign, int sizeOfReal)
    {
        using namespace hkReflect;
        Detail::SizeAlign sa(TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(type));
        int ra = sa.retargetAlignment(sizeOfReal);
        if(ra > curAlign)
        {
            curAlign = ra;
            DLOG("increased align={}", curAlign);
        }
        return curAlign;
    }

    static int computeMaxExplicitClassAlign(_In_ const hkReflect::Type* type, int sizeOfReal)
    {
        int calign = 0;
        for(const hkReflect::Type* t = type; t; t = t->getParent())
        {
            using namespace hkReflect;
            if(hkUlong packed = TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(t))
            {
                Detail::SizeAlign sa(packed);
                int ra = 0;
                switch(sa.m_reqAlignEncoded)
                {
                    case 0: break;
                    case Detail::ALIGN_REQ_REAL4: ra = sizeOfReal*4; break;
                    default: ra = 1 << sa.m_reqAlignEncoded; break;
                }
                if(ra > calign)
                {
                    calign = ra;
                }
            }
            if(TypeDetail::localHasOptional(t, Opt::FORMAT))
            {
                break;
            }
        }
        return calign;
    }

    static int computeMaxFieldAlign(_In_ const hkReflect::Type* type, int sizeofReal, int sizeofPtr)
    {
        int falign = 1;
        for(hkReflect::DeclIter<hkReflect::DataFieldDecl> iter(type); iter.advance();)
        {
            hkReflect::DataFieldDecl f = iter.current();
            falign = computeEffectiveAlignment(f.getType(), falign, sizeofReal);
        }
        for( const hkReflect::Type* t = type; t; t=t->getParent() )
        {
            if( t->hasOwnVtable() )
            {
                falign = hkMath::max2(falign, sizeofPtr);
                break;
            }
        }
        return falign;
    }


    void recomputeFromDecls(_Inout_ hkReflect::Type* type, const hkReflect::RecordLayout::Options& options)
    {
        DLOG_SCOPE("TypeLayout {}", type->getFullName());
        using namespace hkReflect;

        int curOffset = 0;
        int curAlign = 1;
        int classAlign = 0;
        const hkReflect::Type* parent = type->getParent();
        if(parent)
        {
            HK_ASSERT_NO_MSG(0x2cf1d8d8, parent->getAlignOf() >= 1);
            curOffset = parent->getSizeOf();
            curAlign = parent->getAlignOf();

            // If we're at offset 1, it could be because the parent class has no fields
            // try empty base class optimization
            if(options.m_emptyBaseClassOptimization && curOffset == 1)
            {
                curOffset = 0;
                for(const Type* p = parent; p != HK_NULL; p = p->getParent())
                {
                    if(const Detail::DeclsArray* fa = TypeDetail::localGetOptional<Opt::DECLS>(p))
                    {
                        if(fa->getNumDataFields())
                        {
                            
                            
                            

                            // some parent has a field, the optimization doesn't hold, restore the offset.
                            curOffset = 1;
                            break;
                        }
                    }
                }
                if(curOffset == 0)
                {
                    DLOG("applying empty base class optimization");
                }
            }
            // struct X { double d; char c; }; struct Y : public X { int i; }
            // offsetof(Y,i) could be 12 even though sizeof(X) is 16
            if(curOffset != 0)
            {
                if(options.m_reuseParentPadding)
                {
                    int offset = getUsedSizeOf(parent, options.m_pointerSize);
                    if(offset < curOffset)
                    {
                        curOffset = offset;
                        DLOG("applying reuse padding optimization");
                    }
                }
                else if(options.m_reuseClassAlignPadding)
                {
                    // MSVC allows reusing padding when alignment is on the base CLASS not a base FIELD
                    // Check here is the current alignment the result of an alignment on the CLASS.
                    int calign = computeMaxExplicitClassAlign(parent, options.m_sizeofReal);

                    // The current alignment MAY be caused by the class alignment, we have to check all the fields
                    if(curAlign == calign)
                    {
                        int falign = computeMaxFieldAlign(parent, options.m_sizeofReal, options.m_pointerSize);

                        // Layout change only kicks in if class alignment is greater than field alignment.
                        if(calign > falign)
                        {
                            curAlign = falign;
                            classAlign = calign;
                            int offset = getUsedSizeOf(parent, options.m_pointerSize);
                            if(offset < curOffset)
                            {
                                curOffset = offset;
                                DLOG("applying reuse padding optimization2");
                            }
                        }
                    }
                }
            }
        }
        DLOG("starting size={}, align={}", curOffset, curAlign);

        // Check for extra class alignment
        curAlign = computeEffectiveAlignment(type, curAlign, options.m_sizeofReal);

        // reserve space for a vtable if needed
        if(type->hasOwnVtable())
        {
            if(curAlign < options.m_pointerSize)
            {
                curAlign = options.m_pointerSize;
            }
            if(int misalign = curOffset % curAlign)
            {
                curOffset += curAlign - misalign;
            }
            DLOG("vtable @ {}", curOffset);
            curOffset += options.m_pointerSize;
        }

        if(const Detail::InterfaceArray* interfaces = TypeDetail::localGetOptional<Opt::INTERFACES>(type))
        {
            hkArrayView<const Detail::Interface> ifaces = interfaces->items();
            for(int i = 0; i < ifaces.getSize(); ++i)
            {
                const Type* itype = ifaces[i].m_interfaceType;
                int size = itype->getSizeOf();
                int align = itype->getAlignOf();
                HK_ASSERT_NO_MSG(0x65fd864b, size && align);
                if(curAlign < align)
                {
                    curAlign = align;
                }
                if(int misalign = curOffset % align)
                {
                    curOffset += align - misalign;
                }
                const_cast<Detail::Interface&>(ifaces[i]).m_offset = curOffset;
                curOffset += size;
            }
        }

        HK_ON_DEBUG(const Detail::DeclsArray* decls2 = TypeDetail::localGetOptional<Opt::DECLS>(type));
        const Detail::DeclsArray* decls = TypeDetail::decoratorGetOptional<Opt::DECLS>(type);
        HK_ASSERT_NO_MSG(0x4b913be8, decls2 == decls);
        for( auto field : decls->getDataFields() )
        {
            const Type* ftype = field.getType();
            HK_ASSERT_NO_MSG(0x7490a1b9, ftype);
            RecordLayout::recompute(const_cast<hkReflect::Type*>(ftype), options);

            // Check for extra field alignment
            if( const int fieldAlign = computeEffectiveAlignment(ftype, ftype->getAlignOf(), options.m_sizeofReal) )
            {
                HK_ASSERT_NO_MSG(0x797e1d6c, fieldAlign >= 1);
                if(curAlign < fieldAlign)
                {
                    curAlign = fieldAlign;
                }
                if(int misalign = curOffset % fieldAlign)
                {
                    curOffset += fieldAlign - misalign;
                }
            }
            else
            {
                // Fields with 0 align should be Incomplete or Void.
                HK_ASSERT_NO_MSG(0x41a6fc58, ftype->asOpaque() || ftype->asVoid());
            }
            TypeDetail::setFieldOffset(field, curOffset);
            DLOG("{} = {} @ {}", field.getName(), ftype->getSizeOf(), curOffset);
            curOffset += ftype->getSizeOf();
        }

        curOffset = hkMath::max2(curOffset, options.m_minimumRecordSize);
        curAlign = hkMath::max2(classAlign, curAlign);

        if (curAlign)
        {
            // alignment padding at the end
            if(int misalign = curOffset % curAlign)
            {
                curOffset += curAlign - misalign;
            }
        }
        curOffset = hkMath::max2(1, curOffset); // sizeof cannot be zero

        if (parent && curOffset < parent->getSizeOf())
        {
            // This can happen when the parent has fields at the end of the struct that aren't reflected. So
            // sizeof of the parent reports the correct size but we trim off the last bit because the reflection
            // system cannot know about unreflected fields.
            HK_ERROR(0x71544192, "Child's final size is less than parent's! (Do you have a trailing unreflected field?)");
        }

        DLOG("end {} size={}, align={}", type->getFullName(), curOffset, curAlign);
        TypeDetail::localSetSizeAlignPreserveReq(type, curOffset, curAlign);
    }

    struct SizeVisitor : public hkReflect::TypeVisitor< SizeVisitor >
    {
        SizeVisitor(const hkReflect::RecordLayout::Options& opts)
            : m_options(opts)
        {
        }

        void visit(_In_ const hkReflect::Type* t)
        {
            HK_ASSERT_NO_MSG(0x2f5cd378, 0);
        }

        void visit(_In_ const hkReflect::ArrayType* type)
        {
            using namespace hkReflect;
            int count = type->getFixedCount();
            if(count > 0)
            {
                // Repeat type, can't do this until we have the subtype
                const Type* elemType = type->getSubType();
                recompute(const_cast<Type*>(elemType), m_options);

                Detail::SizeAlign sa(TypeDetail::decoratorGetOptional<Opt::SIZE_ALIGN>(elemType));
                sa.m_sizeOf *= count;
                sa.m_sizeOf = hkMath::max2(sa.m_sizeOf, m_options.m_minimumRecordSize);

                TypeDetail::localSetOptionalUlong( const_cast<ArrayType*>(type), Opt::SIZE_ALIGN, sa.asUlong());
            }
            else
            {
                HK_ASSERT_NO_MSG(0x6787bbe5, 0);
            }
        }

        const hkReflect::RecordLayout::Options m_options;
    };
}

void HK_CALL hkReflect::RecordLayout::recompute(_Inout_ hkReflect::Type* typeIn, const Options& options)
{
    Type* type = typeIn;

    // early outs
    if( type->asOpaque() || type->asVoid() )
    {
        HK_ASSERT_NO_MSG(0x524d06e0, type->getSizeOf() == 0);
        HK_ASSERT_NO_MSG(0x549ea357, type->getAlignOf() == 0);
        return;
    }

    // skip decorators, get to something with a size/align
    while( TypeDetail::localHasOptional(type, Opt::SIZE_ALIGN) == false )
    {
        type = const_cast<Type*>(type->getParent());
        HK_ASSERT_NO_MSG(0x6eef150c, type);
    }

    // has it already been computed?
    if( Detail::SizeAlign(TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(type)).m_sizeOf )
    {
        return;
    }
    DLOG_SCOPE("Recompute {}@{}", typeIn->getFullName(), typeIn);

    // ensure parents are computed first
    if( const Type* p = type->getParent() )
    {
        recompute( const_cast<Type*>(p), options );
    }

    // Now the actual work
    if( TypeDetail::localGetOptional<Opt::FORMAT>(type) == 0 )
    {
        // no opt format, it's a field (with an HK_ALIGN on it)
        // copy from the parent, allowing for the extra alignment.
        HK_ASSERT_NO_MSG(0x23d4bee0, type->getParent());
        Detail::SizeAlign par(TypeDetail::decoratorGetOptional<Opt::SIZE_ALIGN>(type->getParent()));
        Detail::SizeAlign cur(TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(type));
        cur.m_sizeOf = par.m_sizeOf;
        cur.m_alignOf = par.m_alignOf;
        cur.m_reqAlignEncoded = hkMath::max2(par.m_reqAlignEncoded, cur.m_reqAlignEncoded);
        cur.m_alignOf = cur.retargetAlignment(options.m_sizeofReal);
        TypeDetail::localSetOptionalUlong(type, Opt::SIZE_ALIGN, cur.asUlong());
    }
    else if(TypeDetail::localGetOptional<Opt::DECLS>(type))
    {
        recomputeFromDecls(type, options);
    }
    else
    {
        SizeVisitor(options).dispatch(type);
    }
}

namespace
{
    struct EmptyBase
    {
    };

    struct InheritsEmptyBase : public EmptyBase
    {
        int x;
    };

    struct EndPadding
    {
        struct Quad
        {
            Quad() { }
            HK_ALIGN16( int x[4] );
        };

        Quad q;
        int y;
    };

    struct InheritsEndPadding : public EndPadding
    {
        int z;
    };

    HK_CLASSALIGN(struct, 16) AlignedBase1
    {
        AlignedBase1();
        hkUint8 tag;
    };

    struct InheritsAlign1 : public AlignedBase1
    {
        hkUint8 val;
    };
}

hkReflect::RecordLayout::Options::Options()
{
    m_pointerSize = -1;
    m_sizeofReal = -1;
    m_reuseParentPadding = true;
    m_reuseClassAlignPadding = true;
    m_emptyBaseClassOptimization = true;
    m_minimumRecordSize = 0;
}


hkReflect::RecordLayout::Options& hkReflect::RecordLayout::Options::setNative()
{
    m_pointerSize = sizeof(void*);
    m_sizeofReal = sizeof(hkReal);
    m_reuseParentPadding = HK_OFFSET_OF(InheritsEndPadding, z) != sizeof(EndPadding);
    m_reuseClassAlignPadding = HK_OFFSET_OF(InheritsAlign1, val) == 1;
    m_emptyBaseClassOptimization = HK_OFFSET_OF(InheritsEmptyBase, x) == 0;
    m_minimumRecordSize = 0;
    return *this;
}

void hkReflect::RecordLayout::recomputeNative(_Inout_ hkReflect::Type* typeInOut)
{
    recompute(typeInOut, Options().setNative());
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
