// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Container/ArrayView/hkArrayView.h>
#include <Common/Base/Container/StringView/hkStringView.h>

class hkAttrData
{
    public:
        /// Describes a field of an attribute.
        class FieldDecl
        {
        public:

            enum FlagBits
            {
                FLAG_NONE       = 0x0,
                FLAG_OPTIONAL   = 0x1,
                FLAG_EXPLICIT   = 0x2
            };
            typedef hkFlags<FlagBits, hkUint8> Flags;

            /// Create a declaration for a scalar field.
            static inline FieldDecl simple(_In_z_ const char* name, Flags flags = FLAG_NONE);

            /// Create a declaration for an array field with fixed size.
            static inline FieldDecl fixedArray(_In_z_ const char* name, int numElements, Flags flags = FLAG_NONE);

            /// Create a declaration for an array field with variable size.
            static inline FieldDecl variableArray(_In_z_ const char* name, Flags flags = FLAG_NONE);

            inline _Ret_z_ const char* getName() const;
            inline bool isOptional() const;
            inline bool isExplicit() const;

            inline bool isSimple() const;
            inline bool isVariableArray() const;
            inline bool isFixedArray() const;

            /// Get the fixed size. Asserts that this field is a fixed array.
            inline int getFixedSize() const;

        private:
            inline FieldDecl(_In_z_ const char* name, int numElements, Flags flags);

            const char* m_name;
            int m_numElements;  
            Flags m_flags;
        };

    private:
        /// Base class for handles to fields in hkAttrData.
        template<typename AttrDataType>
        class FieldBase
        {
            public:

                inline const FieldDecl& getDecl() const;

                /// Return true if this is a valid field.
                inline bool isValid() const;

                /// Return true if parsing has produced a value for this field.
                inline bool isSet() const;

                /// Get the value for this field. Asserts that this field is simple.
                inline hkStringView getValue() const;

                /// Get the values for this field. Asserts that this field is an array.
                inline hkArrayView<const hkStringView> getValuesArray() const;

            protected:
                inline FieldBase(AttrDataType& data, int fieldIdx);

                AttrDataType& m_data;
                int m_fieldIdx;

            friend class hkAttrData;
        };

    public:
        /// Const handle to field;
        typedef FieldBase<const hkAttrData> Field;

        /// Writable handle to field.
        class WritableField : public FieldBase<hkAttrData>
        {
            public:
                /// Set the value for this field. Asserts that this field is simple.
                inline void setValue(const hkStringView& value) const;

                /// Set the values for this field. Asserts that this field is an array and that the size matches.
                inline void setValuesArray(const hkArrayView<const hkStringView>& values) const;

            protected:
                inline WritableField(hkAttrData& data, int fieldIdx);

            friend class hkAttrData;
        };

    public:

        inline void addDecl(const FieldDecl& decl);

        inline Field getField(int fieldIdx) const;
        Field getField(_In_z_ const char* name) const;

        inline WritableField getWritableField(int fieldIdx);

        inline int getNumFields() const;

    private:
        struct FieldData
        {
            FieldData(const FieldDecl& d) : decl(d), begin(-1), end(-1) {}

            FieldDecl decl;
            int begin;
            int end;
        };

        hkInplaceArray<FieldData, 8> m_fields;
        hkInplaceArray<hkStringView, 16> m_values;
};

class HK_EXPORT_COMMON hkAttributeParser
{
    public:
        hkAttributeParser(const hkStringView& attrString);

        /// Moves to the next attribute.
        bool advance();
        /// Writes the name of the current attribute on the provided buffer.
        /// If an error occurs, returns HK_FAILURE and writes an error message into the buffer.
        hkResult currentAttrName(hkStringBuf& buf) const;
        /// Parses the current attribute using the provided declaration and fills the
        /// hkAttrData fields with the corresponding values.
        hkResult currentAttrValue(hkAttrData& attrData, hkStringBuf& errorBuf) const;

    private:
        hkStringView m_nextAttr;
        hkStringView m_currAttrName;
        hkStringView m_currArgs;
};

#include <Common/Base/Reflect/Attributes/hkAttributeParser.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
