// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Attributes/hkAttributeParser.h>

#define HK_ATTR_FAIL_IF(COND, BUF, ...) \
    while(COND) \
    { \
        BUF.appendPrintf(__VA_ARGS__); \
        return HK_FAILURE; \
    }

#define HK_ATTR_FAIL_WITH_VALUE_IF(COND, VALUE, BUF, ...) \
    while(COND) \
    { \
        BUF.appendPrintf(__VA_ARGS__); \
        BUF.append(" (value: '"); \
        BUF.append((VALUE).begin(), (VALUE).getSize()); \
        BUF.append("')"); \
        return HK_FAILURE; \
    }

namespace
{
    
    enum ScopeTypes
    {
        NONE,
        PARENTHESIS,
        CURLY_BRACES,
        SQUARE_BRACES,
        DOUBLE_QUOTES,
        ANGLE_BRACKETS,

        NUM_SCOPE_TYPES,
    };

    ScopeTypes isOpeningScopeDelimiter(const char c, unsigned char count[NUM_SCOPE_TYPES])
    {
        if(count[DOUBLE_QUOTES])
        {
            // We're currently in a quoted string, so we ignore everything until
            // we find our closing double quote.
            return NONE;
        }
        else if(c == '(')
            return PARENTHESIS;
        else if(c == '{')
            return CURLY_BRACES;
        else if(c == '[')
            return SQUARE_BRACES;
        else if(c == '"')
            return DOUBLE_QUOTES;
        else if(c == '<')
            return ANGLE_BRACKETS;
        else
            return NONE;
    }

    ScopeTypes isClosingScopeDelimiter(const char c, unsigned char count[NUM_SCOPE_TYPES])
    {
        if(c == ')')
            return PARENTHESIS;
        else if(c == '}')
            return CURLY_BRACES;
        else if(c == ']')
            return SQUARE_BRACES;
        else if(c == '"' && count[DOUBLE_QUOTES] > 0)
            return DOUBLE_QUOTES;
        else if (c == '>')
            return ANGLE_BRACKETS;
        else
            return NONE;
    }

    union ScopeCount
    {
        hkUint64 isNonZero;
        unsigned char count[NUM_SCOPE_TYPES];
    };

    hkStringView::iterator findWithScoping(hkStringView value, char chr)
    {
        ScopeCount scopes = { 0 };

        hkStringView::iterator strIt = value.begin();
        while(strIt != value.end())
        {
            if (!scopes.isNonZero && *strIt == chr)
                break;

            if(ScopeTypes closeType = isClosingScopeDelimiter(*strIt, scopes.count))
            {
                if (scopes.count[closeType])
                {
                    --scopes.count[closeType];
                }
            }
            else if(ScopeTypes openType = isOpeningScopeDelimiter(*strIt, scopes.count))
            {
                ++scopes.count[openType];
            }

            ++strIt;
        }
        return strIt;
    }

    void splitStringWithScoping(hkStringView value, char splitChar, hkArray<hkStringView>& tokensOut)
    {
        hkStringView::iterator strIt = value.begin();
        while(strIt != value.end())
        {
            hkStringView::iterator tokenStart = strIt;
            strIt = findWithScoping(value.ltrim(strIt), splitChar);
            tokensOut.pushBack(hkStringView(tokenStart, strIt));
            if(strIt != value.end())
                ++strIt;
        }
    }

    hkStringView trimBegin(const hkStringView& str)
    {
        hkStringView::iterator it = str.begin();
        while (it != str.end() && (*it == ' ' || *it == '\t' || *it == '\n'))
        {
            ++it;
        }
        return str.ltrim(it);
    }

    hkStringView trimEnd(const hkStringView& str)
    {
        hkStringView::reverse_iterator it = str.rbegin();
        while (it != str.rend() && (*it == ' ' || *it == '\t' || *it == '\n'))
        {
            ++it;
        }
        return str.rtrim(it.getForward() + 1);
    }

    hkStringView trim(const hkStringView& str)
    {
        return trimEnd(trimBegin(str));
    }

    hkStringView removeScope(const hkStringView& str, char begin, char end)
    {
        if (*str.begin() == begin)
        {
            hkStringView::iterator it = findWithScoping(str.ltrim(1), end);
            if (it == str.end())
            {
                // scope is not closed correctly
                return hkStringView();
            }
            if (it == str.end() - 1)
            {
                return str.slice(1, str.getSize()-2);
            }
        }
        return str;
    }

    hkResult checkValue(const hkStringView& value, hkStringBuf& errorBuf)
    {
        // check that the value is not empty
        HK_ATTR_FAIL_IF(!value, errorBuf, "Found empty value");

        return HK_SUCCESS;
    }

    hkResult setFieldValuesArray(const hkAttrData::WritableField& field, const hkArrayView<const hkStringView>& values, hkStringBuf& errorBuf)
    {
        const hkAttrData::FieldDecl& fieldDecl = field.getDecl();

        // Values can be fewer than the fixed size, but not more.
        HK_ATTR_FAIL_IF( fieldDecl.isFixedArray() && values.getSize() > fieldDecl.getFixedSize(), errorBuf,
            "Found %d elements for a %d-element array", values.getSize(), fieldDecl.getFixedSize() );
        hkInplaceArray<hkStringView, 32> trimmed;
        trimmed.setSize(values.getSize());
        for (int i = 0; i < values.getSize(); ++i)
        {
            hkStringView elem = trim(values[i]);
            hkResult res = checkValue(elem, errorBuf);
            if (!res.isSuccess())
            {
                return HK_FAILURE;
            }
            trimmed[i] = elem;
        }
        field.setValuesArray(trimmed);
        return HK_SUCCESS;
    }

    hkStringView checkIdentifier(const hkStringView& str)
    {
        if (!str || (str[0] != '_' && !(str[0] >= 'A' && str[0] <= 'Z') && !(str[0] >= 'a' && str[0] <= 'z')))
        {
            return hkStringView();
        }

        for (int i = 1; i < str.getSize(); ++i)
        {
            if (str[i] != '_' && !(str[i] >= 'A' && str[i] <= 'Z')  && !(str[i] >= 'a' && str[i] <= 'z') && !(str[i] >= '0' && str[i] <= '9'))
            {
                return str.rtrim(str.getSize() - i);
            }
        }
        return str;
    }

    bool isValidFieldName(const hkStringView& name)
    {
        return name && checkIdentifier(name) == name;
    }

    hkResult setField(const hkStringView& value, const hkAttrData::WritableField& field, hkStringBuf& errorBuf,
        const hkArrayView<const hkStringView>& posArgs, int& posIdx)
    {
        const hkAttrData::FieldDecl& fieldDecl = field.getDecl();

        if (fieldDecl.isSimple())
        {
            // simple value
            hkResult res = checkValue(value, errorBuf);
            HK_ATTR_FAIL_WITH_VALUE_IF(!res.isSuccess(), value, errorBuf, " in value for field '%s'", fieldDecl.getName());
            field.setValue(value);
        }
        else
        {
            hkStringView withoutBraces = removeScope(value, '{', '}');
            HK_ATTR_FAIL_WITH_VALUE_IF(!withoutBraces.data(), value, errorBuf, "Cannot find matching braces in positional argument %d", posIdx);

            if (withoutBraces != value)
            {
                // use the elements inside the braces
                hkInplaceArray<hkStringView, 16> elements;
                splitStringWithScoping(withoutBraces, ',', elements);
                hkResult res = setFieldValuesArray(field, elements, errorBuf);
                HK_ATTR_FAIL_IF(!res.isSuccess(), errorBuf, " when assigning positional argument %d to field '%s'", posIdx, fieldDecl.getName());
            }
            else
            {
                // collect all the positional args

                
                //HK_ATTR_FAIL_IF(attrData.getNumFields() != 1, errorBuf,
                //  "Field '%s' is not the only field and it cannot be filled with positional arguments", fieldDecl.getName());

                hkResult res = setFieldValuesArray(field, posArgs.ltrim(posIdx), errorBuf);
                HK_ATTR_FAIL_IF(!res.isSuccess(), errorBuf, " when assigning positional arguments to field '%s'", fieldDecl.getName());
                posIdx = posArgs.getSize();
            }
        }
        return HK_SUCCESS;
    }
}

hkAttributeParser::hkAttributeParser(const hkStringView& attrString)
    : m_nextAttr(attrString)
{
}

bool hkAttributeParser::advance()
{
    if (!m_nextAttr)
    {
        return false;
    }

    hkStringView::iterator comma = findWithScoping(m_nextAttr, ',');
    hkStringView currAttr = m_nextAttr.rtrim(comma);
    const char* argsBegin = findWithScoping(currAttr, '(');
    m_nextAttr = comma == m_nextAttr.end() ? hkStringView() : m_nextAttr.ltrim(comma + 1);

    m_currAttrName = hkStringView(currAttr.begin(), argsBegin);
    m_currAttrName = trim(m_currAttrName);

    if (*argsBegin == 0)
    {
        return true;
    }

    // args
    m_currArgs = hkStringView(argsBegin, currAttr.end());
    m_currArgs = trimEnd(m_currArgs);

    return true;
}

hkResult hkAttributeParser::currentAttrName(hkStringBuf& buf) const
{
    if (!m_currAttrName)
    {
        buf.set("Empty attribute name");
        return HK_FAILURE;
    }

    // copy the attribute name erasing spaces and checking if it is a valid identifier
    hkStringView remaining(m_currAttrName);
    while (remaining)
    {
        hkStringView nextId = checkIdentifier(remaining);
        if (!nextId)
        {
            buf.printf("Invalid attribute name: '");
            buf.append(nextId.begin(), nextId.getSize());
            buf.append("'");
            return HK_FAILURE;
        }
        buf.append(nextId.data(), nextId.getSize());
        remaining = trimBegin(remaining.ltrim(nextId.getSize()));

        if (remaining.getSize() >= 3 && remaining[0] == ':' && remaining[1] == ':')
        {
            buf.append("::");
            remaining = trimBegin(remaining.ltrim(2));
        }
        else if (remaining.getSize() >= 2 && remaining[0] == '.') 
        {
            buf.append(".");
            remaining = trimBegin(remaining.ltrim(1));
        }
        else if (remaining)
        {
            buf.printf("Invalid attribute name: '");
            buf.append(m_currAttrName.begin(), m_currAttrName.getSize());
            buf.append("'");
            return HK_FAILURE;
        }
    }
    return HK_SUCCESS;
}

hkResult hkAttributeParser::currentAttrValue(hkAttrData& attrData, hkStringBuf& errorBuf) const
{
    if (m_currArgs)
    {
        hkStringView withoutBrackets = removeScope(m_currArgs, '(', ')');
        HK_ATTR_FAIL_IF(!withoutBrackets.data(), errorBuf, "Cannot find matching ')'");

        hkInplaceArray<hkStringView, 32> args;
        splitStringWithScoping(withoutBrackets, ',', args);

        hkInplaceArray<hkStringView, 32> posArgs;
        for (int i = 0; i < args.getSize(); ++i)
        {
            const char* equals = findWithScoping(args[i], '=');
            if (equals != args[i].end())
            {
                // named argument
                hkStringView name = trim(hkStringView(args[i].begin(), equals));
                hkStringView value = trim(hkStringView(equals + 1, args[i].end()));

                HK_ATTR_FAIL_IF(!isValidFieldName(name), errorBuf, "Invalid field name: '%s'", hkStringBuf(name).cString());

                // find the field with the provided name
                int fieldIdx;
                for (fieldIdx = 0; fieldIdx < attrData.getNumFields(); ++fieldIdx)
                {
                    hkAttrData::WritableField field = attrData.getWritableField(fieldIdx);
                    const hkAttrData::FieldDecl& fieldDecl = field.getDecl();
                    if (fieldDecl.getName() == name)
                    {
                        // set the value
                        HK_ATTR_FAIL_WITH_VALUE_IF(field.isSet(), value, errorBuf,
                            "Found value for field '%s' which has already been set", fieldDecl.getName());

                        if (fieldDecl.isSimple())
                        {
                            hkResult res = checkValue(value, errorBuf);
                            HK_ATTR_FAIL_WITH_VALUE_IF(!res.isSuccess(), value, errorBuf, " in value for field '%s'", fieldDecl.getName());
                            field.setValue(value);
                        }
                        else
                        {
                            // check and remove braces
                            hkStringView withoutBraces = removeScope(value, '{', '}');
                            HK_ATTR_FAIL_WITH_VALUE_IF(!withoutBraces.data(), value, errorBuf,
                                "Can't find matching braces in value for field '%s'", fieldDecl.getName());
                            HK_ATTR_FAIL_WITH_VALUE_IF(value == withoutBraces, value, errorBuf, "Value for field '%s' needs braces", fieldDecl.getName());

                            hkInplaceArray<hkStringView, 16> elements;
                            splitStringWithScoping(withoutBraces, ',', elements);

                            hkResult res = setFieldValuesArray(field, elements, errorBuf);
                            HK_ATTR_FAIL_WITH_VALUE_IF(!res.isSuccess(), value, errorBuf, " in value for field '%s'", fieldDecl.getName());
                        }
                        break;
                    }
                }
                HK_ATTR_FAIL_IF(fieldIdx == attrData.getNumFields(), errorBuf, "Field '%s' not found", hkStringBuf(name).cString());
            }
            else
            {
                posArgs.pushBack(trim(args[i]));
            }
        }

        // If there is only a field and only positional arguments try to coerce everything into the field.
        if (attrData.getNumFields() == 1 && !attrData.getWritableField(0).getDecl().isExplicit() &&
            posArgs.getSize() == args.getSize() && posArgs.getSize() > 0)
        {
            int dummy = 0;
            HK_RETURN_IF_FAILED(setField(trim(withoutBrackets), attrData.getWritableField(0), errorBuf, posArgs, dummy));
        }
        else
        {
            // fill empty fields with positional args
            int fieldIdx = 0;
            for (int posIdx = 0; posIdx < posArgs.getSize(); ++posIdx)
            {
                // skip fields which are explicit or have been filled with a named value
                while (fieldIdx < attrData.getNumFields() &&
                    (attrData.getField(fieldIdx).isSet() || attrData.getField(fieldIdx).getDecl().isExplicit()))
                {
                    ++fieldIdx;
                }
                HK_ATTR_FAIL_IF(fieldIdx == attrData.getNumFields(), errorBuf, "Too many positional arguments");

                hkStringView value = posArgs[posIdx];

                const hkAttrData::WritableField& field = attrData.getWritableField(fieldIdx);
                HK_RETURN_IF_FAILED(setField(value, field, errorBuf, posArgs, posIdx));

                ++fieldIdx;
            }
        }
    }

    // check for missing fields
    for (int i = 0; i < attrData.getNumFields(); ++i)
    {
        hkAttrData::Field field = attrData.getField(i);
        bool missing = !field.getDecl().isOptional() && !field.getDecl().isVariableArray() && !field.isSet();
        HK_ATTR_FAIL_IF(missing, errorBuf, "Missing value for field '%s'", field.getDecl().getName());
    }

    return HK_SUCCESS;
}

hkAttrData::Field hkAttrData::getField(_In_z_ const char* name) const
{
    for (int i = 0; i < m_fields.getSize(); ++i)
    {
        if (hkString::strCmp(m_fields[i].decl.getName(), name) == 0)
        {
            return getField(i);
        }
    }
    return Field(*this, -1);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
