// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Object/hkBaseObject.h>
#include <Common/Base/Types/Properties/hkPropertyBag.h>

class hkCriticalSection;

template<class T>
struct hkAutoRemoveReferenceWrapper
{
    HK_INLINE explicit hkAutoRemoveReferenceWrapper (_In_opt_ T* object) : m_object (object) { };
    HK_INLINE ~hkAutoRemoveReferenceWrapper () { if (m_object) m_object->removeReference(); }

    _Ret_maybenull_
    HK_INLINE operator T* () const { return m_object; }

    // If the compiler decides to call copy constructors (no return value optimization), we need to ensure only the latest copy has the ref object
    hkAutoRemoveReferenceWrapper( const hkAutoRemoveReferenceWrapper& h) : m_object(h.m_object) { if (m_object) m_object->addReference(); }
    void operator= ( const hkAutoRemoveReferenceWrapper& h) { if (m_object) m_object->addReference(); }

    mutable T* m_object;

};

/// hkAutoRemoveReference(obj) returns a type-safe wrapper on the pointer that automatically removes a reference once it (the
/// wrapper) goes out of scope. It allows simpler code like "container->addObject( hkAutoRemoveReference( new Obj() ))" without
/// requiring a temporary variable to hold "new Obj()" and manually removing the additional reference after the function call.
template<typename T>
HK_INLINE hkAutoRemoveReferenceWrapper<T> hkAutoRemoveReference(_In_opt_ T* t) { return hkAutoRemoveReferenceWrapper<T>(t); }



/// Base for all classes in the Havok SDK.
/// All core SDK objects that can be owned by multiple owners inherit from this class -
/// rigid bodies, constraints, and actions are all hkReferencedObjects
/// and any object that is memory managed by Havok also inherits from it.
class HK_EXPORT_COMMON hkReferencedObject : public hkBaseObject
{
    public:

        HK_DECLARE_CLASS(hkReferencedObject, New, Reflect);
        HK_RECORD_ATTR( hk::AddOptional(hkReflect::Opt::POINTER_IMPL, &hkReflect::Detail::ReferencedObjectPointerImpl::s_instance) );
        HK_RECORD_ATTR( hk::AddOptional(hkReflect::Opt::ALLOC_IMPL, &hkReflect::Detail::ReferencedObjectAllocImpl::s_instance) );
        HK_RECORD_ATTR(hk::Version(3));

            /// Creates a new referenced object with an initial reference count of 1.
        HK_INLINE hkReferencedObject();

            /// Copy constructor, any default subclass's copy constructor will correctly initialize reference count to 1.
        HK_INLINE hkReferencedObject( const hkReferencedObject& originalObject );

            /// Empty assignment operator - the reference count of this will remain unchanged.
        HK_INLINE void operator=( const hkReferencedObject& originalObject );

            /// Destroy this object.
        virtual ~hkReferencedObject();

            /// Returns the size of this object.
            /// A memory size of 0 is a special case and indicates
            /// that the object should not be freed (as it has
            /// probably been read in from a packed file for instance)
            /// or is embedded in another structure
        HK_ALWAYS_INLINE int getAllocatedSize() const;

            /// Returns the current reference count. Used for debugging only.
        HK_ALWAYS_INLINE int getReferenceCount() const;

            /// Sets the reference count
        inline void setReferenceCount(int newRefCount);

            /// Returns the memory size and flags
        HK_ALWAYS_INLINE int getMemorySizeAndFlags() const;

            /// Sets the memory size and flags
        inline void setMemorySizeAndFlags(int newMemSizeAndFlags);

            /// Sets the memory size, flags and reference count, this function is not thread safe.
        inline void setMemorySizeFlagsAndReferenceCount(int newMemSizeAndFlags, int newRefCount);

            /// Adds a reference to the object - this increments the reference count. This is thread safe.
            /// ###ACCESS_CHECKS###( [(&hkReferencedObjectLock::getInstance()),HK_ACCESS_RW] );
        void addReference() const;

            /// Removes a reference to the object - this decrements the last reference count. This is thread safe.
            /// ###ACCESS_CHECKS###( [(&hkReferencedObjectLock::getInstance()),HK_ACCESS_RW] );
        void removeReference() const;

            /// Adds the references of a full array of referenced, this will call lockAll internally.
        static void HK_CALL addReferences(_In_reads_bytes_(numObjects * pointerStriding) const hkReferencedObject*const* objects, int numObjects, int pointerStriding );

            /// Adds the references of a full array of referenced, this will call lockAll internally.
        template<typename T>
        static void HK_CALL addReferences(_In_reads_bytes_(numObjects * pointerStriding) const T*const* objects, int numObjects, int pointerStriding = sizeof(T*) )
        { addReferences( reinterpret_cast<const hkReferencedObject*const*>(objects), numObjects, pointerStriding );
        HK_COMPILE_TIME_ASSERT( hkTrait::IsConvertibleTo<T*,const hkReferencedObject*>::result ); }

            /// Removes the references of a full array of referenced, this will call lockAll internally.
        static void HK_CALL removeReferences(_In_reads_bytes_(numObjects * pointerStriding)  const hkReferencedObject*const* objects, int numObjects, int pointerStriding );

            /// Removes the references of a full array of referenced, this will call lockAll internally.
        template<typename T>
        static void HK_CALL removeReferences(_In_reads_bytes_(numObjects * pointerStriding) const T*const* objects, int numObjects, int pointerStriding = sizeof(T*) )
        { removeReferences( reinterpret_cast<const hkReferencedObject*const*>(objects), numObjects, pointerStriding );
        HK_COMPILE_TIME_ASSERT( hkTrait::IsConvertibleTo<T*, const hkReferencedObject*>::result ); }

            /// Lock a global lock
        static void HK_CALL lockAll();

            /// Unlock the global lock
        static void HK_CALL unlockAll();

        enum { AUTO_MEMSIZE=0xffff };

    public:

        hkPropertyBag& properties() { return m_propertyBag; }
        const hkPropertyBag& properties() const { return m_propertyBag; }

    protected:

        /// Deletes this object (i.e. delete this;)
        /// Derived objects may customize this behavior, for instance objects may be
        /// queued for deletion rather than deleted immediately.
        virtual void deleteThisReferencedObject();

    private:

        template<typename T>
        friend class hkWeakPtr;

        /// Atomically adds a reference if the reference count is not 0. This is used specifically
        /// by the hkWeakPtr implementation which can rely on a lock in ~hkReferenced, do not use
        /// otherwise.
        /// Returns true if the function added a reference, false otherwise.
        bool addReferenceIfNotZero() const;

        /// The property bag
        hkPropertyBag m_propertyBag HK_ATTR(hk::IncludeInMgd(false));

        /// Memory size, flags
        hkUint16 m_memSizeAndFlags HK_ATTR(hk::Untracked);  //+nosave
        /// reference count
        hkUint16 m_refCount HK_ATTR(hk::Untracked); //+nosave
};

namespace hkMemory
{
    HK_INLINE hk_size_t HK_CALL numBytesToFree(_In_ const hkReferencedObject* obj, hk_size_t nbytes)
    {
        return (obj->getMemorySizeAndFlags() == hkReferencedObject::AUTO_MEMSIZE)
            ? nbytes
            : obj->getMemorySizeAndFlags();
    }
}

#include <Common/Base/Object/hkReferencedObject.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
