// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>
#include <Common/Base/Object/hkSingleton.h>
#include <Common/Base/Types/hkWeakPtr.h>

#if HK_ENDIAN_BIG
#   define ADD_REF_INCREMENT 0x0001
#   define ADD_REF_MASK      0xffff
#else
#   define ADD_REF_INCREMENT 0x00010000
#   define ADD_REF_MASK      0xffff0000
#endif


void HK_CALL hkReferenceCountError(_In_ const hkReferencedObject* o, _In_z_ const char* why)
{
    HK_ERROR(0x2c66f2d8, "Reference count error on object " << (const void*)(o)
        << " with ref count of " << o->getReferenceCount()
        << " in " << why << ".\n"
        << " * Are you calling delete instead of removeReference?\n"
        << " * Have you called removeReference too many times?\n"
        << " * Is this a valid object?\n"
        << " * Do you have more than 32768 references? (unlikely)\n");
}

hkReferencedObject::~hkReferencedObject()
{
#if defined(HK_BUILDING_WITH_ENGINE)
    if(hkWeakPtrDetail::StatusRegistry* registry = hkWeakPtrDetail::StatusRegistry::getInstancePtr())
    {
        registry->setDead(this);
    }
#endif
}

//
//  Atomically increments the reference count

void hkReferencedObject::addReference() const
{
    HK_COMPILE_TIME_ASSERT( HK_OFFSET_OF(hkReferencedObject,m_refCount) - HK_OFFSET_OF(hkReferencedObject,m_memSizeAndFlags) ==sizeof(hkUint16)  );
    HK_COMPILE_TIME_ASSERT( (HK_OFFSET_OF(hkReferencedObject, m_memSizeAndFlags)&3) == 0  );    // check for 4 byte alignment

    // We don't bother keeping references if the reference is going to be ignored.
    if ( m_memSizeAndFlags != 0 )
    {
        hkUint32* addr = const_cast<hkUint32*>(reinterpret_cast<const hkUint32*>(&m_memSizeAndFlags));

        for (hkAtomic::Backoff<> b; ; b.pause())
        {
            // Get old value
            const hkUint32 oldVal = *addr;
            const hkUint32 newVal = oldVal + ADD_REF_INCREMENT;

            // Loop until we can set it
            if ( hkAtomic::compareAndSwap(addr, oldVal, newVal) )
            {
/// Always check/fail on RAZZLE builds.  This is a critical error for a running device
#ifdef HK_COMPILER_RAZZLE
                if (HK_VERY_LIKELY((newVal & ADD_REF_MASK) > (oldVal & ADD_REF_MASK)))
                {
                }
                else
                {
                    HK_FATAL_ERROR(0x0, "Reference counts must be below 2^16");
                }
#endif

#   if defined(HK_DEBUG)
                if ( (newVal & ADD_REF_MASK) < (oldVal & ADD_REF_MASK ) )
                {
                    hkReferenceCountError(this, "addReference");
                }
#   endif
                return;
            }
        }
    }
}

//
//  Atomically increments the reference count if it is not zero
bool hkReferencedObject::addReferenceIfNotZero() const
{
    hkUint32* addr = const_cast<hkUint32*>(reinterpret_cast<const hkUint32*>(&m_memSizeAndFlags));

    for(hkAtomic::Backoff<> b; ; b.pause())
    {
        // Read old state
        const hkUint32 oldVal = *addr;
        const hkUint32 newVal = oldVal + ADD_REF_INCREMENT;

        // Stop if the refcount is 0
        if((oldVal & ADD_REF_MASK) == 0)
        {
            return false;
        }

        // Try to set the reference count. If oldVal goes to 0 between the previous check and the
        // the compareAndSwap then it will fail and we will loop.
        if(hkAtomic::compareAndSwap<hkUint32>(addr, oldVal, newVal))
        {

/// Always check/fail on RAZZLE builds.  This is a critical error for a running device
#ifdef HK_COMPILER_RAZZLE
            if (HK_VERY_LIKELY((newVal & ADD_REF_MASK) > (oldVal & ADD_REF_MASK)))
            {
            }
            else
            {
                HK_FATAL_ERROR(0x0, "Reference counts must be below 2^16");
            }
#endif

            // refcount successfully incremented without ever reaching 0
            return true;
        }
    }
}



//
//  Atomically decrements the reference count

void hkReferencedObject::removeReference() const
{
    if ( m_memSizeAndFlags != 0 )
    {
        hkUint32* addr = const_cast<hkUint32*>(reinterpret_cast<const hkUint32*>(&m_memSizeAndFlags));

        for (hkAtomic::Backoff<> b; ; b.pause())
        {
            // Read old state
            const hkUint32 oldVal = *addr;
            const hkUint32 newVal = oldVal - ADD_REF_INCREMENT;

            // Loop until we can set it
            if ( hkAtomic::compareAndSwap<hkUint32>(addr, oldVal, newVal) )
            {
#   if defined(HK_DEBUG)
                if ( (newVal & ADD_REF_MASK) == ADD_REF_MASK )  // Negative!
                {
                    hkReferenceCountError(this, "removeReference");
                }
#   endif
                if((newVal & ADD_REF_MASK) == 0)
                {
                    hkReferencedObject* thisPtr = const_cast<hkReferencedObject*>(this);
                    thisPtr->deleteThisReferencedObject();
                }

                return;
            }
        }
    }
}


hkBaseObject::~hkBaseObject()
{
}


#include <Common/Base/DebugUtil/MultiThreadCheck/hkMultiThreadCheck.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#define LOCK_MAGIC 0x23df4554

class HK_EXPORT_COMMON hkReferencedObjectLock : public hkReferencedObject
{
    public:
        HK_DECLARE_CLASS(hkReferencedObjectLock, NewOpaque, Singleton);
        typedef hkUint32 ThreadId;
        hkReferencedObjectLock() : m_criticalSection(4000)
        {
            m_multiThreadCheck.enableChecks();
            m_lockCount = 0;
        }

        HK_INLINE void lock()
        {
            m_criticalSection.enter();
            m_lockCount++;
#if defined(HK_DEBUG)
            if ( m_lockCount == 1)
            {
                m_multiThreadCheck.markForWrite();
            }
#endif
        }

        HK_INLINE void unlock()
        {
#if defined(HK_DEBUG)
            if (m_lockCount == 1)
            {
                m_multiThreadCheck.unmarkForWrite();
            }
#endif
            m_lockCount--;
            m_criticalSection.leave();
        }

    public:
        int m_lockCount;

        hkMultiThreadCheck m_multiThreadCheck;
        hkCriticalSection  m_criticalSection;
};

HK_SINGLETON_IMPLEMENTATION(hkReferencedObjectLock);

void hkReferencedObject::lockAll()
{
    hkReferencedObjectLock* lock = &hkReferencedObjectLock::getInstance();
    lock->lock();
}

void hkReferencedObject::unlockAll()
{
    hkReferencedObjectLock* lock = &hkReferencedObjectLock::getInstance();
    lock->unlock();
}

//
//  Deletes this object (i.e. delete this;)

void hkReferencedObject::deleteThisReferencedObject()
{
    delete this;
}

void hkReferencedObject::addReferences(_In_reads_bytes_(numObjects * pointerStriding) const hkReferencedObject*const* objects, int numObjects, int pointerStriding)
{
    for (int i = 0; i < numObjects; i++)
    {
        objects[0]->addReference();
        objects = hkAddByteOffsetConst(objects, pointerStriding);
    }
}

void hkReferencedObject::removeReferences(_In_reads_bytes_(numObjects * pointerStriding) const hkReferencedObject*const* objects, int numObjects, int pointerStriding )
{
    for (int i = 0; i < numObjects; i++)
    {
        objects[0]->removeReference();
        objects = hkAddByteOffsetConst(objects, pointerStriding);
    }
}

/*
HK_INLINE hkReferencedObject::~hkReferencedObject()
{
#ifdef HK_DEBUG
extern void HK_CALL hkRemoveReferenceError(const hkReferencedObject*, const char*);

// if we are in a dtor and the do not delete flag
// is set then we have call delete on an object that does not own its memory and
// the dtor should never have been called.
// But objects which are not new'd will all have random values in this size
// param and we can't set it in the ctor to something other than 0 as the
// the ctor is called after the alloc in hkMemory (where the size mem is set..)
// if ( m_memSizeAndFlags == 0 ) ..

// reference count is either zero because this method was called
// from removeReference or one because we are a local variable.
if( (m_referenceCount & (~1)) != 0)
{
hkRemoveReferenceError(this,"hkReferencedObject destructor");
}
// After calling delete the reference count should always be zero. This catches
// cases where delete is explicitly called.
m_referenceCount = 0;
#endif
}
*/

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
