// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef HKBASE_HKMONITOR_STREAM_H
#define HKBASE_HKMONITOR_STREAM_H

#include <Common/Base/Config/hkConfigMonitors.h>
#include <Common/Base/Config/hkConfigThread.h>
#include <Common/Base/Thread/Thread/hkThreadLocalData.h>

template<typename KEY, typename VALUE> class hkHashMap;

// use the following line to enable 64-timers. 64-bits are useful for profiling "slow" functions on the order of seconds.
// Since we removed the second timer, 64bit timer does not actually make much difference, especially on 64bit platforms
// where the command string ptr is 64 anyway and thus can end up with unused padding anyway.
#if defined(HK_ARCH_X64) \
    || defined(HK_PLATFORM_UWP) \
    || defined(HK_SUPPORT_DWM) \
    || ((defined(HK_PLATFORM_NX)||defined(HK_PLATFORM_IOS)) && defined(HK_ARCH_ARM_64))

    #define HK_ENABLE_64BIT_TIMERS 1
    #if (HK_POINTER_SIZE==4)
        #define HK_ENABLE_IMPICT_64BIT_PADDING_IN_TIMERS 1
    #endif
#endif

#ifndef HK_ENABLE_IMPICT_64BIT_PADDING_IN_TIMERS
#   define HK_ENABLE_IMPICT_64BIT_PADDING_IN_TIMERS 0
#endif

#ifndef HK_ENABLE_64BIT_TIMERS
#   define HK_ENABLE_64BIT_TIMERS 0
#endif

// By default, platform timers are enabled for non-release builds only, but this behavior can be
// toggled here to enable profiling of release builds. Note that some hardware vendors require
// profiling libs be stripped in order to pass certification
#if defined (HK_DEBUG)
    #define HK_ENABLE_PLATFORM_TIMERS 1
#else
    #define HK_ENABLE_PLATFORM_TIMERS 0
#endif

// Rarely if ever used
#define HK_ENABLE_EXTRA_TIMER 0

struct HK_EXPORT_COMMON hkTimerData
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkTimerData);
    HK_DECLARE_POD_TYPE();

    const char* m_streamBegin;
    const char* m_streamEnd;
};

/// Abstract interface for collecting timer information.
struct HK_EXPORT_COMMON hkTimerProvider
{
    HK_DECLARE_CLASS(hkTimerProvider, New, ReflectVirtualBase);

    /// Append timer data objects to the input array. The data returned should not be modified until the next call to clearTimers().
    virtual void appendTimerData(hkArray<hkTimerData>& timerDataInOut) = 0;

    /// Clear timers.
    virtual void clearTimerData() = 0;
};

/// This class is used to capture all monitor information for a single frame.
/// You can use monitors with the following macros
///
/// HK_TIMER_BEGIN( NAME, OBJECT )
/// HK_TIMER_END()
///
/// HK_TIMER_BEGIN_LIST( NAME, OBJECT )
/// HK_TIMER_SPLIT_LIST( NAME )
/// HK_TIMER_END_LIST( )
///
/// HK_MONITOR_PUSH_DIR( PATH )
/// HK_MONITOR_POP_DIR()
/// HK_MONITOR_ADD_VALUE( NAME, VALUE, MONITOR_TYPE )
///
/// You can use the hkMonitorStreamAnalyzer to store multiple frames
/// and to pretty print the information.
///
/// Notes:
///  - You must initialize this class on a per thread basis after calling hkBaseSystem::init or hkBaseSystem::initThread
///  - The hkMonitorStreamAnalyzer is singled threaded
///  - The hkMonitorStreamAnalyzer has utilities for analyzing multiple monitor streams for use in a multithreaded environment
class hkMonitorStream
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MONITOR, hkMonitorStream);

#ifdef HK_DYNAMIC_DLL
        HK_EXPORT_COMMON static hkMonitorStream* HK_CALL getInstancePtr();
        HK_EXPORT_COMMON static int HK_CALL getLockCount();
        HK_EXPORT_COMMON static void HK_CALL setLockCount( int i );
#else
        HK_INLINE static hkMonitorStream* HK_CALL getInstancePtr()
        {
            extern HK_THREAD_LOCAL( hkMonitorStream* ) hkMonitorStream__m_instance;
            return HK_THREAD_LOCAL_GET(hkMonitorStream__m_instance);
        }

        HK_INLINE static int HK_CALL getLockCount()
        {
            extern HK_THREAD_LOCAL( hkInt32 ) hkMonitorStream__m_lockCount;
            return HK_THREAD_LOCAL_GET( hkMonitorStream__m_lockCount );
        }

        HK_INLINE static void HK_CALL setLockCount( int i )
        {
            extern HK_THREAD_LOCAL( hkInt32 ) hkMonitorStream__m_lockCount;
            HK_THREAD_LOCAL_SET( hkMonitorStream__m_lockCount, i );
        }
#endif

        HK_ALWAYS_INLINE static hkMonitorStream& HK_CALL getInstance()  {   return *getInstancePtr();       }



        HK_INLINE static void HK_CALL addLock()
        {
            const int lockCount = getLockCount();
            HK_ASSERT_NO_MSG(0x4b855d50, lockCount >= 0);
            setLockCount(lockCount + 1);
        }

        HK_INLINE static void HK_CALL removeLock()
        {
            const int lockCount = getLockCount();
            HK_ASSERT_NO_MSG(0xbdd13fb, lockCount > 0);
            setLockCount(lockCount - 1);
        }

        HK_INLINE hkMonitorStream() { m_tag = hkUint32( -1 ); }
        HK_EXPORT_COMMON ~hkMonitorStream();

            /// Clear all memory. This must be called per thread, before each frame.
        HK_EXPORT_COMMON void reset();

            /// Sets the size of the stream used. By default the stream size is 0. If this is the
            /// case, or if the stream is full, no data is captured for any timer system (neither
            /// monitors nor platform events, such as PIX timers)
            /// Note that this memory *isn't actually used* by the platform specific timers, but
            /// still disables these timers in order to maintain legacy behavior.
        HK_EXPORT_COMMON void resize( int newSize );

            /// Insert a unique tag in the stream and return it.
            /// The returned value can be used in e.g. HK_TIMER_ENTER_TAG().
        HK_EXPORT_COMMON hkUint32 addTag();

            // We embed frame id in streams, on reset (first cmd), and allows us to correlate with gpu timers that may come in in a few frames time for instance
            // this is different per thread. Will never be 0, so can use that for invalid num.
        HK_INLINE hkUint32 getCurrentFrameId() const    {   return m_frameId;   }

            // Starts a platform-specific profiler block with the specified name
        HK_EXPORT_COMMON void platformBeginEvent( _In_z_ const char* name );

            // Ends the most recently platform-specific profiler block
        HK_EXPORT_COMMON void platformEndEvent();

    public:

        //
        //  These methods are for internal use only, and are called by the macros listed above.
        //

        // Get the pointer to the beginning of the memory used for timers
        HK_ALWAYS_INLINE _Ret_maybenull_ char* getStart() { return m_start; }

        // Get the pointer to the current write pointer
        HK_ALWAYS_INLINE _Ret_maybenull_ char* getEnd() { return m_end; }

        // Get the pointer to the end of the timer memory minus 16
        HK_ALWAYS_INLINE _Ret_maybenull_ char* getCapacityMinus16() { return m_capacityMinus16; }

        // Get the pointer to the capacity of timer memory
        HK_ALWAYS_INLINE _Ret_maybenull_ char* getCapacity() { return m_capacity; }

        // Check whether at least 16 bytes are free
        HK_ALWAYS_INLINE hkBool memoryAvailable()   {   return getEnd() < getCapacityMinus16();     }

        // Check whether at least numBytes bytes are free
        HK_ALWAYS_INLINE hkBool memoryAvailable(int numBytes)   {   return (getEnd()+numBytes) < getCapacity();     }

        HK_ALWAYS_INLINE int getCurrentMonitorDataSize() const  {   return (int)(m_end - m_start);  }

        // Get a piece of memory from the timer memory
        HK_INLINE _Ret_notnull_ void* expandby( int size ) { char* h = m_end; char* newEnd = h + size; m_end = newEnd; return h; }

        // Set the pointer to the current write pointer
        HK_ALWAYS_INLINE void setEnd(_In_ char* ptr) { m_end = ptr; }

        // Roundup end such that can be aligned for next char* cmd
        HK_INLINE void padEnd() {  hkUlong e = (hkUlong)m_end;
            #if HK_ENABLE_64BIT_TIMERS
                e = HK_NEXT_MULTIPLE_OF(8, e); m_end = (char*)e;
            #else
                e = HK_NEXT_MULTIPLE_OF(sizeof(void*), e); m_end = (char*)e;
            #endif
        }

        // Is the buffer allocated on the heap or not
        HK_ALWAYS_INLINE hkBool isBufferAllocatedOnTheHeap() const { return m_isBufferAllocatedOnTheHeap; }

    #if HK_ENABLE_64BIT_TIMERS
        typedef hkUint64 TimerValue;
    #else
        typedef hkUint32 TimerValue;
    #endif

    public:

        // Set the buffer to be a static, preallocated, buffer.
        HK_EXPORT_COMMON void HK_CALL setStaticBuffer( _Inout_updates_( bufferSize ) char* buffer, int bufferSize, bool clearBuffer = true );

        struct CommandStreamConfig
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MONITOR, CommandStreamConfig);

            const hkHashMap< hkUint64, const char* >* m_stringMap;
            const hkHashMap< hkUint64, const hkReflect::Type* >* m_typeMap;

            HK_EXPORT_COMMON CommandStreamConfig();

            bool m_pointersAre64Bit;
            bool m_timersAre64Bit;
            bool m_needEndianSwap;
            bool m_twoTimers;
            hkUint32 m_protocolVersion;
        };

        HK_EXPORT_COMMON static _Ret_maybenull_ const char* readCommandString( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static float readCommandFloat( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static hkUint16 readCommandUInt16( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static hkUint32 readCommandUInt32( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static hkUint64 readCommandUInt64( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static TimerValue readCommandTimer( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static _Ret_maybenull_ void* readCommandPointer( const char **current, const CommandStreamConfig& config );
        HK_EXPORT_COMMON static void padCommand( const char **current, const CommandStreamConfig& config );

        /// Computes the factor to use for converting tick counts recorded on the current platform
        /// to milliseconds.
        HK_EXPORT_COMMON static hkReal getLocalTimerFactor();

    public:

        // Free internal buffers
        HK_EXPORT_COMMON void clear();

        // Called by hkBaseSystem
        HK_EXPORT_COMMON static void HK_CALL init();

        // Called by hkBaseSystem
        HK_EXPORT_COMMON void quit();

    public:

        char* m_start;
        char* m_end;
        char* m_capacity;
        char* m_capacityMinus16;
        hkBool m_isBufferAllocatedOnTheHeap;

    private:

        hkUint32 m_tag;
        static hkUint32 s_tagBase;

        hkUint32 m_frameId;
        static hkUint32 s_frameIdBase;

    public:

        struct Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, Command );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            const char* m_commandAndMonitor;
        };

        struct FrameIdCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, FrameIdCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                m_id = hkMonitorStream::readCommandUInt32(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            hkUint32 m_id;
        };

        struct TagCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, TagCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                m_tag = hkMonitorStream::readCommandUInt32(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            hkUint32 m_tag;
        };

        struct AddValueCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, AddValueCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                m_value = hkMonitorStream::readCommandFloat(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            float m_value;
        };

        struct AddStructCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, AddStructCommand );

            _Ret_maybenull_ void* read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig);

            const hkReflect::Type* m_type;
            // variable size payload follows
        };

        struct AddGpuHandleCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, AddGpuHandleCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                if (!streamConfig.m_pointersAre64Bit)
                {
                    (*stream) += 4; // account for padding.
                }
                m_gpuHandle = readCommandUInt64(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            #if (HK_POINTER_SIZE==4)
                hkUint32 m_padding;
            #endif
            hkUint64 m_gpuHandle;
        };

        struct TimerCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, hkMonitorStream::TimerCommand );

            HK_INLINE void setTime();

            static HK_INLINE TimerValue subtract( TimerValue a, TimerValue b )
            {
                
                if(b > a) // wrap around
                {
                    return a + ( TimerValue(-1) - b);
                }
                else
                {
                    return a-b;
                }
            }

            HK_INLINE TimerCommand& operator += ( const TimerCommand& rhs )
            {
                m_time0 += rhs.m_time0;
                #if HK_ENABLE_EXTRA_TIMER
                    m_time1 += rhs.m_time1; 
                #endif
                return *this;
            }

            HK_INLINE TimerCommand& operator -= ( const TimerCommand& rhs )
            {
                m_time0 = subtract( m_time0, rhs.m_time0 );
                #if HK_ENABLE_EXTRA_TIMER
                    m_time1 = subtract( m_time1, rhs.m_time1 );
                #endif
                return *this;
            }

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                if (streamConfig.m_timersAre64Bit && !streamConfig.m_pointersAre64Bit)
                {
                    (*stream) += 4; // account for padding.
                }
                m_time0 = hkMonitorStream::readCommandTimer(stream, streamConfig);
                if ( streamConfig.m_twoTimers )
                {
                #if HK_ENABLE_EXTRA_TIMER
                    m_time1 =
                #endif // have to read it anyway to keep correct place in stream, even if we wont store it
                    hkMonitorStream::readCommandTimer(stream, streamConfig);
                }
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            #if HK_ENABLE_IMPICT_64BIT_PADDING_IN_TIMERS
                hkUint32 m_padding;
            #endif

            TimerValue m_time0;

            #if HK_ENABLE_EXTRA_TIMER
                TimerValue m_time1;
            #endif
        };

        struct TimerDrawCallCommand : public TimerCommand
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, TimerDrawCallCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                if (streamConfig.m_timersAre64Bit && !streamConfig.m_pointersAre64Bit)
                {
                    (*stream) += 4; // account for padding.
                }
                m_time0 = hkMonitorStream::readCommandTimer(stream, streamConfig);
                if (streamConfig.m_twoTimers)
                {
                    #if HK_ENABLE_EXTRA_TIMER
                        m_time1 =
                    #endif // have to read it anyway to keep correct place in stream, even if we wont store it
                        hkMonitorStream::readCommandTimer(stream, streamConfig);
                }
                m_gpuHandle = readCommandUInt64(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            hkUint64 m_gpuHandle;
        };

        struct MultiTimerCommand
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, MultiTimerCommand );

            HK_INLINE void clear()
            {
                m_callCount = 0;
                m_timerCommand.m_time0 = 0;
                #if HK_ENABLE_EXTRA_TIMER
                    m_timerCommand.m_time1 = 0;
                #endif
            }

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_timerCommand.m_commandAndMonitor = readCommandString(stream, streamConfig);
                if (streamConfig.m_timersAre64Bit && !streamConfig.m_pointersAre64Bit)
                {
                    (*stream) += 4; // account for padding.
                }
                m_timerCommand.m_time0 = readCommandTimer(stream, streamConfig);
                if (streamConfig.m_twoTimers)
                {
                    #if HK_ENABLE_EXTRA_TIMER
                        m_timerCommand.m_time1 =
                    #endif
                        readCommandTimer(stream, streamConfig);
                }
                m_callCount = readCommandUInt32(stream, streamConfig);
                padCommand(stream, streamConfig);
            }

            hkMonitorStream::TimerCommand m_timerCommand;
            hkUint32 m_callCount;
        };

        struct TimerBeginListCommand : public TimerCommand
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, TimerBeginListCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                if (streamConfig.m_timersAre64Bit && !streamConfig.m_pointersAre64Bit)
                {
                    (*stream) += 4; // account for padding.
                }
                m_time0 = hkMonitorStream::readCommandTimer(stream, streamConfig);
                if (streamConfig.m_twoTimers)
                {
                    #if HK_ENABLE_EXTRA_TIMER
                        m_time1 =
                    #endif
                            hkMonitorStream::readCommandTimer(stream, streamConfig);
                }
                m_nameOfFirstSplit = hkMonitorStream::readCommandString(stream, streamConfig);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            const char* m_nameOfFirstSplit;
        };

        struct TimerBeginObjectNameCommand : public TimerCommand
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, TimerBeginObjectNameCommand );

#if (HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED) // DSFL jfk
            const char* read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig);
#else
            inline const char* read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                return "";
            }
#endif

            hkUint16 m_objectNameSize; // String length including zero terminator
            // variable size payload follows
        };

        struct MemoryCommand : public Command
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, MemoryCommand );

            HK_INLINE void read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
            {
                m_commandAndMonitor = readCommandString(stream, streamConfig);
                HK_ASSERT_NOT_IMPLEMENTED(0x2347d504);
                hkMonitorStream::padCommand(stream, streamConfig);
            }

            void* m_ptr;
            int m_sizeAndFlags; // 1<<31 bit means deallocate
            #if HK_ENABLE_IMPICT_64BIT_PADDING_IN_TIMERS
                int m_padding;
            #endif
        };
};


struct HK_POSSIBLY_UNUSED hkDisableTimersScope
{
    HK_INLINE hkDisableTimersScope()
    {
        m_streamPtr = hkMonitorStream::getInstancePtr();
        HK_ON_DEBUG(m_streamPtr->addLock());
        m_capacity          = m_streamPtr->m_capacity;
        m_capacityMinus16   = m_streamPtr->m_capacityMinus16;
        m_streamPtr->m_capacityMinus16 = m_streamPtr->getEnd();
        m_streamPtr->m_capacity        = m_streamPtr->getEnd();
    }
    HK_INLINE ~hkDisableTimersScope()
    {
        m_streamPtr->m_capacityMinus16 = m_capacityMinus16;
        m_streamPtr->m_capacity = m_capacity;
        HK_ON_DEBUG(m_streamPtr->removeLock());
    }
    char* m_capacity;
    char* m_capacityMinus16;
    hkMonitorStream* m_streamPtr;

};

#include <Common/Base/Monitor/hkMonitorStream.inl>

#endif // HKBASE_HKMONITOR_STREAM_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
