// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Fwd/hkcstdarg.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>

#include <Common/Visualize/hkVersionReporter.h>
#include <Common/Visualize/Process/hkStatisticsProcess.h>

hkUint32 hkMonitorStream::s_tagBase = 0;
hkUint32 hkMonitorStream::s_frameIdBase = 1;

HK_THREAD_LOCAL( hkMonitorStream* ) hkMonitorStream__m_instance;
HK_THREAD_LOCAL( hkInt32 ) hkMonitorStream__m_lockCount;

#if HK_ENABLE_64BIT_TIMERS
// this is so that even if ptr size is not 8 bytes, we can still use 8 byte timers easily

// HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::Command) % 8 == 0); // We make sure if this is used to pad stream, special case as base class for the others
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::FrameIdCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::TagCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::AddValueCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::AddStructCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::AddGpuHandleCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::TimerCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::TimerDrawCallCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::MultiTimerCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::TimerBeginListCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::TimerBeginObjectNameCommand) % 8 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(hkMonitorStream::MemoryCommand) % 8 == 0);

#endif

#ifdef HK_DYNAMIC_DLL
hkMonitorStream* hkMonitorStream::getInstancePtr()
{
    extern HK_THREAD_LOCAL( hkMonitorStream* ) hkMonitorStream__m_instance;
    return HK_THREAD_LOCAL_GET( hkMonitorStream__m_instance );
}

int HK_CALL hkMonitorStream::getLockCount()
{
    extern HK_THREAD_LOCAL( hkInt32 ) hkMonitorStream__m_lockCount;
    return HK_THREAD_LOCAL_GET( hkMonitorStream__m_lockCount );
}

void HK_CALL hkMonitorStream::setLockCount( int i )
{
    extern HK_THREAD_LOCAL( hkInt32 ) hkMonitorStream__m_lockCount;
    HK_THREAD_LOCAL_SET( hkMonitorStream__m_lockCount, i );
}
#endif


hkMonitorStream::CommandStreamConfig::CommandStreamConfig()
{
    m_pointersAre64Bit = HK_POINTER_SIZE == 8;
#if HK_ENABLE_64BIT_TIMERS
    m_timersAre64Bit = true;
#else
    m_timersAre64Bit = false;
#endif
    m_needEndianSwap = false;
    m_stringMap = HK_NULL;
    m_typeMap = HK_NULL;

    m_twoTimers = HK_ENABLE_EXTRA_TIMER != 0;
    m_protocolVersion = hkVersionReporter::m_protocolVersion;
}

// If using platform timers, implement calls to platform-specific timers
#if HK_ENABLE_PLATFORM_TIMERS

#if defined(HK_PLATFORM_DURANGO)
#   define USE_PIX
#   include <Common/Base/Fwd/hkwindows.h>
#   include <Pix.h>
#elif defined(HK_PLATFORM_PS4)
#   include <perf.h>
#endif

void hkMonitorStream::platformBeginEvent(const char* name)
{
#if defined(HK_PLATFORM_DURANGO)
    PIXBeginEvent(PIX_COLOR(0, 0, 0), L"%hs", name);
#elif defined(HK_PLATFORM_PS4)
  #if HK_PS4_SDK_VERSION < 0x04500000u
    sceRazorCpuPushMarker(name, SCE_RAZOR_COLOR_WHITE, SCE_RAZOR_MARKER_ENABLE_HUD);
  #else
    sceRazorCpuPushMarker(name, SCE_RAZOR_CPU_COLOR_WHITE, SCE_RAZOR_CPU_MARKER_ENABLE_HUD);
  #endif
#endif
}

void hkMonitorStream::platformEndEvent()
{
#if defined(HK_PLATFORM_DURANGO)
    PIXEndEvent();
#elif defined(HK_PLATFORM_PS4)
    sceRazorCpuPopMarker();
#endif
}

// Otherwise, define empty stub functions
#else //HK_ENABLE_PLATFORM_TIMERS

// Empty stub function
void hkMonitorStream::platformBeginEvent(const char* name)
{
}

// Empty stub function
void hkMonitorStream::platformEndEvent()
{
}

#endif //HK_ENABLE_PLATFORM_TIMERS

#if (HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED)

    #include <Common/Base/Memory/Router/hkMemoryRouter.h>

    _Ret_maybenull_ void* hkMonitorStream::AddStructCommand::read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
    {
        m_commandAndMonitor = readCommandString(stream, streamConfig);
        m_type = (const hkReflect::Type*) readCommandPointer(stream, streamConfig);
        if (streamConfig.m_typeMap)
        {
            m_type = streamConfig.m_typeMap->getWithDefault( (hkUlong)m_type, HK_NULL);
        }
        void* data = HK_NULL;
        if (m_type)
        {
            data = (void*)( HK_NEXT_MULTIPLE_OF(m_type->getAlignOf(), hkUlong(*stream)) );
            *stream = reinterpret_cast<char*>(data) + m_type->getSizeOf();
        }
        hkMonitorStream::padCommand(stream, streamConfig);
        return data;
    }

    const char* hkMonitorStream::TimerBeginObjectNameCommand::read( const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig )
    {
        if ( streamConfig.m_protocolVersion < hkProtocolVersion::REFLECTION_2015_1 )
        {
            m_commandAndMonitor = hkMonitorStream::readCommandString( stream, streamConfig );
            m_time0 = hkMonitorStream::readCommandTimer( stream, streamConfig );
            /*m_time1 = */hkMonitorStream::readCommandTimer( stream, streamConfig );
            m_objectNameSize = 0;
            const char* data = hkMonitorStream::readCommandString( stream, streamConfig );
            hkMonitorStream::padCommand( stream, streamConfig );
            return data;
        }
        else
        {
            m_commandAndMonitor = readCommandString( stream, streamConfig );
            if ( streamConfig.m_timersAre64Bit && !streamConfig.m_pointersAre64Bit )
            {
                ( *stream ) += 4; // account for padding.
            }
            m_time0 = hkMonitorStream::readCommandTimer( stream, streamConfig );
            if ( streamConfig.m_twoTimers )
            {
#if HK_ENABLE_EXTRA_TIMER
                m_time1 =
#endif
                hkMonitorStream::readCommandTimer( stream, streamConfig );
            }
            m_objectNameSize = hkMonitorStream::readCommandUInt16( stream, streamConfig );
            hkMonitorStream::padCommand( stream, streamConfig );
            const char* data = *stream;
            ( *stream ) += m_objectNameSize;
            hkMonitorStream::padCommand( stream, streamConfig );
            return data;
        }
    }

    hkMonitorStream::~hkMonitorStream()
    {
        clear();
    }

    void hkMonitorStream::init()
    {
        hkMonitorStream* instance = hkMemDebugBlockAlloc<hkMonitorStream>(1);
        new (instance) hkMonitorStream();

        HK_MEMORY_TRACKER_ADD(hkMonitorStream, instance);
        HK_THREAD_LOCAL_SET( hkMonitorStream__m_instance, instance );
        HK_THREAD_LOCAL_SET( hkMonitorStream__m_lockCount, 0);
        instance->m_isBufferAllocatedOnTheHeap = false;
        instance->m_start = HK_NULL;
        instance->m_capacity = HK_NULL;
        instance->m_end = HK_NULL;
        instance->m_capacityMinus16 = HK_NULL;
        instance->m_tag = hkUint32(-1); // triggers a rebase in addTag()
    }

    void hkMonitorStream::clear()
    {
        if ( m_start && m_isBufferAllocatedOnTheHeap )
        {
            HK_MEMORY_TRACKER_REMOVE_RAW(getStart());
            hkMemDebugBufFree(m_start, int( m_capacity-m_start));
        }
        m_isBufferAllocatedOnTheHeap = false;
        m_start = HK_NULL;
        m_capacity = HK_NULL;
        m_end = HK_NULL;
        m_capacityMinus16 = HK_NULL;
    }

    void hkMonitorStream::quit()
    {
        hkMonitorStream* s = HK_THREAD_LOCAL_GET( hkMonitorStream__m_instance );
        s->clear();

        HK_MEMORY_TRACKER_REMOVE(s);
        s->~hkMonitorStream();
        hkMemDebugBlockFree(s,1);
        HK_THREAD_LOCAL_SET( hkMonitorStream__m_instance, HK_NULL ); // CK: cleanup, so we can find non-init situations faster
        HK_THREAD_LOCAL_SET( hkMonitorStream__m_lockCount, 0);
    }

    void hkMonitorStream::resize( int newSize )
    {
        if ( newSize == getCapacity() - getStart() )
        {
            return;
        }
        clear();

        if (newSize > 0)
        {
            m_isBufferAllocatedOnTheHeap = true;
            m_start = hkMemDebugBufAlloc<char>(newSize);
            m_end = m_start;
            m_capacity = m_start + newSize;
            m_capacityMinus16 = m_capacity - 32;
            m_frameId = hkAtomic::exchangeAdd32(&s_frameIdBase, 1);
            if (m_frameId == 0)
            {
                m_frameId = hkAtomic::exchangeAdd32(&s_frameIdBase, 1);
            }
            HK_MONITOR_ADD_FRAME_ID();

            HK_MEMORY_TRACKER_ADD_RAW("buffer_hkMonitorStream", m_start, newSize);
        }
    }

    void hkMonitorStream::setStaticBuffer(_Inout_updates_(bufferSize)char* buffer, int bufferSize, bool clearBuffer)
    {
        if (clearBuffer)
        {
            clear();
        }
        m_isBufferAllocatedOnTheHeap = false;
        m_start = buffer ;
        m_end = buffer;
        m_capacity = m_start + bufferSize;
        m_capacityMinus16 = m_capacity - 32 ;
    }

    void hkMonitorStream::reset()
    {
        // Make sure the stream does not get reset if this thread has a lock on it
        HK_ON_DEBUG(const int lockCount = getLockCount());
        HK_ASSERT_NO_MSG(0x63a8449e, lockCount == 0);

        m_end = m_start;
        m_frameId = hkAtomic::exchangeAdd32(&s_frameIdBase, 1);
        if (m_frameId == 0)
        {
            m_frameId = hkAtomic::exchangeAdd32(&s_frameIdBase, 1);
        }
        HK_MONITOR_ADD_FRAME_ID();

        
    #if defined(HK_DEBUG)
        #ifdef HK_DEBUG_SLOW
        HK_MONITOR_ADD_VALUE( "Debug build. Use Release for profiling.", 0.0f, HK_MONITOR_TYPE_SINGLE );
        #else
        HK_MONITOR_ADD_VALUE( "Dev build. Use Release for profiling.", 0.0f, HK_MONITOR_TYPE_SINGLE );
        #endif
    #endif
        if(hkMemorySystem::getInstance().getDebugInterface() != HK_NULL)
        {
            HK_MONITOR_ADD_VALUE("Using a debug/slow memory system.", 0.0f, HK_MONITOR_TYPE_SINGLE);
        }
    }

    hkUint32 hkMonitorStream::addTag()
    {
        // Note: Tags must be:
        // * unique across all (per-thread) streams
        // * not equal to zero - this value has special meaning in HK_MONITOR_COMMAND_GOTO_TAG()
        if( HK_VERY_UNLIKELY( (m_tag & 0xffff) == 0xffff ) )
        {
            m_tag = hkAtomic::exchangeAdd32( &s_tagBase, 0x10000 );
        }
        m_tag++;
        HK_MONITOR_ADD_TAG( (this), m_tag );
        return m_tag;
    }

#else // #if (HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_DISABLED)

    hkMonitorStream::~hkMonitorStream()
    {
    }

    void hkMonitorStream::init()
    {
        // we need this so that hkMonitorStream::getInstance() will return a valid pointer
        static hkMonitorStream _instance;
        HK_THREAD_LOCAL_SET( hkMonitorStream__m_instance, &_instance );
        HK_THREAD_LOCAL_SET( hkMonitorStream__m_lockCount, 0);
    }

    void hkMonitorStream::quit()
    {
    }

    void hkMonitorStream::clear()
    {}

    void hkMonitorStream::setStaticBuffer( _Inout_updates_(bufferSize) char* buffer, int bufferSize, bool clearBuffer )
    {
    }

    void hkMonitorStream::reset()
    {
    }

    void hkMonitorStream::resize( int newSize )
    {
    }

    hkUint32 hkMonitorStream::addTag()
    {
        return 0;
    }

    _Ret_maybenull_ void* hkMonitorStream::AddStructCommand::read(const char** stream, const hkMonitorStream::CommandStreamConfig& streamConfig)
    {
        return HK_NULL;
    }

#endif

    namespace HK_UNITY_ANONYMOUS_NAMESPACE
    {

        static hkUint64 swap64( hkUint64 v )
        {
            union SwapUnion
            {
                hkUint64 val;
                hkUint8 b[8];
            };

            SwapUnion in; in.val = v;
            SwapUnion out;

            out.b[0] = in.b[7];
            out.b[1] = in.b[6];
            out.b[2] = in.b[5];
            out.b[3] = in.b[4];
            out.b[4] = in.b[3];
            out.b[5] = in.b[2];
            out.b[6] = in.b[1];
            out.b[7] = in.b[0];

            return out.val;
        }

        static hkUint32 swap32( hkUint32 v )
        {
            union SwapUnion
            {
                hkUint32 val;
                hkUint8 b[4];
            };
            SwapUnion in; in.val = v;
            SwapUnion out;
            out.b[0] = in.b[3];
            out.b[1] = in.b[2];
            out.b[2] = in.b[1];
            out.b[3] = in.b[0];
            return out.val;
        }

        static hkUint16 swap16( hkUint16 v )
        {
            union SwapUnion
            {
                hkUint16 val;
                hkUint8 b[2];
            };
            SwapUnion in; in.val = v;
            SwapUnion out;
            out.b[0] = in.b[1];
            out.b[1] = in.b[0];
            return out.val;
        }
    }

    _Ret_maybenull_ const char* hkMonitorStream::readCommandString(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        if(streamConfig.m_pointersAre64Bit)
        {
            hkUint64 stringId = *(hkUint64*)(*current);
            (*current) += 8;

            const char* outputString = (const char*)(hkUlong(stringId));
            if(streamConfig.m_stringMap)
            {
                if(streamConfig.m_needEndianSwap)
                {
                    stringId = swap64(stringId);
                }
                if(streamConfig.m_stringMap->get( stringId, &outputString ).isFailure())
                {
                    return HK_NULL;
                }
            }

            return outputString;
        }
        else
        {
            hkUint32 stringId = *(hkUint32*)(*current);
            (*current) += 4;

            const char* outputString = (const char*)(hkUlong(stringId));
            if(streamConfig.m_stringMap)
            {
                if(streamConfig.m_needEndianSwap)
                {
                    stringId = swap32(stringId);
                }
                if(streamConfig.m_stringMap->get( stringId, &outputString ).isFailure())
                {
                    return HK_NULL;
                };
            }

            return outputString;
        }
    }

    float hkMonitorStream::readCommandFloat(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        hkUint32 value = *(hkUint32*)(*current);
        (*current) += 4;

        if(streamConfig.m_needEndianSwap)
        {
            value = swap32(value);
        }

        return *(hkFloat32*)(&value);
    }

    hkUint16 hkMonitorStream::readCommandUInt16(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        hkUint16 value = *(hkUint16*)(*current);
        (*current) += 2;

        if (streamConfig.m_needEndianSwap)
        {
            value = swap16(value);
        }

        return value;

    }


    hkUint32 hkMonitorStream::readCommandUInt32(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        hkUint32 value = *(hkUint32*)(*current);
        (*current) += 4;

        if(streamConfig.m_needEndianSwap)
        {
            value = swap32(value);
        }

        return value;
    }

    hkUint64 hkMonitorStream::readCommandUInt64(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        hkUint64 value = *(hkUint64*)(*current);
        (*current) += 8;

        if(streamConfig.m_needEndianSwap)
        {
            value = swap64(value);
        }

        return value;
    }

    hkMonitorStream::TimerValue hkMonitorStream::readCommandTimer(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        if(streamConfig.m_timersAre64Bit)
        {
            hkUint64 timerValue = *(hkUint64*)(*current);
            (*current) += 8;

            if(streamConfig.m_needEndianSwap)
            {
                timerValue = swap64(timerValue);
            }

            return (TimerValue)timerValue;
        }
        else
        {
            hkUint32 timerValue = *(hkUint32*)(*current);
            (*current) += 4;

            if(streamConfig.m_needEndianSwap)
            {
                timerValue = swap32(timerValue);
            }

            return (TimerValue)timerValue;
        }
    }

    _Ret_maybenull_ void* hkMonitorStream::readCommandPointer(const char **current, const CommandStreamConfig& streamConfig)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        if(streamConfig.m_pointersAre64Bit)
        {
            hkUint64 ptrValue = *(hkUint64*)(*current);
            (*current) += 8;

            if(streamConfig.m_needEndianSwap)
            {
                ptrValue = swap64(ptrValue);
            }

            return (void*)(hkUlong)ptrValue;
        }
        else
        {
            hkUint32 ptrValue = *(hkUint32*)(*current);
            (*current) += 4;

            if(streamConfig.m_needEndianSwap)
            {
                ptrValue = swap32(ptrValue);
            }

            return (void*)(hkUlong)ptrValue;
        }
    }

    void hkMonitorStream::padCommand(const char **current, const CommandStreamConfig& streamConfig)
    {
        if(streamConfig.m_pointersAre64Bit || streamConfig.m_timersAre64Bit)
        {
            while( (hkUlong)(*current) & 0x7)
            {
                (*current)++;
            }
        }
        else
        {
            while( (hkUlong)(*current) & 0x3)
            {
                (*current)++;
            }
        }
    }

    hkReal hkMonitorStream::getLocalTimerFactor()
    {
#if defined(HK_SUPPORT_DWM) && defined(HK_ARCH_IA32)
        LARGE_INTEGER freq;
        QueryPerformanceFrequency(&freq);
        return 1e3f / freq.QuadPart;
#else
        return 1e3f / hkStopwatch::getTicksPerSecond();
#endif
    }

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
