// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamParser.h>


void hkMonitorStreamParserUtil::extractMapsFromStreams( const hkMonitorStream::CommandStreamConfig& config, hkArrayView<const hkTimerData> timerData, hkMonitorStreamStringMap& smap, hkMonitorStreamTypeMap& tmap)
{
    hkHashMap< hkUint64, const char* >& _stringMap = smap.m_runtimeMap;
    hkHashMap< hkUint64, const hkReflect::Type* >& _typeMap = tmap.m_runtimeMap;

    if (!config.m_stringMap || !config.m_typeMap) // if config has maps already, then not a local dump
    {
        for (int thr = 0; thr < timerData.getSize(); ++thr)
        {
            const char* current = timerData[thr].m_streamBegin;
            const char* end = timerData[thr].m_streamEnd;

            while ( current < end )
            {
                hkUint32 rawValue = hkMonitorStream::readCommandUInt32(&current, config);
                current -= 4;
                // Skip commands used to mark the start of an elf
                if ( rawValue <= HK_MAX_ELF_ID)
                {
                    continue;
                }

                const char* currentBeforeString = current;
                const char* string = hkMonitorStream::readCommandString(&current, config);
                current = currentBeforeString; // the cmd read funcs all do the string bit again. Not great.

                if (!config.m_stringMap)
                {
                    _stringMap.insert( hkUlong( (void*)string ), string);
                }

                switch( string[0] )
                {
                case 'F':       // frame id
                    {
                        hkMonitorStream::FrameIdCommand com;
                        com.read(&current, config);
                        break;
                    }

                case 'S':       // split list
                case 'E':       // timer end
                case 'l':       // list end
                case 'T':       // timer begin
                case 'R':       // timer begin with self
                    {
                        if ( string[1] == 't' )
                        {
                            hkMonitorStream::TimerCommand com;
                            com.read(&current, config);
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x2a61a1c8, string[1] == 'g');
                            hkMonitorStream::TimerDrawCallCommand com;
                            com.read(&current, config);
                        }

                        break;
                    }
                case 'O': // object name
                    {
                        hkMonitorStream::TimerBeginObjectNameCommand com;
                        /*const char* name =*/ com.read(&current, config);
                        break;
                    }

                case 'L':       // timer list begin
                    {
                        hkMonitorStream::TimerBeginListCommand com;
                        com.read(&current, config);

                        if (!config.m_stringMap)
                        {
                            _stringMap.insert( hkUlong( (void*)com.m_nameOfFirstSplit ), com.m_nameOfFirstSplit );
                        }
                        break;
                    }

                case 'W':       // multi timer end
                    {
                        hkMonitorStream::MultiTimerCommand com;
                        com.read(&current, config);

                        break;
                    }

                case 'M':
                    {
                        hkMonitorStream::AddValueCommand com;
                        com.read(&current, config);

                        break;
                    }

                case 'X': // timer with tag
                case 'G': // timer with tag
                    {
                        hkMonitorStream::TagCommand com;
                        com.read(&current, config);
                        break;
                    }

                case 'P':
                case 'p':
                case 'N':
                case 'Y':   // multi timer begin
                    {
                        hkMonitorStream::Command com;
                        com.read(&current, config);
                        break;
                    }
                case 'A':
                    {
                        hkMonitorStream::AddStructCommand com;
                        com.read(&current, config);
                        if ( !config.m_typeMap )
                        {
                            _typeMap.insert((hkUlong)com.m_type, com.m_type);
                        }
                        break;
                    }
                case 'H':
                    {
                        hkMonitorStream::AddGpuHandleCommand com;
                        com.read(&current, config);
                        break;
                    }

                default:
                    HK_ASSERT(0x5120d10a, 0, "Inconsistent Monitor capture data" );     return;
                }
            }
        }
    }

    const hkHashMap< hkUint64, const char* >* stringMap = config.m_stringMap? config.m_stringMap : &_stringMap;
    const hkHashMap< hkUint64, const hkReflect::Type* >* typeMap = config.m_typeMap? config.m_typeMap : &_typeMap;

    // Fill in serializable maps
    {
        hkHashMap< hkUint64, const char* >::Iterator iter = stringMap->getIterator();
        smap.m_map.reserve( stringMap->getSize() );
        smap.m_map.setSize(0);
        while (stringMap->isValid(iter))
        {
            hkMonitorStreamStringMap::StringMap& s = smap.m_map.expandOne();
            s.m_id = stringMap->getKey(iter);
            s.m_string = stringMap->getValue(iter);
            iter = stringMap->getNext(iter);
        }

        hkHashMap< hkUint64, const hkReflect::Type* >::Iterator titer = typeMap->getIterator();
        tmap.m_map.reserve( typeMap->getSize() );
        tmap.m_map.setSize(0);
        while (typeMap->isValid(titer))
        {
            hkMonitorStreamTypeMap::TypeMap& t = tmap.m_map.expandOne();
            t.m_id = typeMap->getKey(titer);
            t.m_type = typeMap->getValue(titer);
            titer = typeMap->getNext(titer);
        }
    }
}


namespace {

    using namespace hkMonitorStreamParser;

    static HK_INLINE _Ret_maybenull_ Node* _findChildUsingTagAndName(_Inout_ Node* destParent, _In_ const Node* searchNode)
    {
        const char* name = searchNode->m_name;

        for (int i = 0; i < destParent->m_children.getSize(); ++i)
        {
            Node* child = destParent->m_children[i];
            if ( (child->m_moveToTag == searchNode->m_moveToTag) && hkString::strCmp(child->m_name, name) == 0)
            {
                return child;
            }
        }
        return HK_NULL;
    }

    static _Ret_notnull_ Node* _findOrCreateChildWithHint(_Inout_ Node* destParent, _In_ const Node* searchNode, int& childId, bool ignoreTags)
    {
        const char* name = searchNode->m_name;
        {
            for (int i = 0; i < destParent->m_children.getSize(); ++i)
            {
                Node* child = destParent->m_children[i];
                if ( (ignoreTags || (child->m_moveToTag == searchNode->m_moveToTag)) &&
                    hkString::strCmp(child->m_name, name) == 0)
                {
                    return child;
                }
            }
        }
        Node* child = new Node(destParent, name, searchNode->m_type );
        child->m_moveToTag = ignoreTags ? 0 : searchNode->m_moveToTag;
        return child;
    }

    static HK_INLINE void _mapAndCopyTags(_In_ const Node* source, _Inout_ Node* matchedNode, _Inout_ Tree* destTree)
    {
        if ( source->m_tags )
        {
            if (!matchedNode->m_tags)
            {
                matchedNode->m_tags = new TagInfo;
            }
            for (int i=0; i < source->m_tags->m_tags.getSize(); i++ )
            {
                TagType tag = source->m_tags->m_tags[i];
                if ( matchedNode->m_tags->m_tags.indexOf(tag) < 0 )
                {
                    matchedNode->m_tags->m_tags.pushBack( tag );
                }
                destTree->m_tagToNode.insert( tag, matchedNode );
            }
        }
    }

    static void HK_CALL _mergeThreadTree(_Inout_ Tree* destTree, _Inout_ Node* destNode, _In_ const Tree* sourceRoot, _In_ const Node* sourceNode, int srcThreadId, _Inout_opt_ hkPointerMap<const Node*, const Node*>* nodeMapping)
    {
        int childId = 0; // Also use this as a hint to hasChild - usually the children will be in the same order.

        for ( int i = 0; i < sourceNode->m_children.getSize(); ++i )
        {
            const Node* subTreeNode = sourceNode->m_children[i];

            Node* matchedNode = _findOrCreateChildWithHint( destNode, subTreeNode, childId, false );

            if (nodeMapping)
            {
                nodeMapping->insert(sourceNode, matchedNode);
            }

            // copy tags
            _mapAndCopyTags(subTreeNode, matchedNode, destTree);

            // copy tagged nodes
            TagType tag = subTreeNode->m_moveToTag;
            if ( tag )
            {
                matchedNode->m_moveToTag = tag;
                destTree->m_taggedNodes.insert( matchedNode, tag );
            }

            // have not set thread id yet and is used on main thread
            if ((matchedNode->m_value > 0) && (matchedNode->m_threadFlags.get() == 0))
            {
                matchedNode->m_threadFlags.orWith( 1 );
            }

            if (subTreeNode->m_count > 0)
            {
                matchedNode->m_value += subTreeNode->m_value;
                matchedNode->m_count += subTreeNode->m_count;
                matchedNode->m_threadMaxValue = hkMath::max2(subTreeNode->m_value, matchedNode->m_threadMaxValue);
                matchedNode->m_threadMinValue = hkMath::min2(subTreeNode->m_value, matchedNode->m_threadMinValue);
                matchedNode->m_threadFlags.orWith( 1<<srcThreadId );
            }

            if (!matchedNode->m_gpuHandle)
            {
                matchedNode->m_gpuHandle = subTreeNode->m_gpuHandle; // so at least have some idea, but merged trees not normally used for gpu info like this
            }

            _mergeThreadTree( destTree, matchedNode, sourceRoot, subTreeNode, srcThreadId, nodeMapping );

            childId = (childId < sourceNode->m_children.getSize() - 1 ) ? childId+1 : childId;
        }
    }

    void HK_CALL _mergeSubTree(_Inout_ Tree* tree, _Inout_ Node* dest, _Inout_ Node* source, _Inout_opt_ hkPointerMap<const Node*, const Node*>* nodeMapping)
    {
        Node* matchedNode = _findChildUsingTagAndName( dest, source );
        if ( matchedNode == source )
        {
            return; // found myself, nothing to do
        }

        Node* oldParent = source->m_parent;
        oldParent->m_children.removeAllAndCopy(source);
        source->m_parent = HK_NULL;

        if ( !matchedNode )
        {
            // reuse node
            // just move my child including all children
            dest->m_children.pushBack(source);
            source->m_parent = dest;
        }
        else
        {
            if ((matchedNode->m_value > 0) && (matchedNode->m_threadFlags.get() == 0))
            {
                matchedNode->m_threadFlags.orWith( 1 );
            }

            if (source->m_count > 0)
            {
                matchedNode->m_value  += source->m_value;
                matchedNode->m_count  += source->m_count;
                matchedNode->m_threadMaxValue = hkMath::max2(source->m_threadMaxValue, matchedNode->m_threadMaxValue);
                matchedNode->m_threadMinValue = hkMath::min2(source->m_threadMinValue, matchedNode->m_threadMinValue);
                matchedNode->m_threadFlags.orWith( source->m_threadFlags );
            }

            if (!matchedNode->m_gpuHandle)
            {
                matchedNode->m_gpuHandle = source->m_gpuHandle; // so at least have some idea, but merged trees not normally used for gpu info like this
            }

            if (nodeMapping)
            {
                nodeMapping->insert(source, matchedNode);
            }

            while ( source->m_children.getSize() )
            {
                Node* subTreeNode = source->m_children[0];
                _mergeSubTree( tree, matchedNode, subTreeNode, nodeMapping );
            }

            // move tags
            _mapAndCopyTags(source, matchedNode, tree);

            // move tagged nodes
            TagType tag = source->m_moveToTag;
            if ( tag )
            {
                tree->m_taggedNodes.insert( source, 0 );
            }

            // no longer needed
            delete source;
        }
    }

    static void _findTagsRec(_Inout_ Node* node, hkArray<Node*>& nodes )
    {
        if ( node->m_moveToTag )
        {
            nodes.pushBack(node);
        }
        for (int i =0; i < node->m_children.getSize(); i++ )
        {
            _findTagsRec( node->m_children[i], nodes );
        }
    }

    static void _fixupTags(_Inout_ Tree* tree, _Inout_opt_ hkPointerMap<const Node*, const Node*>* nodeMapping)
    {
        const Tree::TaggedNodesMap& tn = tree->m_taggedNodes;

        hkInplaceArray<Node*,256> nodesToMove;
        _findTagsRec( tree, nodesToMove );
        for (int i = 0; i < nodesToMove.getSize(); i++ )
        {
            Node* node = nodesToMove[i];
            TagType tag = tn.getWithDefault( node, 0 );
            if ( tag == 0 )
            {
                continue; // already resolved;
            }
            Node* newParent = tree->m_tagToNode.getWithDefault( tag, HK_NULL );
            if ( newParent )
            {
                if ( node->m_parent != newParent)
                {
                    //Propagate the timer values up to the new parent
                    Node* propagateTo = newParent;
                    while(propagateTo)
                    {
                        propagateTo->m_value += node->m_value;
                        propagateTo->m_threadMaxValue = hkMath::max2(propagateTo->m_threadMaxValue, node->m_threadMaxValue);
                        propagateTo->m_threadMinValue = hkMath::min2(propagateTo->m_threadMinValue, node->m_threadMinValue);
                        // don't prop m_threadCount ?
                        // don't prop the count ?
                        propagateTo = propagateTo->m_parent;
                    }

                    _mergeSubTree( tree, newParent, node, nodeMapping);
                }
            }
        }

        // do it again, this time removing all tags
        for (int i = 0; i < nodesToMove.getSize(); i++ )
        {
            Node* node = nodesToMove[i];
            TagType tag = tn.getWithDefault( node, 0 );
            if ( tag == 0 )
            {
                continue; // already resolved;
            }
            Node* newParent = tree->m_tagToNode.getWithDefault( tag, HK_NULL );
            if ( newParent )
            {
                HK_ASSERT_NO_MSG ( 0xf046fdfd, node->m_parent == newParent);
                node->m_moveToTag = 0;
                _mergeSubTree( tree, newParent, node, nodeMapping );
            }
        }
    }

    static _Ret_maybenull_ const hkMonitorStreamParser::Node* _getMapping(_In_opt_ const hkMonitorStreamParser::Node* src, const hkPointerMap<const hkMonitorStreamParser::Node*, const hkMonitorStreamParser::Node*>& nodeMap)
    {
        const hkMonitorStreamParser::Node* mappedNode = src;
        if (src)
        {
            while (const hkMonitorStreamParser::Node* queryMap = nodeMap.getWithDefault(mappedNode, HK_NULL))
            {
                mappedNode = queryMap;
            }
        }
        return mappedNode;
    }

    static void _mergeGpuMaps(_Inout_ Tree* mainTree, hkArrayView<Tree*> otherTrees, const hkPointerMap<const hkMonitorStreamParser::Node*, const hkMonitorStreamParser::Node*>& nodeMap)
    {
        // make the mainTree m_gpuHandleCache representative of the merge ( the tags fixup could move them)
        hkMonitorStreamGpuHandleCache& mainCache = mainTree->m_gpuHandleCache;
        for (int c=0; c < mainCache.m_gpuData.getSize(); ++c)
        {
            hkMonitorStreamGpuHandleCache::Mapping& m = mainCache.m_gpuData[c];
            m.m_addHandleNode = _getMapping(m.m_addHandleNode, nodeMap);
            m.m_timerNode = _getMapping(m.m_timerNode, nodeMap);
        }

        // then merge in any new handle data we got from other threads
        for (int t=0; t < otherTrees.getSize(); ++t)
        {
            hkMonitorStreamGpuHandleCache& threadCache = otherTrees[t]->m_gpuHandleCache;
            hkPointerMap<hkUint64, int>::Iterator handleIter = threadCache.m_gpuHandleToIndex.getIterator();
            while (threadCache.m_gpuHandleToIndex.isValid(handleIter))
            {
                hkUint64 gpuHandle = threadCache.m_gpuHandleToIndex.getKey(handleIter);
                int gpuMapIndex = threadCache.m_gpuHandleToIndex.getValue(handleIter);
                hkMonitorStreamGpuHandleCache::Mapping* gpuMap = mainCache.get(gpuHandle);
                const hkMonitorStreamGpuHandleCache::Mapping& gpuThreadMap = threadCache.m_gpuData[gpuMapIndex];
                if (!gpuMap->m_addHandleNode) // if don't have one already
                {
                     gpuMap->m_addHandleNode = _getMapping(gpuThreadMap.m_addHandleNode, nodeMap);
                }
                if (!gpuMap->m_timerNode) // if don't have one already
                {
                     gpuMap->m_timerNode = _getMapping(gpuThreadMap.m_timerNode, nodeMap);
                }

                handleIter = threadCache.m_gpuHandleToIndex.getNext(handleIter);
            }
        }
    }

} // anon namespace

void hkMonitorStreamParserUtil::combineTrees(_Inout_ Tree* mainTree, hkArrayView<Tree*> otherTrees )
{
    hkPointerMap<const hkMonitorStreamParser::Node*, const hkMonitorStreamParser::Node*> nodeMap;
    for (int t=0; t < otherTrees.getSize(); ++t)
    {
        _mergeThreadTree( mainTree, mainTree, otherTrees[t], otherTrees[t], t, otherTrees[t]->m_gpuHandleCache.getSize() > 0? &nodeMap : HK_NULL);
    }

    bool haveNodeMap = nodeMap.getSize() > 0;

    _fixupTags( mainTree, haveNodeMap? &nodeMap : HK_NULL );

    if (haveNodeMap)
    {
        _mergeGpuMaps( mainTree, otherTrees, nodeMap );
    }
}

void hkMonitorStreamParserUtil::findMetaData(_In_ const hkMonitorStreamParser::Node* n, int levels, hkArray<hkReflect::Var>& result )
{
    if (n->m_metaData)
    {
        result.pushBack( n->m_metaData );
    }

    if (levels > 0)
    {
        for (int c = 0; c < n->m_children.getSize(); ++c)
        {
            findMetaData(n->m_children[c], levels - 1, result );
        }
    }
}

void hkMonitorStreamParserUtil::setFlagOnNodesByName(_Inout_ hkMonitorStreamParser::Node* n, _In_z_ const char* str, int flagToSet, int flagForParents)
{
    if (n->m_name && hkString::strStr(n->m_name, str))
    {
        n->m_flags.orWith( (hkUint16)flagToSet);
        if (flagForParents)
        {
            hkMonitorStreamParser::Node* p = n->m_parent;
            while (p)
            {
                p->m_flags.orWith( (hkUint16)flagForParents);
                p = p->m_parent;
            }
        }
    }

    for (int c=0; c < n->m_children.getSize(); ++c)
    {
        setFlagOnNodesByName(n->m_children[c], str, flagToSet, flagForParents);
    }
}

void hkMonitorStreamParserUtil::clearFlagOnNodes(_Inout_ hkMonitorStreamParser::Node* n, int flagToClear)
{
    n->m_flags.clear( (hkUint16)flagToClear );
    for (int c=0; c < n->m_children.getSize(); ++c)
    {
        clearFlagOnNodes(n->m_children[c], flagToClear);
    }
}

namespace
{
    static void hkMonitorStreamParser_accumulateMetadata(hkReflect::Var parent, hkReflect::Var metadata)
    {
        HK_ASSERT_NO_MSG(0x418a6e10, parent.getType()->equals(metadata.getType()));
        if (const hkReflect::RecordType* recordType = metadata.getType()->asRecord())
        {
            // Search for the first bool attribute marked with hkms::AccumulateOp::Tag; if found, use its
            // value and the value of the corresponding parent attribute to determine whether we need to
            // accumulate the values in this object in the parent object. If both this and the parent object
            // are marked as having accumulated values already, accumulation must be skipped in order to not
            // create wrong data.
            for (hkReflect::DeclIter<hkReflect::DataFieldDecl> it(recordType); it.advance(); )
            {
                const hkReflect::FieldDecl field = it.current();
                if (const hkms::Accumulate* accumulate = field.getType()->findAttribute<hkms::Accumulate>())
                {
                    if (accumulate->m_value == hkms::AccumulateOp::Tag)
                    {
                        hkReflect::BoolVar fieldAsBool(metadata[field]);
                        if (fieldAsBool)
                        {
                            hkReflect::BoolVar parentFieldAsBool(parent[field]);
                            if (fieldAsBool.getValue() && parentFieldAsBool && parentFieldAsBool.getValue())
                            {
                                return;
                            }

                            fieldAsBool.setValue(true);
                            break;
                        }
                    }
                }
            }

            // Do the actual accumulation tasks
            for( hkReflect::DeclIter<hkReflect::DataFieldDecl> it(recordType); it.advance(); )
            {
                const hkReflect::FieldDecl field = it.current();
                if (const hkms::Accumulate* accumulate = field.getType()->findAttribute<hkms::Accumulate>())
                {
                    hkReflect::IntVar fieldAsInt(metadata[field]);
                    if (fieldAsInt)
                    {
                        hkReflect::IntVar parentFieldAsInt(parent[field]);
                        if (parentFieldAsInt)
                        {
                            switch (accumulate->m_value)
                            {
                            case hkms::AccumulateOp::Add:
                                parentFieldAsInt.setValue(parentFieldAsInt.getValue() + fieldAsInt.getValue());
                                break;
                            default:
                                HK_ASSERT_NO_MSG(0x6021932f, false);
                            }
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x4f7a2821, false);
                        }
                    }
                }
            }
        }
    }

    static int hkMonitorStreamParser_findParentMetadata(hkReflect::Var metadata, hkArrayView<hkReflect::Var> parents)
    {
        for (int parentIndex = 0; parentIndex < parents.getSize(); ++parentIndex)
        {
            if (parents[parentIndex].getType()->equals(metadata.getType()))
            {
                return parentIndex;
            }
        }
        return -1;
    }
}

void HK_CALL hkMonitorStreamParserUtil::accumulateMetaData(_In_ const hkMonitorStreamParser::Node* node, hkArrayView<hkReflect::Var> parents)
{
    hkArray<hkReflect::Var> localParents;
    localParents.append(parents.begin(), parents.getSize());
    for (int childIndex = 0; childIndex < node->m_children.getSize(); childIndex++)
    {
        hkMonitorStreamParser::Node* current = node->m_children[childIndex];
        if (current->m_type == hkMonitorStreamParser::Node::NODE_TYPE_META_DATA)
        {
            hkReflect::Var metadata = current->m_metaData;
            if (metadata)
            {
                int parentIndex = hkMonitorStreamParser_findParentMetadata(metadata, localParents);
                (parentIndex != -1 ? localParents[parentIndex] : localParents.expandOne()) = metadata;
            }
        }
    }

    for (int childIndex = 0; childIndex < node->m_children.getSize(); childIndex++)
    {
        hkMonitorStreamParser::Node* current = node->m_children[childIndex];

        // Do not further process nodes of type NODE_TYPE_DRAW_CALL_HANDLE and NODE_TYPE_META_DATA as these types do not have any children and are processed together with the parent node.
        if (current->m_type != hkMonitorStreamParser::Node::NODE_TYPE_GPU_HANDLE && current->m_type != hkMonitorStreamParser::Node::NODE_TYPE_META_DATA)
        {
            accumulateMetaData(current, localParents);
        }
    }

    for (int childIndex = 0; childIndex < node->m_children.getSize(); childIndex++)
    {
        hkMonitorStreamParser::Node* current = node->m_children[childIndex];
        if (current->m_type == hkMonitorStreamParser::Node::NODE_TYPE_META_DATA)
        {
            hkReflect::Var metadata = current->m_metaData;
            if (metadata)
            {
                int parentIndex = hkMonitorStreamParser_findParentMetadata(metadata, parents);
                if (parentIndex != -1)
                {
                    hkMonitorStreamParser_accumulateMetadata(parents[parentIndex], metadata);
                }
            }
        }
    }
}

void HK_CALL hkMonitorStreamParserUtil::makeStringsLocal(_Inout_ hkMonitorStreamParser::Node* node, hkPointerMap<char*, char*>& map)
{
    if (!map.getWithDefault((char*)node->m_name, HK_NULL))
    {
        char* localStr = hkString::strDup(node->m_name);
        map.insert(localStr, localStr);
        node->m_name = localStr;
    }
    for (int c = 0; c < node->m_children.getSize(); ++c)
    {
        makeStringsLocal(node->m_children[c], map);
    }
}

void HK_CALL hkMonitorStreamParserUtil::deallocateLocalStrings(hkPointerMap<char*, char*>& map)
{
    hkPointerMap<char*, char*>::Iterator iter = map.getIterator();
    while (map.isValid(iter))
    {
        hkDeallocate<char>(map.getKey(iter));
        iter = map.getNext(iter);
    }
    map.clear();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
