// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Reflect/Util/hkReflectAny.h>
#include <Common/Base/Container/PointerMap/hkPointerMap.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamAttributes.h>

// For backward compatibility with monitor streams generated on PlayStation(R)3:
// Each elf stores a unique ID in the monitor stream at startup to allow for string fix-up later on PPU. This maximum
// value is used to distinguish these IDs from normal commmandAndMonitor pointers.
// The maximum value is reserved to indicate that no string fix-up should be performed in any of the following commands
// until a new elf ID is encountered.
#define HK_MAX_ELF_ID 255


struct HK_EXPORT_COMMON hkGpuProfilingGranularity
{
    HK_DECLARE_CLASS(hkGpuProfilingGranularity, New);

    enum Enum
    {
        PROFILE_SCOPES = 0,
        PROFILE_RENDERABLES = 1,
        PROFILE_DRAW_CALL_RECORDING = 2,
        PROFILE_DRAW_CALL_EXECUTION = 3,
        PROFILE_PIXEL_COUNT = 4,

        COUNT
    };
};


class HK_EXPORT_COMMON hkMonitorStreamStringMap : public hkReferencedObject
{
    public:
        HK_DECLARE_CLASS(hkMonitorStreamStringMap, New, Reflect, EmptyCtor );
        HK_RECORD_ATTR(hk::Version(1));

        struct HK_EXPORT_COMMON StringMap
        {
            HK_DECLARE_CLASS(StringMap, New, Reflect, EmptyCtor );
            HK_ALIGN(hkUint64 m_id,8); // no ulong in serialization yet, and void* will be zero
            hkStringPtr m_string;
        };

        void clear();
        void createRuntime();
        void merge( const hkMonitorStreamStringMap& other );

        hkArray<StringMap> m_map;
        hkHashMap< hkUint64, const char* > m_runtimeMap HK_ATTR( hk::Reflect(false) );
};

class HK_EXPORT_COMMON hkMonitorStreamTypeMap : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS(hkMonitorStreamTypeMap, New, Reflect, EmptyCtor );
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

        struct HK_EXPORT_COMMON TypeMap
        {
            HK_DECLARE_CLASS(TypeMap, New, Reflect, EmptyCtor );
            HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

            HK_ALIGN(hkUint64 m_id, 8); // no ulong in serialization yet, and void* will be zero
            const hkReflect::Type* m_type;
        };

        void clear();
        void createRuntime();
        void merge(const hkMonitorStreamTypeMap& other);

        hkArray<TypeMap> m_map;
        hkHashMap< hkUint64, const hkReflect::Type* > m_runtimeMap HK_ATTR( hk::Reflect(false) );
};


class HK_EXPORT_COMMON hkMonitorStreamColorTable : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS(hkMonitorStreamColorTable, New, Reflect );

        hkMonitorStreamColorTable()
        {
            m_defaultColor = 0xFFFFFFFF;
        }

            /// A mapping of timer names to colors
        struct HK_EXPORT_COMMON ColorPair
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, hkMonitorStreamColorTable::ColorPair );
            HK_DECLARE_REFLECTION();

            hkStringPtr m_colorName;
            hkColor::Argb m_color;

            ColorPair() : m_color(hkColor::WHITE) {}
            ColorPair( const char* name, hkColor::Argb color )
                :   m_colorName(name),
                    m_color( color ){}
        };

        hkArray< struct ColorPair > m_colorPairs;
        hkUint32 m_defaultColor;

        hkColor::Argb findColor(_In_z_ const char* color );

        void addColor(_In_z_ const char* name, hkColor::Argb color );

        void setupDefaultColorTable();
};

struct HK_EXPORT_COMMON hkMonitorStreamColorTableCache
{
        bool findColor(_In_z_ const char* color, hkColor::Argb& c ) const;
        void cacheColor(_In_z_ const char* n, hkColor::Argb& c) const;

        hkRefPtr<hkMonitorStreamColorTable> m_colorTable;

        mutable hkPointerMap<const char*, hkColor::Argb> m_colorCache;

        // used for debugging lots of missing named colors:
        mutable hkPointerMap<const char*, hkColor::Argb> m_unknownColorCache;
};

/// Information about GPU timings.
class HK_EXPORT_COMMON hkGpuTraceResult
{
    public:

        HK_DECLARE_CLASS(hkGpuTraceResult, New, Reflect, EmptyCtor );

        enum ScopeType
        {
            SCOPE_PROBE = 0, // has a proper start and end
            SCOPE_CALL  // has a proper end, but start is from previous call end
        };

        HK_ALIGN(hkUint64 m_id,8);
        double m_gpuTimeBegin;
        double m_gpuTimeEnd;
        hkUint32 m_numPixelsTouched;
        hkEnum<ScopeType, hkUint16> m_type;
        hkUint16 m_threadId;
        hkRefPtr<hkReferencedObject> m_meta; // used to be a hkReflect::Any but ownership got messy
};

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkGpuTraceResult::ScopeType);

namespace hkMonitorStreamParser
{
    struct Node;
}

struct HK_EXPORT_COMMON hkMonitorStreamGpuHandleCache
{
    struct Mapping
    {
        const hkMonitorStreamParser::Node* m_timerNode;
        const hkMonitorStreamParser::Node* m_addHandleNode;
    };

    inline _Ret_maybenull_ Mapping* get(hkUint64 gpuHandle, bool insert = true)
    {
        int index = m_gpuHandleToIndex.getWithDefault(gpuHandle, -1);
        if (index >= 0)
        {
            return m_gpuData.begin() + index;
        }

        if ( insert )
        {
            m_gpuHandleToIndex.insert( gpuHandle, m_gpuData.getSize() );
            Mapping* m = m_gpuData.expandBy(1);
            m->m_timerNode = m->m_addHandleNode = HK_NULL;
            return m;
        }

        return HK_NULL;
    }

    void clear()
    {
        m_gpuData.setSize(0);
        m_gpuHandleToIndex.clear();
    }

    int getSize() const
    {
        return m_gpuData.getSize();
    }

    hkArray< Mapping > m_gpuData;
    hkPointerMap<hkUint64, int> m_gpuHandleToIndex;
};

namespace hkMonitorStreamParser
{
        typedef hkUlong TagType;    // 32 bit in 32bit mode, 64 bit in 64 bit mode

        /// Helper struct for collect tags. Because of multithreading, the same timer
        /// could be issued on multiple threads, so once you merge threads, the same node
        /// might get multiple tags, therefor we need to use an array of tags.
        struct HK_EXPORT_COMMON TagInfo
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, hkMonitorStreamParser::TagInfo );
            hkArray<TagType> m_tags;
        };

        struct HK_EXPORT_COMMON NodeBase
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, NodeBase );
            virtual ~NodeBase() {}
        };

        struct HK_EXPORT_COMMON Node : public NodeBase
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, Node );

            enum NodeType
            {
                NODE_TYPE_TIMER,
                NODE_TYPE_TIMER_DRAW_CALL,
                NODE_TYPE_DIRECTORY,
                NODE_TYPE_SINGLE,
                NODE_TYPE_META_DATA,
                NODE_TYPE_GPU_HANDLE
            };

            enum NodeFlags
            {
                FLAGS_TIME_IS_NOT_ABSOLUTE = 1<<0, // Basically is a MultiTimer, which should be avoided if you want to use FlameGraph etc..
                FLAGS_USER_0 = 1<<1,
                FLAGS_USER_1 = 1<<2,
                FLAGS_USER_2 = 1<<3,
                FLAGS_USER_3 = 1<<4,
                FLAGS_USER_4 = 1<<5,
                FLAGS_USER_5 = 1<<6
            };

            Node(_In_opt_ Node* parent, _In_z_ const char* name, NodeType type);
            Node();
            Node(const Node& rhs);
            virtual ~Node();

            Node& operator=( const Node& other );

            virtual void clear();

            void setTimers( hkReal timerFactor, const hkMonitorStream::TimerCommand& start, const hkMonitorStream::TimerCommand& end);
            void setTimers( hkReal timerFactor, const hkMonitorStream::MultiTimerCommand& multiTimer);

            float                       m_value;  // when m_threadFlags not empty, this is sum across all threads
            hkUint32                    m_count;  // when m_threadFlags not empty, this is sum across all threads
            float                       m_threadMaxValue;
            float                       m_threadMinValue;
            hkFlags<NodeFlags,hkUint32> m_threadFlags;  // flag as to which thread we ran on (where ran == m_value > 0), only used in combined nodes
            hkUint64                    m_gpuHandle;
            double                      m_absoluteStartTime;
            const char*                 m_name;

            Node*                       m_parent;
            TagInfo*                    m_tags;
            TagType                     m_moveToTag;

            hkEnum<NodeType,hkUint16>   m_type;
            hkFlags<NodeFlags,hkUint16> m_flags;
            hkArray<struct Node*>       m_children;
            hkReflect::Var              m_metaData;
        };

        struct HK_EXPORT_COMMON Tree : public Node
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MONITOR, Tree );

            Tree( ) : Node(HK_NULL, "/", NODE_TYPE_DIRECTORY ) {}
            virtual ~Tree() { }

            typedef hkPointerMap<Node*,TagType> TaggedNodesMap;

            virtual void clear();

            hkPointerMap<TagType, Node*> m_tagToNode;
            TaggedNodesMap m_taggedNodes;   // nodes which we need to move to be a child of a tagNode

            hkMonitorStreamGpuHandleCache m_gpuHandleCache;
            hkUint32 m_frameId;
        };

        // You either make a Tree of Nodes from the trace, or you can gather some raw data (for either CPU or GPU timer use).
        // Gathering the values directly is fastest, but using the Nodes is best for general use and will have more information

        HK_EXPORT_COMMON Tree* HK_CALL makeTree(
            const hkMonitorStream::CommandStreamConfig& config,
            const hkTimerData& timerData, hkReal timerFactor,
            _In_z_ const char* rootNodeName = "/", bool reuseNodesIfPossible = true, bool trackGpuHandles = false, hkArrayView<const char*> ignoredNodes = hkArrayView<const char*>());

        // If you are just interested in some total valaues for given timer names, such as for a line graph
        // you can just parse the stream directly and not make Nodes
        HK_EXPORT_COMMON void HK_CALL gatherRawCpuTimers(
            const hkMonitorStream::CommandStreamConfig& config,
            const hkTimerData& timerData, hkReal timerFactor,
            hkArrayView<const char*> timerNames, _Out_writes_(_Inexpressible_(timerNames.getSize())) hkReal* valuesOut );

        // For gpu timers, where you just want to work out which gpu id is used by given cpu node names
        struct HK_EXPORT_COMMON TimerNameMap
        {
            const char* name; // will be == timerNames array values in below call, so can avoid strcmp if you like
            hkUint64 id;
        };
        HK_EXPORT_COMMON void HK_CALL gatherRawGpuIds(
            const hkMonitorStream::CommandStreamConfig& config,
            const hkTimerData& timerData, hkArrayView<const char*> timerNames, hkArray<TimerNameMap>& map );
};


namespace hkMonitorStreamParserUtil
{
    // In order to send a stats dump to a different pc, or even just to file and back, we neeed
    // to gather any pointers in the stream. Currently that is the char* for the commands and names, but then also
    // any Type* that we have embedded in the stream, as meta data
    HK_EXPORT_COMMON void HK_CALL extractMapsFromStreams(
        const hkMonitorStream::CommandStreamConfig& config, hkArrayView<const hkTimerData> timerData,
        hkMonitorStreamStringMap& smap, hkMonitorStreamTypeMap& tmap );

    // Combined trees can loose some information, such as Meta and gpuhandles.
    // They are normally just for tree overviews, use the exact per thread tress in a flame graph etc to see exact info per thread
    // Will delete other trees.
    HK_EXPORT_COMMON void HK_CALL combineTrees(_Inout_ hkMonitorStreamParser::Tree* mainTree, hkArrayView<hkMonitorStreamParser::Tree*> otherTrees);

    // Search node (and children down to 'levels') for a valid meta data vars
    HK_EXPORT_COMMON void HK_CALL findMetaData(_In_ const hkMonitorStreamParser::Node* n, int levels, hkArray<hkReflect::Var>& result);

    // Set a flag on nodes with given text in name (and all parents of those nodes)
    HK_EXPORT_COMMON void HK_CALL setFlagOnNodesByName(_Inout_ hkMonitorStreamParser::Node* n, _In_z_ const char* str, int flagToSet, int flagForParents);
    HK_EXPORT_COMMON void HK_CALL clearFlagOnNodes(_Inout_ hkMonitorStreamParser::Node* n, int flagToClear);

    HK_EXPORT_COMMON void HK_CALL accumulateMetaData(_In_ const hkMonitorStreamParser::Node* node, hkArrayView<hkReflect::Var> parents);

    HK_EXPORT_COMMON void HK_CALL makeStringsLocal(_Inout_ hkMonitorStreamParser::Node* node, hkPointerMap<char*, char*>& map);

    HK_EXPORT_COMMON void HK_CALL deallocateLocalStrings( hkPointerMap<char*, char*>& map );
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
