// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamParser.h>

#define DEBUG_LOG_IDENTIFIER "base.monitor"
#include <Common/Base/System/Log/hkLog.hxx>


namespace
{
    struct hkMonitorStreamParser_TempNode
    {
        hkMonitorStreamParser::Node* m_node;
        hkMonitorStreamParser::TagType m_moveToTag;
    };

    HK_INLINE void hkMonitorStreamParser_push(hkArray<hkMonitorStreamParser_TempNode>& stack, _In_ hkMonitorStreamParser::Node* node, hkMonitorStreamParser::TagType moveToTag)
    {
        hkMonitorStreamParser_TempNode& n = stack.expandOne();
        n.m_node = node;
        n.m_moveToTag = moveToTag;
    }

    HK_INLINE void hkMonitorStreamParser_pop( hkArray<hkMonitorStreamParser_TempNode>& stack, _Outptr_ hkMonitorStreamParser::Node** node, _Out_ hkMonitorStreamParser::TagType* moveToTag )
    {
        hkMonitorStreamParser_TempNode& n = stack.back();
        *node = n.m_node;
        *moveToTag = n.m_moveToTag;
        stack.popBack();
    }


    static _Ret_notnull_ hkMonitorStreamParser::Node* createNewNode(_Inout_ hkMonitorStreamParser::Tree* root, _Inout_opt_ hkMonitorStreamParser::Node* parent,
        _In_z_ const char* name, hkMonitorStreamParser::Node::NodeType type, bool wantNodeReuse, hkMonitorStreamParser::TagType tag)
    {
        // see if we have added this node already to this parent. If so reuse and augment
        // to prevent needless splitting.
        if ( name[0] == '/' )
        {
            name++;
            parent = root;
        }

        if( wantNodeReuse)
        {
            for (int c=0; parent && name && (c < parent->m_children.getSize()); ++c)
            {
                hkMonitorStreamParser::Node* child = parent->m_children[c];
                if ( (tag == child->m_moveToTag) &&  child->m_name && (hkString::strCmp(child->m_name, name) == 0) )
                {
                    return child;
                }
            }
        }

        hkMonitorStreamParser::Node* node = new hkMonitorStreamParser::Node( parent, name, type );
        if ( tag )
        {
            node->m_moveToTag = tag;
            root->m_taggedNodes.insert(node, tag);
        }
        return node;
    }

    /// A structure used in hkMonitorStreamParser::gatherRawTimers() to collect timer values
    struct hkMonitorStreamParser_RawTimerCollector
    {
        void HK_INLINE init(_In_z_ const char* timerName)
        {
            m_node = hkMonitorStreamParser::Node( HK_NULL, timerName, hkMonitorStreamParser::Node::NODE_TYPE_TIMER );
            m_stringLength = hkString::strLen( timerName ) + 1;
            m_startCommand.m_commandAndMonitor = HK_NULL;
            m_depth = 0;
        }

        hkMonitorStreamParser::Node         m_node;
        int                                 m_stringLength; // including terminator. For memcmp() instead of strcmp().
        hkMonitorStream::TimerCommand       m_startCommand;
        int                                 m_depth;
    };
}

void hkMonitorStreamStringMap::clear()
{
    m_runtimeMap.clear();
    m_map.clear();
}

void hkMonitorStreamStringMap::createRuntime()
{
    for (int s=0; s < m_map.getSize(); ++s)
    {
        m_runtimeMap.insert( m_map[s].m_id, m_map[s].m_string);
    }
}

void hkMonitorStreamStringMap::merge(const hkMonitorStreamStringMap& other)
{
    for (int s = 0; s < other.m_map.getSize(); ++s)
    {
        if (!m_runtimeMap.hasKey(other.m_map[s].m_id)) //assumes all char* are const once connection open
        {
            m_map.pushBack(other.m_map[s]);
            m_runtimeMap.insert(m_map.back().m_id, m_map.back().m_string.cString());
        }
    }
}

void hkMonitorStreamTypeMap::clear()
{
    m_runtimeMap.clear();
    m_map.clear();
}

void hkMonitorStreamTypeMap::createRuntime()
{
    for (int s=0; s < m_map.getSize(); ++s)
    {
        m_runtimeMap.insert( m_map[s].m_id, m_map[s].m_type);
    }
}


void hkMonitorStreamTypeMap::merge(const hkMonitorStreamTypeMap& other)
{
    for (int s = 0; s < other.m_map.getSize(); ++s)
    {
        if (!m_runtimeMap.hasKey(other.m_map[s].m_id)) //assumes all type* are const once connection open
        {
            m_map.pushBack(other.m_map[s]);
            m_runtimeMap.insert(m_map.back().m_id, m_map.back().m_type); //XX ownership of Type* is ill defined
        }
    }
}

/// Setting up a color table.
/// WARNING: The demo framework caches this color table
/// in a file in the root demo directory.
void hkMonitorStreamColorTable::setupDefaultColorTable()
{
    //
    // Physics 2012
    //

    addColor( "Physics 2012",       hkColor::GREY25 );

    // Blue for collision detection
    addColor( "NarrowPhase",        hkColor::CORNFLOWERBLUE );
    addColor( "Broadphase",         hkColor::BLUE );
    addColor( "3AxisSweep",         hkColor::MIDNIGHTBLUE );
    addColor( "WaitForExport",      hkColor::DARKSLATEBLUE );

    // Yellow / brown for single threaded work
    addColor( "SplitIsle",          hkColor::BROWN );
    addColor( "PostCollide",        hkColor::BEIGE );
    addColor( "Maintenance",        hkColor::YELLOW );
    addColor( "InterIsland",        hkColor::YELLOW );
    addColor( "PendingOps",         hkColor::YELLOW );
    addColor( "ConstraintCallbacks",hkColor::YELLOW );
    addColor( "SingleObj",          hkColor::TAN );
    addColor( "TOIs",               hkColor::GOLD );
    addColor( "PostSimCB",          hkColor::WHEAT);

    // Green for solve related jobs
    addColor( "Integrate",          hkColor::GREEN );
    addColor( "Actions",            hkColor::SPRINGGREEN );
    addColor( "SetupJobs",          hkColor::LIMEGREEN );
    addColor( "BuildJacTask",       hkColor::SEAGREEN );
    addColor( "BuildAccumulators",  hkColor::OLIVE );
    addColor( "BuildJacobians",     hkColor::FORESTGREEN );
    addColor( "Solver",             hkColor::PALEGREEN );
    addColor( "Solve",              hkColor::PALEGREEN );
    addColor( "IntegrateMotions",   hkColor::LIGHTGREEN );
    addColor( "SolverExport",       hkColor::DARKSEAGREEN );

    // Red for locking etc
    addColor( "GetNextJob",             hkColor::DARKRED);
    addColor( "GetNextTask",            hkColor::DARKRED);
    addColor( "FinishJob",              hkColor::DARKRED);
    addColor( "FinishTask",             hkColor::DARKRED);
    addColor( "FinishJobAndGetNext",    hkColor::DARKRED);
    addColor( "FinishTaskAndGetNext",   hkColor::DARKRED);
    addColor( "WaitForSignal",          hkColor::MAROON);
    addColor( "WaitForTasks",           hkColor::MAROON);
    addColor( "LockQueue",              hkColor::RED);
    addColor( "CriticalLock",           hkColor::CRIMSON );

    // Raycasting
    addColor( "hkSpuMoppRaycastQuery", hkColor::MEDIUMPURPLE);

    // Vehicle
    addColor( "VehicleManager", hkColor::PALEGOLDENROD );
    addColor( "Vehicle", hkColor::WHEAT );
    addColor( "VehicleJob", hkColor::DARKKHAKI );
    addColor( "ApplyVehicleForces", hkColor::KHAKI );

    //
    // Physics
    //

    addColor( "Physics",            0xFFFFB300 );

    addColor( "PreCollide",         hkColor::BEIGE );
    addColor( "DispatchCommands",   hkColor::GOLD );
    addColor( "PostCollideEvent",   hkColor::WHEAT );
    addColor( "BroadPhase",         hkColor::BLUE );
    addColor( "NarrowPhase",        hkColor::CORNFLOWERBLUE );
    addColor( "GenerateWorkItems", hkColor::SALMON);
    addColor( "ProcessSecondRoundResults", hkColor::MEDIUMSLATEBLUE );
    addColor( "GridTask",           hkColor::CORNFLOWERBLUE );
    addColor( "PairsTask",          hkColor::LIGHTSTEELBLUE );

    addColor( "PrepareConstraints",     hkColor::LIMEGREEN );
    addColor( "BuildConstraintJacobians",   hkColor::LIMEGREEN );

    addColor( "PreSolve",           hkColor::OLIVE );
    addColor( "Solve",              hkColor::PALEGREEN );
    addColor( "SolveConstraints",   hkColor::LIGHTGREEN );
    addColor( "SubIntegrate",       hkColor::LIMEGREEN );
    addColor( "UpdateBodies",       hkColor::DARKSEAGREEN );
    addColor( "PostSolve",          hkColor::OLIVE );
    addColor( "PostSimulateEvent",  hkColor::WHEAT );
    addColor( "Deactivation",       hkColor::LIGHTSALMON );

    // Particles
    addColor( "PartitionTreePrepareBuildContextBuffers", hkColor::HOTPINK );
    addColor( "hkcdSimdTree::BuildFirstNLevels",         hkColor::HOTPINK );
    addColor( "hkcdSimdTree::BuildRanges",               hkColor::HOTPINK );
    addColor( "hkcdSimdTree::FinalizeTree",              hkColor::HOTPINK );
    addColor( "StaticCollisions", 0xFF66CC66 );
    addColor( "FindDynamicNeighbors", 0xFFFFB300);
    addColor( "DynamicCollisions", 0xFFFFB300 );
    addColor( "ParticleParticleCollisions", 0xFFFFFF00 );
    addColor( "ParticleParticleInterbranchSolveTask", 0xFFFFFF00 );
    addColor( "Query dirty AABBs", hkColor::MAGENTA );
    addColor( "Update dirty AABBs", hkColor::TEAL );

    // Waits
    addColor( "WaitForMainThread",      hkColor::DARKRED);
    addColor( "WaitForWorkerThreads",   hkColor::DARKRED);
    addColor( "WaitForOtherTasks",      hkColor::DARKRED);

    // Semaphore release timers. It can be very useful for debugging purposes to see the release count in the timers.
    addColor( "ReleaseSemaphore",       hkColor::MEDIUMPURPLE);
    addColor( "ReleaseSemaphore_1",     hkColor::MEDIUMPURPLE);
    addColor( "ReleaseSemaphore_2",     hkColor::MEDIUMPURPLE);
    addColor( "ReleaseSemaphore_3",     hkColor::MEDIUMPURPLE);
    addColor( "ReleaseSemaphore_4",     hkColor::MEDIUMPURPLE);
    addColor( "ReleaseSemaphore_5+",    hkColor::MEDIUMPURPLE);

    //
    // Animation
    //

    addColor( "hkaQuantizedSampleAndCombineJob",    hkColor::ORCHID);
    addColor( "SampleAndBlend", hkColor::MEDIUMPURPLE);
    addColor( "hkaCpuSampleAnimationJob",   hkColor::YELLOW);
    addColor( "hkaCpuSampleAndCombineJob",  hkColor::PALEGREEN);
    addColor( "SpuAnim",    hkColor::LIGHTBLUE);

    //
    // Cloth
    //

    addColor( "Cloth", hkColor::MEDIUMPURPLE);
    addColor( "Simulate", hkColor::LIGHTGREEN);
    addColor( "Accumulate Actions", hkColor::SPRINGGREEN);
    addColor( "Collide", hkColor::DARKGREEN);
    addColor( "Collide And Solve", hkColor::DARKGREEN);
    addColor( "Input Conversion", hkColor::PINK );
    addColor( "Output Conversion", hkColor::PINK );
    addColor( "Mesh Mesh Deform", hkColor::LIGHTBLUE );
    addColor( "Mesh Bone Deform", hkColor::LIGHTBLUE );
    addColor( "Skin", hkColor::BLUE);
    addColor( "Gather All Vertices", hkColor::YELLOW);
    addColor( "Gather Some Vertices", hkColor::YELLOW);
    addColor( "Copy Vertices", hkColor::YELLOW);
    addColor( "Recalculate Some Normals", hkColor::ORANGE);
    addColor( "Recalculate All Normals", hkColor::ORANGE);
    addColor( "Update Some Vertex Frames (N)", hkColor::ORANGE);
    addColor( "Update Some Vertex Frames (T)", hkColor::DARKORANGE);
    addColor( "Update Some Vertex Frames (TB)", hkColor::DARKORANGE);
    addColor( "Update All Vertex Frames (N)", hkColor::ORANGE);
    addColor( "Update All Vertex Frames (T)", hkColor::DARKORANGE);
    addColor( "Update All Vertex Frames (TB)", hkColor::DARKORANGE);

    // ENGINE / TOOLS
    addColor( "hktApplet3d::renderViewports", hkColor::SEAGREEN);
    addColor( "hkeWorld", hkColor::LIGHTGREEN);
    addColor( "hkeRenderPlugin::render", hkColor::BROWN);
    addColor( "hkrRenderPassTask", hkColor::ORANGERED);
    addColor( "hkrRenderPassSortTask", hkColor::BLUEVIOLET);
    addColor( "executeCommandBuffer", hkColor::DARKBLUE);

}



hkColor::Argb hkMonitorStreamColorTable::findColor(_In_z_ const char* name )
{
    const char* colorLookUp = name[0] == '/' ? name + 1 : name;
    for (int i = 0; i < m_colorPairs.getSize(); ++i)
    {
        if ( hkString::strCasecmp(m_colorPairs[i].m_colorName, colorLookUp) == 0)
        {
            return m_colorPairs[i].m_color;
        }
    }
    return m_defaultColor;
}

void hkMonitorStreamColorTable::addColor(_In_z_ const char* name, hkColor::Argb color )
{
    const char* colorLookUp = name[0] == '/' ? name + 1 : name;
    for (int i = 0; i < m_colorPairs.getSize(); ++i)
    {
        if ( hkString::strCasecmp(m_colorPairs[i].m_colorName, colorLookUp) == 0)
        {
            m_colorPairs[i].m_color = color;
            return;
        }
    }

    m_colorPairs.pushBack( ColorPair(name, color) );
}

bool hkMonitorStreamColorTableCache::findColor(_In_z_ const char* n, hkColor::Argb& c) const
{
    hkPointerMap<const char*, hkColor::Argb>::Iterator iter = m_colorCache.findKey(n);
    if (m_colorCache.isValid(iter))
    {
        c = m_colorCache.getValue(iter);
        return true;
    }

    // see if in color table to cache
    c = m_colorTable->findColor(n);
    if (c != m_colorTable->m_defaultColor)
    {
        m_colorCache.insert(n, c);
        return true;
    }

    // unknown color
    iter = m_unknownColorCache.findKey(n);
    if (!m_unknownColorCache.isValid(iter))
    {
        m_unknownColorCache.insert(n, c);
    }
    return false;
}

void hkMonitorStreamColorTableCache::cacheColor(_In_z_ const char* n, hkColor::Argb& c) const
{
    m_colorCache.insert(n, c);
}

hkMonitorStreamParser::Tree* hkMonitorStreamParser::makeTree(const hkMonitorStream::CommandStreamConfig& config,
    const hkTimerData& timerData, hkReal timerFactor,
    _In_z_ const char* rootNodeName, bool reuseNodesIfPossible, bool trackGpuHandles,
    hkArrayView<const char*> ignoredNodes)
{

    const char* currentStreamPtr = timerData.m_streamBegin;
    Tree* rootNode = new Tree( );
    Node* currentNode = rootNode;
    Node* lastBeginListNodeCreated = HK_NULL;   // helps debugging
    hkInplaceArray<hkMonitorStream::TimerCommand,16> timerStack;


    hkInplaceArray<hkMonitorStreamParser_TempNode,16> nodeStack;

    // The stack below is just for debugging - it holds all the last timers parsed. If you are missing an 'end' macro
    // then the culprit *is likely* to be one of debugMismatchCheckerTimerStack[0] to debugMismatchCheckerTimerStack[ current size of debugMismatchCheckerTimerStack - 1]
    // and is most likely the most recently altered file. However further monitor parsing may have overwritten the culprit so it may not be in this array.
    // The only way to match things perfectly and always identify the culprit is to use HK_TIMER_NAMED_END
    hkInplaceArray<hkMonitorStream::TimerCommand,16> debugMismatchCheckerTimerStack;
    TagType moveToTag = 0;

    while( currentStreamPtr < timerData.m_streamEnd )
    {
        hkUint32 rawValue = hkMonitorStream::readCommandUInt32(&currentStreamPtr, config);
        currentStreamPtr -= 4;

        const char* currentBeforeString = currentStreamPtr;
        const char* string = hkMonitorStream::readCommandString(&currentStreamPtr, config);

        // Skip commands used to mark the start of an elf
        {
            if ( rawValue <= HK_MAX_ELF_ID)
            {
                //int elfId = rawValue;
                continue;
            }
        }

        currentStreamPtr = currentBeforeString;
        if(0){      // seriously helps with debugging
            int offset = int(currentStreamPtr-timerData.m_streamBegin);
            if ( offset == 0 )
            {
                Log_Info( "\n\n\n" );
            }

            char buffer[256];
            int numTabs = timerStack.getSize() + nodeStack.getSize();
            for (int i =0; i < numTabs; i++) {buffer[i] = '\t';}
            hkString::snPrintf(buffer+numTabs, HK_COUNT_OF(buffer)-numTabs, 255-numTabs, "%s\t%i", string, offset );
            Log_Info( buffer );
        }
        switch(string[0])
        {
        case 'F':
            {
                hkMonitorStream::FrameIdCommand com; com.read(&currentStreamPtr, config);
                rootNode->m_frameId = com.m_id;
                break;
            }
        case 'X':
            {
                hkMonitorStream::TagCommand com; com.read(&currentStreamPtr, config);
                if ( moveToTag && !currentNode->m_parent)   // as we cannot move the root node just shortcut the move
                {
                    currentNode = createNewNode( rootNode, currentNode, "Dummy: Missing timer around hkTaskQueue::submitHandles()", hkMonitorStreamParser::Node::NODE_TYPE_DIRECTORY, true, moveToTag );
                    moveToTag = 0;
                }
                rootNode->m_tagToNode.insert(com.m_tag, currentNode);
                if (!currentNode->m_tags)
                {
                    currentNode->m_tags = new TagInfo;
                }
                currentNode->m_tags->m_tags.pushBack(com.m_tag);
                break;
            }

        case 'G':   // timer with tag
            {
                hkMonitorStream::TagCommand com;
                com.read(&currentStreamPtr, config);
                moveToTag = com.m_tag;
                break;
            }

        case 'R': // timer begin with self
        case 'T': // timer begin
            {
                hkMonitorStreamParser_push( nodeStack, currentNode, moveToTag );
                if (string[1] == 't')
                {
                    currentNode = createNewNode( rootNode, currentNode, string + 2, hkMonitorStreamParser::Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                    moveToTag = 0;

                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, config);

                    timerStack.pushBack( com );
                    debugMismatchCheckerTimerStack.pushBack( com );

                    // if ( string[0] == 'R' ) // old Node::FLAGS_REQUEST_SELF, should not be needed if display is good enough
                }
                else
                {
                    HK_ASSERT_NO_MSG(0xa895bea, string[1] == 'g');
                    currentNode = createNewNode( rootNode, currentNode, string + 2, hkMonitorStreamParser::Node::NODE_TYPE_TIMER_DRAW_CALL, reuseNodesIfPossible, moveToTag );

                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, config);

                    timerStack.pushBack( com );
                    debugMismatchCheckerTimerStack.pushBack( com );

                    currentNode->m_gpuHandle = com.m_gpuHandle;

                    if (trackGpuHandles)
                    {
                        hkMonitorStreamGpuHandleCache::Mapping* gpuData = rootNode->m_gpuHandleCache.get( currentNode->m_gpuHandle );
                        gpuData->m_timerNode = currentNode;
                    }
                }
                break;
            }

        case 'E': // timer end
            {
                if ( timerStack.getSize() == 0)
                {
                    HK_WARN(0xfafe7975, "Unmatched HK_TIMER_END() macro (with no HK_TIMER_BEGIN()) in timed code");
                    return rootNode;
                }
                const hkMonitorStream::TimerCommand& start = timerStack[timerStack.getSize() - 1];

                if ( string[2] && hkString::strCmp( start.m_commandAndMonitor+2, string+2) !=0 )
                {
                    HK_WARN( 0xf03edefe, "Unmatched timer command: '" << start.m_commandAndMonitor+1 << "' =! '" << string+1 );
                    return rootNode;
                }

                if (string[1] == 'g') // gpu timer
                {
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, config);

                    if (timerFactor != 0.f)
                        currentNode->setTimers(timerFactor, start, com);
                }
                else
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, config);

                    if (timerFactor != 0.f)
                        currentNode->setTimers(timerFactor, start, com);
                }

                hkMonitorStreamParser_pop( nodeStack, &currentNode, &moveToTag );
                timerStack.popBack();
                debugMismatchCheckerTimerStack.popBack();
                break;
            }

        case 'Y': // multi timer begin
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);

                hkMonitorStreamParser_push( nodeStack, currentNode, moveToTag );
                currentNode = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                moveToTag = 0;
                currentNode->m_flags.orWith( Node::FLAGS_TIME_IS_NOT_ABSOLUTE ); // notify the timer node that it doesn't hold an honest absolute time!
                break;
            }

        case 'W': // multi timer end
            {
                hkMonitorStream::MultiTimerCommand com;
                com.read(&currentStreamPtr, config);

                if (timerFactor != 0.f)
                    currentNode->setTimers( timerFactor, com );

                hkMonitorStreamParser_pop( nodeStack, &currentNode, &moveToTag );
                break;
            }

        case 'O': // object name
            {
                hkMonitorStream::TimerBeginObjectNameCommand com;
                const char* name = com.read(&currentStreamPtr, config);

                timerStack.pushBack( com );
                hkMonitorStreamParser_push( nodeStack, currentNode, moveToTag );
                currentNode = createNewNode( rootNode, currentNode, name, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                moveToTag = 0;
                debugMismatchCheckerTimerStack.pushBack( com );
                break;
            }

        case 'L': // timer list begin
            {
                hkMonitorStream::TimerBeginListCommand com;
                com.read(&currentStreamPtr, config);

                {
                    timerStack.pushBack( com );
                    debugMismatchCheckerTimerStack.pushBack( com );
                    hkMonitorStreamParser_push( nodeStack, currentNode, moveToTag );
                    currentNode = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                    moveToTag = 0;
                    lastBeginListNodeCreated = currentNode;
                }

                {
                    hkMonitorStream::TimerCommand com2 = com;
                    com2.m_commandAndMonitor = com.m_nameOfFirstSplit;
                    timerStack.pushBack( com2 );
                    debugMismatchCheckerTimerStack.pushBack( com2 );
                    hkMonitorStreamParser_push( nodeStack, currentNode, moveToTag );
                    currentNode = createNewNode( rootNode, currentNode, com2.m_commandAndMonitor+2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                }

                break;
            }

        case 'S': // split list
            {
                if ( timerStack.getSize() == 0)
                {
                    HK_WARN(0xfafe79fa, "Unmatched HK_TIMER_SPLIT_LIST() macro (with no HK_TIMER_BEGIN_LIST()) in timed code");
                    return rootNode;
                }
                hkMonitorStream::TimerCommand& start = timerStack[ timerStack.getSize()- 1 ];

                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, config);

                if (timerFactor != 0.f)
                    currentNode->setTimers(timerFactor, start, com);

                currentNode = createNewNode( rootNode, currentNode->m_parent, string + 2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                lastBeginListNodeCreated = currentNode;
                start = com;
                break;
            }

        case 'l': // list end
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, config);

                {
                    if ( timerStack.getSize() <= 1)
                    {
                        HK_ON_DEBUG( const char* lastName = (lastBeginListNodeCreated) ? lastBeginListNodeCreated->m_name : "ROOT" );
                        HK_WARN(0xfafe79fb, "Unmatched HK_TIMER_END_LIST() macro (with no HK_TIMER_BEGIN_LIST()) in timed code, last beginList created was " << lastName);
                        return rootNode;
                    }

                    const hkMonitorStream::TimerCommand& start = timerStack[ timerStack.getSize()- 1 ];
                    if ( start.m_commandAndMonitor[0] != 'S')
                    {
                        HK_WARN( 0xf0323454, "Mismatched HK_TIMER_END_LIST()/HK_TIMER_BEGIN_LIST() found. Probably missing a HK_TIMER_BEGIN_LIST() *or* timer '" << start.m_commandAndMonitor + 2 << "' has no END macro. Otherwise check debugMismatchCheckerTimerStack in this function for likely culprits.");
                        return rootNode;
                    }

                    if (timerFactor != 0.f)
                        currentNode->setTimers(timerFactor, start, com);
                    hkMonitorStreamParser_pop( nodeStack, &currentNode, &moveToTag );
                    timerStack.popBack();
                    debugMismatchCheckerTimerStack.setSize( debugMismatchCheckerTimerStack.getSize() - 1 );
                }
                {
                    const hkMonitorStream::TimerCommand& start = timerStack[ timerStack.getSize()- 1 ];
                    if (start.m_commandAndMonitor[0] != 'L')
                    {
                        HK_WARN(0xf0323454, "Mismatched HK_TIMER_END_LIST()/HK_TIMER_BEGIN_LIST() found. Probably missing a HK_TIMER_BEGIN_LIST() *or* timer '" << start.m_commandAndMonitor + 2 << "' has no END macro. Otherwise check debugMismatchCheckerTimerStack in this function for likely culprits.");
                        return rootNode;
                    }
                    if (timerFactor != 0.f)
                        currentNode->setTimers(timerFactor, start, com);
                    hkMonitorStreamParser_pop(nodeStack, &currentNode, &moveToTag);
                    timerStack.popBack();
                    debugMismatchCheckerTimerStack.setSize( debugMismatchCheckerTimerStack.getSize() - 1 );
                }
                break;
            }

        case 'M':   //HK_MONITOR_COMMAND_ADD_VALUE
            {
                hkMonitorStream::AddValueCommand com;
                com.read(&currentStreamPtr, config);

                Node *node = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_SINGLE, reuseNodesIfPossible, moveToTag);
                node->m_value += com.m_value;
                node->m_count += 1;
                break;
            }
        case 'A':
            {
                hkMonitorStream::AddStructCommand com;
                void* data = com.read(&currentStreamPtr, config);

                Node *node = createNewNode( rootNode, currentNode, string + 2, hkMonitorStreamParser::Node::NODE_TYPE_META_DATA, reuseNodesIfPossible, moveToTag );
                node->m_metaData = hkReflect::Var(data, com.m_type);
                break;
            }
        case 'H':
            {
                hkMonitorStream::AddGpuHandleCommand com;
                com.read(&currentStreamPtr, config);

                Node* node = createNewNode( rootNode, currentNode, string + 1, hkMonitorStreamParser::Node::NODE_TYPE_GPU_HANDLE, reuseNodesIfPossible, moveToTag);
                node->m_gpuHandle = com.m_gpuHandle;
                if (trackGpuHandles)
                {
                    hkMonitorStreamGpuHandleCache::Mapping* gpuData = rootNode->m_gpuHandleCache.get( node->m_gpuHandle );
                    gpuData->m_addHandleNode = currentNode;
                }

                break;
            }
        case 'P': //Push Dir
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);

                hkMonitorStreamParser_push( nodeStack, currentNode, moveToTag );
                currentNode = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_DIRECTORY, reuseNodesIfPossible, moveToTag);
                moveToTag = 0;
                break;
            }

        case 'p':
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);

                if ( nodeStack.isEmpty() )
                {
                    HK_WARN(0xf023df45, "Mismatched HK_TIMER_POP() function");
                    return rootNode;
                }
                hkMonitorStreamParser_pop( nodeStack, &currentNode, &moveToTag );
                break;
            }

        case 'N': // nop
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);
                break;
            }

        case 'm': // memory
            {
                hkMonitorStream::MemoryCommand com;
                com.read(&currentStreamPtr, config);
                break;
            }

        default:
            HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
            return rootNode;
        }
    }

    if ( timerStack.getSize() )
    {
        HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data. Probably missing HK_TIMER_END*s eg. for timer '" << timerStack[timerStack.getSize() - 1].m_commandAndMonitor + 2 << "' Check timerStack in this function for likely culprits." );
    }

    return rootNode;
}


// This impl of gatherRawTimers is a bit slower than the old hkMonitorStreamAnalyser ver
// but his version can work across diffrerent PCs/Console etc as uses read() rather than casts
// so can be used in VDB
void HK_CALL hkMonitorStreamParser::gatherRawCpuTimers(const hkMonitorStream::CommandStreamConfig& config, const hkTimerData& timerData, hkReal timerFactor, hkArrayView<const char*> timerNames, _Out_writes_(_Inexpressible_(timerNames.getSize())) hkReal* valuesOut)
{
    int numTimers = timerNames.getSize();
    if (numTimers <= 0 )
        return;

    // Create some temporary storage to collect the timings
    hkArray<hkMonitorStreamParser_RawTimerCollector>::Temp collectors( numTimers );
    for( int i=0; i<numTimers; ++i )
    {
        collectors[i].init( timerNames[i] );
    }

    // Walk the stream
    const char* currentStreamPtr = timerData.m_streamBegin;
    while( currentStreamPtr < timerData.m_streamEnd )
    {

        hkUint32 rawValue = hkMonitorStream::readCommandUInt32(&currentStreamPtr, config);
        // Skip commands used to mark the start of an elf
        if ( rawValue <= HK_MAX_ELF_ID)
        {
            //int elfId = rawValue;
            continue;
        }
        currentStreamPtr -= 4;

        const char* currentBeforeString = currentStreamPtr;
        const char* string = hkMonitorStream::readCommandString(&currentStreamPtr, config);
        const char* monitorName = string + 2;
        currentStreamPtr = currentBeforeString; // the cmd read funcs all do the string bit again. Not great.

        switch( string[0] )
        {

        case 'F':
            {
                hkMonitorStream::FrameIdCommand com; com.read(&currentStreamPtr, config);
                break;
            }

        case 'X':
            {
                hkMonitorStream::TagCommand com; com.read(&currentStreamPtr, config);
                break;
            }

        case 'G':   // timer with tag
            {
                hkMonitorStream::TagCommand com;
                com.read(&currentStreamPtr, config);
                break;
            }

        case 'R': // timer begin with self
        case 'T': // timer begin
            {
                if (string[1] == 't')
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, config);

                    for( int i=0; i<numTimers; ++i )
                    {
                        if( collectors[i].m_startCommand.m_commandAndMonitor )
                        {
                            collectors[i].m_depth++;
                        }
                        else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                        {
                            collectors[i].m_startCommand = com;
                            collectors[i].m_depth = 0;
                        }
                    }
                }
                else // gpu
                {
                    HK_ASSERT_NO_MSG(0x372112f9, string[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, config);
                }
                break;
            }

        case 'E': // timer end
            {
                if (string[1] == 't')
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, config);
                    for( int i=0; i<numTimers; ++i )
                    {
                        if( collectors[i].m_startCommand.m_commandAndMonitor  )
                        {
                            if( collectors[i].m_depth == 0 )
                            {
                                collectors[i].m_node.setTimers( timerFactor, collectors[i].m_startCommand, com);
                                collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                            }
                            else
                            {
                                collectors[i].m_depth--;
                            }
                        }
                    }
                }
                else // Gpu
                {
                    HK_ASSERT_NO_MSG(0x7a67f97d, string[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, config);
                }


                break;
            }

        case 'Y': // multi timer begin
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor)
                    {
                        collectors[i].m_depth++;
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        // Note that this pointer isn't used by the multi timer's end event,
                        // but setting it to a non-NULL value is needed for the code logic to work
                        collectors[i].m_startCommand.m_commandAndMonitor = (const char*)0x1;
                        collectors[i].m_depth = 0;
                    }
                }
                break;
            }

        case 'W': // multi timer end
            {
                hkMonitorStream::MultiTimerCommand com;
                com.read(&currentStreamPtr, config);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor)
                    {
                        if( collectors[i].m_depth == 0 )
                        {
                            collectors[i].m_node.setTimers( timerFactor, com );
                            collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                        }
                        else
                        {
                            collectors[i].m_depth--;
                        }
                    }
                }


                break;
            }

        case 'O': // object name
            {
                hkMonitorStream::TimerBeginObjectNameCommand com;
                /*const char* name = */ com.read(&currentStreamPtr, config);

                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        collectors[i].m_depth++;
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 0;
                    }
                }

                break;
            }

        case 'L': // timer list begin
            {
                hkMonitorStream::TimerBeginListCommand com;
                com.read(&currentStreamPtr, config);

                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        collectors[i].m_depth += 2;
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 1;
                    }
                    else if( hkString::memCmp( com.m_nameOfFirstSplit + 2, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 0;
                    }
                }

                break;
            }

        case 'S': // split list
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, config);

                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor  )
                    {
                        if( collectors[i].m_depth == 0 )
                        {
                            collectors[i].m_node.setTimers( timerFactor, collectors[i].m_startCommand, com );
                            collectors[i].m_startCommand.m_commandAndMonitor  = HK_NULL;
                        }
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 0;
                    }
                }
                break;
            }

        case 'l': // list end
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, config);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        if( collectors[i].m_depth <= 1 )
                        {
                            collectors[i].m_node.setTimers( timerFactor, collectors[i].m_startCommand, com );
                            collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                        }
                        collectors[i].m_depth -= 2;
                    }
                }

                break;
            }

        case 'M':   //HK_MONITOR_COMMAND_ADD_VALUE
            {
                hkMonitorStream::AddValueCommand com;
                com.read(&currentStreamPtr, config);
                for( int i=0; i<numTimers; ++i )
                {
                    if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_node.m_type = hkMonitorStreamParser::Node::NODE_TYPE_SINGLE;
                        collectors[i].m_node.m_value = com.m_value;
                        collectors[i].m_node.m_count += 1;
                    }
                }

                break;
            }
        case 'A':
            {
                hkMonitorStream::AddStructCommand com;
                /*void* data = */com.read(&currentStreamPtr, config);
                break;
            }
        case 'H':
            {
                hkMonitorStream::AddGpuHandleCommand com;
                com.read(&currentStreamPtr, config);
                break;
            }
        case 'P': //Push Dir
        case 'p':
        case 'N': // nop
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);
                break;
            }

        case 'm': // memory
            {
                hkMonitorStream::MemoryCommand com;
                com.read(&currentStreamPtr, config);
                break;
            }

        default:
            HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
            return;
        }
    }

    // Copy values out
    for( int i=0; i<numTimers; ++i )
    {
        valuesOut[i] = collectors[i].m_node.m_value;
    }
}

namespace
{
    static inline _Ret_maybenull_z_ const char* _nameMatch(_In_z_ const char* monitorName, hkArrayView<const char*> timerNames)
    {
        if (!monitorName) return HK_NULL;
        for (int t=0; t < timerNames.getSize(); ++t)
        {
            if (hkString::strCmp( timerNames[t], monitorName ) == 0)
            {
                return timerNames[t];
            }
        }
        return HK_NULL;
    }
}

void HK_CALL hkMonitorStreamParser::gatherRawGpuIds( const hkMonitorStream::CommandStreamConfig& config,
                                            const hkTimerData& timerData, hkArrayView<const char*> timerNames, hkArray<TimerNameMap>& map )
{
    int numTimers = timerNames.getSize();
    if (numTimers <= 0 )
        return;

    // Walk the stream
    const char* currentStreamPtr = timerData.m_streamBegin;

    hkArray<const char*> matchedNames;
    matchedNames.reserve(20);

    while( currentStreamPtr < timerData.m_streamEnd )
    {

        hkUint32 rawValue = hkMonitorStream::readCommandUInt32(&currentStreamPtr, config);
        // Skip commands used to mark the start of an elf
        if ( rawValue <= HK_MAX_ELF_ID)
        {
            //int elfId = rawValue;
            continue;
        }
        currentStreamPtr -= 4;

        const char* currentBeforeString = currentStreamPtr;
        const char* string = hkMonitorStream::readCommandString(&currentStreamPtr, config);
        const char* monitorName = string + 2;
        currentStreamPtr = currentBeforeString; // the cmd read funcs all do the string bit again. Not great.

        switch( string[0] )
        {

        case 'F':
            {
                hkMonitorStream::FrameIdCommand com; com.read(&currentStreamPtr, config);
                break;
            }

        case 'X':
            {
                hkMonitorStream::TagCommand com; com.read(&currentStreamPtr, config);
                break;
            }

        case 'G':   // timer with tag
            {
                hkMonitorStream::TagCommand com;
                com.read(&currentStreamPtr, config);
                break;
            }

        case 'R': // timer begin with self
        case 'T': // timer begin
            {
                if (string[1] == 't')
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, config);

                    matchedNames.pushBack( _nameMatch( monitorName, timerNames ) );
                }
                else // gpu
                {
                    HK_ASSERT_NO_MSG(0x78fb9281, string[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, config);
                    // name of this drawcall is normally just 'DrawCall' so not much use to us, so we use current in scope one above

                    if ( matchedNames.back() )
                    {
                        TimerNameMap& tnm = map.expandOne();
                        tnm.id = com.m_gpuHandle;
                        tnm.name = matchedNames.back();
                    }
                }
                break;
            }

        case 'E': // timer end
            {
                if (string[1] == 't')
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, config);

                    matchedNames.popBack();
                }
                else // Gpu
                {
                    HK_ASSERT_NO_MSG(0x1e7de8e5, string[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, config);

                    // should have got from start
                }


                break;
            }

        case 'Y': // multi timer begin
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);

                matchedNames.pushBack( _nameMatch( monitorName, timerNames ) );

                break;
            }

        case 'W': // multi timer end
            {
                hkMonitorStream::MultiTimerCommand com;
                com.read(&currentStreamPtr, config);

                matchedNames.popBack();

                break;
            }

        case 'O': // object name
            {
                hkMonitorStream::TimerBeginObjectNameCommand com;
                const char* name = com.read(&currentStreamPtr, config);

                matchedNames.pushBack( _nameMatch( name, timerNames) );

                break;
            }

        case 'L': // timer list begin
            {
                hkMonitorStream::TimerBeginListCommand com;
                com.read(&currentStreamPtr, config);

                matchedNames.pushBack( _nameMatch( com.m_nameOfFirstSplit, timerNames) );

                break;
            }

        case 'S': // split list
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, config);

                matchedNames.popBack();
                matchedNames.pushBack( _nameMatch( monitorName, timerNames) );

                break;
            }

        case 'l': // list end
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, config);

                matchedNames.popBack();

                break;
            }

        case 'M':   //HK_MONITOR_COMMAND_ADD_VALUE
            {
                hkMonitorStream::AddValueCommand com;
                com.read(&currentStreamPtr, config);

                break;
            }
        case 'A':
            {
                hkMonitorStream::AddStructCommand com;
                /*void* data = */com.read(&currentStreamPtr, config);
                break;
            }
        case 'H':
            {
                hkMonitorStream::AddGpuHandleCommand com;
                com.read(&currentStreamPtr, config);

                if ( matchedNames.back() )
                {
                    TimerNameMap& tnm = map.expandOne();
                    tnm.id = com.m_gpuHandle;
                    tnm.name = matchedNames.back();
                }

                break;
            }
        case 'P': //Push Dir
        case 'p':
        case 'N': // nop
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, config);
                break;
            }

        case 'm': // memory
            {
                hkMonitorStream::MemoryCommand com;
                com.read(&currentStreamPtr, config);
                break;
            }

        default:
            HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
            return;
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
