// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamCache.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamParser.h>

hkMonitorStreamCache::hkMonitorStreamCache(const hkMonitorStream::CommandStreamConfig& config )
: m_back(0)
{
    m_config = config;
}

void hkMonitorStreamCache::clearAndSetConfig( const hkMonitorStream::CommandStreamConfig& config )
{
    m_config = config;
    for (int f = 0; f < m_frameData.getSize(); ++f)
    {
        m_frameData[f].clear();
    }
    m_back = 0;
}

void hkMonitorStreamCache::resize( int numFrames, int bytesPerFrame )
{
    if ( (m_frameData.getSize() == numFrames) && ( (numFrames==0) || (m_frameData[0].m_data.getSize() == bytesPerFrame)) )
        return; // already that size, no need to resize (and in doing so wipe contents)

    m_frameData.setSize(numFrames);
    for (int f=0; f< m_frameData.getSize(); ++f)
    {
        m_frameData[f].clear();
        m_frameData[f].m_data.setSize(bytesPerFrame);
    }
    m_back = 0;
}

void hkMonitorStreamCache::getSize( int& numFrames, int& bytesPerFrame ) const
{
    numFrames = m_frameData.getSize();
    if (numFrames > 0)
    {
        bytesPerFrame = m_frameData[0].m_data.getSize();
    }
    else
    {
        bytesPerFrame = 0;
    }
}

float hkMonitorStreamCache::getUsage() const
{
    int usedBytes = 0;
    int numFrames = 0;
    int bytesPerFrame = 0;
    getSize( numFrames, bytesPerFrame );

    int totalBytes = numFrames * bytesPerFrame;
    for (int f=0; f < numFrames; ++f)
    {
        usedBytes += m_frameData[f].m_usedDataSize;
    }

    if ( (usedBytes > 0) && (totalBytes > 0) )
    {
        return float( double(usedBytes) / double(totalBytes) );
    }
    return 0;
}

bool hkMonitorStreamCache::captureCpuTrace( const hkArrayView<hkTimerData>& timers )
{
    if (m_frameData.getSize() <= (int)m_back )
    {
        HK_WARN(0x3a3c2fa7, "Trying to capture a frame on an hkMonitorStreamCache that has no allocation to store it");
        return false;
    }

    m_frameData[m_back].clear();

    // fit in as many threads as we can
    hkUlong availableStorage = m_frameData[m_back].m_data.getSize();
    for (int t=0; t < timers.getSize(); ++t)
    {
        hkUlong neededSize = (hkUlong)( timers[t].m_streamEnd - timers[t].m_streamBegin );
        if (availableStorage >= neededSize)
        {
            hkTimerData& td = m_frameData[m_back].m_timerData.expandOne();
            const char* start = m_frameData[m_back].m_data.begin() + m_frameData[m_back].m_usedDataSize;
            td.m_streamBegin = start;
            td.m_streamEnd = start + neededSize;
            if (neededSize)
            {
                hkString::memCpy( (void*)start, timers[t].m_streamBegin, (int)neededSize);
            }
            m_frameData[m_back].m_usedDataSize += (int)neededSize;
            availableStorage -= neededSize;

            const char* strStart = start;
            const char* firstString = neededSize ? hkMonitorStream::readCommandString(&strStart, m_config) : HK_NULL;
            if ( firstString && (firstString[0] == 'F') )
            {
                hkMonitorStream::FrameIdCommand com; com.read(&start, m_config);
                m_frameData[m_back].m_timerFrameIds.pushBack( com.m_id );
            }
            else
            {
                m_frameData[m_back].m_timerFrameIds.pushBack( 0 );
            }
        }
        else
        {
            // not enough space. we could store partial but will get corrupt end timers etc.
            HK_WARN_ALWAYS(0xabba379c, "Had to ignore thread timer data as hkMonitorStreamCache size not big enough");
            m_frameData[m_back].m_complete = false;
            break;
        }
    }

    bool ok = (m_frameData[m_back].m_usedDataSize > 0);
    m_back = (m_back + 1) % m_frameData.getSize();
    return ok;
}

int hkMonitorStreamCache::captureGpuTrace( const hkArrayView<hkGpuTraceResult>& trace)
{
    if (trace.getSize() == 0)
    {
        return -1;
    }

    if (m_frameData.getSize() <= (int)m_back )
    {
        HK_WARN(0x64c0a4d7, "Trying to capture a frame on an hkMonitorStreamCache that has no allocation to store it");
        return -1;
    }


    // find the cpu frame for this gpu trace, usually a few frames back

    int lastFilledFrame = getIndexFromFrame(0);
    if ( (m_frameData[lastFilledFrame].m_gpuTrace.getSize() != 0) || ( m_frameData[lastFilledFrame].m_usedDataSize == 0 ) )
    {
        //  Don't warn as happens a lot when running view but only grabbing a few of the cpu/gpu frames
        //  HK_WARN_ALWAYS(0xabba8bd3, "Captured more GPU trace the CPU trace.. will start to loose data.");
        return -1;
    }

    // the gpu trace can be from any cpu thread really   as far as  we know here, but
    // we just want to find the right frame (so set of thread captures)
    // as such searching on the first trace should be enough
    hkUint32 frameIdLower = trace[0].m_threadId; // uint16 ->32

    int framesBack = 0;
    int frame = lastFilledFrame;
    bool foundFrame = false;
    while ( !foundFrame && (framesBack < m_frameData.getSize()) )
    {
        for (int fi=0; fi < m_frameData[frame].m_timerFrameIds.getSize(); ++fi)
        {
            const hkUint32& id = m_frameData[frame].m_timerFrameIds[fi];
            if ( (id & 0x0000FFFF) == frameIdLower )
            {
                foundFrame = true;
                break;
            }
        }

        if (!foundFrame)
        {
            ++framesBack;
            frame = (frame + m_frameData.getSize() - 1) % m_frameData.getSize();
        }
    }

    if (foundFrame)
    {
        m_frameData[frame].m_gpuTrace.setSize(trace.getSize());
        hkString::memCpy(m_frameData[frame].m_gpuTrace.begin(), trace.begin(), trace.getSize()*sizeof(hkGpuTraceResult) );
        // take a clone of any meta data var in there
        for (int t = 0; t < m_frameData[frame].m_gpuTrace.getSize(); ++t)
        {
            if (m_frameData[frame].m_gpuTrace[t].m_meta)
            {
                m_frameData[frame].m_gpuTrace[t].m_meta->addReference(); // we would have missed that in the memcpy
            }
        }
    }
    else
    {
    //  Don't warn as happens a lot when running view but only grabbning a few of the cpu/gpu frames
    //  HK_WARN_ALWAYS(0xabba3e4a, "Captured  GPU trace with no CPU trace.. will start to loose data.");
    }
    return foundFrame ? framesBack : -1;
}


int hkMonitorStreamCache::getNumCapturedFrames() const
{
    int count = 0;
    for (int index = 0; index < m_frameData.getSize(); ++index)
    {
        if (m_frameData[index].m_usedDataSize > 0)
        {
            ++count;
        }
    }
    return count;
}

int hkMonitorStreamCache::getNumPendingGpuCaptures() const
{
    int startFrame = (m_back + m_frameData.getSize() - 1 ) % m_frameData.getSize();
    int frame = startFrame;
    int ret = 0;
    while ( (m_frameData[frame].m_usedDataSize > 0) && (m_frameData[frame].m_gpuTrace.getSize() == 0) )
    {
        ret++;
        frame = (frame + m_frameData.getSize() - 1) % m_frameData.getSize();

        if ( frame == startFrame )
            break;
    }
    return ret;
}


hkArrayView<hkTimerData> hkMonitorStreamCache::getCpuTimers( int numFramesBack, bool& isComplete ) const
{
    isComplete = true;
    if ( (numFramesBack < 0) || (numFramesBack >= m_frameData.getSize()) )
    {
        return hkArrayView<hkTimerData>(); // no longer here or had no room for it at the time
    }

    int frameOfInterest = (m_back + m_frameData.getSize() - (numFramesBack + 1) ) % m_frameData.getSize() ; // the one just before m_back is newest
    HK_ASSERT_NO_MSG(0x39bcb602, (frameOfInterest >= 0) && (m_frameData.getSize() > frameOfInterest));
    if (m_frameData[frameOfInterest].m_usedDataSize == 0 )
    {
        return hkArrayView<hkTimerData>(); // no longer here or had no room for it at the time
    }

    isComplete = m_frameData[frameOfInterest].m_complete; // did we manage to store the whole dump?

    return m_frameData[frameOfInterest].m_timerData;
}

hkArrayView<hkGpuTraceResult> hkMonitorStreamCache::getGpuTimers( int numFramesBack ) const
{
    if ( (numFramesBack < 0) || (numFramesBack >= m_frameData.getSize()) )
    {
        return hkArrayView<hkGpuTraceResult>(); // no longer here or had no room for it at the time
    }

    int frameOfInterest = getIndexFromFrame(numFramesBack);
    HK_ASSERT_NO_MSG(0x2127bc6d, (frameOfInterest >= 0) && (m_frameData.getSize() > frameOfInterest));

    return m_frameData[frameOfInterest].m_gpuTrace; // may be empty
}

int hkMonitorStreamCache::getNumThreads(int numFramesBack) const
{
    bool complete = true;
    return getCpuTimers(numFramesBack, complete).getSize();
}

int hkMonitorStreamCache::getMaxNumThreads() const
{
    int maxThreads= 0;
    for (int index = 0; index < m_frameData.getSize(); ++index)
    {
        if (m_frameData[index].m_usedDataSize > 0)
        {
            maxThreads = hkMath::max2( maxThreads, m_frameData[index].m_timerData.getSize() );
        }
    }
    return maxThreads;
}

bool hkMonitorStreamCache::save(int numFramesBack, int numFramesToSave, hkMonitorStreamContainer& c) const
{
    if ( (numFramesBack < 0) || (numFramesBack >= m_frameData.getSize()) )
    {
        return false;
    }

    int maxFrames = getNumCapturedFrames() - numFramesBack;
    if (numFramesToSave > maxFrames )
    {
        numFramesToSave = maxFrames;
    }

    if (numFramesToSave < 1)
    {
        return false;
    }

    c.m_littleEndian = ( HK_ENDIAN_LITTLE && !m_config.m_needEndianSwap) || ( !HK_ENDIAN_LITTLE && m_config.m_needEndianSwap);
    c.m_pointersAre64Bit = m_config.m_pointersAre64Bit;
    c.m_timersAre64Bit = m_config.m_timersAre64Bit;
    c.m_timerFactor = hkMonitorStream::getLocalTimerFactor();


    int totalCpuSize = 0;
    int totalGpuSize = 0;
    int totalCpuThreads = 0;
    for (int f=0; f < numFramesToSave; ++f)
    {
        int idx = getIndexFromFrame(f + numFramesBack);
        const FrameData& fd = m_frameData[idx];
        totalCpuSize += fd.m_usedDataSize;
        totalGpuSize += fd.m_gpuTrace.getSize();
        if (fd.m_usedDataSize > 0)
        {
            totalCpuThreads += fd.m_timerData.getSize();
        }

        // if we have string map, keep it, otherwise make it
        hkMonitorStreamParserUtil::extractMapsFromStreams( m_config, fd.m_timerData, c.m_stringMap, c.m_typeMap );
    }

    c.m_trace.setSize( totalCpuSize );
    c.m_gpuTrace.setSize( totalGpuSize );
    c.m_threadStartOffsets.reserve( totalCpuThreads );
    c.m_frameStartOffsets.reserve( numFramesToSave );
    c.m_gpuFrameStartOffsets.reserve( numFramesToSave );

    hkUlong currentCpuFrameStartOffset = 0;
    hkUlong currentGpuFrameStartOffset = 0;
    for (int fi=numFramesToSave-1; fi >= 0; --fi)
    {
        int idx = getIndexFromFrame(fi + numFramesBack);
        const FrameData& fd = m_frameData[idx];
        c.m_frameStartOffsets.pushBack(c.m_threadStartOffsets.getSize());
        c.m_gpuFrameStartOffsets.pushBack( (hkUint32)currentGpuFrameStartOffset );

        if (fd.m_usedDataSize)
        {
            for (int thr=0; thr < fd.m_timerData.getSize(); ++thr)
            {
                hkUlong offset = fd.m_timerData[thr].m_streamBegin - fd.m_timerData[0].m_streamBegin;
                c.m_threadStartOffsets.pushBack( (hkUint32)(offset + currentCpuFrameStartOffset) );
            }
            hkString::memCpy( c.m_trace.begin() + currentCpuFrameStartOffset, fd.m_data.begin(), fd.m_usedDataSize );
            hkString::memCpy( c.m_gpuTrace.begin() + currentGpuFrameStartOffset, fd.m_gpuTrace.begin(), fd.m_gpuTrace.getSize() * sizeof(hkGpuTraceResult) );

            // take a clone of any meta data var in there, otherwise ownership gets v messy
            for (int t = 0; t <  fd.m_gpuTrace.getSize(); ++t)
            {
                if ( fd.m_gpuTrace[t].m_meta)
                {
                    c.m_gpuTrace[int(t + currentGpuFrameStartOffset)].m_meta->addReference();
                }
            }

            currentCpuFrameStartOffset += fd.m_usedDataSize;
            currentGpuFrameStartOffset += fd.m_gpuTrace.getSize();
        }
    }
    return true;
}


void hkMonitorStreamCache::FrameData::clear()
{
    m_timerData.setSize(0);
    m_complete = true;
    m_usedDataSize = 0;
    m_gpuTrace.setSize(0);
    m_timerFrameIds.setSize(0);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
