// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamAnalyzer.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>

#include <Common/Base/Reflect/Core/hkReflectVar.h>
#include <Common/Base/Reflect/Core/hkReflectCallable.h>

#define DEBUG_LOG_IDENTIFIER "base.monitor"
#include <Common/Base/System/Log/hkLog.hxx>

//static const char* HK_CAPTURE_FRAME_STRING = "Fd";
//static const char* HK_CAPTURE_PARAMS_STRING = "Ii";

hkReal hkMonitorStreamAnalyzer::s_lastFrameTime = 16666.6f;

//#define SHOW_MAX_TIMER_VALUE

hkMonitorStreamAnalyzer::CombinedThreadSummaryOptions::CombinedThreadSummaryOptions()
        :   m_indentationToFirstTimerValue(0)
        ,   m_timerColumnWidth(16)
        ,   m_tabSpacingForTimerNames(4)
        ,   m_tabSpacingForTimerValues(2)
        ,   m_displayPartialTree(false)
        ,   m_activeNode(HK_NULL)
        ,   m_rightArrowChar('>')
        ,   m_downArrowChar('_')
        ,   m_useTabsNotSpacesForColumns(false)
        ,   m_showUnaccountedForTimes(false)
        ,   m_maxDepth(HK_INT32_MAX) // no max depth by default
{
}

//#define HK_MONITOR_SHOW_TOTAL_SUBTREE_CALL_COUNTS // this display "mode" helps quickly find large call counts

static void HK_CALL hkMonitorStreamAnalyzer_printGpuTiming( hkOstream& outstream, const hkArrayView<hkGpuTraceResult>& traces, hkUint64 guid );
static void HK_CALL hkMonitorStreamAnalyzer_writeNodeRec( hkOstream& outstream, _In_ const hkMonitorStreamAnalyzer::Node* node, int recDepth, float factor, hkArrayView<hkStringPtr> ignoredNames, _In_opt_ const hkMonitorStreamAnalyzer::DrawCallHandleMap* drawCallMappings, _In_opt_ const hkArrayView<hkGpuTraceResult>* traceResults );

//
// Node constructors and destructor
//

hkMonitorStreamAnalyzer::Node::Node()
: m_gpuHandle( 0xFFFFFFFFFFFFFFFFull ),
  m_parent(HK_NULL), m_name( HK_NULL ), m_userFlags(0), m_type( NODE_TYPE_TIMER )
{
    m_threadData.setSize( 1 );
    m_moveToTag = 0;
    m_absoluteStartTime = 0;
    m_flags.clear();
    m_tags = 0;
    m_gpuHandle = 0;
}

hkMonitorStreamAnalyzer::Node::Node(_In_opt_ Node* parent, _In_z_ const char* name, NodeType type)
: m_gpuHandle( 0xFFFFFFFFFFFFFFFFull ),
  m_parent(parent), m_name( name ), m_userFlags(0), m_type( type )
{
    if (parent)
    {
        parent->m_children.pushBack(this);
    }

    m_threadData.setSize( 1 );
    m_moveToTag = 0;
    m_absoluteStartTime = 0;
    m_flags.clear();
    m_tags = 0;
    m_gpuHandle = 0;
}

hkMonitorStreamAnalyzer::Node::Node(const Node& rhs)
    : m_absoluteStartTime(rhs.m_absoluteStartTime)
    , m_parent(rhs.m_parent)
    , m_name(rhs.m_name)
    , m_userFlags(rhs.m_userFlags)
    , m_type(rhs.m_type)
{
    m_threadData.append( rhs.m_threadData );
    m_children = rhs.m_children;
    m_moveToTag = rhs.m_moveToTag;
    m_tags = 0;
    m_gpuHandle = rhs.m_gpuHandle;
    m_metaData = rhs.m_metaData;
}

hkMonitorStreamAnalyzer::Node::~Node()
{
    for (int i=0; i< m_children.getSize();i++)
    {
        delete m_children[i];
    }
    m_children.clear();
    if ( m_tags )
    {
        delete m_tags;
    }
}

char* hkMonitorStreamAnalyzer::getStreamBegin()
{
    // If there have been no calls to HK_TIMER_BEGIN, the size of m_data will
    // be 0. Calling m_data[0] would cause an assert in such a case.
    if (m_data.getSize() == 0)
    {
        return HK_NULL;
    }
    else
    {
        return &m_data[0];
    }
}



//
// Monitor stream analyzer constructor and destructor
//
hkMonitorStreamAnalyzer::hkMonitorStreamAnalyzer( int memorySize, int numThreads )
{
    s_lastFrameTime = 16666.6f;

    m_data.reserve( memorySize );
    m_nodeIdForFrameOverview = "Physics 2012";
    resetNumThreads( numThreads );
}

void hkMonitorStreamAnalyzer::resetNumThreads( int numThreads )
{
    m_frameInfos._setSize( hkContainerHeapAllocator().get(&m_frameInfos), numThreads );
    m_numThreads = numThreads;
    reset();
    HK_ASSERT(0x9876fa45, numThreads > 0 && numThreads <= 8, "Num threads out of bounds" );
}

void hkMonitorStreamAnalyzer::reset()
{
    m_data.clear();

    for (int i = 0; i < m_frameInfos.getSize(); ++i)
    {
        m_frameInfos[i].clear();
    }
}

void hkMonitorStreamAnalyzer::resetFrames( int frameCount )
{
    if (frameCount == 0)
        return;

    int size = 0;
    int lastStreamEnd = 0;

    // Accumulate memory for N frames and shift remaining data to the beginning of the arrays.
    for (int threadId = 0; threadId < m_frameInfos.getSize(); ++threadId)
    {
        const int count = hkMath::min2(frameCount, m_frameInfos[threadId].getSize());
        if (count == 0)
            continue;

        for (int i=0; i<count; ++i)
        {
            hkMonitorStreamFrameInfo& currentFrameInfo = m_frameInfos[threadId][i];

            // Note, also accumulate possible padding added in captureFrameDetails.
            size += ( currentFrameInfo.m_frameStreamEnd - lastStreamEnd );

            lastStreamEnd = currentFrameInfo.m_frameStreamEnd;
        }

        m_frameInfos[threadId].removeAtAndCopy(0, count);
    }

    // The size of one complete frame is always aligned (see captureFrameDetails).
    //size = HK_NEXT_MULTIPLE_OF( HK_REAL_ALIGNMENT, size );

    // Update stream offsets depending on the size that has been removed.
    for (int threadId=0; threadId<m_frameInfos.getSize(); ++threadId)
    {
        for (int i=0; i<m_frameInfos[threadId].getSize(); ++i)
        {
            hkMonitorStreamFrameInfo& currentFrameInfo = m_frameInfos[threadId][i];
            currentFrameInfo.m_frameStreamStart -= size;
            currentFrameInfo.m_frameStreamEnd -= size;
        }
    }

    if ( size > 0 )
        m_data.removeAtAndCopy(0, size);
}


hkMonitorStreamFrameInfo::hkMonitorStreamFrameInfo()
:   m_heading( "Unknown Heading" ),
    m_indexOfTimer0(0),
    m_indexOfTimer1(1),
    m_absoluteTimeCounter(ABSOLUTE_TIME_TIMER_0),
    m_timerFactor0(1.0f),
    m_timerFactor1(1.0f),
    m_threadId(0),
    m_frameStreamStart(0),
    m_frameStreamEnd(0)
{
}

union b4Uint32
{
    hkUint8 b[4];
    hkUint32 d;
};

union b4Uint64
{
    hkUint8 b[8];
    hkUint64 d;
};

union b4Float
{
    hkUint8 b[4];
    hkReal d;
};

inline void _byteSwapUint64( hkUint64& v )
{
    union b4Uint64 dataIn, dataOut;
    dataIn.d = v;
    dataOut.b[0] = dataIn.b[7];
    dataOut.b[1] = dataIn.b[6];
    dataOut.b[2] = dataIn.b[5];
    dataOut.b[3] = dataIn.b[4];
    dataOut.b[4] = dataIn.b[3];
    dataOut.b[5] = dataIn.b[2];
    dataOut.b[6] = dataIn.b[1];
    dataOut.b[7] = dataIn.b[0];
    v = dataOut.d;
}

inline void _byteSwapUint32( hkUint32& v )
{
    union b4Uint32 dataIn, dataOut;
    dataIn.d = v;
    dataOut.b[0] = dataIn.b[3];
    dataOut.b[1] = dataIn.b[2];
    dataOut.b[2] = dataIn.b[1];
    dataOut.b[3] = dataIn.b[0];
    v = dataOut.d;
}

template< typename T >
inline void _byteSwapTimerValue( T& v )
{
    _byteSwapUint32(v);
}

template<>
inline void _byteSwapTimerValue<hkUint64> ( hkUint64& v )
{
    _byteSwapUint64(v);
}

inline void _byteSwapFloat( float& v )
{
    union b4Float dataIn, dataOut;
    dataIn.d = v;
    dataOut.b[0] = dataIn.b[3];
    dataOut.b[1] = dataIn.b[2];
    dataOut.b[2] = dataIn.b[1];
    dataOut.b[3] = dataIn.b[0];
    v = float(dataOut.d);
}

void hkMonitorStreamAnalyzer::extractStringMap(_In_reads_to_ptr_(frameEnd) const char* frameStart, _In_ const char* frameEnd, hkPointerMap<const void*, const char*>& map)
{
    hkMonitorStream::CommandStreamConfig streamConfig;
    const char* current = frameStart;
    const char* end = frameEnd;

    while ( current < end )
    {
        const hkMonitorStream::Command* command = reinterpret_cast<const hkMonitorStream::Command*>( current );

        if ( hkUlong(command->m_commandAndMonitor) <= HK_MAX_ELF_ID )
        {
            int* elfId = (int*)command;
            current = (char*)(elfId + 1);
            continue;
        }

        // this assumes the stream was generated on the same architecture than the one calling extractStringMap
        char* str = *(char**)current;
        map.insert(str, str);

        switch( str[0] )
        {
        case 'F':
            {
                hkMonitorStream::FrameIdCommand com;
                /*const char* name =*/ com.read(&current, streamConfig);
                break;
            }

        case 'S':       // split list
        case 'E':       // timer end
        case 'l':       // list end
        case 'T':       // timer begin
        case 'R':       // timer begin with self
            {
                if ( command->m_commandAndMonitor[1] == 't' )
                {
                hkMonitorStream::TimerCommand com;
                com.read(&current, streamConfig);
                }
                else
                {
                    HK_ASSERT_NO_MSG(0x53874ee2, command->m_commandAndMonitor[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&current, streamConfig);
                }

                break;
            }
        case 'O': // object name
            {
                hkMonitorStream::TimerBeginObjectNameCommand com;
                /*const char* name =*/ com.read(&current, streamConfig);
                break;
            }

        case 'L':       // timer list begin
            {
                hkMonitorStream::TimerBeginListCommand com;
                com.read(&current, streamConfig);

                map.insert(com.m_nameOfFirstSplit, com.m_nameOfFirstSplit );
                break;
            }

        case 'W':       // multi timer end
            {
                hkMonitorStream::MultiTimerCommand com;
                com.read(&current, streamConfig);

                break;
            }

        case 'M':
            {
                hkMonitorStream::AddValueCommand com;
                com.read(&current, streamConfig);

                break;
            }

        case 'X': // timer with tag
        case 'G': // timer with tag
            {
                hkMonitorStream::TagCommand com;
                com.read(&current, streamConfig);
                break;
            }

        case 'P':
        case 'p':
        case 'N':
        case 'Y':   // multi timer begin
            {
                hkMonitorStream::Command com;
                com.read(&current, streamConfig);

                break;
            }
        case 'A':
            {
                hkMonitorStream::AddStructCommand com;
                com.read(&current, streamConfig);
                break;
            }
        case 'H':
            {
                hkMonitorStream::AddGpuHandleCommand com;
                com.read(&current, streamConfig);
                break;
            }
        case 'Q':   // pause/unpause
            {
                hkMonitorStream::TimerCommand com;
                com.read(&current, streamConfig);
                break;
            }

            default:
            HK_ASSERT(0x5120d10a, 0, "Inconsistent Monitor capture data" );     return;
        }
    }
}

hkBool hkMonitorStreamAnalyzer::captureFrameDetails(_In_ const char* monitorStreamBegin, _In_ const char* monitorStreamEnd, const hkMonitorStreamFrameInfo& info )
{
    // make sure there is enough capacity for the actual capture data (+params +end of frame)
    int size = static_cast<int>( monitorStreamEnd - monitorStreamBegin);
    if( (m_data.getCapacity() - m_data.getSize()) < size  )
    {
        HK_WARN_ONCE(0x98efaed7, "Out of memory in the monitor stream analyzer. Timer data will not be captured");
        return false;
    }

    HK_ASSERT(0xaceca543, info.m_threadId < m_frameInfos.getSize(), "hkMonitorStreamAnalyzer not initialized to the correct number of threads." );
    // copy the frame info
    hkMonitorStreamFrameInfo& newInfo = m_frameInfos[info.m_threadId]._expandOne( hkContainerHeapAllocator().get(&m_frameInfos) );
    newInfo = info;
    newInfo.m_frameStreamStart = m_data.getSize();
    newInfo.m_frameStreamEnd =  newInfo.m_frameStreamStart + size;

    // allocate the space
    char* data = m_data.expandByUnchecked( size  );

    // copy the capture data
    hkString::memCpy( data , monitorStreamBegin, size );


    return true;
}


//
//  build tree
//

static _Ret_notnull_ hkMonitorStreamAnalyzer::Node* createNewNode(_Inout_ hkMonitorStreamAnalyzer::Tree* root, _Inout_opt_ hkMonitorStreamAnalyzer::Node* parent,
    const char* name, hkMonitorStreamAnalyzer::Node::NodeType type, bool wantNodeReuse, hkMonitorStreamAnalyzer::TagType tag )
{
    // see if we have added this node already to this parent. If so reuse and augment
    // to prevent needless splitting.
    if(wantNodeReuse)
    {
        if ( name[0] == '/' )
        {
            name++;
            parent = root;
        }

        for (int c=0; parent && name && (c < parent->m_children.getSize()); ++c)
        {
            hkMonitorStreamAnalyzer::Node* child = parent->m_children[c];
            if ( (tag == child->m_moveToTag) &&  child->m_name && (hkString::strCmp(child->m_name, name) == 0) )
            {
                return child;
            }
        }
    }

    hkMonitorStreamAnalyzer::Node* node = new hkMonitorStreamAnalyzer::Node( parent, name, type );
    if ( tag )
    {
        node->m_moveToTag = tag;
        root->m_nodesWhichNeedToBeMovedToATag.insert(node, tag);
    }
    return node;
}



void hkMonitorStreamAnalyzer::Node::setTimers( const hkMonitorStreamFrameInfo& frameInfo, const hkMonitorStream::CommandStreamConfig& streamConfig, const hkMonitorStream::TimerCommand& start, const hkMonitorStream::TimerCommand& end)
{
    int id0 = frameInfo.m_indexOfTimer0;
    // note we use += here as we may be sharing a previous node.
    if ( id0 >= 0 )
    {
        hkMonitorStream::TimerValue delta = end.m_time0 - start.m_time0;
#ifdef HK_ARCH_X64
        if (!streamConfig.m_timersAre64Bit)
        {
            // Handle 32-bit timer wrap, so we still get the correct interval if start > end
            delta = delta & 0xffffffff;
        }
#endif
        m_threadData[id0].m_value += float(frameInfo.m_timerFactor0 * (delta));
        m_threadData[id0].m_count += 1;
    }
#if HK_ENABLE_EXTRA_TIMER
    int id1 = frameInfo.m_indexOfTimer1;
    if ( id1 >= 0 )
    {
        m_threadData[id1].m_value += frameInfo.m_timerFactor1 * float( end.m_time1 - start.m_time1 );
        m_threadData[id1].m_count += 1;
    }
#endif

    if (frameInfo.m_absoluteTimeCounter == hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0)
    {
        //HK_ASSERT_NO_MSG(0x70c5e11d, m_absoluteStartTime == 0);
        m_absoluteStartTime = double(frameInfo.m_timerFactor0) * double( start.m_time0 );
    }
#if HK_ENABLE_EXTRA_TIMER
    else if (frameInfo.m_absoluteTimeCounter == hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_1)
    {
        //HK_ASSERT_NO_MSG(0x1a153430, m_absoluteStartTime == 0);
        m_absoluteStartTime = double(frameInfo.m_timerFactor1) * double( start.m_time1 );
    }
#endif
}


void hkMonitorStreamAnalyzer::Node::setTimers( const hkMonitorStreamFrameInfo& frameInfo, const hkMonitorStream::MultiTimerCommand& multiTimer)
{
    int id0 = frameInfo.m_indexOfTimer0;
    // note we use += here as we may be sharing a previous node.
    if ( id0 >= 0 )
    {
        m_threadData[id0].m_value += float(frameInfo.m_timerFactor0 * ( multiTimer.m_timerCommand.m_time0 ));
        m_threadData[id0].m_count += multiTimer.m_callCount;
    }

#if HK_ENABLE_EXTRA_TIMER
    int id1 = frameInfo.m_indexOfTimer1;
    if ( id1 >= 0 )
    {
        m_threadData[id1].m_value += frameInfo.m_timerFactor1 * float( multiTimer.m_timerCommand.m_time1 );
        m_threadData[id1].m_count += multiTimer.m_callCount;
    }
#endif
}

namespace
{
    float applyDeductionsToTree_impl(hkMonitorStreamAnalyzer::Node* node, int id, hkMonitorStreamAnalyzer::TimerDeductionNodeRemovalStrategy timerDeductionNodeRemovalStrategy)
    {
        float deduction = 0;

        for(int i = node->m_children.getSize()-1; i >= 0; i--)
        {
            hkMonitorStreamAnalyzer::Node* child = node->m_children[i];

            if(child->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER_DEDUCTION)
            {
                HK_ASSERT(0x531c0a5f, child->m_children.isEmpty(), "Deduction nodes must have no children");
                deduction += child->m_threadData[id].m_value;
                if(timerDeductionNodeRemovalStrategy == hkMonitorStreamAnalyzer::REMOVE_TIMER_DEDUCTION_NODES)
                {
                    node->m_children.removeAt(i);
                }
            }
            else
            {
                deduction += applyDeductionsToTree_impl(child, id, timerDeductionNodeRemovalStrategy);
            }
        }

        if(node->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER)
        {
            node->m_threadData[id].m_value -= deduction;

            if(node->m_flags.anyIsSet(hkMonitorStreamAnalyzer::Node::FLAGS_REQUEST_SELF)
                && !node->m_children.isEmpty()
                && node->m_children[0]->m_flags.anyIsSet(hkMonitorStreamAnalyzer::Node::FLAGS_IS_SELF))
            {
                hkMonitorStreamAnalyzer::Node* self = node->m_children[0];
                self->m_threadData[id].m_value -= deduction;
            }
        }

        return deduction;
    }
}

void hkMonitorStreamAnalyzer::applyDeductionsToTree(hkMonitorStreamAnalyzer::Node* node, const hkMonitorStreamFrameInfo& frameInfo, TimerDeductionNodeRemovalStrategy timerDeductionNodeRemovalStrategy)
{
    applyDeductionsToTree_impl(node, frameInfo.m_indexOfTimer0, timerDeductionNodeRemovalStrategy);
}

void hkMonitorStreamAnalyzer::Node::createSelfTimer(const hkMonitorStreamFrameInfo& frameInfo, _Inout_ Tree* rootNode)
{
    int id0 = frameInfo.m_indexOfTimer0;
    // note we use += here as we may be sharing a previous node.
    if ( id0 >= 0 )
    {
        // sum all children
        hkReal sum = 0;
        Node* selfNode = HK_NULL;
        int i = 0;
        if ( m_children.getSize() && hkString::strCmp( m_children[0]->m_name, "self" ) == 0 )
        {
            selfNode = m_children[0];
            i = 1;
        }
        for (; i < m_children.getSize(); i++ )
        {
            sum += m_children[i]->m_threadData[id0].m_value;
        }
        hkReal self = m_threadData[id0].m_value - sum;
        if (!selfNode )
        {
            selfNode = createNewNode( rootNode, HK_NULL, "self", NODE_TYPE_TIMER, false, 0 );
            selfNode->m_parent = this;
            selfNode->m_flags.orWith(FLAGS_IS_SELF);
            m_children.insertAt(0, selfNode );
        }
        selfNode->m_threadData[id0].m_count += 1;
        selfNode->m_threadData[id0].m_value = float(self);
    }
}

void hkMonitorStreamAnalyzer::Node::getPath( hkStringBuf& pathOut ) const
{
    pathOut.clear();
    const hkMonitorStreamAnalyzer::Node* parent = m_parent;
    while ( parent )
    {
        pathOut.prepend( "/" );
        pathOut.prepend( parent->m_name );
        parent = parent->m_parent;
    }
}

struct hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer::Node::findChild( const char* pathIn )
{
    if ( ( pathIn != HK_NULL ) && ( m_children.getSize() > 0 ) )
    {
        hkStringBuf nodeDir = pathIn;
        hkStringBuf nodeSubPath;
        int recurseIdx = nodeDir.indexOf( '/' );
        if ( recurseIdx != -1 )
        {
            nodeSubPath = nodeDir;
            nodeSubPath.slice( recurseIdx + 1, nodeDir.getLength() - recurseIdx - 1 );
            nodeDir.slice( 0, recurseIdx );
        }
        for ( int i = 0; i < m_children.getSize(); i++ )
        {
            if ( Node* child = m_children[i] )
            {
                if ( nodeDir.compareToIgnoreCase( child->m_name ) == 0 )
                {
                    if ( nodeSubPath.getLength() )
                    {
                        return child->findChild( nodeSubPath );
                    }
                    else
                    {
                        return child;
                    }
                }
            }
        }
    }
    return HK_NULL;
}

static HK_INLINE hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer_findChildWithHint(hkMonitorStreamAnalyzer::Node* destParent, const hkMonitorStreamAnalyzer::Node* searchNode, int& childId )
{
    const char* name = searchNode->m_name;
    if ( childId < destParent->m_children.getSize() )
    {
        hkMonitorStreamAnalyzer::Node* child = destParent->m_children[childId];
        if( (child->m_moveToTag == searchNode->m_moveToTag) && hkString::strCmp(child->m_name, name) == 0)
        {
            return child;
        }
    }

    for (int i = 0; i < destParent->m_children.getSize(); ++i)
    {
        hkMonitorStreamAnalyzer::Node* child = destParent->m_children[i];
        if ( (child->m_moveToTag == searchNode->m_moveToTag) && hkString::strCmp(child->m_name, name) == 0)
        {
            childId = i;
            return child;
        }
    }
    return HK_NULL;
}

static HK_INLINE hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer_findChildUsingTagAndName(hkMonitorStreamAnalyzer::Node* destParent, const hkMonitorStreamAnalyzer::Node* searchNode )
{
    const char* name = searchNode->m_name;

    for (int i = 0; i < destParent->m_children.getSize(); ++i)
    {
        hkMonitorStreamAnalyzer::Node* child = destParent->m_children[i];
        if ( (child->m_moveToTag == searchNode->m_moveToTag) && hkString::strCmp(child->m_name, name) == 0)
        {
            return child;
        }
    }
    return HK_NULL;
}

static _Ret_notnull_ hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer_findOrCreateChildWithHint(
    _Inout_ hkMonitorStreamAnalyzer::Node* destParent, _In_ const hkMonitorStreamAnalyzer::Node* searchNode, int& childId, bool ignoreTags)
{
    const char* name = searchNode->m_name;
    {
        for (int i = 0; i < destParent->m_children.getSize(); ++i)
        {
            hkMonitorStreamAnalyzer::Node* child = destParent->m_children[i];
            if ( (ignoreTags || (child->m_moveToTag == searchNode->m_moveToTag)) &&
                hkString::strCmp(child->m_name, name) == 0)
            {
                return child;
            }
        }
    }

    hkMonitorStreamAnalyzer::Node* child = new hkMonitorStreamAnalyzer::Node(destParent, name, searchNode->m_type );
    child->m_threadData.setSize( searchNode->m_threadData.getSize() );
    child->m_moveToTag = ignoreTags ? 0 : searchNode->m_moveToTag;
    return child;
}

void HK_CALL hkMonitorStreamAnalyzer::scaleTimerValuesRec(_Inout_ hkMonitorStreamAnalyzer::Node* node, hkReal scale)
{
    for (int i=0; i < node->m_threadData.getSize(); i++ )
{
        node->m_threadData[i].m_value *= (float)scale;
        node->m_threadData[i].m_count = 0;
    }
    for ( int i = 0; i < node->m_children.getSize(); ++i)
    {
        scaleTimerValuesRec( node->m_children[i], scale );
    }
}

static hkMonitorStreamAnalyzer::Node* HK_CALL hkMonitorStreamAnalyzer_findChildByName(
    _Inout_ hkMonitorStreamAnalyzer::Node* node, _In_z_ const char* childName, bool searchAnyChild = false )
{
    // search child
    for( int j = 0; j < node->m_children.getSize(); j++ )
    {
        if( hkString::strCmp( childName, node->m_children[j]->m_name ) == 0 )
        {
            return node->m_children[j];
        }
    }
    if ( searchAnyChild && node->m_children.getSize()>0 )
    {
        // find the most expensive
        float maxValue = 0;
        int maxIndex = 0;
        for( int j = 1; j < node->m_children.getSize(); j++ )
        {
            hkMonitorStreamAnalyzer::Node* child = node->m_children[j];
            if ( child->m_threadData[0].m_value > maxValue )
            {
                maxValue = child->m_threadData[0].m_value;
                maxIndex = j;
            }
        }
        return node->m_children[maxIndex];
    }
    return HK_NULL;
}

static HK_INLINE void HK_CALL hkMakeSum(_Inout_ hkMonitorStreamAnalyzer::Node* sum, _In_ const hkMonitorStreamAnalyzer::Node* node)
{
    if( sum->m_threadData.getSize() < node->m_threadData.getSize() )
    {
        sum->m_threadData.setSize( node->m_threadData.getSize() );
    }
    for (int j = 0; j < node->m_threadData.getSize(); j++)
    {
        sum->m_threadData[j].m_value += node->m_threadData[j].m_value;
        sum->m_threadData[j].m_count += node->m_threadData[j].m_count;
    }
}

// This duplicates the structure of tree into sum (combining children with the same parent of the same name)
static void HK_CALL hkMakeSumRecursive(_Inout_ hkMonitorStreamAnalyzer::Node* sum, _In_ const hkMonitorStreamAnalyzer::Node* tree)
{
    for( int i = 0; i < tree->m_children.getSize(); i++ )
    {
        const hkMonitorStreamAnalyzer::Node* childIn = tree->m_children[i];
        hkMonitorStreamAnalyzer::Node* childOut = hkMonitorStreamAnalyzer_findChildByName( sum, childIn->m_name, false);

        if( !childOut )
        {
            childOut = new hkMonitorStreamAnalyzer::Node( sum, childIn->m_name, childIn->m_type );
        }
        hkMakeSum( childOut, childIn );
        hkMakeSumRecursive( childOut, childIn );
    }
}

/// This functions takes all the tags from source, copies the tags in mathedNode and
/// updates the destTree->m_tagToNode with those new tags to point to the matched Node
HK_INLINE
void HK_CALL hkMonitorStreamAnalyzer_mapAndCopyTags(_In_ const hkMonitorStreamAnalyzer::Node* source, _Inout_ hkMonitorStreamAnalyzer::Node* matchedNode, _Inout_ hkMonitorStreamAnalyzer::Tree* destTree)
{
    if ( source->m_tags )
    {
        if (!matchedNode->m_tags)
        {
            matchedNode->m_tags = new hkMonitorStreamAnalyzer::TagInfo;
        }
        for (int i=0; i < source->m_tags->m_tags.getSize(); i++ )
        {
            hkMonitorStreamAnalyzer::TagType tag = source->m_tags->m_tags[i];
            matchedNode->m_tags->m_tags.pushBack( tag );
            destTree->m_tagToNode.insert( tag, matchedNode );
        }
    }
}

void HK_CALL hkMonitorStreamAnalyzer_getPath(_In_ const hkMonitorStreamAnalyzer::Node* node, hkStringBuf& path)
{
    hkInplaceArray<const hkMonitorStreamAnalyzer::Node*,32> parents;
    for ( const hkMonitorStreamAnalyzer::Node* p = node; p->m_parent; p = p->m_parent )
    {
        parents.pushBack(p);
    }
    for (int i =parents.getSize()-1; i>=0; i-- )
    {
        const hkMonitorStreamAnalyzer::Node* p = parents[i];
        path.append( p->m_name );
        if ( i>0 )
        {
            path.append("/");
        }
    }
}


// this function is used to move part of a tree to another place,
// since there might be already data in the target place, this function merges one tree into another
void HK_CALL hkMonitorStreamAnalyzer_mergeSubTree(_Inout_ hkMonitorStreamAnalyzer::Tree* tree, _Inout_ hkMonitorStreamAnalyzer::Node* dest, _Inout_ hkMonitorStreamAnalyzer::Node* source)
{

    hkMonitorStreamAnalyzer::Node* matchedNode = hkMonitorStreamAnalyzer_findChildUsingTagAndName( dest, source );
    if ( matchedNode == source )
    {
        return; // found myself, nothing to do
    }

    hkMonitorStreamAnalyzer::Node* oldParent = source->m_parent;
    oldParent->m_children.removeAllAndCopy(source);
    source->m_parent = HK_NULL;

    if ( !matchedNode )
    {
        // reuse node
        // just move my child including all children
        dest->m_children.pushBack(source);
        source->m_parent = dest;
    }
    else
    {
        hkMakeSum( matchedNode, source );

        while ( source->m_children.getSize() )
        {
            hkMonitorStreamAnalyzer::Node* subTreeNode = source->m_children[0];
            hkMonitorStreamAnalyzer_mergeSubTree( tree, matchedNode, subTreeNode );
        }

        // move tags
        hkMonitorStreamAnalyzer_mapAndCopyTags(source, matchedNode, tree);

        // move tagged nodes
        hkMonitorStreamAnalyzer::TagType tag = source->m_moveToTag;
        if ( tag )
        {
            tree->m_nodesWhichNeedToBeMovedToATag.insert( source, 0 );
        }
        // no longer needed
        delete source;
    }
}

void hkMonitorStreamAnalyzer_findTagsRec(_Inout_ hkMonitorStreamAnalyzer::Node* node, hkArray<hkMonitorStreamAnalyzer::Node*>& nodes)
{
    if ( node->m_moveToTag )
    {
        nodes.pushBack(node);
    }
    for (int i =0; i < node->m_children.getSize(); i++ )
    {
        hkMonitorStreamAnalyzer_findTagsRec( node->m_children[i], nodes );
    }
}

void hkMonitorStreamAnalyzer::fixupTags(_Inout_ Tree* tree )
{
    bool warnMissingTags = false;

    const Tree::TaggedNodesMap& tn = tree->m_nodesWhichNeedToBeMovedToATag;

    hkInplaceArray<hkMonitorStreamAnalyzer::Node*,256> nodesToMove;
    hkMonitorStreamAnalyzer_findTagsRec( tree, nodesToMove );
    for (int i = 0; i < nodesToMove.getSize(); i++ )
    {
        Node* node = nodesToMove[i];
        TagType tag = tn.getWithDefault( node, 0 );
        if ( tag == 0 )
        {
            continue; // already resolved;
        }
        Node* newParent = tree->m_tagToNode.getWithDefault( tag, HK_NULL );
        if ( newParent )
        {
            if ( node->m_parent != newParent)
            {
                // Propagate the timer values up to the new parent
                Node* propagateTo = newParent;
                while(propagateTo)
                {
                    if( propagateTo->m_threadData.getSize() < node->m_threadData.getSize() )
                    {
                        propagateTo->m_threadData.setSize( node->m_threadData.getSize() );
                    }
                    for(int v = 0; v < node->m_threadData.getSize(); v++)
                    {
                        propagateTo->m_threadData[v].m_value += node->m_threadData[v].m_value;
                    }
                    propagateTo = propagateTo->m_parent;
                }

                hkMonitorStreamAnalyzer_mergeSubTree( tree, newParent, node );
            }
        }
    }

    // do it again, this time removing all tags
    for (int i = 0; i < nodesToMove.getSize(); i++ )
    {
        Node* node = nodesToMove[i];
        TagType tag = tn.getWithDefault( node, 0 );
        if ( tag == 0 )
        {
            continue; // already resolved;
        }
        Node* newParent = tree->m_tagToNode.getWithDefault( tag, HK_NULL );
        if ( !newParent )
        {
            warnMissingTags = true;
        }

        HK_ASSERT_NO_MSG ( 0xf046fdfd, !newParent || node->m_parent == newParent);
        node->m_moveToTag = 0;
        hkMonitorStreamAnalyzer_mergeSubTree( tree, node->m_parent, node );
    }

    // If there were any nodes that should be moved to a tag but the tag couldn't be found, add a warning to the stats
    if ( warnMissingTags )
    {
        const char* missingTagsWarningText = "Warning! Tags were not found, some timers may be missing.";

        // Check if the warning already exists in the tree
        bool found = false;
        for ( int i = 0; i < tree->m_children.getSize(); ++i )
        {
            hkMonitorStreamAnalyzer::Node* child = tree->m_children[i];
            if ( hkString::strCmp( child->m_name, missingTagsWarningText ) == 0 )
            {
                child->m_threadData[0].m_count = 1;
                found = true;
                break;
            }
        }

        // If not, add it
        if (!found)
        {
            Node* warningNode = new Node();
            warningNode->m_name = missingTagsWarningText;
            warningNode->m_threadData[0].m_count = 1;
            tree->m_children.insertAt( 0, &warningNode, 1 );
        }
    }
}

void hkMonitorStreamAnalyzer::removeTagsFromNodeRec(_Inout_ Node* node)
{
    if(node->m_tags)
    {
        node->m_tags->m_tags.clear();
    }

    for(int i = 0; i < node->m_children.getSize(); i++)
    {
        removeTagsFromNodeRec(node->m_children[i]);
    }
}

void hkMonitorStreamAnalyzer::Tree::clearTags()
{
    removeTagsFromNodeRec(this);
    m_nodesWhichNeedToBeMovedToATag.clear();
    m_tagToNode.clear();
}

// merge a tree produced by a single thread into another tree.
static void HK_CALL hkMonitorStreamAnalyzer_mergeThreadTree(
    _Inout_ hkMonitorStreamAnalyzer::Tree* destTree, _Inout_ hkMonitorStreamAnalyzer::Node* destNode,
    _In_ const hkMonitorStreamAnalyzer::Tree* sourceRoot, _In_ const hkMonitorStreamAnalyzer::Node* sourceNode, int destTimerId, int sourceTimerId)
{
    typedef hkMonitorStreamAnalyzer::Node Node;

    int childId = 0; // Also use this as a hint to hasChild - usually the children will be in the same order.

    // copy tags
    hkMonitorStreamAnalyzer_mapAndCopyTags(sourceNode, destNode, destTree);

    for ( int i = 0; i < sourceNode->m_children.getSize(); ++i )
    {
        const Node* subTreeNode = sourceNode->m_children[i];
        Node* matchedNode = hkMonitorStreamAnalyzer_findOrCreateChildWithHint( destNode, subTreeNode, childId, false );

        // copy tagged nodes
        hkMonitorStreamAnalyzer::TagType tag = subTreeNode->m_moveToTag;
        if ( tag )
        {
            matchedNode->m_moveToTag = tag;
            destTree->m_nodesWhichNeedToBeMovedToATag.insert( matchedNode, tag );
        }

        if( matchedNode->m_threadData.getSize() <= destTimerId )
        {
            matchedNode->m_threadData.setSize(destTimerId+1);
        }

        matchedNode->m_threadData[destTimerId] = subTreeNode->m_threadData[sourceTimerId];
        hkMonitorStreamAnalyzer_mergeThreadTree(destTree, matchedNode, sourceRoot, subTreeNode, destTimerId, sourceTimerId );

        childId = (childId < sourceNode->m_children.getSize() - 1 ) ? childId+1 : childId;
    }
}


void hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary(_Inout_ hkMonitorStreamAnalyzer::Tree* mainTree, _In_ const hkMonitorStreamAnalyzer::Tree* subTree, int destTimerId, int sourceTimerId)
{
    hkMonitorStreamAnalyzer_mergeThreadTree( mainTree, mainTree, subTree, subTree, destTimerId, sourceTimerId );
}

_Ret_notnull_ hkMonitorStreamAnalyzer::Tree* HK_CALL hkMonitorStreamAnalyzer::createTreeOneFrameManyThreads(
    const hkMonitorStream::CommandStreamConfig& streamConfig, const hkMonitorStreamFrameInfo& frameInfo,
    _In_reads_(numThreads) const hkTimerData* threadStreams, int numThreads,
    TimerDeductionNodeHandling timerDeductionNodeHandling)
{
    Tree* result;
    {
        const hkTimerData& td = threadStreams[0];
        result = makeStatisticsTreeForSingleFrame( streamConfig, td.m_streamBegin, td.m_streamEnd, frameInfo );
        if(timerDeductionNodeHandling != IGNORE_TIMER_DEDUCTION_NODES)
        {
            applyDeductionsToTree(
                result,
                frameInfo,
                timerDeductionNodeHandling == APPLY_AND_DELETE_TIMER_DEDUCTION_NODES ? REMOVE_TIMER_DEDUCTION_NODES : DONT_REMOVE_TIMER_DEDUCTION_NODES);
        }

    }

    for (int i = 1; i < numThreads; i++ )
    {
        const hkTimerData& td = threadStreams[i];
        Tree* tree = makeStatisticsTreeForSingleFrame( streamConfig, td.m_streamBegin, td.m_streamEnd, frameInfo );
        if(timerDeductionNodeHandling != IGNORE_TIMER_DEDUCTION_NODES)
        {
            applyDeductionsToTree(
                tree,
                frameInfo,
                timerDeductionNodeHandling == APPLY_AND_DELETE_TIMER_DEDUCTION_NODES ? REMOVE_TIMER_DEDUCTION_NODES : DONT_REMOVE_TIMER_DEDUCTION_NODES);
        }
        hkMonitorStreamAnalyzer_mergeThreadTree( result, result, tree, tree, i, 0 );
        delete tree;
    }

    fixupTags( result );

    return result;
}

void HK_CALL hkMonitorStreamAnalyzer_blendTreeRec(_Inout_ hkMonitorStreamAnalyzer::Node* smoothedTreeInOut, _In_ const hkMonitorStreamAnalyzer::Node* frameTree, hkReal smoothingFactor)
{
    smoothedTreeInOut->m_threadData.setSize( frameTree->m_threadData.getSize() );

    int childId = 0; // Also use this as a hint to hasChild - usually the children will be in the same order.
    float f1 = 1.0f - float(smoothingFactor);
    for ( int c = 0; c < frameTree->m_children.getSize(); ++c )
    {
        const hkMonitorStreamAnalyzer::Node* subTreeNode = frameTree->m_children[c];
        hkMonitorStreamAnalyzer::Node* matchedNode = hkMonitorStreamAnalyzer_findOrCreateChildWithHint( smoothedTreeInOut, subTreeNode, childId, true );

        for( int i = 0; i < matchedNode->m_threadData.getSize() && i < subTreeNode->m_threadData.getSize(); i++ )
        {
            matchedNode->m_threadData[i].m_value += f1 * subTreeNode->m_threadData[i].m_value;
            matchedNode->m_threadData[i].m_count += subTreeNode->m_threadData[i].m_count;
        }
        hkMonitorStreamAnalyzer_blendTreeRec( matchedNode, subTreeNode, smoothingFactor );

        childId = (childId < frameTree->m_children.getSize() - 1 ) ? childId+1 : childId;
    }
}

void HK_CALL hkMonitorStreamAnalyzer::blendTree(_Inout_ Node* smoothedTreeInOut, _In_ const Node* frameTree, hkReal smoothingFactor)
{
    scaleTimerValuesRec( smoothedTreeInOut, smoothingFactor );
    hkMonitorStreamAnalyzer_blendTreeRec( smoothedTreeInOut, frameTree, smoothingFactor );

    #if 0
{
        hkOstream ostream( "trees.txt");
        hkMonitorStreamAnalyzer_writeNodeRec( ostream, frameTree, 0, 1.0f, hkArrayView<hkStringPtr>(), HK_NULL, HK_NULL  );
        ostream.flush();
    }
    #endif
}

namespace
{
    template<bool EXPAND>
    void collapseExpandMonitorsInternal( hkMonitorStreamAnalyzer::Node* nodeIn )
    {
        if ( !nodeIn )
        {
            return;
        }

        hkMonitorStreamAnalyzer::Node* parent = nodeIn;
        while ( parent->m_parent )
        {
            parent = parent->m_parent;
            if ( EXPAND )
            {
                parent->m_userFlags |= 1;
            }
            else if ( parent->m_parent )
            {
                parent->m_userFlags &= ~1;
            }
        }
    }
}

void hkMonitorStreamAnalyzer::collapseMonitors( Node* nodeIn )
{
    collapseExpandMonitorsInternal<false>( nodeIn );
}

void hkMonitorStreamAnalyzer::expandMonitors( Node* nodeIn )
{
    collapseExpandMonitorsInternal<true>( nodeIn );
}

_Ret_maybenull_ hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer::navigateMonitors(const hkMonitorStreamAnalyzer::CursorKeys& keys, _Inout_ hkMonitorStreamAnalyzer::Node* activeNodeIn)
{
    hkMonitorStreamAnalyzer::Node* activeNode = activeNodeIn;

    if ( activeNode == HK_NULL )
    {
        return HK_NULL;
    }

    if ( keys.m_upPressed )
    {
        hkMonitorStreamAnalyzer::Node* f = activeNode->m_parent->m_children[0];
        // If there is a previous child to go to, move to it
        if ( f != activeNode )
        {
            // find the previous child
            for (int i= 0; i < activeNode->m_parent->m_children.getSize(); ++i)
            {
                if ( activeNode->m_parent->m_children[i] == activeNode)
                {
                    f = activeNode->m_parent->m_children[i - 1];
                    break;
                }
            }

            // if this value is unfolded, go into it
            while (true)
            {
                if ( ( f->m_children.getSize() > 0) && f->m_userFlags & 1 )
                {
                    f = f->m_children[f->m_children.getSize() - 1];
                    continue;
                }
                break;
            }
            activeNode = f;
        }
        else
        {
            if ( activeNode->m_parent->m_parent )
            {
                activeNode = activeNode->m_parent;
            }
        }
    }

    // test for down
    if ( keys.m_downPressed )
    {
        hkMonitorStreamAnalyzer::Node* f  = activeNode;
        if ( ( f->m_children.getSize() > 0 ) && f->m_userFlags & 1 )
        {
            activeNode = f->m_children[0];
        }
        else
        {
            bool foundChild = false;
            while (!foundChild)
            {
                for (int i = 0; i < f->m_parent->m_children.getSize(); ++i)
                {
                    if ( (f->m_parent->m_children[i] == f) && i < (f->m_parent->m_children.getSize() - 1) )
                    {
                        activeNode = f->m_parent->m_children[i + 1];
                        foundChild = true;
                        break;
                    }
                }
                if ( !foundChild && f->m_parent->m_parent )
                {
                    f = f->m_parent;
                    continue;
                }
                break;
            }
        }
    }

    // test for left
    if ( keys.m_leftPressed )
    {
        if ( activeNode->m_userFlags & 1 )
        {
            activeNode->m_userFlags &= ~1;
        }
        else
        {
            if ( activeNode->m_parent->m_parent )
            {
                activeNode = activeNode->m_parent;
                activeNode->m_userFlags &= ~1;
            }
        }
    }

    // test for right
    if ( keys.m_rightPressed )
    {
        // unfold the whole subtree
        activeNode->m_userFlags |= 1;
    }

    return activeNode;
}

int hkMonitorStreamAnalizer_findMaxTimerNameIndent(_Inout_ hkMonitorStreamAnalyzer::Node* node, int recursionDepth, int spacing, bool displayPartialTree)
{
    int indent = spacing * recursionDepth + hkString::strLen(node->m_name) + 8; // Add a few characters - we don't know the (count) value which will be displayed after the timer

    if( !displayPartialTree || ( node->m_userFlags & 1 ) )
    {
        for ( int i = 0; i < node->m_children.getSize(); ++i )
        {
            int childIndent = hkMonitorStreamAnalizer_findMaxTimerNameIndent(  node->m_children[i], recursionDepth + 1, spacing, displayPartialTree );
            if ( childIndent > indent )
            {
                indent = childIndent;
            }
        }
    }
    return indent;

}

void hkMonitorStreamAnalyzer_printField( hkArray<char>& textLine, int recursionDepth, const hkMonitorStreamAnalyzer::CombinedThreadSummaryOptions& options, int column, hkReal value, int count)
{
    hkOstream lineOs( textLine );

    if (options.m_useTabsNotSpacesForColumns)
    {
        lineOs << '\t';
    }
    else
    {
        int extraSpacing = options.m_indentationToFirstTimerValue + options.m_timerColumnWidth * column + options.m_tabSpacingForTimerValues * ( recursionDepth - 1) - textLine.getSize();
        for (int j = 0; j < extraSpacing; ++j) lineOs << ' ';
    }

    lineOs.printf("%-6.3f (%i)", value, count );
}

void hkMonitorStreamAnalyzer::showCombinedThreadSummaryRec( hkOstream& os, _Inout_ hkMonitorStreamAnalyzer::Node* node, int recursionDepth, int numThreads, const CombinedThreadSummaryOptions& options )
{
    hkArray<char> textLine;
    hkOstream lineOs( textLine );

    if (recursionDepth != 0)
    {
        {
            // Show arrow for active node if only displaying part of the tree

            if (options.m_displayPartialTree)
            {
                lineOs << char((options.m_activeNode == node) ? options.m_rightArrowChar  : ' ');
            }

            int extraSpacing = options.m_tabSpacingForTimerNames * (recursionDepth - 1);
            for(int i=0; i < extraSpacing; ++i)     lineOs << ' ';
        }

        // Show expandable tree sub parts if only displaying part of the tree
        if (options.m_displayPartialTree )
        {
            if( node->m_children.getSize() > 0 )
            {
                lineOs << ((node->m_userFlags & 1) ? options.m_downArrowChar : options.m_rightArrowChar );
            }
            else
            {
                lineOs << ' ';
            }
        }

        // Print the timer names, and total count for all threads for each timer
        int count = 0;
        hkReal sum = 0.0f;
        {
            for ( int i = 0; i < node->m_threadData.getSize(); ++i )
            {
                count += node->m_threadData[i].m_count;
#if defined(SHOW_MAX_TIMER_VALUE)
                sum = hkMath::max2( sum, node->m_threadData[i].m_value );
#else
                sum += node->m_threadData[i].m_value;
#endif
            }
        }

        if(options.m_showUnaccountedForTimes && node->m_type == Node::NODE_TYPE_TIMER)
        {
            hkReal  nodeSum = node->valuesSum();
            hkReal  childrenSum = 0;
            for(int i=0; i<node->m_children.getSize(); ++i)
            {
                if(node->m_children[i]->m_type == Node::NODE_TYPE_TIMER)
                {
                    childrenSum += node->m_children[i]->valuesSum();
                }
            }
            int     unaccountedFor = 0;
            if(count && childrenSum > 0.0f)
            {
                hkReal  leftOver = nodeSum - childrenSum;
                unaccountedFor = (int) ((leftOver / nodeSum) * 100.0f + 0.5f);
            }

            // You can enable reporting of % time unaccounted for. Disabled for now.
            if(0 && (unaccountedFor >= 1))
            {
#ifdef HK_MONITOR_SHOW_TOTAL_SUBTREE_CALL_COUNTS
                lineOs.printf("%s (%i)  (~%d%%) ", node->m_name, node->m_totalCallCountForChildren, unaccountedFor );
#else
                lineOs.printf("%s (~%d%%) ", node->m_name, unaccountedFor );
#endif
            }
            else
            {
#ifdef HK_MONITOR_SHOW_TOTAL_SUBTREE_CALL_COUNTS
                lineOs.printf("%s (%i) ", node->m_name, node->m_totalCallCountForChildren );
#else
                lineOs.printf("%s ", node->m_name );
#endif
            }
        }
        else
        {
#ifdef HK_MONITOR_SHOW_TOTAL_SUBTREE_CALL_COUNTS
            lineOs.printf("%s (%i) ", node->m_name, node->m_totalCallCountForChildren_ );
#else
            lineOs.printf("%s ", node->m_name );
#endif
        }

        // Print timer values
        for ( int i = 0; i < numThreads && i < node->m_threadData.getSize(); ++i )
        {
            hkMonitorStreamAnalyzer_printField( textLine, recursionDepth, options, i, node->m_threadData[i].m_value, node->m_threadData[i].m_count );
        }
        for ( int i = node->m_threadData.getSize(); i < numThreads; ++i )
        {
            hkMonitorStreamAnalyzer_printField( textLine, recursionDepth, options, i, 0.0f, 0 );
        }

        if (numThreads > 1)
        {
            hkMonitorStreamAnalyzer_printField( textLine, recursionDepth, options, numThreads, sum, count );
        }

        os << textLine.begin();
        os << " \n";
    }

    if( (!options.m_displayPartialTree || ( node->m_userFlags & 1 )) &&
        (recursionDepth < options.m_maxDepth) )
    {
        for ( int i = 0; i < node->m_children.getSize(); ++i )
        {
            showCombinedThreadSummaryRec( os, node->m_children[i], recursionDepth + 1, numThreads, options );
        }
    }
}

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    static hkUint32 swap32( hkUint32 v )
    {
        union SwapUnion
        {
            hkUint32 val;
            hkUint8 b[4];
        };
        SwapUnion in; in.val = v;
        SwapUnion out;
        out.b[0] = in.b[3];
        out.b[1] = in.b[2];
        out.b[2] = in.b[1];
        out.b[3] = in.b[0];
        return out.val;
    }

    static float swap32F( float v )
    {
        union SwapUnion
        {
            float val;
            hkUint8 b[4];
        };
        SwapUnion in; in.val = v;
        SwapUnion out;
        out.b[0] = in.b[3];
        out.b[1] = in.b[2];
        out.b[2] = in.b[1];
        out.b[3] = in.b[0];
        return out.val;
    }


    static hkUint64 swap64( hkUint64 v )
    {
        union SwapUnion
        {
            hkUint64 val;
            hkUint8 b[8];
        };

        SwapUnion in; in.val = v;
        SwapUnion out;

        out.b[0] = in.b[7];
        out.b[1] = in.b[6];
        out.b[2] = in.b[5];
        out.b[3] = in.b[4];
        out.b[4] = in.b[3];
        out.b[5] = in.b[2];
        out.b[6] = in.b[1];
        out.b[7] = in.b[0];

        return out.val;
    }

    struct hkMonitorStreamAnalyzer_Node
    {
        hkMonitorStreamAnalyzer::Node* m_node;
        hkMonitorStreamAnalyzer::TagType m_moveToTag;
    };

    HK_INLINE void hkMonitorStreamAnalyzer_push(hkArray<hkMonitorStreamAnalyzer_Node>& stack, _In_ hkMonitorStreamAnalyzer::Node* node, hkMonitorStreamAnalyzer::TagType moveToTag)
    {
        hkMonitorStreamAnalyzer_Node& n = stack.expandOne();
        n.m_node = node;
        n.m_moveToTag = moveToTag;
    }
    HK_INLINE void hkMonitorStreamAnalyzer_pop( hkArray<hkMonitorStreamAnalyzer_Node>& stack, _Outptr_ hkMonitorStreamAnalyzer::Node** node, _Out_ hkMonitorStreamAnalyzer::TagType* moveToTag )
    {
        hkMonitorStreamAnalyzer_Node& n = stack.back();
        *node = n.m_node;
        *moveToTag = n.m_moveToTag;
        stack.popBack();
    }
}


_Ret_maybenull_ hkMonitorStreamAnalyzer::Tree* hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(const hkMonitorStream::CommandStreamConfig& streamConfig,
    _In_ const char* frameStart,
    _In_ const char* frameEnd,
    const hkMonitorStreamFrameInfo& frameInfo,
    _In_z_ const char* rootNodeName,
    bool reuseNodesIfPossible,
    _Inout_opt_ DrawCallHandleMap* drawCallMappings)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    const char* currentStreamPtr = frameStart;
    Tree* rootNode = new Tree( );
    Node* currentNode = rootNode;
    Node* lastBeginListNodeCreated = HK_NULL;   // helps debugging
    hkInplaceArray<hkMonitorStream::TimerCommand,16> timerStack;


    hkInplaceArray<hkMonitorStreamAnalyzer_Node,16> nodeStack;

    // The stack below is just for debugging - it holds all the last timers parsed. If you are missing an 'end' macro
    // then the culprit *is likely* to be one of debugMismatchCheckerTimerStack[0] to debugMismatchCheckerTimerStack[ current size of debugMismatchCheckerTimerStack - 1]
    // and is most likely the most recently altered file. However further monitor parsing may have overwritten the culprit so it may not be in this array.
    // The only way to match things perfectly and always identify the culprit is to use HK_TIMER_NAMED_END
    hkInplaceArray<hkMonitorStream::TimerCommand,16> debugMismatchCheckerTimerStack;
    TagType moveToTag = 0;

    HK_ON_DEBUG(bool expectUnpause = false;)

    while( currentStreamPtr < frameEnd )
    {
        hkUint32 rawValue = hkMonitorStream::readCommandUInt32(&currentStreamPtr, streamConfig);
        currentStreamPtr -= 4;

        const char* currentBeforeString = currentStreamPtr;
        const char* string = hkMonitorStream::readCommandString(&currentStreamPtr, streamConfig);

        // Skip commands used to mark the start of an elf
        {
            if ( rawValue <= HK_MAX_ELF_ID)
            {
                //int elfId = rawValue;
                continue;
            }
        }

        currentStreamPtr = currentBeforeString;

        if(!string)
        {
            HK_WARN(0xc04e8b8, "Failed to read the monitor stream, all statistics for this thread will be lost!");
            delete rootNode;
            return HK_NULL;
        }

        if(0){      // seriously helps with debuggin
            int offset = int(currentStreamPtr-frameStart);
            if ( offset == 0 )
            {
                Log_Info( "\n\n\n" );
            }

            char buffer[256];
            int numTabs = timerStack.getSize() + nodeStack.getSize();
            for (int i =0; i < numTabs; i++) {buffer[i] = '\t';}
            hkString::snPrintf(buffer+numTabs, HK_COUNT_OF(buffer) - numTabs, 255-numTabs, "%s\t%i", string, offset );
            Log_Info( buffer );
        }

        HK_ASSERT(0x7a15a298, expectUnpause == (string[0] == 'Q' && string[1] == 'u'), "Pause/unpause must come in pairs");

        switch(string[0])
        {
        case 'F':
            {
                hkMonitorStream::FrameIdCommand com; com.read(&currentStreamPtr, streamConfig);
                // com.m_frameId
                break;
            }

        case 'X':
            {
                hkMonitorStream::TagCommand com; com.read(&currentStreamPtr, streamConfig);
                if ( moveToTag )
                {
                    // Nodes tagged to move to com.m_tag should go to the node tagged with moveToTag, but we don't know where it will be yet.
                    // This typically happens if a task graph is submitted in another hkTask::process() without any timers surrounding it.
                    // To get correct statistics the user needs to make sure that they submit their task graph inside of a timer.
                    currentNode = createNewNode( rootNode, currentNode, "Dummy: Missing timer around hkTaskQueue::submitHandles()", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY, true, moveToTag );
                    moveToTag = 0;
                }
                rootNode->m_tagToNode.insert(com.m_tag, currentNode);
                if (!currentNode->m_tags)
                {
                    currentNode->m_tags = new TagInfo;
                }
                currentNode->m_tags->m_tags.pushBack(com.m_tag);
                break;
            }

        case 'G':   // enter/exit tag
            {
                hkMonitorStream::TagCommand com;
                com.read(&currentStreamPtr, streamConfig);
                moveToTag = com.m_tag;
                break;
            }

        case 'R': // timer begin with self
        case 'T': // timer begin
            {
                hkMonitorStreamAnalyzer_push( nodeStack, currentNode, moveToTag );
                if (string[1] == 't')
                {
                    currentNode = createNewNode( rootNode, currentNode, string + 2, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                    moveToTag = 0;

                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, streamConfig);

                    timerStack.pushBack( com );
                debugMismatchCheckerTimerStack.pushBack( com );

                    if ( string[0] == 'R' )
                    {
                        currentNode->m_flags.orWith( Node::FLAGS_REQUEST_SELF );
                    }
                }
                else
                {
                    HK_ASSERT_NO_MSG(0xc04e8b8, string[1] == 'g');
                    currentNode = createNewNode( rootNode, currentNode, string + 2, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER_DRAW_CALL, reuseNodesIfPossible, moveToTag );

                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, streamConfig);

                    timerStack.pushBack( com );
                    debugMismatchCheckerTimerStack.pushBack( com );

                    currentNode->m_gpuHandle = com.m_gpuHandle;

                    if (drawCallMappings != HK_NULL)
                    {
                        DrawCallHandleMap::Iterator it = drawCallMappings->findKey(com.m_gpuHandle);
                        if (drawCallMappings->isValid(it))
                        {
                            hkTuple<Node*, Node*>& data = drawCallMappings->getValue(it);
                            data.m_0 = currentNode;
                        }
                        else
                        {
                            hkTuple<Node*, Node*> data;
                            data.m_0 = currentNode;
                            data.m_1 = HK_NULL;
                            drawCallMappings->insert(com.m_gpuHandle, data);
                        }
                    }
                }
                break;
            }

        case 'E': // timer end
            {
                if ( timerStack.getSize() == 0)
                {
                    HK_WARN(0xfafe7975, "Unmatched HK_TIMER_END() macro (with no HK_TIMER_BEGIN()) in timed code");
                    return rootNode;
                }
                const hkMonitorStream::TimerCommand& start = timerStack[timerStack.getSize() - 1];

                if ( string[2] && hkString::strCmp( start.m_commandAndMonitor+2, string+2) !=0 )
                {
                    HK_WARN( 0xf03edefe, "Unmatched timercommand: '" << start.m_commandAndMonitor+1 << "' =! '" << string+1 );
                    return rootNode;
                }

                if (string[1] == 'g') // gpu timer
                {
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, streamConfig);

                    currentNode->setTimers( frameInfo, streamConfig, start, com );
                }
                else
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, streamConfig);

                    currentNode->setTimers( frameInfo, streamConfig, start, com );
                }
                if ( currentNode->m_flags.anyIsSet(Node::FLAGS_REQUEST_SELF ))
                {
                    currentNode->createSelfTimer( frameInfo, rootNode );
                }
                hkMonitorStreamAnalyzer_pop( nodeStack, &currentNode, &moveToTag );
                timerStack.popBack();
                debugMismatchCheckerTimerStack.popBack();
                break;
            }

        case 'Y': // multi timer begin
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, streamConfig);

                hkMonitorStreamAnalyzer_push( nodeStack, currentNode, moveToTag );
                currentNode = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                moveToTag = 0;
                currentNode->m_flags.orWith( Node::FLAGS_TIME_IS_NOT_ABSOLUTE ); // notify the timer node that it doesn't hold an honest absolute time!
                break;
            }

        case 'W': // multi timer end
            {
                hkMonitorStream::MultiTimerCommand com;
                com.read(&currentStreamPtr, streamConfig);

                currentNode->setTimers( frameInfo, com );
                hkMonitorStreamAnalyzer_pop( nodeStack, &currentNode, &moveToTag );
                break;
            }

        case 'O': // object name
            {
                hkMonitorStream::TimerBeginObjectNameCommand com;
                const char* name = com.read( &currentStreamPtr, streamConfig );

                timerStack.pushBack( com );
                hkMonitorStreamAnalyzer_push( nodeStack, currentNode, moveToTag );
                currentNode = createNewNode( rootNode, currentNode, name, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                moveToTag = 0;
                debugMismatchCheckerTimerStack.pushBack( com );

                break;
            }

        case 'L': // timer list begin
            {
                hkMonitorStream::TimerBeginListCommand com;
                com.read(&currentStreamPtr, streamConfig);

                {
                    timerStack.pushBack( com );
                    debugMismatchCheckerTimerStack.pushBack( com );
                    hkMonitorStreamAnalyzer_push( nodeStack, currentNode, moveToTag );
                    currentNode = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                    moveToTag = 0;
                    lastBeginListNodeCreated = currentNode;
                }

                {
                    hkMonitorStream::TimerCommand com2 = com;
                    com2.m_commandAndMonitor = com.m_nameOfFirstSplit;
                    timerStack.pushBack( com2 );
                    debugMismatchCheckerTimerStack.pushBack( com2 );
                    hkMonitorStreamAnalyzer_push( nodeStack, currentNode, moveToTag );
                    currentNode = createNewNode( rootNode, currentNode, com2.m_commandAndMonitor+2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                }

                break;
            }

        case 'S': // split list
            {
                if ( timerStack.getSize() == 0)
                {
                    HK_WARN(0xfafe79fa, "Unmatched HK_TIMER_SPLIT_LIST() macro (with no HK_TIMER_BEGIN_LIST()) in timed code");
                    return rootNode;
                }
                hkMonitorStream::TimerCommand& start = timerStack[ timerStack.getSize()- 1 ];

                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, streamConfig);

                currentNode->setTimers( frameInfo, streamConfig, start, com );
                currentNode = createNewNode( rootNode, currentNode->m_parent, string + 2, Node::NODE_TYPE_TIMER, reuseNodesIfPossible, moveToTag );
                lastBeginListNodeCreated = currentNode;
                start = com;
                break;
            }

        case 'l': // list end
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, streamConfig);

                {
                    if ( timerStack.getSize() <= 1)
                    {
                        HK_ON_DEBUG( const char* lastName = (lastBeginListNodeCreated) ? lastBeginListNodeCreated->m_name : "ROOT" );
                        HK_WARN(0xfafe79fb, "Unmatched HK_TIMER_END_LIST() macro (with no HK_TIMER_BEGIN_LIST()) in timed code, last beginList created was " << lastName);
                        return rootNode;
                    }

                    const hkMonitorStream::TimerCommand& start = timerStack[ timerStack.getSize()- 1 ];
                    if ( start.m_commandAndMonitor[0] != 'S')
                    {
                        HK_WARN( 0xf0323454, "Mismatched HK_TIMER_END_LIST()/HK_TIMER_BEGIN_LIST() found. Probably missing a HK_TIMER_BEGIN_LIST() *or* timer '" << start.m_commandAndMonitor + 2 << "' has no END macro. Otherwise check debugMismatchCheckerTimerStack in this function for likely culprits.");
                        return rootNode;
                    }

                    currentNode->setTimers( frameInfo, streamConfig, start, com );
                    hkMonitorStreamAnalyzer_pop( nodeStack, &currentNode, &moveToTag );
                    timerStack.popBack();
                    debugMismatchCheckerTimerStack.setSize( debugMismatchCheckerTimerStack.getSize() - 1 );
                }
                {
                    const hkMonitorStream::TimerCommand& start = timerStack[ timerStack.getSize()- 1 ];
                    if (start.m_commandAndMonitor[0] != 'L')
                    {
                        HK_WARN(0xf0323454, "Mismatched HK_TIMER_END_LIST()/HK_TIMER_BEGIN_LIST() found. Probably missing a HK_TIMER_BEGIN_LIST() *or* timer '" << start.m_commandAndMonitor + 2 << "' has no END macro. Otherwise check debugMismatchCheckerTimerStack in this function for likely culprits.");
                        return rootNode;
                    }
                    currentNode->setTimers( frameInfo, streamConfig, start, com );
                    hkMonitorStreamAnalyzer_pop(nodeStack, &currentNode, &moveToTag);
                    timerStack.popBack();
                    debugMismatchCheckerTimerStack.setSize( debugMismatchCheckerTimerStack.getSize() - 1 );
                }
                break;
            }

        case 'M':   //HK_MONITOR_COMMAND_ADD_VALUE
            {
                hkMonitorStream::AddValueCommand com;
                com.read(&currentStreamPtr, streamConfig);

                Node *node = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_SINGLE, reuseNodesIfPossible, moveToTag);
                node->m_threadData[0].m_value += com.m_value;
                node->m_threadData[0].m_count += 1;
                break;
            }
        case 'A':
            {
                hkMonitorStream::AddStructCommand com;
                void* data = com.read(&currentStreamPtr, streamConfig);

                Node *node = createNewNode( rootNode, currentNode, string + 2, hkMonitorStreamAnalyzer::Node::NODE_TYPE_META_DATA, reuseNodesIfPossible, moveToTag );
                //XX This wont work with VDB until COM-2826 is fixed
                node->m_metaData.setFromVar( hkReflect::Var(data, com.m_type) );
                break;
            }
        case 'H':
            {
                hkMonitorStream::AddGpuHandleCommand com;
                com.read(&currentStreamPtr, streamConfig);

                Node* node = createNewNode( rootNode, currentNode, string + 1, hkMonitorStreamAnalyzer::Node::NODE_TYPE_GPU_HANDLE, reuseNodesIfPossible, moveToTag);
                node->m_gpuHandle = com.m_gpuHandle;

                if (drawCallMappings != HK_NULL)
                {
                    DrawCallHandleMap::Iterator it = drawCallMappings->findKey(com.m_gpuHandle);
                    if (drawCallMappings->isValid(it))
                    {
                        hkTuple<Node*, Node*>& data = drawCallMappings->getValue(it);
                        data.m_1 = currentNode;
                    }
                    else
                    {
                        hkTuple<Node*, Node*> data;
                        data.m_0 = HK_NULL;
                        data.m_1 = currentNode;
                        drawCallMappings->insert(com.m_gpuHandle, data);
                    }
                }


                break;
            }
        case 'P': //Push Dir
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, streamConfig);

                hkMonitorStreamAnalyzer_push( nodeStack, currentNode, moveToTag );
                currentNode = createNewNode( rootNode, currentNode, string + 2, Node::NODE_TYPE_DIRECTORY, reuseNodesIfPossible, moveToTag);
                moveToTag = 0;
                break;
            }

        case 'p':
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, streamConfig);

                if ( nodeStack.isEmpty() )
                {
                    HK_WARN(0xf023df45, "Mismatched HK_TIMER_POP() function");
                    return rootNode;
                }
                hkMonitorStreamAnalyzer_pop( nodeStack, &currentNode, &moveToTag );
                break;
            }

        case 'N': // nop
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, streamConfig);
                break;
            }

        case 'm': // memory
            {
                hkMonitorStream::MemoryCommand com;
                com.read(&currentStreamPtr, streamConfig);
                break;
            }

        case 'Q': // Pause/unpause
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, streamConfig);

                switch(string[1])
                {
                    case 'p': // pause
                        {
                            hkMonitorStreamAnalyzer_push(nodeStack, currentNode, moveToTag);
                            currentNode = createNewNode(rootNode, currentNode, string + 2, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER_DEDUCTION, reuseNodesIfPossible, moveToTag);
                            moveToTag = 0;
                            timerStack.pushBack(com);

                            HK_ON_DEBUG(expectUnpause = true;)
                            break;
                        }
                    case 'u': // unpause
                        {
                            const hkMonitorStream::TimerCommand& start = timerStack[timerStack.getSize() - 1];
                            currentNode->setTimers(frameInfo, streamConfig, start, com);
                            hkMonitorStreamAnalyzer_pop(nodeStack, &currentNode, &moveToTag);
                            timerStack.popBack();

                            HK_ON_DEBUG(expectUnpause = false;)
                            break;
                        }
                    default:
                        HK_WARN(0x92735944, "Inconsistent Monitor capture data");
                        return rootNode;
                }
                break;
            }

        default:
            HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
            return rootNode;
        }
    }

    if ( timerStack.getSize() )
    {
        HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data. Probably missing HK_TIMER_END*s eg. for timer '" << timerStack[timerStack.getSize() - 1].m_commandAndMonitor + 2 << "' Check timerStack in this function for likely culprits." );
    }

    return rootNode;
}


_Ret_notnull_ hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer::makeStatisticsTreeForMultipleFrames(int threadId, hkBool reuseNodesIfPossible, _Inout_opt_ DrawCallHandleMap* drawCallMappings)
{
    Node* rootNode = new Node( HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY );
    rootNode->m_children.setSize(m_frameInfos[threadId].getSize());


    for (int i = 0; i < m_frameInfos[threadId].getSize(); ++i)
    {
        hkMonitorStreamFrameInfo& currentFrameInfo = m_frameInfos[threadId][i];

        const char* start = m_data.begin() + currentFrameInfo.m_frameStreamStart;
        const char* end   = m_data.begin() + currentFrameInfo.m_frameStreamEnd;


        hkMonitorStream::CommandStreamConfig streamConfig;
        rootNode->m_children[i] = makeStatisticsTreeForSingleFrame(streamConfig, start, end, currentFrameInfo, currentFrameInfo.m_heading, reuseNodesIfPossible, drawCallMappings);
    }

    return rootNode;
}



/////////////////////////////////////////////////////////////////////////////////
//
// Find the total call counts
//
/////////////////////////////////////////////////////////////////////////////////


void hkMonitorStreamAnalyzer_traverseCompleteTreeToFindTotalCallCounts( _Inout_ hkMonitorStreamAnalyzer::Node* node )
{
    if ( !node->m_children.getSize() )
    {
        // this is a leaf
        node->m_totalCallCountForChildren = 0;
        for ( int i = 0; i < node->m_threadData.getSize(); ++i )
        {
            node->m_totalCallCountForChildren += node->m_threadData[i].m_count;
        }
    }
    else
    {
        node->m_totalCallCountForChildren = 0;
        for ( int i = 0; i < node->m_children.getSize(); ++i )
        {
            hkMonitorStreamAnalyzer_traverseCompleteTreeToFindTotalCallCounts( node->m_children[i] );
            node->m_totalCallCountForChildren += node->m_children[i]->m_totalCallCountForChildren;
        }
    }
}



/////////////////////////////////////////////////////////////////////////////////
//
// Text output utilities
//
/////////////////////////////////////////////////////////////////////////////////

void hkMonitorStreamAnalyzer_printColumnHeader(hkArray<char>& textLine, hkMonitorStreamAnalyzer::CombinedThreadSummaryOptions& options, int column, _In_z_ const char* header)
{
    hkOstream lineOs( textLine );

    if (options.m_useTabsNotSpacesForColumns)
    {
        lineOs << '\t';
    }
    else
    {
        int extraSpacing = options.m_indentationToFirstTimerValue + options.m_timerColumnWidth * column - textLine.getSize();
        for (int j = 0; j < extraSpacing; ++j) lineOs << ' ';
    }

    lineOs.printf(header);
}

void hkMonitorStreamAnalyzer::showCombinedThreadSummaryForSingleFrame( _Inout_ Node* node, int numThreads, hkOstream& os, CombinedThreadSummaryOptions& options )
{
    options.m_indentationToFirstTimerValue = hkMonitorStreamAnalizer_findMaxTimerNameIndent( node, 0, options.m_tabSpacingForTimerNames, options.m_displayPartialTree );

    hkArray<char> textLine;
    hkOstream lineOs( textLine );

    // Display table title row
    // Note: extra spaces in column headers help in empty case so text isn't flush against next column or border
    lineOs.printf("Timer Name  ");

    if ( numThreads > 1 )
    {
        hkStringBuf header;

        for ( int i = 0; i < numThreads; ++i )
        {
            header.printf("Thread %d  ", i);
            hkMonitorStreamAnalyzer_printColumnHeader( textLine, options, i, header.cString() );
        }

#if defined(SHOW_MAX_TIMER_VALUE)
        hkMonitorStreamAnalyzer_printColumnHeader( textLine, options, numThreads, "Max  " );
#else
        hkMonitorStreamAnalyzer_printColumnHeader( textLine, options, numThreads, "Sum  " );
#endif

    }
    os << textLine.begin();
    os.printf("\n\n");

#ifdef HK_MONITOR_SHOW_TOTAL_SUBTREE_CALL_COUNTS
    hkMonitorStreamAnalyzer_traverseCompleteTreeToFindTotalCallCounts( node );
#endif

    showCombinedThreadSummaryRec(os, node, 0, numThreads, options );
}


// The nodes array is arranged as follows:
// For each thread, there is one element in the list
// The root node has the frame info heading
// T

// This function produces several output lists:
//      - a total hierarchical average
//      - a per frame single node
//      - a summary per frame
//      - the detailed view

// (reordered and with these protos for SN compiler:)
static _Ret_maybenull_ hkMonitorStreamAnalyzer::Node* HK_CALL hkMonitorStreamAnalyzer_findNextChildByName(_Inout_ hkMonitorStreamAnalyzer::Node* parent, _In_z_ const char* childName, const _In_opt_ hkMonitorStreamAnalyzer::Node* oldChild);





// per name: size
    // disable certain items
// sort by size
// for each elem: find all nodes with a name
    // make child statistics
    // make parent statistics

struct HK_EXPORT_COMMON hkMonitorStreamAnalyzerPrintByTypeInfo
{
    HK_DECLARE_CLASS(hkMonitorStreamAnalyzerPrintByTypeInfo, NewManual);

    hkMonitorStreamAnalyzerPrintByTypeInfo() : m_this( HK_NULL, 0, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER) {}
    bool operator< ( const hkMonitorStreamAnalyzerPrintByTypeInfo& other ) const { return m_this.m_threadData[1].m_value < other.m_this.m_threadData[1].m_value; }

    hkMonitorStreamAnalyzer::Node m_this;
    hkArray<hkMonitorStreamAnalyzer::Node>::Debug m_parents;
    hkArray<hkMonitorStreamAnalyzer::Node>::Debug m_children;
};

HK_MEMORY_TRACKER_MANUAL_BEGIN(3, hkMonitorStreamAnalyzerPrintByTypeInfo);
    HK_MEMORY_TRACKER_MANUAL_FIELD(m_this, hkMonitorStreamAnalyzer::Node);
    HK_MEMORY_TRACKER_MANUAL_FIELD(m_parents, hkArray<hkMonitorStreamAnalyzer::Node>::Debug);
    HK_MEMORY_TRACKER_MANUAL_FIELD(m_children, hkArray<hkMonitorStreamAnalyzer::Node>::Debug);
HK_MEMORY_TRACKER_MANUAL_END();

HK_INLINE bool hkCompareInfosBySize(_In_ const hkMonitorStreamAnalyzerPrintByTypeInfo* a, _In_ const hkMonitorStreamAnalyzerPrintByTypeInfo* b)
{
    return *a < *b;
}

static void HK_CALL hkBuildSizePerName(_In_ const hkMonitorStreamAnalyzer::Node* node, hkStringMap<hkMonitorStreamAnalyzerPrintByTypeInfo*>& mapOut )
{
    const char* name = node->m_name;

    hkMonitorStreamAnalyzerPrintByTypeInfo* info = mapOut.getWithDefault( name, HK_NULL );
    if ( !info )
    {
        info = new hkMonitorStreamAnalyzerPrintByTypeInfo;
        info->m_this.m_name = name;
        mapOut.insert( name, info );
    }
    hkMakeSum( &info->m_this, node );

    for (int i = 0; i < node->m_children.getSize(); i++ )
    {
        const hkMonitorStreamAnalyzer::Node* child = node->m_children[i];
        hkBuildSizePerName( child, mapOut );

        // insert child as my child as well
        {
            const char* childName = child->m_name;
            int c;
            hkMonitorStreamAnalyzer::Node* childSum = HK_NULL;
            for (c = info->m_children.getSize()-1; c>=0; c-- )
            {
                childSum = &info->m_children[c];
                if ( hkString::strCmp( childName, childSum->m_name) == 0)
                {
                    break;
                }
            }
            if ( c < 0)
            {
                childSum = new (info->m_children.expandBy(1)) hkMonitorStreamAnalyzer::Node( HK_NULL, childName, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER );
            }
            hkMakeSum( childSum, child);
        }
    }

    // insert parent as my child as well
    if ( node->m_parent)
    {
        const char* parentName = node->m_parent->m_name;
        int c;
        hkMonitorStreamAnalyzer::Node* parentSum = HK_NULL;
        for (c = info->m_parents.getSize()-1; c>=0; c-- )
        {
            parentSum = &info->m_parents[c];
            if ( hkString::strCmp( parentName, parentSum->m_name) == 0)
            {
                break;
            }
        }
        if ( c < 0)
        {
            parentSum = new (info->m_parents.expandBy(1)) hkMonitorStreamAnalyzer::Node( HK_NULL, parentName, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER );
        }
        hkMakeSum( parentSum, node->m_parent);
    }
}
static void printSingleNodeValues( hkOstream& outstream, hkReal maxMem, _In_ const hkMonitorStreamAnalyzer::Node* node )
{
    hkUint32 maxCount = 0;
    int columns = 0;

    {
        for (int i = 0; i < node->m_threadData.getSize(); i++)
        {
            if ( !node->m_threadData[i].m_count )
            {
                continue;
            }
            columns ++;
            if ( node->m_threadData[i].m_count > maxCount) { maxCount = node->m_threadData[i].m_count; }
        }
    }
    const char* sep = "\t\t\t";

    char buffer[256];

    if ( maxMem > 0 )
    {
        hkString::snPrintf( buffer, HK_COUNT_OF(buffer), 200, "%s (%i) %4.1f%%", node->m_name, maxCount, 100.0f * node->m_threadData[1].m_value / maxMem );
    }
    else
    {
        hkString::snPrintf( buffer, HK_COUNT_OF(buffer), 200, "%s (%i)", node->m_name, maxCount);
    }


    outstream.printf("%-34s%s", buffer,sep);

    for (int i = 0; i < node->m_threadData.getSize(); i++)
    {
        if ( !node->m_threadData[i].m_count )
        {
            continue;
        }
        outstream.printf("% 12.3f: ", node->m_threadData[i].m_value);
    }
    outstream.printf("\n");
}



static void printStatisticsByType(hkOstream& outstream, _Inout_ hkMonitorStreamAnalyzer::Node* rootNode, hkReal maxImportance = 0.01f)
{
    hkArray<hkMonitorStreamAnalyzerPrintByTypeInfo*>::Temp stats;
    {
        hkStringMap<hkMonitorStreamAnalyzerPrintByTypeInfo*> map;
        hkBuildSizePerName( rootNode, map );
        stats.reserve( map.getSize() );
        for ( hkStringMap<hkMonitorStreamAnalyzerPrintByTypeInfo*>::Iterator i = map.getIterator(); map.isValid(i); i = map.getNext(i))
        {
            stats.pushBackUnchecked( map.getValue(i) );
        }
        hkSort( stats.begin(), stats.getSize(), hkCompareInfosBySize );
    }

    hkReal maxVal = stats.back()->m_this.m_threadData[1].m_value;

    outstream.printf("\n\n");
    outstream.printf("************************************\n" );
    outstream.printf("********** Per Type Times    *******\n" );
    outstream.printf("************************************\n" );

    for (int i = stats.getSize()-1; i >=0; i--)
    {
        const hkMonitorStreamAnalyzerPrintByTypeInfo* info = stats[i];
        hkReal val = info->m_this.m_threadData[1].m_value;
        if ( val > maxVal * maxImportance )
        {
            const hkMonitorStreamAnalyzer::Node* node = &info->m_this;
            outstream.printf("\n");
            outstream.printf("************************************\n\n" );
            {
                for (int p = 0; p < info->m_parents.getSize(); p++ )
                {
                    outstream.printf("\t\t");
                    printSingleNodeValues( outstream, 0.0f, &info->m_parents[p] );
                }
            }
            printSingleNodeValues( outstream, maxVal, node );
            {
                for (int c = 0; c < info->m_children.getSize(); c++ )
                {
                    outstream.printf("\t\t");
                    printSingleNodeValues( outstream, 0.0f, &info->m_children[c] );
                }
            }

        }
        delete info;
    }
}

// Normally the \a nodes are the timings for each thread, the first level children are the frames
void hkMonitorStreamAnalyzer::writeStatisticsDetails(hkOstream& outstream, hkArrayBase<ThreadRootNodes>& perThreadNodes, int numThreads, int reportLevel, _In_z_ const char* nodeIdForFrameOverview, bool showMultithreadedSummary, _Inout_opt_ DrawCallHandleMap* drawCallMappings, _In_opt_ const hkArrayView<hkGpuTraceResult>* traceResultsPerFrame)
{
    // Print the version of Havok that created these statistics.
    outstream.printf("Havok version: %s\n", HAVOK_SDK_VERSION_STRING);

    // The first two analyzes only work for one thread at the moment.
    if ((1 || perThreadNodes.getSize() == 1) && perThreadNodes[0].m_node)
    {
        const hkArrayBase<Node*>& childNodes = perThreadNodes[0].m_node->m_children;

        // the average value
        hkReal avgValue = 0.0f;
        //
        //  summarize everything
        //

        {
            Node sum( 0, "Sum", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY ); // sum over all frames
            float f = 1.0f / childNodes.getSize();
            for ( int i = 0; i < childNodes.getSize();i++ )
            {
                hkMakeSumRecursive( &sum, childNodes[i] );
            }
            for ( Node* node = hkMonitorStreamAnalyzer_findChildByName( &sum, nodeIdForFrameOverview, true ); node; node = hkMonitorStreamAnalyzer_findNextChildByName( &sum, nodeIdForFrameOverview, node ) )
            {
                avgValue += node->m_threadData[0].m_value * f;
            }
            if ( avgValue <= 0.0f ) { avgValue = 1000.0f; }

            if( reportLevel & REPORT_TOTAL )
            {
                outstream.printf("\n\n");
                outstream.printf("*********************************\n" );
                outstream.printf("**** Total Times (Thread 0) *****\n" );
                outstream.printf("*********************************\n" );
                outstream.printf("*** Timers are added together ***\n\n");
                hkMonitorStreamAnalyzer_writeNodeRec( outstream, &sum, 0, f, hkArrayView<hkStringPtr>(), HK_NULL, HK_NULL );
            }
        }

        //
        // print each frame in a single line
        //
        if( reportLevel & REPORT_PERFRAME_TIME )
        {
            outstream.printf("\n\n");
            outstream.printf("*********************************\n" );
            outstream.printf("*** Per Frame Time (Thread 0) ***\n" );
            outstream.printf("*********************************\n" );
            outstream.printf("Ascii Art all frames overview for node %s\n", nodeIdForFrameOverview );

            const int GRAPH_SIZE = 40;
            char buffer[GRAPH_SIZE+10];
            hkString::memSet( buffer, ' ', GRAPH_SIZE );
            buffer[GRAPH_SIZE] = 0;

            for ( int i = 0; i < childNodes.getSize();i++ )
            {
                Node* frameRoot = childNodes[i];
                if ( frameRoot->m_children.getSize() == 0 )
                {
                    continue;
                }

                hkReal val = 0.0f;
                const char* nodeName = "Unknown";
                {
                    for ( Node* node = hkMonitorStreamAnalyzer_findChildByName( frameRoot, nodeIdForFrameOverview, true ); node; node = hkMonitorStreamAnalyzer_findNextChildByName( frameRoot, nodeName, node ) )
                    {
                        nodeName = node->m_name;
                        val  += node->m_threadData[0].m_value;
                    }
                }
                //
                // draw graph
                //
                {
                    hkReal relVal = 0.5f * GRAPH_SIZE * val / avgValue;
                    int index = int(relVal);
                    if ( index < 0) index = 0;
                    if (index >=GRAPH_SIZE ) index = GRAPH_SIZE-1;
                    char *p = buffer;
                    int j = 0;
                    for (; j +4 < index; j+=4){ *(p++) = '\t'; }
                    int j2 = j;
                    for (; j2 < index; j2+=1){ *(p++) = ' '; }
                    *(p++) = '#';
                    j2++;
                    j += 4;
                    for (; j2 < j; j2+=1){ *(p++) = ' '; }
                    for (; j < GRAPH_SIZE; j+=4){ *(p++) = '\t'; }
                    *(p) = 0;
                    outstream.printf(buffer);
                }
                outstream.printf("%i %-12s %f\n" , i, nodeName, val );
            }
        }
    }
    int numFrames = perThreadNodes[0].m_node->m_children.getSize();
    if( reportLevel & REPORT_PERFRAME_SUMMARY )
    {
        if (perThreadNodes.getSize() == 1 || !showMultithreadedSummary)
        {
            // For each frame
            for ( int frameId = 0; frameId < numFrames; frameId++ )
            {
                // For each thread
                for (int threadId = 0; threadId < perThreadNodes.getSize(); ++threadId )
                {
                    Node* node = perThreadNodes[threadId].m_node;
                    if ( node && frameId < node->m_children.getSize() )
                    {
                        Tree tree;

                        const Tree* singleTree = static_cast<const Tree*>(node->m_children[frameId]);
                        mergeTreesForCombinedThreadSummary(&tree, singleTree, threadId, 0 );
                        fixupTags(&tree);

                        outstream.printf("\n");
                        outstream.printf("****************************************\n" );
                        outstream.printf("****** Summary Frame:%i Thread:%i ******\n",frameId, threadId );
                        outstream.printf("****************************************\n" );
                        outstream.printf("%s\n", node->m_children[frameId]->m_name ); // should be the heading from the frameinfo
                        hkMonitorStreamAnalyzer_writeNodeRec( outstream, &tree, 0, 1.0f, hkArrayView<hkStringPtr>(), HK_NULL, HK_NULL );

                        if( reportLevel & REPORT_PERFRAME_PERTYPE )
                        {
                            printStatisticsByType( outstream, &tree );
                        }
                    }
                }
            }
        }
        else if(perThreadNodes[0].m_node)
        {
            //
            //  Multi threaded summary per frame
            //
            for ( int frameId = 0; frameId < perThreadNodes[0].m_node->m_children.getSize(); frameId++ )
            {
                outstream.printf("\n");
                outstream.printf("****************************************\n" );
                outstream.printf("****** Summary Frame:%i ******\n",frameId);
                outstream.printf("****************************************\n" );
                //outstream.printf("%s\n", node->m_children[i]->m_name ); // should be the heading from the frameinfo

                Tree tree;
                {
                    for (int threadId = 0; threadId < perThreadNodes.getSize(); ++threadId )
                {
                        if ( frameId < perThreadNodes[threadId].m_node->m_children.getSize() )
                    {
                            const Tree* singleTree = static_cast<const Tree*>(perThreadNodes[threadId].m_node->m_children[frameId]);
                            mergeTreesForCombinedThreadSummary(&tree, singleTree, threadId, 0 );
                        }
                    }
                }
                fixupTags( &tree );

                CombinedThreadSummaryOptions options;
                options.m_tabSpacingForTimerNames = 4;
                options.m_tabSpacingForTimerValues = 2;
                options.m_timerColumnWidth = 16;
                options.m_displayPartialTree = false;
                options.m_useTabsNotSpacesForColumns = false;
                showCombinedThreadSummaryForSingleFrame( &tree, numThreads, outstream, options );
                if( reportLevel & REPORT_PERFRAME_PERTYPE )
                {
                    printStatisticsByType( outstream, &tree );
                }
            }
        }
    }


    //
    //  detailed view
    //
    if( reportLevel & REPORT_PERFRAME_DETAIL )
    {
        // For each frame
        for ( int i = 0; perThreadNodes[0].m_node && i < perThreadNodes[0].m_node->m_children.getSize(); i++ )
        {
            const hkArrayView<hkGpuTraceResult>* gpuTrace = HK_NULL;
            if (traceResultsPerFrame != HK_NULL)
            {
                gpuTrace = &( traceResultsPerFrame[i] );
            }

            // For each thread
            for (int j = 0; j < perThreadNodes.getSize(); ++j )
            {
                Node* node = perThreadNodes[j].m_node;
                if (node && i < node->m_children.getSize() )
                {
                    outstream.printf("\n\n");
                    outstream.printf("***************************************\n" );
                    if (j < numThreads )
                    {
                        outstream.printf("***** Details Frame-%i Thread:%i ******\n", i, j );
                    }

                    outstream.printf("***************************************\n" );
                    outstream.printf("%s\n", node->m_children[i]->m_name );
                    hkMonitorStreamAnalyzer_writeNodeRec( outstream, node->m_children[i], 0, 1.0f, hkArrayView<hkStringPtr>(), drawCallMappings, gpuTrace );
                }
            }
        }
    }
}


static void HK_CALL hkMonitorStreamAnalyzer_writeNodeRec( hkOstream& outstream, _In_ const hkMonitorStreamAnalyzer::Node* node, int recDepth, float factor, hkArrayView<hkStringPtr> ignoredNames, const _In_opt_ hkMonitorStreamAnalyzer::DrawCallHandleMap* drawCallMappings, _In_opt_ const hkArrayView<hkGpuTraceResult>* traceResults )
{
    if( recDepth )
    {
        for( int j = 1; j < recDepth; j++ )
        {
            outstream.printf("\t" );
        }

        //
        //  Find maximum count
        //  and the number of columns used
        //
        hkUint32 maxCount = 0;
        int columns = 0;
        {
            for (int i = 0; i < node->m_threadData.getSize(); i++)
            {
                if ( !node->m_threadData[i].m_count )
                {
                    continue;
                }
                columns ++;
                if ( node->m_threadData[i].m_count > maxCount) { maxCount = node->m_threadData[i].m_count; }
            }
        }


        //
        //  print name
        //
        {
        char buffer[256];

        if ( maxCount <= 1 )
        {
            hkString::snPrintf( buffer, HK_COUNT_OF(buffer), 200, "%s", node->m_name);
        }
        else if ( factor == 1.0f)
        {
            hkString::snPrintf( buffer, HK_COUNT_OF(buffer), 200, "%s (%i)", node->m_name, maxCount);
        }
        else
        {
            hkString::snPrintf( buffer, HK_COUNT_OF(buffer), 200, "%s (%4.1f)", node->m_name, maxCount * factor);
        }
        outstream.printf("%-32s", buffer);

        for( int j = 5 /*RecDepth*/; j < 8; j++ )
        {
            outstream.printf("\t" );
        }
        }

        // tags
        if ( node->m_tags )
        {
            outstream.printf("{" );
            for (int i =0; i < node->m_tags->m_tags.getSize(); i++ )
            {
                outstream.printf("%i,", node->m_tags->m_tags[i] );
            }
            outstream.printf("}" );
        }
        if ( node->m_moveToTag )
        {
            outstream.printf("[%i]", node->m_moveToTag );
        }


        //
        //  rescale and print values
        //
        if ( columns >0 )
        {
            int columnsToPrint = columns;
            if (1)
            {
                for ( int i = 0; i < node->m_threadData.getSize(); i++)
                {
                    hkUint32 c = node->m_threadData[i].m_count;
                    if ( !c )
                    {
                        continue;
                    }
                    hkReal val = node->m_threadData[i].m_value * factor;
                    if ( c < maxCount)
                    {
                        val *= hkReal(maxCount)/hkReal(c);
                    }
                    columnsToPrint--;
                    if (columnsToPrint)
                    {
                        outstream.printf("% 12.3f: ", val);
                    }
                    else
                    {
                        outstream.printf("% 12.3f\n", val);
                    }
                }
            }
        }
        else
        {
            outstream.printf("% 12.3f\n", 0.0f);
        }

        if (drawCallMappings != HK_NULL)
        {
            for ( int i = 0; i < node->m_children.getSize(); i++ )
            {
                const hkMonitorStreamAnalyzer::Node* current = node->m_children[i];

                if (current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_GPU_HANDLE)
                {
                    hkTuple<hkMonitorStreamAnalyzer::Node*, hkMonitorStreamAnalyzer::Node*> link;
                    if (drawCallMappings->get(current->m_gpuHandle, &link).isSuccess())
                    {
                        if (link.m_0 != HK_NULL)
                        {
                            HK_ASSERT_NO_MSG(0x5edee432, link.m_0->m_threadData[0].m_count > 0);
                            outstream.printf(" [ DrawCall %.3f ]", link.m_0->m_threadData[0].m_value);
                        }

                        if (traceResults != HK_NULL)
                        {
                             hkMonitorStreamAnalyzer_printGpuTiming(outstream, *traceResults, current->m_gpuHandle);
                        }
                    }
                }
                else if (current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_META_DATA)
                {
                    bool formatted = false;
                    hkStringBuf buffer;
                    if (hkReflect::RecordVar record = current->m_metaData.var())
                    {
                        hkReflect::CallableIterator it(record.getType(), hkReflect::CALLABLE_METHODS);
                        while (it.advance())
                        {
                            const hkReflect::NamedCallable* callable = it.current();
                            if (hkString::strCmp(callable->getName(), "format") == 0 && callable->getNumParams() == 1 &&
                                callable->getParamType(0)->equals<hkStringBuf*>())
                            {
                                hkStringBuf* bufferPtr = &buffer;
                                callable->callMethod(record.getAddress(), &bufferPtr, HK_NULL);
                                formatted = true;
                                break;
                            }
                        }
                    }
                    if (!formatted)
                    {
                        current->m_metaData.var().toString(buffer);
                    }
                    outstream.printf(" [ %s ]", buffer.cString());
                }
            }

            if (node->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER_DRAW_CALL)
            {
                hkTuple<hkMonitorStreamAnalyzer::Node*, hkMonitorStreamAnalyzer::Node*> link;
                if (drawCallMappings->get(node->m_gpuHandle, &link).isSuccess())
                {
                    if (traceResults != HK_NULL)
                    {
                        hkMonitorStreamAnalyzer_printGpuTiming(outstream, *traceResults, node->m_gpuHandle);
                    }

                    hkStringBuf buffer;

                    const hkMonitorStreamAnalyzer::Node* current = link.m_1;
                    while (current != HK_NULL)
                    {
                        buffer.insert(0, "/");
                        buffer.insert(0, current->m_name);
                        current = current->m_parent;
                    }

                    outstream.printf(" [ %s ]", buffer.cString());
                }
            }
            outstream.printf("\n");
        }
    }

    for ( int i = 0; i < node->m_children.getSize(); i++ )
    {
        hkMonitorStreamAnalyzer::Node* current = node->m_children[i];

        // Do not further process nodes of type NODE_TYPE_DRAW_CALL_HANDLE and NODE_TYPE_META_DATA as these types do not have any children and are processed together with the parent node.
        bool ignored = current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_GPU_HANDLE || current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_META_DATA;
        for (int n = 0; !ignored && n < ignoredNames.getSize(); ++n)
        {
            if (ignoredNames[n] == current->m_name)
                ignored = true;
        }
        if (ignored)
        {
            continue;
        }
        hkMonitorStreamAnalyzer_writeNodeRec( outstream, node->m_children[i], recDepth + 1, factor, ignoredNames, drawCallMappings, traceResults );
    }
}

static void hkMonitorStreamAnalyzer_accumulateMetadata(hkReflect::Var parent, hkReflect::Var metadata)
{
    HK_ASSERT_NO_MSG(0x4f0a57b5, parent.getType()->equals(metadata.getType()));
    if (const hkReflect::RecordType* recordType = metadata.getType()->asRecord())
    {
        for (hkReflect::DeclIter<hkReflect::FieldDecl> it(recordType); it.advance(); )
        {
            const hkReflect::FieldDecl& field = it.current();
            if (const hkms::Accumulate* accumulate = field.getType()->findAttribute<hkms::Accumulate>())
            {
                hkReflect::IntVar parentFieldAsInt(parent[field]);
                if (parentFieldAsInt)
                {
                    switch (accumulate->m_value)
                    {
                        case hkms::AccumulateOp::Add:
                            parentFieldAsInt.setValue(parentFieldAsInt.getValue() + hkReflect::IntVar(metadata[field]).getValue());
                            break;
                        default:
                            HK_ASSERT_NO_MSG(0x1329386a, false);
                    }
                }
                else
                {
                    HK_ASSERT_NO_MSG(0x43d2e1b8, false);
                }
            }
        }
    }
}

static int hkMonitorStreamAnalyzer_findParentMetadata(hkReflect::Var metadata, hkArrayView<hkReflect::Var> parents)
{
    for (int parentIndex = 0; parentIndex < parents.getSize(); ++parentIndex)
    {
        if (parents[parentIndex].getType()->equals(metadata.getType()))
        {
            return parentIndex;
        }
    }
    return -1;
}

void HK_CALL hkMonitorStreamAnalyzer::accumulateMetaData( hkMonitorStreamAnalyzer::Node* node, hkArrayView<hkReflect::Var> parents )
{
    hkArray<hkReflect::Var> localParents;
    localParents.append(parents.begin(), parents.getSize());
    for (int childIndex = 0; childIndex < node->m_children.getSize(); childIndex++)
    {
        hkMonitorStreamAnalyzer::Node* current = node->m_children[childIndex];
        if (current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_META_DATA)
{
            hkReflect::Var metadata = current->m_metaData.var();
            int parentIndex = hkMonitorStreamAnalyzer_findParentMetadata(metadata, localParents);
            (parentIndex != -1 ? localParents[parentIndex] : localParents.expandOne()) = metadata;
        }
    }

    for (int childIndex = 0; childIndex < node->m_children.getSize(); childIndex++)
    {
        hkMonitorStreamAnalyzer::Node* current = node->m_children[childIndex];

        // Do not further process nodes of type NODE_TYPE_DRAW_CALL_HANDLE and NODE_TYPE_META_DATA as these types do not have any children and are processed together with the parent node.
        if (current->m_type != hkMonitorStreamAnalyzer::Node::NODE_TYPE_GPU_HANDLE && current->m_type != hkMonitorStreamAnalyzer::Node::NODE_TYPE_META_DATA)
        {
            accumulateMetaData(current, localParents);
        }
    }

    for (int childIndex = 0; childIndex < node->m_children.getSize(); childIndex++)
    {
        hkMonitorStreamAnalyzer::Node* current = node->m_children[childIndex];
        if (current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_META_DATA)
        {
            hkReflect::Var metadata = current->m_metaData.var();
            int parentIndex = hkMonitorStreamAnalyzer_findParentMetadata(metadata, parents);
            if (parentIndex != -1)
    {
                hkMonitorStreamAnalyzer_accumulateMetadata(parents[parentIndex], metadata);
            }
        }
    }
}

static _Ret_maybenull_ hkMonitorStreamAnalyzer::Node* HK_CALL hkMonitorStreamAnalyzer_findNextChildByName(_Inout_ hkMonitorStreamAnalyzer::Node* node, _In_z_ const char* childName, const _In_opt_ hkMonitorStreamAnalyzer::Node* oldChild)
{
    // search child
    int j;
    for( j = 0; j < node->m_children.getSize(); j++ )
    {
        if ( node->m_children[j] == oldChild )
        {
            break;
        }
    }
    j++;
    for( ; j < node->m_children.getSize(); j++ )
    {
        if( hkString::strCmp( childName, node->m_children[j]->m_name ) == 0 )
        {
            return node->m_children[j];
        }
    }
    return HK_NULL;
}



static void HK_CALL hkMonitorStreamAnalyzer_searchTopLevelNodesRec(_Inout_ hkMonitorStreamAnalyzer::Node* node, hkArrayView<hkGpuTraceResult> scopeTraces, hkArray< hkMonitorStreamAnalyzer::Node* >& topLevelNodes)
{
    if (node->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER_DRAW_CALL)
    {
        topLevelNodes.pushBack(node);
        return;
    }

    for (int i=0; i<node->m_children.getSize(); ++i)
    {
        hkMonitorStreamAnalyzer::Node* current = node->m_children[i];
        if (current->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_GPU_HANDLE)
        {
            // Search for scope, the found position is the correct execution order as the scope traces array is ordered.
            for (int idx=0; idx < scopeTraces.getSize(); ++idx)
{
                if (current->m_gpuHandle == scopeTraces[idx].m_id)
    {
                    topLevelNodes[idx] = node;
                    // Note, early out! This skips all sub-scopes.
                    return;
                }
            }

            HK_ASSERT_NO_MSG(0x468eef70, "GPU handle not found in scope traces!");
        }

        // Process children if no top level scope has been found.
        hkMonitorStreamAnalyzer_searchTopLevelNodesRec( current, scopeTraces, topLevelNodes );
    }
}

static void HK_CALL hkMonitorStreamAnalyzer_printGpuTiming( hkOstream& outstream, const hkArrayView<hkGpuTraceResult>& traces, hkUint64 guid )
{
    const hkUint32 count = traces.getSize();
    for (hkUint32 idx=0; idx<count; ++idx)
    {
        const hkGpuTraceResult& current = traces[idx];
        if (traces[idx].m_id == guid)
        {
            const double gpuTime = current.m_gpuTimeEnd - current.m_gpuTimeBegin;
            outstream.printf(" [ GPU %g ms (%d) ]", gpuTime * 1e3, current.m_numPixelsTouched);
            break;
        }
    }
}


void hkMonitorStreamAnalyzer::writeStatistics( hkOstream& outStream, int reportLevel )
{
    checkAllThreadsCapturedSameNumFrames();

    hkArray<ThreadRootNodes> threadRoots;

    for (int i = 0; i < m_frameInfos.getSize(); ++i )
    {
        Node* node =  hkMonitorStreamAnalyzer::makeStatisticsTreeForMultipleFrames( i, false );
        ThreadRootNodes root;
        root.m_node = node;
        threadRoots.pushBack(root);
    }

    hkMonitorStreamAnalyzer::writeStatisticsDetails( outStream, threadRoots, m_numThreads, reportLevel, m_nodeIdForFrameOverview, true );

    for ( int i = 0; i < threadRoots.getSize();i++ )
    {
        delete threadRoots[i].m_node;
    }
}

void hkMonitorStreamAnalyzer::writeStatisticsForFrames(hkOstream& outStream, int numFrames, int reportLevel /* = REPORT_ALL */, _In_opt_ const hkArrayView<hkGpuTraceResult>* traceResults /* = HK_NULL */)
{
    checkAllThreadsCapturedSameNumFrames();

    hkArray<ThreadRootNodes> perThreadNodes;
    DrawCallHandleMap drawCallHandleMappings;

    hkMonitorStream::CommandStreamConfig config;
    for (int threadId = 0; threadId < m_frameInfos.getSize(); ++threadId )
    {
        const int count = hkMath::min2(numFrames, m_frameInfos[threadId].getSize());
        if (count == 0)
            continue;

        Node* rootNode = new Node( HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY );
        rootNode->m_children.setSize(count);

        for (int i = 0; i < count; ++i)
        {
            hkMonitorStreamFrameInfo& currentFrameInfo = m_frameInfos[threadId][i];

            const char* start = m_data.begin() + currentFrameInfo.m_frameStreamStart;
            const char* end   = m_data.begin() + currentFrameInfo.m_frameStreamEnd;

            rootNode->m_children[i] = makeStatisticsTreeForSingleFrame( config, start, end, currentFrameInfo, currentFrameInfo.m_heading, false, &drawCallHandleMappings );
        }

        ThreadRootNodes root;
        root.m_node = rootNode;
        perThreadNodes.pushBack(root);
    }

    hkMonitorStreamAnalyzer::writeStatisticsDetails(
        outStream, perThreadNodes, m_numThreads, reportLevel, m_nodeIdForFrameOverview, true, &drawCallHandleMappings, traceResults );

    for ( int i = 0; i < perThreadNodes.getSize();i++ )
    {
        delete perThreadNodes[i].m_node;
    }
}

void hkMonitorStreamAnalyzer::writeGpuStatistics(hkOstream& outStream, int numFrames, _In_reads_(numFrames) hkArrayView<hkGpuTraceResult>* traceResultsPerFrame, hkArrayView<hkReflect::Var> rootMetadata, hkArrayView<hkStringPtr> ignoredNames)
{
    checkAllThreadsCapturedSameNumFrames();

    hkArray<Tree*> nodes;
    DrawCallHandleMap drawCallHandleMappings;
    hkMonitorStream::CommandStreamConfig config;
    for (int threadId = 0; threadId < m_frameInfos.getSize(); ++threadId )
    {
        const int count = hkMath::min2(numFrames, m_frameInfos[threadId].getSize());

        Tree* rootNode = new Tree();
        rootNode->m_children.setSize(count);

        for (int i = 0; i < count; ++i)
        {
            hkMonitorStreamFrameInfo& currentFrameInfo = m_frameInfos[threadId][i];

            const char* start = m_data.begin() + currentFrameInfo.m_frameStreamStart;
            const char* end   = m_data.begin() + currentFrameInfo.m_frameStreamEnd;

            rootNode->m_children[i] = makeStatisticsTreeForSingleFrame( config, start, end, currentFrameInfo, currentFrameInfo.m_heading, false, &drawCallHandleMappings );
        }

        accumulateMetaData( rootNode, rootMetadata );
        nodes.pushBack(rootNode);
    }

    hkInplaceArray<Node*, 16> topLevelNodes;
    {
        // For each frame
        for (int frame=0; frame<nodes[0]->m_children.getSize(); ++frame)
        {
            // Reserve enough elements because hkSearchTopLevelNodesRec inserts the found elements in the correct order.
            topLevelNodes.setSize(traceResultsPerFrame[frame].getSize(), HK_NULL);

            outStream.printf("***** GPU Frame-%i ******\n", frame);

            // For each thread
            for (int threadId = 0; threadId < nodes.getSize(); ++threadId)
            {
                Tree* node = nodes[threadId];
                if (frame < node->m_children.getSize())
                {
                    hkMonitorStreamAnalyzer_searchTopLevelNodesRec( node->m_children[frame], traceResultsPerFrame[frame], topLevelNodes );
                }
            }

            // Output GPU details for all top level nodes found for this frame
            for (int i=0; i<topLevelNodes.getSize(); ++i)
            {
                Node* current = topLevelNodes[i];
                if (current != HK_NULL)
                {
                    hkMonitorStreamAnalyzer_writeNodeRec( outStream, current, 1, 1.0f, ignoredNames, &drawCallHandleMappings, traceResultsPerFrame? &( traceResultsPerFrame[frame] ) : HK_NULL );
                }
            }

            topLevelNodes.clear();
        }
    }

    for (int i=0; i<nodes.getSize(); ++i)
    {
        delete nodes[i];
    }
}


namespace
{
    struct Indenter
    {
        hkArray<char>::Debug m_array;

        Indenter()
        {
            m_array.reserve(64);
            m_array.pushBack(0);
        }
        const char* get() const
        {
            HK_ASSERT_NO_MSG(0x2d7d3f27, m_array.getSize());
            return m_array.begin();
        }
        void left()
        {
            m_array.popBack();
            m_array.popBack();
            m_array.back() = 0;
        }
        void right()
        {
            m_array.back() = ' ';
            m_array.pushBack(' ');
            m_array.pushBack(0);
        }
        void clear()
        {
            m_array.setSize(1);
            m_array[0] = 0;
        }
    };
}

void hkMonitorStreamAnalyzer::writeRawStatistics( hkOstream& os )
{
    hkMonitorStream::CommandStreamConfig streamConfig;
    checkAllThreadsCapturedSameNumFrames();
    Indenter indenter;
    int numFrames = m_frameInfos[0].getSize();
    os.printf("StatisticsDumpInfo(num_threads=%i, num_frames=%i)\n", m_numThreads, numFrames );

    for( int frameIndex = 0; frameIndex < numFrames; ++frameIndex )
    {
        for( int threadIndex = 0; threadIndex < m_frameInfos.getSize(); ++threadIndex )
        {
            const hkArrayBase< hkMonitorStreamFrameInfo >& frameInfos = m_frameInfos[threadIndex];
            const hkMonitorStreamFrameInfo& frame = frameInfos[frameIndex];

            os.printf("\n\n\n\nFrameInfo(heading='%s', frame=%i, thread_id=%i, time_counter=%i)\n", frame.m_heading.cString(), frameIndex, frame.m_threadId, (int)frame.m_absoluteTimeCounter);

            const char* cur = m_data.begin() + frame.m_frameStreamStart;
            const char* end   = m_data.begin() + frame.m_frameStreamEnd;

            indenter.clear();
            while( cur < end )
            {
            const char* curBeforeString = cur;
            const char* string = hkMonitorStream::readCommandString(&cur, streamConfig);
            cur = curBeforeString;

                switch(string[0])
            {
                case 'F': // Frame id
                {
                        hkMonitorStream::FrameIdCommand com;
                        com.read(&cur, streamConfig);
                        break;
                    }
                case 'R': // timer begin with self
                case 'T': // timer begin
                    {
                        if ( string[1] == 't' )
                        {
                        hkMonitorStream::TimerCommand com;
                        com.read(&cur, streamConfig);

                            #if HK_ENABLE_EXTRA_TIMER
                                os.printf("%sTimerBegin('%s',%u,%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1 );
                            #else
                                os.printf("%sTimerBegin('%s',%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0 );
                            #endif
                            indenter.right();
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x70220c6d, string[1] == 'g');
                            hkMonitorStream::TimerDrawCallCommand com;
                            com.read(&cur, streamConfig);
                            #if HK_ENABLE_EXTRA_TIMER
                                os.printf("%sTimerBegin('%s',%u,%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1 );
                            #else
                                os.printf("%sTimerBegin('%s',%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0 );
                            #endif
                            indenter.right();
                        }
                        break;
                    }
                    case 'E': // timer end
                    {
                        if ( string[1] == 't' )
                        {
                            hkMonitorStream::TimerCommand com;
                            com.read(&cur, streamConfig);

                            indenter.left();
                            #if HK_ENABLE_EXTRA_TIMER
                                os.printf("%sTimerEnd('%s',%llu,%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1 );
                            #else
                                os.printf("%sTimerEnd('%s',%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0 );
                            #endif
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x70220c6d, string[1] == 'g');
                            hkMonitorStream::TimerDrawCallCommand com;
                            com.read(&cur, streamConfig);
                            indenter.left();
                            #if HK_ENABLE_EXTRA_TIMER
                                os.printf("%sTimerEnd('%s',%llu,%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1 );
                            #else
                                os.printf("%sTimerEnd('%s',%llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0 );
                            #endif
                        }
                        break;
                    }
                    case 'Y': // multi timer begin
                    {
                        hkMonitorStream::Command com;
                        com.read(&cur, streamConfig);

                        indenter.right();
                        os.printf("%sMultiTimerBegin('%s')\n", indenter.get(), com.m_commandAndMonitor+2 );
                        break;
                    }

                    case 'X': // timer with tag
                    case 'G': // timer with tag
                        {
                            hkMonitorStream::TagCommand com;
                            com.read(&cur, streamConfig);
                            if ( string[0] =='X')
                            {
                                os.printf("%sPlaceTag('%x')\n", indenter.get(), com.m_tag );
                            }
                            else
                            {
                                os.printf("%sJumpToTag('%x')\n", indenter.get(), com.m_tag );
                            }

                            break;
                        }

                    case 'W': // multi timer end
                    {
                        hkMonitorStream::MultiTimerCommand com;
                        com.read(&cur, streamConfig);

                        HK_ASSERT_NO_MSG(0x70220c6d, com.m_timerCommand.m_commandAndMonitor[1] == 't');
                        indenter.left();
                        #if HK_ENABLE_EXTRA_TIMER
                        os.printf("%sMultiTimerEnd('%s',%u,%llu,%u)\n", indenter.get(),
                                  com.m_timerCommand.m_commandAndMonitor+2, com.m_timerCommand.m_time0, com.m_timerCommand.m_time1, com.m_callCount );
                        #else
                            os.printf("%sMultiTimerEnd('%s',%llu,%u)\n", indenter.get(),
                                  com.m_timerCommand.m_commandAndMonitor+2, com.m_timerCommand.m_time0, com.m_callCount );
                        #endif
                        break;
                    }
                    case 'O': // object name  (use tags instead)
                    {
                        hkMonitorStream::TimerBeginObjectNameCommand com;
                        com.read(&cur, streamConfig);
                        break;
                    }

                    case 'L': // timer list begin
                    {
                        hkMonitorStream::TimerBeginListCommand com;
                        com.read(&cur, streamConfig);

                        HK_ASSERT_NO_MSG(0x70220c6d, com.m_commandAndMonitor[1] == 't');
                        HK_ASSERT_NO_MSG(0x70220c6d, com.m_nameOfFirstSplit[1] == 't');
                        #if HK_ENABLE_EXTRA_TIMER
                        os.printf("%sTimerBegin('%s', %u, %llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1);
                        os.printf("%sTimerSplit('%s', %u, %llu)\n", indenter.get(), com.m_nameOfFirstSplit+2, com.m_time0, com.m_time1);
                        #else
                            os.printf("%sTimerBegin('%s', %llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0);
                            os.printf("%sTimerSplit('%s', %llu)\n", indenter.get(), com.m_nameOfFirstSplit+2, com.m_time0);
                        #endif

                        indenter.right();
                        break;
                    }
                    case 'S': // split list
                    {
                        hkMonitorStream::TimerCommand com;
                        com.read(&cur, streamConfig);

                        HK_ASSERT_NO_MSG(0x70220c6d, com.m_commandAndMonitor[1] == 't');
                        indenter.left();
                        #if HK_ENABLE_EXTRA_TIMER
                        os.printf("%sTimerSplit('%s', %llu, %llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1);
                        #else
                            os.printf("%sTimerSplit('%s', %llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0);
                        #endif
                        indenter.right();
                        break;
                    }
                    case 'l': // list end
                    {
                        hkMonitorStream::TimerCommand com;
                        com.read(&cur, streamConfig);

                        HK_ASSERT_NO_MSG(0x70220c6d, com.m_commandAndMonitor[1] == 't');
                        indenter.left();
                        #if HK_ENABLE_EXTRA_TIMER
                        os.printf("%sTimerEnd('%s', %llu, %llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0, com.m_time1);
                        #else
                            os.printf("%sTimerEnd('%s', %llu)\n", indenter.get(), com.m_commandAndMonitor+2, com.m_time0);
                        #endif
                        break;
                    }
                    case 'M':
                    {
                        hkMonitorStream::AddValueCommand com;
                        com.read(&cur, streamConfig);

                        os.printf("%sAddValue(%s,%f)\n", indenter.get(), com.m_commandAndMonitor+1, com.m_value);
                        break;
                    }
                    case 'P':
                    {
                        HK_ASSERT_NO_MSG(0x7a28795d, 0);
                        break;
                    }
                    case 'p':
                    {
                        HK_ASSERT_NO_MSG(0x43e6b574, 0);
                        break;
                    }
                    case 'A':
                    {
                        hkMonitorStream::AddStructCommand com;
                        com.read(&cur, streamConfig);
                        break;
                    }
                    case 'H':
                    {
                        hkMonitorStream::AddGpuHandleCommand com;
                        com.read(&cur, streamConfig);
                        break;
                    }
                    case 'N': // nop
                    {
                        hkMonitorStream::Command com;
                        com.read(&cur, streamConfig);
                        break;
                    }
                    case 'm': // memory
                    {
                        hkMonitorStream::MemoryCommand com;
                        com.read(&cur, streamConfig);

                        int size = com.m_sizeAndFlags&0x7fffffff;
                        hkBool32 isFree = com.m_sizeAndFlags & ~0x7fffffff;
                        const char* what = isFree ? "Free" : "Alloc";
                        os.printf("%s%s%s(ptr=0x%p, nbytes=%i)\n", indenter.get(), what, com.m_commandAndMonitor+1, com.m_ptr, size);
                        break;
                    }
                    case 'Q': // Pause/unpause
                    {
                        hkMonitorStream::TimerCommand com;
                        com.read(&cur, streamConfig);
                        break;
                    }
                    default:
                    {
                        HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
                        return;
                    }
                }
            }
        }
    }
}

static void hkMonitorStreamAnalyzer_writeJsonBlock(hkOstream& outStream, _In_z_ const char* name, _In_z_ const char* phaseTag, int threadIndex, hkUint64 timestamp)
{
    outStream.printf("\t{\n");
    outStream.printf("\t\t\"name\": \"%s\",\n", name);
    outStream.printf("\t\t\"pid\": \"1\",\n");
    outStream.printf("\t\t\"tid\": \"%d\",\n", threadIndex);
    outStream.printf("\t\t\"ts\": \"%llu\",\n", timestamp);
    outStream.printf("\t\t\"ph\": \"%s\"\n", phaseTag);
    outStream.printf("\t},\n");
}

void hkMonitorStreamAnalyzer::writeJsonStatistics( hkOstream& outStream, hkUint64 ticksPerSecond )
{
    checkAllThreadsCapturedSameNumFrames();

    double timerFactor = 1e6 / static_cast<double>(ticksPerSecond);

    hkUint32 tmp = 0;

    outStream.printf("[\n");

    for( int threadIndex = 0; threadIndex < m_frameInfos.getSize(); ++threadIndex )
    {
        const hkArrayBase< hkMonitorStreamFrameInfo >& frameInfos = m_frameInfos[threadIndex];
        for( int frameIndex = 0; frameIndex < frameInfos.getSize(); ++frameIndex )
        {
            const hkMonitorStreamFrameInfo& frame = frameInfos[frameIndex];

            const void* cur = m_data.begin() + frame.m_frameStreamStart;
            const void* end   = m_data.begin() + frame.m_frameStreamEnd;

            while( cur < end )
            {
                switch( static_cast<const hkMonitorStream::Command*>(cur)->m_commandAndMonitor[0] )
                {
                case 'F': // Frame id
                    {
                        const hkMonitorStream::FrameIdCommand* com = static_cast<const hkMonitorStream::FrameIdCommand*>( cur );
                        cur = com + 1;
                        break;
                    }

                case 'R': // timer begin with self
                case 'T': // timer begin
                    {
                        if ( static_cast<const hkMonitorStream::Command*>(cur)->m_commandAndMonitor[1] == 't' )
                        {
                            const hkMonitorStream::TimerCommand* com = static_cast<const hkMonitorStream::TimerCommand*>( cur );
                            hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_commandAndMonitor+2, "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                            cur = com + 1;
                        }
                        else
                        {
                            const hkMonitorStream::TimerDrawCallCommand* com = static_cast<const hkMonitorStream::TimerDrawCallCommand*>( cur );
                            HK_ASSERT_NO_MSG(0x1bc5f5c0, com->m_commandAndMonitor[1] == 'g');
                            hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_commandAndMonitor+2, "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                            cur = com + 1;
                        }
                        break;
                    }
                case 'E': // timer end
                    {
                        if ( static_cast<const hkMonitorStream::Command*>(cur)->m_commandAndMonitor[1] == 't' )
                        {
                            const hkMonitorStream::TimerCommand* com = static_cast<const hkMonitorStream::TimerCommand*>( cur );
                            hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_commandAndMonitor+2, "E", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                            cur = com+1;
                        }
                        else
                        {
                            const hkMonitorStream::TimerDrawCallCommand* com = static_cast<const hkMonitorStream::TimerDrawCallCommand*>( cur );
                            HK_ASSERT_NO_MSG(0x69f773ef, com->m_commandAndMonitor[1] == 'g');
                            hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_commandAndMonitor+2, "E", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                            cur = com+1;
                        }
                        break;
                    }
                case 'X': // timer with tag
                case 'G': // timer with tag
                    {
                        const hkMonitorStream::TagCommand* com = static_cast<const hkMonitorStream::TagCommand*>( cur );
                        //hkMonitorStreamAnalyzer_writeJsonBlock(outStream, (const char*)(com+1), "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        cur = com+1;
                        break;
                    }

                case 'Y': // multi timer begin
                    {
                        const hkMonitorStream::Command* com = static_cast<const hkMonitorStream::Command*>( cur );
                        cur = com + 1;
                        break;
                    }
                case 'W': // multi timer end
                    {
                        const hkMonitorStream::MultiTimerCommand* com = static_cast<const hkMonitorStream::MultiTimerCommand*>( cur );
                        HK_ASSERT_NO_MSG(0x5778a6da, com->m_timerCommand.m_commandAndMonitor[1] == 't');
                        cur = com + 1;
                        break;
                    }
                case 'O': // object name
                    {
                        const hkMonitorStream::TimerBeginObjectNameCommand* com = static_cast<const hkMonitorStream::TimerBeginObjectNameCommand*>( cur );
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, (const char*)(com+1), "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        cur = (const char *)(com + 1) + com->m_objectNameSize;
                        cur = (const char*)( HK_NEXT_MULTIPLE_OF(sizeof(void*), (hkUlong)cur) );
                        break;
                    }
                case 'L': // timer list begin
                    {
                        const hkMonitorStream::TimerBeginListCommand* com = static_cast<const hkMonitorStream::TimerBeginListCommand*>( cur );
                        HK_ASSERT_NO_MSG(0x6406e903, com->m_commandAndMonitor[1] == 't');
                        HK_ASSERT_NO_MSG(0x346a20b3, com->m_nameOfFirstSplit[1] == 't');
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_commandAndMonitor+2, "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_nameOfFirstSplit+2, "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        cur = com + 1;
                        break;
                    }
                case 'S': // split list
                    {
                        const hkMonitorStream::TimerCommand* com = static_cast<const hkMonitorStream::TimerCommand*>( cur );
                        HK_ASSERT_NO_MSG(0x2a1ac051, com->m_commandAndMonitor[1] == 't');
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, "", "E", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, com->m_commandAndMonitor+2, "B", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        cur = com + 1;
                        break;
                    }
                case 'l': // list end
                    {
                        const hkMonitorStream::TimerCommand* com = static_cast<const hkMonitorStream::TimerCommand*>( cur );
                        HK_ASSERT_NO_MSG(0x336699e3, com->m_commandAndMonitor[1] == 't');
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, "", "E", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        hkMonitorStreamAnalyzer_writeJsonBlock(outStream, "", "E", threadIndex, static_cast<hkUint64>(com->m_time0 * timerFactor));
                        cur = com + 1;
                        break;
                    }
                case 'M':
                    {
                        const hkMonitorStream::AddValueCommand* com = static_cast<const hkMonitorStream::AddValueCommand*>( cur );
                        cur = com + 1;
                        break;
                    }
                case 'P':
                case 'p':
                case 'N':
                    {
                        const hkMonitorStream::Command* com = static_cast<const hkMonitorStream::Command*>( cur );
                        cur = com + 1;
                        break;
                    }
                case 'A':
                    {
                        const hkMonitorStream::AddStructCommand* com = static_cast<const hkMonitorStream::AddStructCommand*>( cur );
                        cur = reinterpret_cast<const char*>(HK_NEXT_MULTIPLE_OF(com->m_type->getAlignOf(), hkUlong(com + 1))) + com->m_type->getSizeOf();
                        cur = (const char*)(HK_NEXT_MULTIPLE_OF(sizeof(void*), (hkUlong)cur));
                        break;
                    }
                case 'H':
                    {
                        const hkMonitorStream::AddGpuHandleCommand* com = static_cast<const hkMonitorStream::AddGpuHandleCommand*>( cur );
                        cur = com + 1;
                        break;
                    }
                case 'm': // memory
                    {
                        const hkMonitorStream::MemoryCommand* com = static_cast<const hkMonitorStream::MemoryCommand*>( cur );
                        cur = com + 1;
                        break;
                    }
                case 'Q': // pause/unpause
                    {
                        const hkMonitorStream::TimerCommand* com = static_cast<const hkMonitorStream::TimerCommand*>( cur );
                        cur = com + 1;
                        break;
                    }
                default:
                    {
                        HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
                        return;
                    }
                }

                ++tmp;
            }
        }
    }

    outStream.printf("{}]\n");
}



//
// Drawing utilities
//

inline _Ret_maybenull_ hkMonitorStreamAnalyzer::Node* hkMonitorStreamAnalyzer_findChildByAbsoluteTime(_Inout_ hkMonitorStreamAnalyzer::Node* currentNode, double sampleTime, int absoluteTimeIndex)
{

    // preselect the children
    int start = 0;
    int end = currentNode->m_children.getSize();
    while ( end - start > 2 )
    {
        int mid = (end+start)>>1;
        hkMonitorStreamAnalyzer::Node* child = currentNode->m_children[mid];
        if ( child->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_SINGLE )
        {
            break;
        }

        if ( child->m_absoluteStartTime > sampleTime )
        {
            end = mid;
        }
        else
        {
            start = mid;
        }
    }

    for (int i = start; i < end; ++i )
    {
        hkMonitorStreamAnalyzer::Node* child = currentNode->m_children[i];
        if ( child->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_SINGLE )
        {
            continue;
        }

        double endTime = child->m_absoluteStartTime + child->m_threadData[absoluteTimeIndex].m_value;
        if ( sampleTime <= endTime )
        {
            if ( sampleTime >= child->m_absoluteStartTime )
            {
                return child;
            }
            return HK_NULL;
        }
    }
    return HK_NULL;
}



_Ret_maybenull_ hkMonitorStreamAnalyzer::Node* getNodeAtSample(_Inout_ hkMonitorStreamAnalyzer::Node* currentNode, _In_opt_ hkMonitorStreamAnalyzer::Node* lastHit, double sampleTime, int absoluteTimeIndex)
{
    HK_ASSERT_NO_MSG(0x43cedd3e, currentNode->m_absoluteStartTime <= sampleTime);
    if ( lastHit  )
    {
        double  endTime = lastHit->m_absoluteStartTime + lastHit->m_threadData[absoluteTimeIndex].m_value;
        if ( sampleTime <= endTime )
        {
            currentNode = lastHit;
        }
    }

    while ( currentNode )
    {

        hkMonitorStreamAnalyzer::Node* child = hkMonitorStreamAnalyzer_findChildByAbsoluteTime( currentNode, sampleTime, absoluteTimeIndex );
        if ( child != HK_NULL )
        {
            currentNode = child;
            continue;
        }
        break;
    }
    return currentNode;
}


static void HK_CALL outputStatsForFrame(
    _Inout_ hkMonitorStreamAnalyzer::Node* root,
    hkDouble64 startTime,
    hkReal timeInc,
    int maxFrames,
    hkArray<hkMonitorStreamAnalyzer::Node*>& timerNodesAtTicks,
    int absoluteTimeIndex )
{
    hkMonitorStreamAnalyzer::Node* firstNode = root->m_children[0];
    int fni = 0;
    while(firstNode && (firstNode->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_SINGLE))
    {
        ++fni;
        firstNode = fni < root->m_children.getSize() ? root->m_children[fni] : HK_NULL;
    }

    hkMonitorStreamAnalyzer::Node* lastNode  = root->m_children.back();
    fni = root->m_children.getSize() - 1;
    while(lastNode && (lastNode->m_type == hkMonitorStreamAnalyzer::Node::NODE_TYPE_SINGLE))
    {
        --fni;
        lastNode = fni >= 0 ? root->m_children[fni] : HK_NULL;
    }

    if (!lastNode || !firstNode)
        return;

    // Wait until the current time starts
    // adding null names makes the entry in the texture the background texture
    double currentTimeNorm = firstNode->m_absoluteStartTime - startTime;
    double sampleTimeNorm = 0;
    // renormalize times to around 0 (As when rec a demo that has been running for some time
    // the inc may be say 4.0e-5, whereas time may be in the 1000s, thus never incing at all..
    int maxLoops = maxFrames;
    while ( maxLoops && (currentTimeNorm > sampleTimeNorm) )
    {
        sampleTimeNorm += timeInc;
        timerNodesAtTicks.pushBack(HK_NULL);
        --maxLoops;
    }

    double endTimeNorm = lastNode->m_absoluteStartTime + lastNode->m_threadData[absoluteTimeIndex].m_value - startTime;

    // reset root (frame, as it normally does not have any proper timer info, in case the lookup needs to come back up to parent)
    root->m_absoluteStartTime = firstNode->m_absoluteStartTime;
    root->m_threadData[absoluteTimeIndex].m_value = float((endTimeNorm + startTime) - root->m_absoluteStartTime + 1.0f);

    hkMonitorStreamAnalyzer::Node* node = HK_NULL;
    while ( maxLoops && (endTimeNorm > sampleTimeNorm) ) // more to sample
    {
        node = getNodeAtSample( root, node, sampleTimeNorm + startTime, absoluteTimeIndex );
        if (node)
        {
            timerNodesAtTicks.pushBack(node);
        }

        sampleTimeNorm += timeInc;
        --maxLoops;
    }

}


struct TargaHeader2
{
    unsigned char  IDLength;
    unsigned char  ColormapType;
    unsigned char  ImageType;
    unsigned char  ColormapSpecification[5];
    unsigned short XOrigin;
    unsigned short YOrigin;
    unsigned short ImageWidth;
    unsigned short ImageHeight;
    unsigned char  PixelDepth;
    unsigned char  ImageDescriptor;
};

#define GET_ENDIAN_SWAPPED_16(x)  ((((x) & 0xff) << 8) | (( (x) & 0xff00) >> 8))

bool saveToTGA(_In_reads_(width * height) const int* data, hkOstream& s, int width, int height)
{
    // Header
    TargaHeader2 tga;
    hkString::memSet(&tga, 0, sizeof(tga));
    tga.ImageType  = 2; // raw

#if HK_ENDIAN_BIG
    tga.ImageHeight = (unsigned short)GET_ENDIAN_SWAPPED_16(height);
    tga.ImageWidth = (unsigned short)GET_ENDIAN_SWAPPED_16(width);
    for (int h=0; h< height;++h)
    {
        for (int w=0;w<width; ++w)
        {
            char* datac = (char*)( &data[h*width + w] );
            char r = datac[0];
            char g = datac[1];
            char b = datac[2];
            char a = datac[3];
            datac[0] = a;
            datac[1] = b;
            datac[2] = g;
            datac[3] = r;
        }
    }
#else
    tga.ImageHeight = (unsigned short)height;
    tga.ImageWidth = (unsigned short)width;
#endif

    tga.PixelDepth = (unsigned char)32;
    s.write((char*)&tga, sizeof(tga));

    s.write((char*)data, height * width * 4);
    return true;
}


static const hkUint32 number_0[7] =
{
    0x0000800,
    0x0008080,
    0x0080008,
    0x0080008,
    0x0080008,
    0x0008080,
    0x0000800,
};
static const hkUint32 number_1[7] =
{
    0x0008800,
    0x0080800,
    0x0000800,
    0x0000800,
    0x0000800,
    0x0000800,
    0x0088888,
};
static const hkUint32 number_2[7] =
{
    0x0008880,
    0x0080008,
    0x0000008,
    0x0000080,
    0x0000800,
    0x0008000,
    0x0088888,
};
static const hkUint32 number_3[7] =
{
    0x0088880,
    0x0000008,
    0x0000008,
    0x0000888,
    0x0000008,
    0x0000008,
    0x0088880,
};
static const hkUint32 number_4[7] =
{
    0x0000080,
    0x0000880,
    0x0008080,
    0x0008080,
    0x0088888,
    0x0000080,
    0x0000080,
};
static const hkUint32 number_5[7] =
{
    0x0008888,
    0x0008000,
    0x0008000,
    0x0008880,
    0x0000008,
    0x0000008,
    0x0008880,
};
static const hkUint32 number_6[7] =
{
    0x0000880,
    0x0008000,
    0x0080000,
    0x0080880,
    0x0088008,
    0x0080008,
    0x0008880,
};
static const hkUint32 number_7[7] =
{
    0x0088888,
    0x0000008,
    0x0000080,
    0x0000800,
    0x0008000,
    0x0008000,
    0x0008000,
};

static const hkUint32 number_8[7] =
{
    0x0008880,
    0x0080008,
    0x0080008,
    0x0008880,
    0x0080008,
    0x0080008,
    0x0008880,
};

static const hkUint32 number_9[7] =
{
    0x0008880,
    0x0080008,
    0x0080008,
    0x0008888,
    0x0000008,
    0x0000080,
    0x0088800,
};

static const hkUint32* numbers[10] = {
    &number_0[0],
    &number_1[0],
    &number_2[0],
    &number_3[0],
    &number_4[0],
    &number_5[0],
    &number_6[0],
    &number_7[0],
    &number_8[0],
    &number_9[0]
};


static void HK_CALL drawDigit( int nr, int currentY, int outputPixelWidth, _Inout_updates_(_Inexpressible_()) int* texture )
{
    const hkUint32* pattern = numbers[nr];
    for ( int x =0; x < 8; x++)
    {
        for (int y=0; y < 7; y++)
        {
            if ( (pattern[6-y]<<(4*x))&0xf0000000 )
            {
                texture[ x + y * outputPixelWidth] = 0xFF000000;
            }
        }
    }
}

static void HK_CALL drawNumber( int nr, int currentY, int outputPixelWidth, _Inout_updates_(_Inexpressible_()) int* texture )
{
    int x = 0;
    for (int i=1000; i >= 1; i=i/10)
    {
        int digit = (nr/i)%10;
        drawDigit( digit, currentY, outputPixelWidth, texture + x);
        x += 7;
    }
}

static hkMonitorStreamAnalyzer::Node* HK_CALL getNodeSampledAtTick( hkMonitorStreamFrameInfo& info,
                                            int frameIndex, int tick,
                                            const hkArrayBase<hkMonitorStreamAnalyzer::Node*>& nodes,
                                            int maxX,
                                            hkDouble64 frameTime,
                                            hkDouble64 absoluteFrameStartTimes )
{
    hkReal timeIncrement = hkReal(frameTime / (hkReal)maxX);
    hkArray<hkMonitorStreamAnalyzer::Node*> timerNodesAtTicks;
    timerNodesAtTicks.reserveExactly( maxX );

    hkMonitorStreamAnalyzer::Node* sampledNode  = HK_NULL;

    if (nodes[frameIndex]->m_children.getSize() > 0)
    {
        timerNodesAtTicks.clear();

        HK_ASSERT(0x8f258165, info.m_absoluteTimeCounter != hkMonitorStreamFrameInfo::ABSOLUTE_TIME_NOT_TIMED, \
                "You cannot draw statistics unless one of your timers is absolute time");

        int absoluteTimeIndex = (info.m_absoluteTimeCounter == hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0) ? info.m_indexOfTimer0 : info.m_indexOfTimer1;

        outputStatsForFrame(nodes[frameIndex], absoluteFrameStartTimes, timeIncrement, maxX, timerNodesAtTicks, absoluteTimeIndex );

        int numSamplesThisFrame = timerNodesAtTicks.getSize();

        if ( tick < numSamplesThisFrame )
        {
            sampledNode = timerNodesAtTicks[tick];
        }
    }

    return sampledNode;
}

static void HK_CALL drawStatistics(const hkMonitorStreamFrameInfo& info,
    int frameIndex,
    const hkArrayBase<hkMonitorStreamAnalyzer::Node*>& nodes,
    _Inout_updates_(_Inexpressible_()) int* texture,
    int height,
    hkMonitorStreamColorTable& colorTable,
    int pixelWidth,
    int maxX,
    hkReal frameTime,
    hkDouble64 absoluteFrameStartTimes,
    hkPointerMap<const char*, hkColor::Argb>& unknownColorMap)
{
    hkMonitorStreamAnalyzer::Node* rootNode = nodes[frameIndex];
    if (rootNode->m_children.getSize() == 0)
    {
        return;
    }

    HK_ASSERT(0x8f258165, info.m_absoluteTimeCounter != hkMonitorStreamFrameInfo::ABSOLUTE_TIME_NOT_TIMED, \
               "You cannot draw statistics unless one of your timers is absolute time");

    hkPointerMap<const char*, hkColor::Argb> colorMap;
    hkReal timeIncrement = frameTime / (hkReal)maxX;
    hkArray<hkMonitorStreamAnalyzer::Node*> timerNodesAtTicks;
    timerNodesAtTicks.reserveExactly( maxX );

    {
        int absoluteTimeIndex = (info.m_absoluteTimeCounter == hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0) ?
                                info.m_indexOfTimer0 : info.m_indexOfTimer1;
        outputStatsForFrame(rootNode, absoluteFrameStartTimes, timeIncrement, maxX, timerNodesAtTicks, absoluteTimeIndex);
    }

    int numSamplesThisFrame = timerNodesAtTicks.getSize();

    hkMonitorStreamAnalyzer::Node* prevNode = HK_NULL;
    hkColor::Argb prevColor = hkColor::WHITE;
    for (int j = 0; (j < numSamplesThisFrame) && (j < maxX); j++ )
    {
        hkMonitorStreamAnalyzer::Node* node = timerNodesAtTicks[j];

        // Leave the background texture when there is no timer info
        if (node == HK_NULL || node->m_type != hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER)
        {
            continue;
        }

        hkColor::Argb color;
        if (node == prevNode)
        {
            color = prevColor;
        }
        else
        {
            prevNode = node;

            // Use white for timers not registered in the color map
            color = hkColor::WHITE;

            // For this node - try to match the name of the deepest known timer against
            // the color table.  Use 2 maps to speed the process.
            while (node != HK_NULL)
            {
                if ( node->m_flags.anyIsSet( hkMonitorStreamAnalyzer::Node::FLAGS_TIME_IS_NOT_ABSOLUTE ) )
                {
                    // this is a multi-timer, we ignore it and fall through to parent node
                    node = node->m_parent;
                    continue;
                }

                const char* name = node->m_name;
                if ( colorMap.get(name, &color).isFailure() )
                {
                    if (unknownColorMap.get(name, &color).isSuccess() )
                    {
                        // If the color is unknown try the parent
                        node = node->m_parent;
                        continue;
                    }

                    // Color not in cached map yet - look it up in the table
                    bool colorFound = false;
                    for (int i = 0; i < colorTable.m_colorPairs.getSize(); ++i)
                    {
                        if ( hkString::strCasecmp(colorTable.m_colorPairs[i].m_colorName, name) == 0)
                        {
                            color = colorTable.m_colorPairs[i].m_color;
                            colorFound = true;
                            break;
                        }
                    }
                    if ( colorFound )
                    {
                        colorMap.insert(name, color );
                        break; // Found color
                    }
                    else
                    {
                        unknownColorMap.insert(name, color);
                        node = node->m_parent;
                    }
                }
                else
                {
                    break; // Found color
                }
            }

            // Draw first sample of a node with a darkened version of the color to be able to see the boundaries
            // between nodes with the same color.
            prevColor = color;
            color = hkColor::darken(color, 1);
        }

        for (int k = 0; k < height; k++)
        {
            texture[ k * pixelWidth + j ] = color;
        }
    }
}

static inline hkReal __hkMin( hkReal a, hkReal b)
{
    return (a<b)?a:b;
}

static inline hkReal __hkMax( hkReal a, hkReal b)
{
    return (a>b)?a:b;
}

static inline double __hkMinD( double a, double  b)
{
    return (a<b)?a:b;
}

static inline double __hkMaxD( double  a, double  b)
{
    return (a>b)?a:b;
}

void hkMonitorStreamAnalyzer::checkAllThreadsCapturedSameNumFrames() const
{
    int numFrames = m_frameInfos[0].getSize();
    for (int i = 1; i < m_frameInfos.getSize(); ++i)
    {
        if ( m_frameInfos[i].getSize() != numFrames )
        {
            HK_WARN_ONCE(0xbebf8746, "Inconsistent number of captured frames between threads, timer analysis data may be corrupt" );
        }
    }
}

int hkMonitorStreamAnalyzer::ThreadDrawInput::computePerThreadHeightToFit( int textureHeight, int numFrames, int numThreads, int frameGap, int threadGap )
{
    int perFrameHeight = ( textureHeight / numFrames ) - frameGap;
    int perThreadHeight = ( perFrameHeight / numThreads ) - threadGap;
    return perThreadHeight;
}

static inline hkUint32 hkRoundUpPow2(hkUint32 n)
{
    n--;
    n |= n >> 1;
    n |= n >> 2;
    n |= n >> 4;
    n |= n >> 8;
    n |= n >> 16;
    n++;
    return n;
}

static void _rebase( hkArray<hkMonitorStreamAnalyzer::Node*>& rootNodes, double hdt, double dt)
{
    for (int n=0; n < rootNodes.getSize(); ++n)
    {
        if ( rootNodes[n]->hasValidStartTime() )
        {
            if (rootNodes[n]->m_absoluteStartTime < hdt)
                rootNodes[n]->m_absoluteStartTime += dt;
        }
        _rebase(rootNodes[n]->m_children, hdt, dt);
    }
}

static bool _needsRebase(hkArray<hkMonitorStreamAnalyzer::Node*>& rootNodes, double halfRange, double& maxSoFar )
{
    for (int n=0; n < rootNodes.getSize(); ++n)
    {
        if ( rootNodes[n]->hasValidStartTime() )
        {
            if (rootNodes[n]->m_absoluteStartTime < halfRange)
            {
                if ((maxSoFar - rootNodes[n]->m_absoluteStartTime) > halfRange)
                {
                    // low abs value, with over half the range to small value, small values are in fact overflows
                    return true;
                }
            }

            maxSoFar = __hkMaxD(maxSoFar, rootNodes[n]->m_absoluteStartTime);

        }
        if (_needsRebase(rootNodes[n]->m_children, halfRange, maxSoFar))
            return true;
    }
    return false;
}

void hkMonitorStreamAnalyzer::getTimerLimits(const hkArrayBase<ThreadRootNodes>& perThreadNodes, const ThreadDrawInput& input, hkArray< hkArray< hkMonitorStreamFrameInfo > >& frameInfos,
                                             hkDouble64& maxFrameTime,
                                             hkArray<hkDouble64>& startTimes)
{
    int numThreads = perThreadNodes.getSize();
    int frameEnd = input.m_frameStart + input.m_numFrames;
    int numFramesInStream = 0;
    for(int c = 0; c < perThreadNodes.getSize(); ++c)
    {
        if(perThreadNodes[c].m_node)
        {
            numFramesInStream = perThreadNodes[c].m_node->m_children.getSize();
            break;
        }
    }

    int numFrames = input.m_numFrames;

    if (frameEnd > numFramesInStream )
    {
        numFrames = numFramesInStream - input.m_frameStart;
        frameEnd  = numFramesInStream;
    }
    if ( numFrames <= 0)
    {
        return;
    }

    // Get first start time for the thread so that they are calibrated to
    // each other. Can be very noticeable on Xbox360 for instance.
    startTimes.setSize(input.m_numFrames, 0.0f);
    maxFrameTime = 0;

    {
        for (int j = input.m_frameStart; j < frameEnd; ++j)
        {
            int fzero = j -input.m_frameStart;
            int absoluteTimeIndex = (frameInfos[0][j].m_absoluteTimeCounter == hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0) ? frameInfos[0][j].m_indexOfTimer0 : frameInfos[0][j].m_indexOfTimer1;

            // This for loop accounts for wrap around in the unit32 timers, specifically for the timerbase (the timer value will be ok)
            // First we detect if the range is too much and we have some small timers, if so assume we need rebase
            // Note this limits the amount of time we can time for, but it is long enough for realtime
            {
                bool needRebase = false;
                hkMonitorStream::TimerValue fullRangeInt = hkMonitorStream::TimerValue(-1);
#ifdef HK_ARCH_X64
                if (!input.m_streamConfig->m_pointersAre64Bit)
                {
                    fullRangeInt = 0xffffffff;
                }
#endif
                double fullRange = double( fullRangeInt ) * ((frameInfos[0][j].m_absoluteTimeCounter == hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0) ? frameInfos[0][j].m_timerFactor0 : frameInfos[0][j].m_timerFactor1);
                double halfRange = fullRange * 0.5;
                double maxSoFar = 0.0;
                for (int i = 0; i < numThreads; ++i )
                {
                    if(perThreadNodes[i].m_node)
                    {
                        hkArray<Node*>& threadIframeJ = perThreadNodes[i].m_node->m_children[j]->m_children;
                        if (_needsRebase(threadIframeJ,halfRange,maxSoFar))
                        {
                            needRebase = true;
                            break;
                        }
                    }
                }
                if (needRebase)
                {
                    for (int ii = 0; ii < numThreads; ++ii )
                    {
                        if(perThreadNodes[ii].m_node)
                        {
                            hkArray<Node*>& threadIframeJ = perThreadNodes[ii].m_node->m_children[j]->m_children;
                            _rebase(threadIframeJ,halfRange,fullRange);
                        }
                    }
                }
            }

            hkArray<double> threadStartTimes(numThreads,HK_REAL_MAX);
            for (int i = 0; i < numThreads; ++i )
            {
                if(perThreadNodes[i].m_node)
                {
                    hkArray<Node*>& threadIframeJ = perThreadNodes[i].m_node->m_children[j]->m_children;

                    // take first node's start time (frame node not normally a proper timed node)
                    for (int iv=0; iv < threadIframeJ.getSize(); ++iv)
                    {
                        if ( threadIframeJ[iv]->hasValidStartTime() )
                        {
                            double st = threadIframeJ[iv]->m_absoluteStartTime;
                            threadStartTimes[i] = __hkMinD( st, threadStartTimes[i]);
                        }
                    }
                }
            }

            startTimes[fzero] = HK_REAL_MAX;
            for (int si = 0; si < numThreads; ++si )
            {
                startTimes[fzero] = hkMath::min2( threadStartTimes[si], startTimes[fzero] );
            }

            for (int iii = 0; iii < numThreads; ++iii )
            {
                if(perThreadNodes[iii].m_node)
                {
                    hkArray<Node*>& threadIframeJ = perThreadNodes[iii].m_node->m_children[j]->m_children;
                    for (int iv=0; iv < threadIframeJ.getSize(); ++iv)
                    {
                        if (threadIframeJ[iv]->m_type != Node::NODE_TYPE_SINGLE)
                        {
                            double endTime = threadIframeJ[iv]->m_absoluteStartTime + threadIframeJ[iv]->m_threadData[absoluteTimeIndex].m_value - startTimes[fzero];
                            maxFrameTime = hkMath::max2( endTime, maxFrameTime);
                        }
                    }
                }
            }
        }
    }

    if (input.m_limitStartTime > 0.0f)
    {
        for (int j = input.m_frameStart; j < frameEnd; ++j)
        {
            int fzero = j -input.m_frameStart;
            startTimes[fzero] += input.m_limitStartTime;
        }
    }

    if (input.m_limitFrameTime > 0.0f)
    {
        maxFrameTime = input.m_limitFrameTime;
    }
    else if (numFrames == 1)
    {
        // Timescale is assumed to be in microseconds i.e. timerFactor = 1e6 / ticksPerSec
        // Adjust frame time slowly if in visual mode

        hkReal maxFrameTimeRounded = hkRoundUpPow2( (int)(maxFrameTime / 1000.0f) + 1) * 1000.0f;

        if ( maxFrameTimeRounded > s_lastFrameTime * 3.0f )
        {
            maxFrameTime = maxFrameTimeRounded;
        }
        else if ( maxFrameTimeRounded < s_lastFrameTime * .3f )
        {
            maxFrameTime = maxFrameTimeRounded;
        }
        else
        {
            maxFrameTime = s_lastFrameTime + (maxFrameTimeRounded - s_lastFrameTime) * 0.05f;
        }

        //Clamp min to 60Hz frame
        maxFrameTime = hkMath::max2(16666, maxFrameTime);

        // Only set last when not doing zoom
        s_lastFrameTime  = hkReal(maxFrameTime);
    }
}

void hkMonitorStreamAnalyzer::writeStatisticsDetailsToTexture(const hkArrayBase<ThreadRootNodes>& perThreadNodes, const ThreadDrawInput& input, hkArray< hkArray< hkMonitorStreamFrameInfo > >& frameInfos, int*& texture, int& height, _Inout_opt_ SampleInfo* sampleInfo)
{
    bool capturedThreads = false;
    for(int c = 0; c < perThreadNodes.getSize(); ++c)
    {
        if(perThreadNodes[c].m_node)
        {
            capturedThreads = true;
            break;
        }
    }
    if (!capturedThreads)
        return; // nothing to do..

    hkDouble64 maxFrameTime = -1.0f;
    hkArray<hkDouble64> startTimes;
    getTimerLimits( perThreadNodes, input, frameInfos, maxFrameTime, startTimes );

    int numCaptured = frameInfos[0].getSize() - input.m_frameStart;
    int numFrames = input.m_numFrames > numCaptured ? numCaptured : input.m_numFrames;
    int frameEnd = input.m_frameStart + numFrames;

    int pixelHeightPerThread = input.m_heightPerThread + input.m_gapBetweenThreads;
    int pixelHeightPerFrame  = pixelHeightPerThread * perThreadNodes.getSize() + input.m_gapBetweenFrames;
    int pixelHeight          = pixelHeightPerFrame * numFrames;

    int currentY = pixelHeight;
    int startX   = numFrames > 1 ? 32 : 0 ;
    int maxX = input.m_outputPixelWidth-startX;

    int numTotalPixels = pixelHeight * (input.m_outputPixelWidth + 1);
    texture = hkAllocate<int>(numTotalPixels, HK_MEMORY_CLASS_DEMO);
    height = pixelHeight;
    hkString::memSet(texture, 0x00, numTotalPixels * 4);

    if (sampleInfo)
    {
        sampleInfo->m_maxSampleTime = hkReal(maxFrameTime);
    }

    hkPointerMap<const char*, hkColor::Argb> unknownColorMap;
    {
        for (int f = input.m_frameStart; f < frameEnd; f++ )
        {
            currentY -= input.m_gapBetweenFrames;

            if (numFrames > 1)
            {
                drawNumber( f, currentY, input.m_outputPixelWidth, &texture[ (currentY-pixelHeightPerThread) * input.m_outputPixelWidth] );
            }

            for (int i = 0; i < perThreadNodes.getSize(); ++i )
            {
                currentY -= pixelHeightPerThread;

                if(perThreadNodes[i].m_node)
                {
                    int* output = &texture[currentY * input.m_outputPixelWidth+startX];
                    drawStatistics(
                        frameInfos[i][0],
                        f,
                        perThreadNodes[i].m_node->m_children,
                        output,
                        input.m_heightPerThread,
                        *input.m_colorTable,
                        input.m_outputPixelWidth,
                        maxX,
                        (hkReal)maxFrameTime,
                        startTimes[f-input.m_frameStart],
                        unknownColorMap );
                }
            }
        }
    }

    HK_ASSERT_NO_MSG( 0xf0212343, currentY == 0);

    //
    //  Draw 1 msec lines
    //
    hkReal pixelsPerMs = hkReal((1000 * maxX) / maxFrameTime);
    int pixelOffset = int( hkReal( int(input.m_limitStartTime) % 1000) * 0.001f * pixelsPerMs );
    {
        if (pixelsPerMs > 5)
        {
            int x = startX - pixelOffset;
            int numMsBars = 0;
            while ( x < input.m_outputPixelWidth )
            {
                if (x >= 0)
                {
                    for (int y=0; y < pixelHeight;y++)
                    {
                        texture[y * input.m_outputPixelWidth+x] = 0xff0000ff;
                    }
                }
                numMsBars++;
                x = startX - pixelOffset + int( numMsBars * pixelsPerMs );
            }
        }
    }

    //
    //  Draw 60Hz frame markers
    //
    {
        int pixelsPerFrame = int(16666 * maxX / maxFrameTime);
        if (pixelsPerFrame > 0)
        {
            for (int x = startX - pixelOffset; x < input.m_outputPixelWidth; x+= pixelsPerFrame)
            {
                if (x < 0)
                    continue;

                for (int y=0; y < pixelHeight;y++)
                {
                    texture[y * input.m_outputPixelWidth+x] = 0xff00ff00;
                    texture[y * input.m_outputPixelWidth+x+1] = 0xff00ff00;
                }
            }
        }
    }

    if (input.m_warnAboutMissingTimers)
    {
        // Warn about unknown timers
        for (hkPointerMap<const char*, hkColor::Argb>::Iterator itr = unknownColorMap.getIterator(); unknownColorMap.isValid(itr); itr = unknownColorMap.getNext( itr ) )
        {
            HK_WARN(0x94696eee, "Unknown timer when drawing monitor output: " << unknownColorMap.getKey(itr));
        }
    }


}
void hkMonitorStreamAnalyzer::writeStatisticsDetailsToTga( const hkArrayBase<ThreadRootNodes>& perThreadNodes, const hkMonitorStreamAnalyzer::ThreadDrawInput& input, hkArray< hkArray< hkMonitorStreamFrameInfo > >& frameInfos, hkOstream& outStream, _Inout_opt_ SampleInfo* sampleInfo)
{
    int* texture = HK_NULL;
    int height = 0;

    writeStatisticsDetailsToTexture( perThreadNodes, input, frameInfos, texture, height, sampleInfo  );

    if (texture)
    {
        saveToTGA( texture, outStream, input.m_outputPixelWidth, height );
        hkDeallocate(texture);
    }
}

hkMonitorStreamAnalyzer::Node* HK_CALL hkMonitorStreamAnalyzer::reverseLookupNodeAtTgaSample( int x, int y, const hkArrayBase<hkMonitorStreamAnalyzer::ThreadRootNodes>& perThreadNodes, const ThreadDrawInput& input, hkArray< hkArray< hkMonitorStreamFrameInfo > >& frameInfos )
{
    int numFrames = input.m_numFrames;
    int numThreads = perThreadNodes.getSize();
    if (numThreads < 1)
        return HK_NULL; // nothing to do..

    int startX   = numFrames > 1 ? 32 : 0 ;

    if ( (x < startX) || (x >= input.m_outputPixelWidth))
        return HK_NULL; // out of range

    int maxX = input.m_outputPixelWidth-startX;
    x -= startX;

    int pixelHeightPerThread = input.m_heightPerThread + input.m_gapBetweenThreads;
    int pixelHeightPerFrame  = pixelHeightPerThread * numThreads + input.m_gapBetweenFrames;
    int pixelHeight          = pixelHeightPerFrame * numFrames;

    if ( (y < 0) || (y >= pixelHeight))
        return HK_NULL; // out of range

    // frames and threads start at top, so y inverted
    y = (pixelHeight - 1) - y;
    int yFrame = y / pixelHeightPerFrame;
    int yThread = (y % pixelHeightPerFrame) / pixelHeightPerThread;

    if(yThread >= frameInfos.getSize())
    {
        HK_WARN(0x3f184b66, "Trying to access a thread out of range");
        return HK_NULL;
    }

    hkDouble64 maxFrameTime = -1.0f;
    hkArray<hkDouble64> startTimes;
    getTimerLimits( perThreadNodes, input, frameInfos, maxFrameTime, startTimes );

    if(perThreadNodes[yThread].m_node)
    {
        return getNodeSampledAtTick( frameInfos[yThread][0], yFrame, x, perThreadNodes[yThread].m_node->m_children, maxX, maxFrameTime, startTimes[yFrame] );
    }
    else
    {
        return HK_NULL;
    }
}


namespace
{
    /// A structure used in hkMonitorStreamAnalyzer::gatherRawStreamTimerValues() to collect timer values
    struct hkMonitorStreamAnalyzer_TimerCollector
    {
        void HK_INLINE init(_In_z_ const char* timerName)
        {
            m_node = hkMonitorStreamAnalyzer::Node( HK_NULL, timerName, hkMonitorStreamAnalyzer::Node::NODE_TYPE_TIMER );
            m_stringLength = hkString::strLen( timerName ) + 1;
            m_startCommand.m_commandAndMonitor = HK_NULL;
            m_depth = 0;
        }

        hkMonitorStreamAnalyzer::Node           m_node;
        int                                     m_stringLength; // including terminator. For memcmp() instead of strcmp().
        hkMonitorStream::TimerCommand           m_startCommand;
        int                                     m_depth;
    };
}


void HK_CALL hkMonitorStreamAnalyzer::gatherRawStreamTimerValues(
    _In_reads_to_ptr_(frameEnd) const char* frameStart, _In_ const char* frameEnd, const hkMonitorStreamFrameInfo& frameInfo,
    const char** timerNames, int numTimers, hkReal* valuesOut, const hkMonitorStream::CommandStreamConfig* streamConfig )
{
    if( numTimers <= 0 )
    {
        return;
    }

    // Create some temporary storage to collect the timings
    hkArray<hkMonitorStreamAnalyzer_TimerCollector>::Temp collectors( numTimers );
    for( int i=0; i<numTimers; ++i )
    {
        collectors[i].init( timerNames[i] );
    }

    hkMonitorStream::CommandStreamConfig defaultConfig;
    if ( !streamConfig )
    {
        streamConfig = &defaultConfig;
    }

    // Walk the stream
    const char* currentStreamPtr = frameStart;
    while( currentStreamPtr < frameEnd )
    {
        // Skip commands used to mark the start of an elf
        if( HK_VERY_UNLIKELY( hkMonitorStream::readCommandUInt32(&currentStreamPtr, *streamConfig) <= HK_MAX_ELF_ID ) )
        {
            continue;
        }
        currentStreamPtr -= 4;

        const char* currentBeforeString = currentStreamPtr;
        const char* string = hkMonitorStream::readCommandString(&currentStreamPtr, *streamConfig);
        const char* monitorName = string + 2;
        currentStreamPtr = currentBeforeString; // the cmd read funcs all do the string bit again. Not great.

        switch( string[0] )
        {

        case 'T': // timer begin
        case 'R': // timer begin with self
            {
                if (string[1] == 't')
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, *streamConfig);
                    for( int i=0; i<numTimers; ++i )
                    {
                        if( collectors[i].m_startCommand.m_commandAndMonitor )
                        {
                            collectors[i].m_depth++;
                        }
                        else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                        {
                            collectors[i].m_startCommand = com;
                            collectors[i].m_depth = 0;
                        }
                    }
                }
                else // GPU
                {
                    HK_ASSERT_NO_MSG(0x49fc315b, string[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, *streamConfig);
                }
                break;
            }

        case 'E': // timer end
            {
                if (string[1] == 't')
                {
                    hkMonitorStream::TimerCommand com;
                    com.read(&currentStreamPtr, *streamConfig);
                    for( int i = 0; i < numTimers; ++i )
                    {
                        if( collectors[i].m_startCommand.m_commandAndMonitor )
                        {
                            if( collectors[i].m_depth == 0 )
                            {
                                collectors[i].m_node.setTimers( frameInfo, *streamConfig, collectors[i].m_startCommand, com );
                                collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                            }
                            else
                            {
                                collectors[i].m_depth--;
                            }
                        }
                    }
                }
                else // GPU
                {
                    HK_ASSERT_NO_MSG(0x718c97a8, string[1] == 'g');
                    hkMonitorStream::TimerDrawCallCommand com;
                    com.read(&currentStreamPtr, *streamConfig);
                }
                break;
            }

        case 'Y': // multi timer begin
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor)
                    {
                        collectors[i].m_depth++;
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        // Note that this pointer isn't used by the multi timer's end event,
                        // but setting it to a non-NULL value is needed for the code logic to work
                        collectors[i].m_startCommand.m_commandAndMonitor = (const char*)0x1;
                        collectors[i].m_depth = 0;
                    }
                }
                break;
            }

        case 'W': // multi timer end
            {
                hkMonitorStream::MultiTimerCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor)
                    {
                        if( collectors[i].m_depth == 0 )
                        {
                            collectors[i].m_node.setTimers( frameInfo, com );
                            collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                        }
                        else
                        {
                            collectors[i].m_depth--;
                        }
                    }
                }
                break;
            }

        case 'O': // object name
            {
                hkMonitorStream::TimerBeginObjectNameCommand com;
                const char* name = com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        collectors[i].m_depth++;
                    }
                    else if( hkString::memCmp( name, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 0;
                    }
                }
                break;
            }

        case 'L': // timer list begin
            {
                hkMonitorStream::TimerBeginListCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        collectors[i].m_depth += 2;
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 1;
                    }
                    else if( hkString::memCmp( com.m_nameOfFirstSplit + 2, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 0;
                    }
                }
                break;
            }

        case 'S': // split list
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        if( collectors[i].m_depth == 0 )
                        {
                            collectors[i].m_node.setTimers( frameInfo, *streamConfig, collectors[i].m_startCommand, com );
                            collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                        }
                    }
                    else if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_startCommand = com;
                        collectors[i].m_depth = 0;
                    }
                }
                break;
            }

        case 'l': // list end
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( collectors[i].m_startCommand.m_commandAndMonitor )
                    {
                        if( collectors[i].m_depth <= 1 )
                        {
                            collectors[i].m_node.setTimers( frameInfo, *streamConfig, collectors[i].m_startCommand, com );
                            collectors[i].m_startCommand.m_commandAndMonitor = HK_NULL;
                        }
                        collectors[i].m_depth -= 2;
                    }
                }
                break;
            }

        case 'M': // HK_MONITOR_COMMAND_ADD_VALUE
            {
                hkMonitorStream::AddValueCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                for( int i=0; i<numTimers; ++i )
                {
                    if( hkString::memCmp( monitorName, collectors[i].m_node.m_name, collectors[i].m_stringLength ) == 0 )
                    {
                        collectors[i].m_node.m_type = hkMonitorStreamAnalyzer::Node::NODE_TYPE_SINGLE;
                        collectors[i].m_node.m_threadData[frameInfo.m_indexOfTimer0].m_value = com.m_value;
                        collectors[i].m_node.m_threadData[frameInfo.m_indexOfTimer0].m_count += 1;
                    }
                }
                break;
            }

            //
            // Irrelevant commands
            //

        case 'F':
            {
                hkMonitorStream::FrameIdCommand com;
                com.read( &currentStreamPtr, *streamConfig );
                break;
            }
        case 'X':
            {
                hkMonitorStream::TagCommand com;
                com.read( &currentStreamPtr, *streamConfig );
                break;
            }
        case 'G':
            {
                hkMonitorStream::TagCommand com;
                com.read( &currentStreamPtr, *streamConfig );
                break;
            }
        case 'A':
            {
                hkMonitorStream::AddStructCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                break;
            }
        case 'H':
            {
                hkMonitorStream::AddGpuHandleCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                break;
            }
        case 'm':
            {
                hkMonitorStream::MemoryCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                break;
            }
        case 'P':
        case 'p':
        case 'N':
            {
                hkMonitorStream::Command com;
                com.read(&currentStreamPtr, *streamConfig);
                break;
            }
        case 'Q':
            {
                hkMonitorStream::TimerCommand com;
                com.read(&currentStreamPtr, *streamConfig);
                break;
            }

        default:
            HK_WARN(0x3d7745e3, "Inconsistent Monitor capture data" );
            return;
        }
    }

    // Copy values out
    HK_ASSERT_NO_MSG(0x1c538ca3, frameInfo.m_indexOfTimer0 == 0 );
    for( int i=0; i<numTimers; ++i )
    {
        valuesOut[i] = collectors[i].m_node.m_threadData[0].m_value;
    }
}

void HK_CALL hkMonitorStreamAnalyzer::makeStringsLocal(_Inout_ hkMonitorStreamAnalyzer::Node* node, hkPointerMap<char*, char*>& map)
{
    if (!map.getWithDefault((char*)node->m_name, HK_NULL))
    {
        char* localStr = hkString::strDup(node->m_name);
        map.insert(localStr, localStr);
        node->m_name = localStr;
    }
    for (int c = 0; c < node->m_children.getSize(); ++c)
    {
        makeStringsLocal(node->m_children[c], map);
    }
}

void HK_CALL hkMonitorStreamAnalyzer::deallocateLocalStrings(hkPointerMap<char*, char*>& map)
{
    hkPointerMap<char*, char*>::Iterator iter = map.getIterator();
    while (map.isValid(iter))
    {
        hkDeallocate<char>(map.getKey(iter));
        iter = map.getNext(iter);
    }
    map.clear();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
