// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

class hkMemorySystem;
class hkMemoryTracker;

#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Container/PointerMultiMap/hkPointerMultiMap.h>

class HK_EXPORT_COMMON hkMemoryTrackerSnapshot
{
    public:

        HK_DECLARE_CLASS(hkMemoryTrackerSnapshot, NewPlacement, NonCopyable);

        struct HK_EXPORT_COMMON Block
        {
            HK_DECLARE_CLASS(Block, NewPlacement);

            Block() : m_name(HK_NULL), m_ptr(HK_NULL), m_type(HK_NULL), m_size(0), m_num(-1) {}
            Block(_In_opt_z_ const char* name, _In_ const void* ptr, int size, _In_opt_ const hkReflect::Type* type = HK_NULL, int num = -1)
                : m_name(name), m_ptr(ptr), m_type(type), m_size(size), m_num(num) {}

            const char* m_name;
            const void* m_ptr;
            const hkReflect::Type* m_type;
            int m_size; // 0 if not on heap
            int m_num; // -1 if single instance, >=0 for arrays
        };

        hkMemoryTrackerSnapshot(_In_ hkMemoryAllocator* mem);
        ~hkMemoryTrackerSnapshot();

            /// Add a top-level block to the snapshot.
            /// Blocks can be typed or raw (type = HK_NULL, identified by the label).
            /// If onHeap is true, the block is used to calculate the total used memory and must not overlap with other
            /// blocks; if false, the block is only explored for contained heap blocks (arrays, strings, etc.).
        void addBlock(const Block& block, bool onHeap);

            /// Finalizes the snapshot calculating the links between the blocks.
            /// This should be called after all top-level blocks have been added. Explores each block collecting
            /// information about pointers and calling custom handlers.
        void finalize();

            /// Add a link going out from the current block.
            /// Must be called from block handlers.
        void addLink(_In_opt_ const void* ptr);

            /// Add a new typed/raw block linked by the current block.
            /// Must be called from block handlers.
        void addLinkedBlock(const Block& block, bool onHeap);

            /// Add a linked block corresponding to the easyAlloc overhead for the given pointer.
        void addLinkedEasyOverhead(_In_ const void* ptr);

            /// Add a linked block corresponding to the alignedAlloc overhead for the given pointer.
        void addLinkedAlignedOverhead(_In_ const void* ptr);

            /// Compile and cache a type for pointer exploration.
            /// Returns true if instances of the type may contain links/blocks and will be explored.
        bool addType(_In_opt_ const hkReflect::Type* type);

        void clear();

        bool isEmpty() const;

        typedef int BlockIterator;
        BlockIterator begin() const;
        BlockIterator end() const;
        const Block& getBlock(BlockIterator it) const;

        BlockIterator getBlockFor(_In_ const void* ptr) const;

        typedef hkPointerMultiMap<int, int> LinkMap;
        class LinkIterator
        {
            public:
                HK_DECLARE_CLASS(LinkIterator, NoNew);
                bool advance();
                BlockIterator current() const;
            private:
                HK_ALWAYS_INLINE LinkIterator(const LinkMap& map, LinkMap::Iterator it, int key)
                    : m_map(map), m_it(it), m_key(key), m_current(0) {}
                const LinkMap& m_map;
                LinkMap::Iterator m_it;
                int m_key;
                int m_current;

            friend class hkMemoryTrackerSnapshot;
        };

            /// Returns true if the block contains outgoing links.
        bool containsLinks(BlockIterator it) const;

            /// Gets an iterator for the links in a block.
        LinkIterator getLinksFor(BlockIterator it) const;

        bool isFinalized() const;

    protected:
        void initImpl();
        void quitImpl();

    protected:

        hkMemoryAllocator* m_mem;

            /// Array of tracked blocks.
        hkArray<Block> m_blocks;

            /// Associates memory pointers with the corresponding block.
            
        struct Index : public hkHashMapDetail::Index
        {
            HK_DECLARE_CLASS(Index, NoNew);

            void add(_In_opt_ const void* ptr, int idx);
            int find(_In_opt_ const void* ptr, hkArrayView<const Block> blocks) const;
        };
        Index m_blockIndex;

            /// Associates each block with its pointed blocks.
        LinkMap m_linksFromBlock;

            /// Keeps alive prettified names for the block types.
        hkArray<hkStringPtr> m_prettyNames;

        bool m_finalized;

        struct Impl;
        Impl* m_impl;
};

#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
