// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Memory/Tracker/hkMemoryTracker.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmInterpreter.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

namespace
{
    using hkTypeVm::Program;

    int getStartOffset(_In_opt_ const hkReflect::Type* type)
    {
        if (type == HK_NULL || type->getParent() == HK_NULL)
        {
            return 0;
        }
        if (auto decls = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::DECLS>(type))
        {
            auto fields = decls->getDataFields();
            if (fields.getSize())
            {
                return fields[0].getOffset();
            }
        }
        return type->getParent()->getSizeOf();
    }

    inline Program::DefaultInstruction makeHandlerInstr(_In_ const hkReflect::Type* type, int offset, _In_ const hk::MemoryTracker* attr)
    {
        
        hkTypeVm::Program::DefaultInstruction attrInstr;
        attrInstr.set(hkTypeVm::Program::INST_POINTER, offset, type, offset, type);
        attrInstr.setUserData(reinterpret_cast<hkUlong>(attr->m_handler));
        return attrInstr;
    }

    // Keeps info about pointers, containers and manual handlers.
    // If the type, one of its ancestors or one of its fields has a hk::MemoryTracker attribute, replace the
    // corresponding instructions
    struct FindLinksPass : public hkTypeVm::Pass
    {
        virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType) HK_OVERRIDE
        {
            Program::Iterator it = program.end()-1;
            for (const hkReflect::Type* currContext = srcType; currContext; currContext = currContext->getParent())
            {
                int contextStart = getStartOffset(currContext);

                // 1. Handle MemoryTracker attribute on current ancestor.
                if (const hk::MemoryTracker* attr = hkReflect::TypeDetail::localFindAttribute<hk::MemoryTracker>(currContext))
                {
                    if (attr->m_handler)
                    {
                        program.insertAfter(it, makeHandlerInstr(currContext, 0, attr));
                    }

                    if (attr->m_opaque)
                    {
                        program.remove(program.begin(), it+1);
                        return HK_SUCCESS;
                    }
                }

                // 3. Handle MemoryTracker attribute on fields of the current ancestor.
                for (; it != program.begin()-1 && it->getSrcOffset() >= contextStart; --it)
                {
                    if (const Program::DefaultInstruction* instr = Program::asDefaultInstruction(*it))
                    {
                        // If the field is a Record the pass will process its type independently.
                        if (instr->getInstructionType() != Program::INST_RECORD)
                        {
                            const hkReflect::Type* type = instr->getSrcType();
                            if (const hk::MemoryTracker* attr = hkReflect::TypeDetail::decoratorFindAttribute<hk::MemoryTracker>(type))
                            {
                                if (attr->m_handler)
                                {
                                    // Add an instruction to execute the manual handler.
                                    const Program::DefaultInstruction& attrInstr = makeHandlerInstr(type, instr->getSrcOffset(), attr);

                                    if (attr->m_opaque)
                                    {
                                        // Erase the original instruction.
                                        program.replace(it, it + 1, attrInstr);
                                    }
                                    else
                                    {
                                        program.insertAfter(it, attrInstr);
                                    }
                                }
                                else if (attr->m_opaque)
                                {
                                    // Erase the original instruction.
                                    program.remove(it);
                                }
                            }
                            else if (instr->getInstructionType() == Program::INST_BOOL
                                || instr->getInstructionType() == Program::INST_INT
                                || instr->getInstructionType() == Program::INST_FLOAT)
                            {
                                // Discard.
                                program.remove(it);
                            }
                        }
                    }
                }
            }

            return HK_SUCCESS;
        }
    };

    HK_INLINE bool cmpString(const hkStringPtr& a, const hkStringPtr& b)
    {
        return hkString::strCmp(a.cString(), b.cString()) < 0;
    }

    HK_INLINE bool cmpBlock(const hkMemoryTrackerSnapshot::Block& a, const hkMemoryTrackerSnapshot::Block& b)
    {
        return a.m_ptr < b.m_ptr;
    }
}

struct hkMemoryTrackerSnapshot::Impl : public hkTypeVm::UnaryInterpreter<Impl>
{
    public:

        // Object/array which might contain links to other blocks.
        struct Todo
        {
            Todo(const hkReflect::ArrayValue& data, _In_ const void* owner, Program& program)
                : m_data(data), m_owner(owner), m_program(program), m_linksBegin(0), m_linksNum(0) {}

            hkReflect::ArrayValue m_data;

            // Pointer to the heap block which contains this todo.
            // Links from this todo will be attributed to its owner.
            const void* m_owner;

            Program& m_program;

            // Index into m_allLinks.
            int m_linksBegin;
            int m_linksNum;
        };

        typedef hkArrayView<const void* const> Links;

        Impl(hkMemoryTrackerSnapshot& snapshot)
            : m_snapshot(snapshot), m_current(-1)
        {
            getCompiler().excludeProperties = true;
            getCompiler().addPass(new FindLinksPass);
            getCompiler().addPass(new hkTypeVm::InlineFixedArrayPass);
            getCompiler().addPass(new hkTypeVm::InlineRecordPass);
        };

        HK_INLINE void addTodo(const Block& block, _In_ const void* owner)
        {
            addTodo(hkReflect::ArrayValue(block.m_ptr, block.m_num >= 0 ? block.m_num : 1, block.m_type), owner);
        }

        HK_INLINE void addTodo(const hkReflect::ArrayValue& array, _In_ const void* owner)
        {
            const hkReflect::Type* type = array.getSubType();
            // Add a todo only if the type is compiled into a non-empty program.
            if (hkViewPtr<Program> program = compile(type))
            {
                m_todos.pushBack(Todo(array, owner, *program));
            }
        }

        HK_INLINE hkViewPtr<Program> compile(_In_opt_ const hkReflect::Type* type)
        {
            if (type && type != hkReflect::getType<hkReflect::Detail::Opaque>())
            {
                hkViewPtr<Program> program = getCompiler().compile(type);
                if (!program->isEmpty())
                {
                    return program;
                }
            }
            return HK_NULL;
        }

        // Run the interpreter on all todos. Add new blocks to the end and continue until there are no more todos.
        // Keep track of the todo which is currently being processed.
        void findAllBlocksAndLinks()
        {
            for (m_current = 0; m_current < m_todos.getSize(); ++m_current)
            {
                Todo& todo = m_todos[m_current];
                const int count = todo.m_data.getCount();
                hkReflect::ArrayValue av = todo.m_data;
                hkTypeVm::Program& program = todo.m_program;

                for (int i = 0; i < count; ++i)
                {
                    
                    run(av[i], program);
                }
            }
        }

        HK_INLINE void execPointer(_In_ void* addr, const hkTypeVm::Program::DefaultInstruction& instr)
        {
            if (hk::MemoryTracker::Handler func = reinterpret_cast<hk::MemoryTracker::Handler>(instr.getUserData()))
            {
                // Manual handler, call it on the object.
                func(instr.getSrcVar(addr), m_snapshot);
            }
            else
            {
                // Add the link to the list.
                hkReflect::Var tgt = instr.getSrcVarAs<hkReflect::PointerVar>(addr).getValue();
                addLink(tgt.getAddress());
            }
        }

        HK_INLINE void execArray(_In_ void* addr, const hkTypeVm::Program::ArrayInstruction& instr)
        {
            // Add the elements to the todo list.
            hkReflect::ArrayVar var = instr.getSrcVar(addr);
            HK_ASSERT_NO_MSG(0x6138587, var.getType() && var.getType()->getFixedCount() == 0);
            hkReflect::ArrayValue value = var.getValue();
            if (value.getCount())
            {
                addTodo(value, m_todos[m_current].m_owner);
            }

            // Add the link to the list.
            addLink(value.getAddress());
        }

        HK_INLINE void execString(_In_ const void* addr, const hkTypeVm::Program::DefaultInstruction& instr)
        {
            // Add the link to the list.
            const char* str = instr.getSrcVarAs<hkReflect::StringVar>(addr).getValue();
            addLink(str);
        }

        HK_INLINE void addLink(_In_opt_ const void* ptr)
        {
            if (ptr)
            {
                Todo& todo = m_todos[m_current];

                // Add an element to m_allLinks and update the todo indices.
                if (!todo.m_linksNum)
                {
                    todo.m_linksBegin = m_allLinks.getSize();
                    todo.m_linksNum = 1;
                }
                else
                {
                    HK_ASSERT_NO_MSG(0xf238135, todo.m_linksBegin + todo.m_linksNum == m_allLinks.getSize());
                    ++todo.m_linksNum;
                }

                m_allLinks.pushBack(ptr);
            }
        }

        HK_INLINE int numTodos() const { return m_todos.getSize(); }
        HK_INLINE const Todo& todo(int i) const { return m_todos[i]; }

        HK_INLINE Links links(int i)
        {
            return Links(m_allLinks).slice(m_todos[i].m_linksBegin, m_todos[i].m_linksNum);
        }

        HK_INLINE _Ret_notnull_ const void* currentBlock()
        {
            return m_todos[m_current].m_owner;
        }

        hkMemoryTrackerSnapshot& m_snapshot;
        hkArray<Todo>::Temp m_todos;
        hkArray<const void*>::Temp m_allLinks;
        int m_current;
};


void hkMemoryTrackerSnapshot::Index::add(_In_opt_ const void* ptr, int idx)
{
    hkHashMapDetail::Index::add(hkHashMapDetail::computeHash(ptr), idx);
}

int hkMemoryTrackerSnapshot::Index::find(_In_opt_ const void* ptr, hkArrayView<const Block> blocks) const
{
    hkUint32 hash = hkHashMapDetail::computeHash(ptr);
    for (hkHashMapDetail::Entry* entry = getFirstEntry(hash); true; entry = nextEntry(entry))
    {
        if (entry->isEmpty())
        {
            return -1;
        }
        else if (blocks[entry->idx].m_ptr == ptr)
        {
            return entry->idx;
        }
    }
}

hkMemoryTrackerSnapshot::hkMemoryTrackerSnapshot(_In_ hkMemoryAllocator* mem)
    : m_mem(mem)
    , m_finalized(false)
    , m_impl(HK_NULL)
{
}

void hkMemoryTrackerSnapshot::initImpl()
{
    m_impl = m_mem->create<Impl, hkMemoryTrackerSnapshot&>(*this);
}

void hkMemoryTrackerSnapshot::quitImpl()
{
    m_mem->destroy(m_impl);
    m_impl = HK_NULL;
}

hkMemoryTrackerSnapshot::~hkMemoryTrackerSnapshot()
{
    quitImpl();
}

void hkMemoryTrackerSnapshot::clear()
{
    m_blocks.clearAndDeallocate();
    m_blockIndex.clearAndDeallocate();
    m_linksFromBlock.clear();
    quitImpl();
    m_finalized = false;
}

void hkMemoryTrackerSnapshot::addLink(_In_opt_ const void* ptr)
{
    HK_ASSERT(0x6c8cfe9d, m_impl, "addLink can be called only while finalizing the snapshot");
    m_impl->addLink(ptr);
}

void hkMemoryTrackerSnapshot::addBlock(const Block& block, bool onHeap)
{
    HK_ASSERT(0x46234f3a, !m_impl && !isFinalized(), "Blocks can only be added before finalizing the snapshot");

    // Enqueue the block.
    // We cannot "new" things in this method because otherwise we will modify the contents of the memory tracker,
    // so we just enqueue the block here.
    if (onHeap)
    {
        HK_ASSERT_NO_MSG(0x275951fb, block.m_size > 0);
        // Store the block in the snapshot.
        m_blocks.pushBack(block);
    }
    else
    {
        // Set the size to 0 to mark the block as visit-only.
        Block b(block);
        b.m_size = 0;
        m_blocks.pushBack(b);
    }
}

void hkMemoryTrackerSnapshot::addLinkedBlock(const Block& block, bool onHeap)
{
    HK_ASSERT(0x5c4786a2, m_impl, "addLinkedBlock can be called only while finalizing the snapshot");
    if (onHeap)
    {
        m_blocks.pushBack(block);
        m_impl->addLink(block.m_ptr);
    }
    m_impl->addTodo(block, onHeap ? block.m_ptr : m_impl->currentBlock());
}

bool hkMemoryTrackerSnapshot::addType(_In_opt_ const hkReflect::Type* type)
{
    HK_ASSERT(0xa29b6be, !isFinalized(), "addType can be called only while finalizing the snapshot");
    return m_impl->compile(type) != HK_NULL;
}

void hkMemoryTrackerSnapshot::finalize()
{
    HK_ASSERT_NO_MSG(0x3cbabd95, !m_impl);

    if (isFinalized())
    {
        return;
    }

    initImpl();

    // Add all the collected blocks as todos.
    for (int i = 0; i < m_blocks.getSize(); ++i)
    {
        if (m_blocks[i].m_size > 0)
        {
            m_impl->addTodo(m_blocks[i], m_blocks[i].m_ptr);
        }
        else
        {
            // Not on heap, add it as a todo then remove the block.
            m_impl->addTodo(m_blocks[i], HK_NULL);
            m_blocks.removeAt(i);
            --i;
        }
    }

    // Run the interpreter and get additional blocks and links information in the Impl.
    // We need to find all blocks before analyzing links between blocks.
    m_impl->findAllBlocksAndLinks();

    // Sort the blocks.
    hkSort(m_blocks.begin(), m_blocks.getSize(), &cmpBlock);

#if 0
    // Check that there are no overlapping blocks.
    for (int i = 0; i < m_blocks.getSize() - 1; ++i)
    {
        if (m_blocks[i + 1].m_ptr < hkAddByteOffset(m_blocks[i].m_ptr, m_blocks[i].m_size))
        {
            HK_BREAKPOINT(0);
        }
    }
#endif

    // Build the block index.
    m_blockIndex.reserve(m_blocks.getSize());
    for (int i = 0; i < m_blocks.getSize(); ++i)
    {
        m_blockIndex.add(m_blocks[i].m_ptr, i);
    }

    // Create a map with all the links.
    for (int todoIdx = 0; todoIdx < m_impl->numTodos(); ++todoIdx)
    {
        const Impl::Todo& todo = m_impl->todo(todoIdx);
        Impl::Links links = m_impl->links(todoIdx);
        if (todo.m_owner && links.getSize())
        {
            // The owner of the todo is a heap block and the todo has links.
            int ownerId = m_blockIndex.find(todo.m_owner, m_blocks);
            HK_ASSERT(0x29338cb9, ownerId >= 0, "Todo owner is not a heap block");
            for (int linkIdx = 0; linkIdx < links.getSize(); ++linkIdx)
            {
                // Add a link only if it points to another block heap.
                int pointedId = m_blockIndex.find(links[linkIdx], m_blocks);
                if (pointedId >= 0)
                {
                    m_linksFromBlock.insert(ownerId, pointedId);
                }
            }
        }
    }

    // Polish the blocks names
    for (int i = 0; i < m_blocks.getSize(); ++i)
    {
        const hkReflect::Type* type = m_blocks[i].m_type;
        
        bool usesOldMacro = m_blocks[i].m_name && m_blocks[i].m_name[0] == '!';
        if (type && type != hkReflect::getType<hkReflect::Detail::Opaque>() && type->getName())
        {
            bool nameIsMissing = m_blocks[i].m_name == HK_NULL || usesOldMacro;
            bool nameIsEqual = type->getName() != m_blocks[i].m_name;
            bool nameIsUnscoped = !nameIsEqual && hkString::endsWith(type->getName(), m_blocks[i].m_name);

            // Replace the name if the block name is incomplete. If the block has a custom name leave it like this.
            if (nameIsMissing || nameIsUnscoped)
            {
                hkStringBuf aux;
                const char* fullName = type->getFullName(aux);
                if (m_blocks[i].m_num >= 0)
                {
                    if (fullName != aux.cString())
                    {
                        aux = fullName;
                    }
                    aux.appendPrintf("[%d]", m_blocks[i].m_num);
                    fullName = aux.cString();
                }

                
                int nameIdx = hkAlgorithm::findInsertionIndex(fullName, m_prettyNames.begin(), m_prettyNames.getSize(), &cmpString);
                if (nameIdx == m_prettyNames.getSize() || m_prettyNames[nameIdx] != fullName)
                {
                    m_prettyNames.insertAt(nameIdx, fullName);
                }
                m_blocks[i].m_name = m_prettyNames[nameIdx].cString();
            }
        }
        else if (usesOldMacro)
        {
            m_blocks[i].m_name = "Unnamed (use HK_DECLARE_CLASS)";
        }
    }

    quitImpl();
    m_finalized = true;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
