// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/Memory/System/Util/hkMemorySnapshot.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Memory/Tracker/hkMemoryTracker.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerReportUtil.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>

#include <Common/Base/Fwd/hkctime.h>

namespace
{
    hkSystemTime getSystemTime()
    {
        return static_cast<hkSystemTime>(::time(0));
    }
}

namespace
{
    void HK_CALL writeModuleInfo(_In_z_ const char* text, _Inout_ void* stream)
    {
        *reinterpret_cast<hkOstream*>(stream) << "Module( str=r'''" << text << "''' )\n";
    }

    void HK_CALL writeMemorySystemStats(const hkMemorySnapshot& snapshot, hkOstream& stream)
    {
        const char* prev = snapshot.getMemorySystemStatistics();
        if (prev)
        {
            while( const char* cur = hkString::strChr(prev, '\n') )
            {
                stream << "Statistics( str='";
                stream.write( prev, int(cur-prev) );
                stream << "' )\n";
                cur += 1;
                prev = cur;
            }
        }
    }

    // Internal structure for passing parameters to writeStackTrace
    struct AddressParams
    {
        hkOstream* m_output;
        hkArray<hkUlong>* m_addresses;
        int m_index;
    };

    static void HK_CALL writeStackTrace(_In_z_ const char* text, _Inout_ void* context )
    {
        AddressParams* params = (AddressParams*)context;

        if (params->m_index >= params->m_addresses->getSize())
        {
            (*params->m_output) << "Location( loc=" << "0x00000000" << ", str=r\"\"\"" << "No Stacktrace available" << "\"\"\" )\n";
        }
        else
        {
            hkUlong address = (*params->m_addresses)[params->m_index];
            int index = hkString::lastIndexOf(text, '\n');
            hkStringView subText(text, index >= 0 ? index : hkString::strLen(text));
            (*params->m_output) << "Location( loc=" << hkUint64(address) << ", str=r\"\"\"" << subText << "\"\"\" )\n";
        }

        params->m_index++;
    }
}

void HK_CALL hkMemoryTrackerReportUtil::takeSnapshotsAndGenerateReport(_In_opt_ const hkMemoryTracker* memoryTracker,
                                                                       _In_z_ const char* filename)
{
    hkMallocAllocator allocator;
    hkMemorySnapshot memorySnapshot(&allocator);
    hkMemoryTrackerSnapshot trackerSnapshot(&allocator);

    takeSnapshots(memoryTracker, memorySnapshot, trackerSnapshot);
    hkOstream stream(filename);
    generateReport(memorySnapshot, trackerSnapshot, stream);
}

void HK_CALL hkMemoryTrackerReportUtil::takeSnapshots(_In_opt_ const hkMemoryTracker* memoryTracker,
                                                      hkMemorySnapshot& memorySnapshot,
                                                      hkMemoryTrackerSnapshot& trackerSnapshot)
{
    // takes the system snapshot
    hkMemorySystem::getInstance().getMemorySnapshot(memorySnapshot);
    memorySnapshot.sort();
    if (memoryTracker)
    {
        // takes the tracker snapshot
        memoryTracker->getSnapshot(trackerSnapshot);
    }
}

void HK_CALL hkMemoryTrackerReportUtil::generateReport(const hkMemorySnapshot& memorySnapshot,
                                                       const hkMemoryTrackerSnapshot& trackerSnapshot,
                                                       hkOstream& stream )
{
    HK_ASSERT_NO_MSG(0x543ec0e7, stream.isOk());
    // Maps to determine whether we've seen particular objects before...
    hkStringMap<int> seenTypes; 
    hkPointerMap<hkUlong,int> seenAddresses;
    hkPointerMap<hkUlong,int> seenLocations;
    hkPointerMap<int, int> seenCallstacks;

    hkStackTracer tracer;
    const hkStackTracer::CallTree& callTree = memorySnapshot.getCallTree();

    if ( !trackerSnapshot.isEmpty() && callTree.isEmpty() )
    {
        stream << "#NOTE: Could not retrieve stack information. Are you using the hkCheckingMemorySystem?\n#\n";
    }

    // Boilerplate lines describing contents of file
    stream << "#V <integer> - Version number\n";
    stream << "#Date(ts=<timestamp>) - Date of capture\n";
    stream << "#Module(mod=<platform-dependent module/symbol identifier string>) - Module information\n";
    stream << "#Statistics(str=<string>) - Raw memory system Statistics\n";
    stream << "#Provider(id=<provider id>, name=<name>, parIds=[<parent>*]) - Hierarchy of providers (allocators)\n";
    stream << "#Router(temp=<provider id>, stack=<provider id>, heap=<provider id>, debug=<provider id>, solver=<provider id>) - Memory router wiring\n";
    stream << "#Allocation(addr=<address>, size=<size>, provId=<provider id>, status=<status>, callstackId=<callstack id>) - Allocation report\n";
    stream << "#Type(id=<type id>, name=<type name>) - Block type definition\n";
    stream << "#Block(id=<block id>, typeId=<type id>, addr=<address>, size=<size>) - Tracker block report\n";
    stream << "#References(blockId=<block id>, refIds=[<owned block id>+]) - Blocks referenced by a given block\n";
    stream << "#Callstack(id=<callstack id>, locations=[<location>+]) - Callstack declaration for a specific address\n";
    stream << "#Location(loc=<location>, str=<string name>) - Program location\n";

    // v0 : tags(D,M,R,L,C,T,a,c,o)
    // v1 : adds V and A tags
    stream << "V 1\n"; 

    // Output timestamp
    stream << "Date( ts=" << (hkUint64)getSystemTime() << " )\n";

    // Output module information required to convert addresses to lines later
    tracer.getModuleInfo( writeModuleInfo, &stream );

    // Output raw memory system stats
    writeMemorySystemStats(memorySnapshot, stream);

    // Output the provider hierarchy
    {
        for( int i = 0; i < memorySnapshot.getProviders().getSize(); ++i )
        {
            const hkMemorySnapshot::Provider& provider = memorySnapshot.getProviders()[i];
            stream.printf("Provider( id=%i, name='%s', parIds=[", i, provider.m_name );
            for( int j = 0; j < provider.m_parentIndices.getSize(); ++j )
            {
                stream.printf("%i", provider.m_parentIndices[j]);
                if(j != provider.m_parentIndices.getSize() - 1)
                    stream << ',';
            }
            stream.printf("] )\n");
        }
    }

    // Output the memory router sample wiring
    stream.printf( "Router( temp=%i, stack=%i, heap=%i, debug=%i, solver=%i )\n",
        memorySnapshot.getRouterWiring()[0], memorySnapshot.getRouterWiring()[1], memorySnapshot.getRouterWiring()[2], memorySnapshot.getRouterWiring()[3], memorySnapshot.getRouterWiring()[4] );

    // Output the list of allocations and callstacks when the address is referred
    {
        for( int i = 0; i < memorySnapshot.getAllocations().getSize(); ++i )
        {
            const hkMemorySnapshot::Allocation& a = memorySnapshot.getAllocations()[i];

            // Write the callstack if we haven't seen it
            if((a.m_traceId != -1) && (seenCallstacks.getWithDefault( a.m_traceId, 0 ) == 0 ))
            {
                hkArray<hkUlong> callStack;
                int stackSize = callTree.getCallStackSize( a.m_traceId );
                callStack.setSize(stackSize);
                callTree.getCallStack(a.m_traceId, callStack.begin(), stackSize);

                // Write this callstack out
                stream << "Callstack( id=" << a.m_traceId << ", locations=[";
                for( int c = 0; c < callStack.getSize(); c++ )
                {
                    stream << hkUint64(callStack[c]);
                    if(c != callStack.getSize()-1)
                        stream << ',';
                    // Save address for output later
                    seenAddresses.insert( callStack[c], 0 );
                }
                stream << "] )\n";

                seenCallstacks.insert( a.m_traceId, 1 );
            }

            stream.printf("Allocation( addr=0x%p, size=%i, provId=%i, status=%i", a.m_start, a.m_size, a.m_sourceId, int(a.m_status) );
            if(a.m_traceId != -1)
            {
                stream.printf(", callstackId=%i", a.m_traceId);
            }
            stream << " )\n";
        }
    }

    // Output the list of blocks with their types
    for( hkMemoryTrackerSnapshot::BlockIterator blockIt = trackerSnapshot.begin();
        blockIt != trackerSnapshot.end(); ++blockIt )
    {
        const hkMemoryTrackerSnapshot::Block& block = trackerSnapshot.getBlock(blockIt);
        const char* blockName = block.m_name ? block.m_name : "<unnamed>";

        int typeId = -1;
        {
            // Write the type if we haven't seen it before
            hkResult res = seenTypes.get(blockName, &typeId);
            if( res.isFailure() )
            {
                // insert new type in the map
                typeId = seenTypes.getSize();
                seenTypes.insert(blockName, typeId);

                stream << "Type( id=" << typeId << ", name='";
                stream << blockName;
                stream << "' )\n";
            }
        }

        // Data referring specifically to this block
        stream << "Block( id=" << blockIt <<
            ", typeId=" << typeId <<
            ", addr=0x" << static_cast<const void*>(block.m_ptr) <<
            ", size=" << static_cast<unsigned int>(block.m_size) << " )\n";
    }

    // Write the address-to-line table
    {
        hkArray<hkUlong> uniqueAddresses;
        uniqueAddresses.reserve( seenAddresses.getSize() );
        hkPointerMap<hkUlong,int>::Iterator iter;
        for( iter = seenAddresses.getIterator(); seenAddresses.isValid( iter ); iter = seenAddresses.getNext( iter ) )
        {
            uniqueAddresses.pushBack( seenAddresses.getKey( iter ) );
        }
        AddressParams params;
        params.m_output = &stream;
        params.m_addresses = &uniqueAddresses;
        params.m_index = 0;
        tracer.dumpStackTrace( uniqueAddresses.begin(), uniqueAddresses.getSize(), writeStackTrace, &params );
    }

    // Add references data
    {
        for( hkMemoryTrackerSnapshot::BlockIterator blockIt = trackerSnapshot.begin();
            blockIt != trackerSnapshot.end(); ++blockIt )
        {
            hkArray<int> childIds;

            for(hkMemoryTrackerSnapshot::LinkIterator linkIt = trackerSnapshot.getLinksFor(blockIt); linkIt.advance();)
            {
                int childId = linkIt.current();

                childIds.pushBack(childId);
            }

            if( childIds.getSize() )
            {
                stream << "References( blockId=" << blockIt << ", refIds=[";
                for( int c = 0; c < childIds.getSize(); c++)
                {
                    stream << childIds[c];
                    if(c != childIds.getSize()-1)
                        stream << ',';
                }
                stream << "] )\n";
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
