// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

class hkMemorySystem;
class hkMemoryTrackerSnapshot;
namespace hkReflect { class Type; }

class HK_EXPORT_COMMON hkMemoryTracker
{
    public:

        HK_DECLARE_PLACEMENT_ALLOCATOR();

            /// Virtual Dtor
        virtual ~hkMemoryTracker();
        typedef hkMemoryRouter::Allocator Allocator;

            /// Used to identify a tracked type. Matched with the corresponding reflected type inside hkMemoryTrackerReg.
        typedef const hkUint16* Handle;

            /// Wraps either a Handle or a hkReflect::Type.
            /// We don't want to resolve a Handle into its type or viceversa every time an allocation happens, so we
            /// wrap whatever is used inside a BlockType and pass it to addBlock(), so that it will be resolved only
            /// when/if needed.
        struct BlockType
        {
            public:
                HK_ALWAYS_INLINE BlockType() : m_data(0), m_isType(false) {}
                HK_ALWAYS_INLINE BlockType(Handle handle) : m_data((hkUlong)handle), m_isType(false) {}
                HK_ALWAYS_INLINE BlockType(_In_ const hkReflect::Type* type) : m_data((hkUlong)type), m_isType(true) {}

                HK_ALWAYS_INLINE bool isValid() const { return m_data != 0; }
                HK_ALWAYS_INLINE Handle asHandle() const { return m_isType ? HK_NULL : (Handle)m_data; }
                HK_ALWAYS_INLINE _Ret_maybenull_ const hkReflect::Type* asType() const { return m_isType ? (const hkReflect::Type*)m_data : HK_NULL; }

                    /// Wraps the hkReflect::Type of T if T is reflected, otherwise its trackerHandle.
                template <typename T>
                HK_ALWAYS_INLINE static BlockType of() { return GetBlockType<T>::func(); }

            private:
                hkUlong m_data;
                bool m_isType; 
        };

            /// Add a new object.
        template<typename T>
        HK_INLINE void addTyped(_In_z_ const char* name, _In_opt_bytecount_(size) T* obj, hk_size_t size = sizeof(T), bool onHeap = true)
        {
            addBlock(name, BlockType::of<T>(), obj, size, onHeap);
        }

            /// Add a top-level block to the tracker.
            /// Blocks can be typed or raw (type = HK_NULL, identified by the name).
            /// If onHeap is true, the block is used to calculate the total used memory and must not overlap with other
            /// blocks; if false, the block is only explored for contained heap blocks (arrays, strings, etc.).
        virtual void addBlock(_In_z_ const char* name, const BlockType& type, _In_opt_bytecount_(size) const void* ptr, hk_size_t size, bool onHeap) = 0;

            /// Remove a top-level block from the tracker.
        virtual void removeBlock(_In_opt_ const void* ptr) = 0;

            /// Get a snapshot
        virtual hkResult getSnapshot(hkMemoryTrackerSnapshot& snapshot) const = 0;

            /// Get the instance - may return null if tracking is not initialized.
        static inline _Ret_maybenull_ hkMemoryTracker* HK_CALL getInstancePtr() { return s_singleton; }
            /// Get the instance - if not set, returns a dummy tracker that ignores all events.
        static hkMemoryTracker& HK_CALL getInstance();
            /// Set the instance
        static void HK_CALL setInstance(_In_opt_ hkMemoryTracker* tracker);

    protected:

        template<typename T, bool reflected = hkReflect::Detail::IsReflectedBase<T>::result>
        struct GetBlockType;

        static hkMemoryTracker* s_singleton;
};

#ifdef HK_MEMORY_TRACKER_ENABLE
template<typename T>
struct hkMemoryTracker::GetBlockType<T, true>
{
    static hkMemoryTracker::BlockType func() { return HK_REFLECT_GET_TYPE(T); }
};

template<typename T>
struct hkMemoryTracker::GetBlockType<T, false>
{
    static hkMemoryTracker::BlockType func() { return HK_TRACKER_GET_HANDLE(T); }
};
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
