// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Memory/Router/hkMemoryRouter.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
typedef hkMemoryAllocator::MemoryStatistics hkAllocatorStatistics;

class hkMemorySnapshot;
class hkThreadMemory;
class hkDebugMemorySystem;

//struct hkMemoryPointerInfo;

    /// Interface to high level memory operations.
    /// These operations apply to the system as a whole i.e., initialization
    /// and shutdown, memory statistics and debugging methods.
class HK_EXPORT_COMMON hkMemorySystem
{
    public:

        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true));
        HK_DECLARE_PLACEMENT_ALLOCATOR();

            /// Construction info for hkMemorySystem
        struct HK_EXPORT_COMMON FrameInfo
        {
            HK_RECORD_ATTR(hk::MemoryTracker(opaque=true));

                /// Construct a FrameInfo with the specified physics solver buffer.
                /// See Common Havok Components > 1.1 Havok Base Library > 1.1.7. Memory Customization > 1.1.7.2. Solver Buffer Size
                /// for more information. May be zero if Physics2012 is not being used.
            FrameInfo(int physics2012SolverBufferSize = 0, int stackAllocatorSizeHint = 32 * 1024);

                /// If set, a buffer will be allocated and given to hkSolverAllocator.
            int m_physics2012SolverBufferSize;

                /// This value is used as a hint to the stack allocator to describe the expected maximum size of temporary memory required by the engine
            int m_stackAllocatorSizeHint;
        };

        /// An aggregate of all the memory statistics of underlying providers.
        struct HK_EXPORT_COMMON MemoryStatistics
        {
            HK_DECLARE_PLACEMENT_ALLOCATOR();
            struct Entry
            {
                HK_DECLARE_PLACEMENT_ALLOCATOR();
                hkStringPtr m_allocatorName;
                hkAllocatorStatistics m_allocatorStats;
            };
            hkArray < Entry > m_entries;
        };
            /// Flags for init and quit
        enum FlagBits
        {
                /// Apply to persistent parts, heap(), debug()
            FLAG_PERSISTENT = 1,
                /// Apply to all frame local allocators such as temp(), stack() and solver()
            FLAG_TEMPORARY = 2,
                /// Apply to all allocators.
            FLAG_ALL = FLAG_PERSISTENT|FLAG_TEMPORARY,
                // old names for the above
            FLAG_ROUTER = FLAG_PERSISTENT,
            FLAG_FRAME = FLAG_TEMPORARY,
        };

        // Wrap a critical section around an allocator, any access is synchronised
        // by the section
        class LockedMemoryAllocator : public hkMemoryAllocator
        {
        public:
            HK_DECLARE_CLASS(LockedMemoryAllocator, New);

            LockedMemoryAllocator(hkMemoryAllocator& chainedAlloc);

            virtual _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* blockAlloc(int numBytes) HK_OVERRIDE;
            virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE;
            virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut) void* bufAlloc(int& reqNumBytesInOut) HK_OVERRIDE;
            virtual void bufFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE;
            virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut) void* bufRealloc(_In_reads_bytes_(oldNumBytes) void* pold, int oldNumBytes, int& reqNumBytesInOut) HK_OVERRIDE;
            virtual void blockAllocBatch(_Out_writes_all_(numPtrs) void** ptrsOut, int numPtrs, int blockSize) HK_OVERRIDE;
            virtual void blockFreeBatch(_In_reads_(numPtrs) void** ptrsIn, int numPtrs, int blockSize) HK_OVERRIDE;
            virtual void getMemoryStatistics(MemoryStatistics& u) const HK_OVERRIDE;
            virtual int getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const HK_OVERRIDE;
            virtual void resetPeakMemoryStatistics() HK_OVERRIDE;

            // Extended interface would not be protected so it isn't currently implemented
            virtual _Ret_maybenull_ ExtendedInterface* getExtendedInterface() HK_OVERRIDE { return HK_NULL; }

        private:
            hkMemoryAllocator& m_chainedAllocator;
            mutable hkCriticalSection m_section;
        };
        typedef hkFlags<FlagBits, int> Flags;

            /// Destructor, usually empty as the real work is done in mainQuit.
        virtual ~hkMemorySystem();

            /// Memory system methods.
            /// These are not normal methods of hkMemoryRouter because each instance is
            /// thread local, but the methods operate on the global system.
            /// The code which initializes the memory system will set these pointers
            /// as only it knows how they should be implemented.
        virtual _Ret_notnull_ hkMemoryRouter* mainInit(const FrameInfo& info, Flags f=FLAG_ALL) = 0;

            /// Shut down the memory system.
            /// Some implementations will return HK_FAILURE if leaks are detected.
        virtual hkResult mainQuit(Flags f=FLAG_ALL) = 0;

            /// Method for threads to construct their local instances.
        virtual void threadInit(hkMemoryRouter& r, _In_z_ const char* name, Flags f=FLAG_ALL) = 0;
            /// Method for threads to deinitialize their local instances.
        virtual void threadQuit(hkMemoryRouter& r, Flags f=FLAG_ALL) = 0;

            // Limiting memory

            /// Set the soft limit for the heap.
        virtual hkResult setHeapSoftLimit(int nbytes);

            /// Get the soft limit for the heap.
        virtual int getHeapSoftLimit() const;

            /// Returns true if a certain allocation can be done.
        virtual bool solverCanAllocSingleBlock( int numBytes );

            /// Returns true if a total amount of memory of at least 'size' is available.
        virtual bool heapCanAllocTotal( int numBytes );


            /// Print memory usage statistics to the given ostream.
        virtual void printStatistics(hkOstream& ostr) const = 0;

            /// Collects the memory statistics from underlying allocators.
        virtual void getMemoryStatistics(MemoryStatistics& stats) = 0;

            /// Specialized heap access.
            /// For regular heap access use hkMemoryRouter::getInstance().heap(). This method
            /// accesses the underlying heap beyond the thread local caches. Currently only used
            /// by the PlayStation(R)3 SPU allocator.
        virtual _Ret_maybenull_ hkMemoryAllocator* getUncachedLockedHeapAllocator() = 0;

            /// Try to free any unused thread local memory, see also garbageCollect().
            /// Because this method will typically access thread local which is not protected
            /// with a lock, the calling thread should generally own the memory router. Otherwise
            /// if the router belongs to another thread, it must be ensure that the other thread
            /// does not enter any memory functions or corruption is likely.
        virtual void garbageCollectThread(hkMemoryRouter& router);
            /// Try to free unused memory in the shared area.
            /// This method may be called after one or more threads have called garbageCollectThread.
            /// While this method is thread safe, note that it will generally lock the shared area
            /// for its entire duration.
        virtual void garbageCollectShared();
            /// Default implementation garbage calls garbageCollectThread then garbageCollectShared.
            /// Depending on the memory system, there are often two levels of block caching. The first
            /// level is private to each thread and the second level is shared between all threads.
            /// This method frees the current threads cache and the shared cache. Note that the
            /// other threads caches are not garbage collected. See also garbageCollectThread().
        virtual void garbageCollect();

        // Optional debugging interfaces.

            /// Access to the debug interface (if present).
        virtual _Ret_maybenull_ hkDebugMemorySystem* getDebugInterface() { return HK_NULL; }

            /// Optional interface to retrieve a snapshot of all allocations.
        virtual hkResult getMemorySnapshot(hkMemorySnapshot& snapshot) const { return HK_FAILURE; }

            /// Returns HK_FAILURE if not implemented, or the address wasn't allocated by this system.
            /// callStack should point to a buffer that will receive the call stack. stackSize is the size of the buffer
            /// will be modified to the call stack size written to the buffer.
            /// allocSize returns the size of the allocation.
        virtual hkResult getAllocationCallStack(_In_ const void* ptr, _In_opt_ hkUlong* callStack, int& stackSize, hk_size_t& allocSize) { return HK_FAILURE; }

            /// This virtual function will run through the used heap allocators and
            /// set a new scrub value for heap allocations and deallocations.
            /// By default we will do anything with those values as this function is
            /// currently only used by the hkCheckDeterminismUtil. If a specific memory
            /// system requires handling of this, it can always be overriden.
            /// At the moment only the hkCheckingMemorySystem and the hkFreeListMemorySystem
            /// handle heap scrubbing.
        virtual void setHeapScrubValues(hkUint32 allocValue, hkUint32 freeValue) {}

            /// Perform an internal consistency check.
        virtual hkBool32 isOk() const { return true; }

            /// Set the global instance pointer
        static void HK_CALL replaceInstance(_In_opt_ hkMemorySystem* m );

            /// Get the global instance
        static hkMemorySystem& HK_CALL getInstance();

            /// Get the global instance pointer
        static _Ret_maybenull_ hkMemorySystem* HK_CALL getInstancePtr();

    protected:

        static hkMemorySystem* s_instance;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
