// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/StackTracer/hkStackTracer.h>

class hkMemorySystem;

/// BETA - A memory snapshot which shows all levels of hierarchical allocations.
///
/// A memory snapshot can be obtained by a supported memory system by using the
/// hkMemorySystem::getMemorySnapshot() call. The memory system is the only entity
/// that knows exactly how the allocator hierarchy being used is laid out, and it is
/// therefore its responsibility to fill the memory snapshot properly. Every allocation
/// contains the ID of the entity (provider) it comes from; a full snapshot should
/// always contain an allocation at every level in the allocator hierarchy for each
/// top-level allocation (down to the system allocator).
/// Assuming we have a plain allocator stack like the following one:
///
/// 1. System allocator
/// 2. Large block allocator
/// 3. Padded allocator
/// 4. Memory system
///
/// For every user allocation recorded in the memory system (level 4), we should
/// have a corresponding allocation at level 3 (plus some overhead caused by the padding),
/// which should be contained in an allocation at level 2 (a large block also containing some
/// overhead eventually). And the large block should be obtained from the system allocator
/// itself (with unknown compiler- and OS-driven overhead).
/// Depending on the specific memory system, we might instead end up with an incomplete
/// allocation tree where, for instance, there is no level 3, 2 or 1 allocation containing
/// an allocation registered by the memory system. In this case we assume that at the
/// missing levels we have no overhead (they behave as simple pass-through allocators
/// for that specific allocation).
/// The memory snapshot doesn't need to be complete, but it should be as complete
/// as possible.
///
/// Generally, only allocations from the system heap should be reported; if a memory
/// system (or an allocator used by it) allocates space to the user from some static
/// buffer, those allocations should NOT be reported.
///
class HK_EXPORT_COMMON hkMemorySnapshot
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, hkMemorySnapshot);

        typedef hkStackTracer::CallTree::TraceId TraceId;
        typedef int SourceId;
        typedef int ProviderId;
        typedef int AllocationId;

        enum StatusBits
        {
            STATUS_OVERHEAD,
            STATUS_USED,
            STATUS_UNUSED,
        };
        typedef hkEnum<StatusBits,hkInt8> Status;

        explicit hkMemorySnapshot(_In_opt_ hkMemoryAllocator* a=HK_NULL);
        hkMemorySnapshot(const hkMemorySnapshot& rhs);
        void setAllocator(_In_ hkMemoryAllocator* a);
        void swap(hkMemorySnapshot& m);
        void clear();

        ~hkMemorySnapshot();

        void init(_In_opt_ const hkMemorySystem* memorySystem);

            /// If -1 is used as parent ID then the provider is configured without any parent.
        ProviderId addProvider(_In_z_ const char* name, ProviderId parent = -1);

            /// Add a new parent to the specified provider
        void addParentProvider(ProviderId provider, ProviderId parent);

            

        AllocationId addItem(ProviderId id, Status status, _In_ const void* address, hkLong size);

        AllocationId addAllocation(ProviderId id, _In_ const void* address, hkLong size) { return addItem(id, STATUS_USED, address, size);  }
        AllocationId addUnused(ProviderId id, _In_ const void* address, hkLong size) { return addItem(id, STATUS_UNUSED, address, size);  }
        AllocationId addOverhead(ProviderId id, _In_ const void* address, hkLong size) { return addItem(id, STATUS_OVERHEAD, address, size);  }

            /// Describes how a sample hkMemoryRouter is connected to the provider hierarchy.
        void setRouterWiring(ProviderId stack, ProviderId temp, ProviderId heap, ProviderId debug, ProviderId solver);

        void setCallStack(AllocationId id, _In_reads_(numAddresses) const hkUlong* addresses, int numAddresses);
            /// Get the call tree. Can be used with the m_traceId, to get a call stack for each allocation.
        const hkStackTracer::CallTree& getCallTree() const { return m_callTree; }

        void sort();

        struct Allocation
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,Allocation);

            /// ==
            hkBool operator==(const Allocation& rhs) const;
            /// !=
            hkBool operator!=(const Allocation& rhs) const;

            const void* m_start;    ///< Base of allocation
            hkLong m_size;          ///< Size of allocation
            SourceId m_sourceId;    ///< Index of the allocator it came from
            TraceId m_traceId;      ///< Index of the callstack or -1 if none
            Status m_status;        ///< Status (type of the allocation)
        };

        /// A provider is an entity capable of obtaining allocations; in case the
        /// provider has no parent we assume that the allocations are coming from
        /// the system allocator. If the provider has at least one parent then the
        /// allocations will be obtained from any of the parent providers.
        struct Provider
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,Provider);
            char m_name[32];                    ///< name of the allocator
            hkArrayBase<int> m_parentIndices;   ///< index of a parent or -1 if none
        };

        const hkArrayBase<Allocation>& getAllocations() const { return m_allocations; }

        /// Works out the allocations which are different
        static void HK_CALL allocationDiff(const hkMemorySnapshot& snapA,
            const hkMemorySnapshot& snapB, hkArray<Allocation>& onlyInA, hkArray<Allocation>& onlyInB);

        /// Dump details (including stack trace if there is one) about an allocation
        void dumpAllocation(const Allocation& alloc, hkOstream& stream) const;

        const hkArrayBase<Provider>& getProviders() const { return m_providers; }
        _Ret_notnull_ const ProviderId* getRouterWiring() const { return m_routerWiring; }
        _Ret_maybenull_ const char* getMemorySystemStatistics() const { return m_memSysStatistics.begin(); }

    protected:

        hkMemoryAllocator* m_mem;
        hkArrayBase<Allocation> m_allocations;
        hkArrayBase<Provider> m_providers;
        /// Describes how a sample router is connected to the provider hierarchy
        /// (i.e. which provider is used for the different kind of allocations),
        /// in the following order: stack, temp, heap, debug, solver.
        ProviderId m_routerWiring[5];
        hkStackTracer::CallTree m_callTree;
            /// Memory usage statistics.
        hkArrayBase<char> m_memSysStatistics;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
