// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>

#include <Common/Base/Memory/Allocator/FreeList/hkFreeListAllocator.h>
#include <Common/Base/Memory/System/FreeList/hkFreeListMemorySystem.h>

namespace hkReflect { namespace Version { class PatchRegistry; } }
namespace hkSerialize { namespace Detail { struct FileFormatRegNode; } }
class hkMemoryAllocator;
class hkThreadMemory;
class hkMemoryRouter;
class hkMemorySystem;
class hkMemoryTracker;

    /// Helper methods to initialize and quit the memory system.
    /// These methods exist mainly for brevity in our demos; the memory system initialization
    /// may be inlined in your setup code.
    /// The init methods must be passed a base system allocator.
    /// All Havok memory allocations will ultimately come from
    /// this allocator. hkMallocAllocator::m_defaultMallocAllocator
    /// is usually suitable for simple systems
namespace hkMemoryInitUtil
{
    // When using CheckingMemory, different levels of check can be used
    enum CheckLevel
    {
        // Checks for leaks, pad allocations, and detect use-after-free errors. Include call stacks. Slow.
        CHECK_MAX,
        // Same as CHECK_MAX, without tracking call stacks. Faster.
        CHECK_NOSTACK,
        // Check only for leaks.
        CHECK_FASTEST,

        CHECK_DEFAULT = CHECK_MAX
    };

        /// Initialize with an hkFreeListMemorySystem + leak detection.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initLeakDetect(_In_ hkMemoryAllocator *memoryAllocator, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo());

        /// Initialize with an hkFreeListMemorySystem using a hkFreeListAllocator for heap allocations, and a hkLargeBlockAllocator for
        /// larger allocations that the hkFreeListAllocator can't handle. The hkLargeBlockAllocator will allocate memory from the memoryAllocator.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initFreeListLargeBlock(_In_ hkMemoryAllocator *memoryAllocator, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo(), _In_opt_ const hkFreeListAllocator::Cinfo* cinfo = HK_NULL, hkFreeListMemorySystem::SetupFlags flags = hkFreeListMemorySystem::DEFAULT_SETUP_FLAGS);

        /// Initialize with an hkFreeListMemorySystem using a hkFreeListAllocator for heap allocations, and the memoryAllocator for any allocations that
        /// the hkFreeListAllocator can't handle.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initFreeList(_In_ hkMemoryAllocator *memoryAllocator, _In_opt_ hkMemoryAllocator::ExtendedInterface* memoryAllocatorExtended, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo(), _In_opt_ const hkFreeListAllocator::Cinfo* cinfo = HK_NULL, hkFreeListMemorySystem::SetupFlags flags = hkFreeListMemorySystem::DEFAULT_SETUP_FLAGS);

        /// Initialize with an hkCheckingMemorySystem.
        /// This checks for leaks and some other common errors and is several orders of magnitude
        /// slower than the free list system.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initChecking(_In_ hkMemoryAllocator *memoryAllocator, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo(), CheckLevel checkLevel = CHECK_DEFAULT);

        /// Initialize with an hkCheckingMemorySystem using an hkGuardingAllocator.
        /// This checks for leaks as well as giving the most thorough validation possible, with most read and write out-of-bounds accesses and use-after-free errors being reported
        /// immediately. This has the highest performance and memory overhead of all allocators.
        /// Only supported on the Win64 platform, falling back to the behavior of initChecking on other platforms.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initGuarding(const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo(), CheckLevel checkLevel = CHECK_DEFAULT);

        /// Initialize with an hkSimpleMemorySystem.
        /// This is slower than the free list system and is intended as an example.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initSimple(_In_ hkMemoryAllocator *memoryAllocator, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo());

        /// Initialize with an hkOptimizerMemorySystem.
        /// This is mainly for internal use to detect where the memory system is being used
        /// sub-optimally.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initOptimizer(_In_ hkMemoryAllocator *memoryAllocator, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo());

        /// Initialize using an explicit heap allocator.
        /// The heapAllocator, and heapInteface provide the direct implementation of the heap (in contrast to other methods
        /// such as initFreeList, which uses a hkFreeListAllocator as the heap allocator). The heapInterface provides
        /// methods for garbageCollection/memoryWalk, etc., on the heapAllocator; it is optional and can be passed as HK_NULL.
        /// The memoryAllocator is the allocator which will be used for non heap allocations; for example
        /// the solver's memory block is allocated from the memoryAllocator.
    HK_EXPORT_COMMON _Ret_notnull_ hkMemoryRouter* HK_CALL initHeapAllocator(_In_ hkMemoryAllocator *memoryAllocator, _In_opt_ hkMemoryAllocator *heapAllocator, _In_opt_ hkMemoryAllocator::ExtendedInterface* heapInterface, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo());

        /// Will destroy the memory allocator, by calling mainQuit() and destroying the allocator.
        /// Should only be called if an init call has been made.
    HK_EXPORT_COMMON hkResult HK_CALL quit();

        /// Initialize with the default system, currently hkFreeListMemorySystem.
    HK_EXPORT_COMMON inline _Ret_notnull_ hkMemoryRouter* HK_CALL initDefault(_In_ hkMemoryAllocator *memoryAllocator, const hkMemorySystem::FrameInfo& info = hkMemorySystem::FrameInfo()) { return initFreeListLargeBlock(memoryAllocator, info); }

        /// When using Checking mem, if you load DLLs dynamically you should let the stacktracer know by calling this function
    HK_EXPORT_COMMON void HK_CALL refreshDebugSymbols();

#if defined(HK_MEMORY_TRACKER_ENABLE)
        /// Init the memory tracker.
        /// Call this before the hkMemorySystem has been initialized.
    HK_EXPORT_COMMON void HK_CALL initMemoryTracker();
        /// Quit the memory tracker.
        /// Call this after the hkMemorySystem has been shut down.
    HK_EXPORT_COMMON void HK_CALL quitMemoryTracker();
#endif

    HK_EXPORT_COMMON void HK_CALL outputDebugString(_In_z_ const char* s, _In_opt_ void*);

#ifndef HK_DYNAMIC_DLL

    /// Cross DLL sync info. Used in non-DLL build.
    struct SyncInfo
    {
        HK_DECLARE_CLASS(SyncInfo, NoNew);

        /// Gets the local library info in the SyncInfo.
        static void getLocalInfo(SyncInfo& info);

        enum ExcludeFlagsBits
        {
            EXCLUDE_NONE = 0,
            EXCLUDE_VERSION_REGISTRY = (1<<1)
        };
        SyncInfo();
        typedef hkFlags<ExcludeFlagsBits, hkUint8> ExcludeFlags;

        /// Replaces the local library hkBaseSystem::InitNodes with the ones contained in the SyncInfo.
        static void syncLocalInitNodes(const SyncInfo& info, ExcludeFlags exclude = EXCLUDE_NONE);

        /// Replaces local hkMonitorStream and hkMultiThreadCheck singletons.
        static void syncDebugInfo(const SyncInfo& info);

        hkMemoryRouter* m_memoryRouter;
        hkMemoryRouter* m_fallBackMemoryRouter;
        hkMemorySystem* m_memorySystem;
        hkBaseSystem::InitNode* m_systemInitNodes;
        void* m_stackTracerImpl; // the stack tracer implementation
        hkCriticalSection* m_mtCheckSection; // the multithreading debug check
        hkMonitorStream* m_monitors; // montor ptrs as monitors not a normal singleton
        hkStackTracer::CallTree* m_mtCheckStackTree; // the multithreading debug check
        hkReflect::Version::PatchRegistry* m_patchRegistry; // Patch registry, not always synchronised
        hkMemoryTracker* m_memoryTracker;
        int m_threadNumber;
    };

#endif

    HK_EXPORT_COMMON extern hkMemorySystem* s_system;

    typedef hkResult (HK_CALL *onQuitFunc)( void );
    HK_EXPORT_COMMON extern onQuitFunc s_onQuitFunc;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
