// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/System/StackTracer/hkStackTracer.h>
#include <Common/Base/DebugUtil/MultiThreadCheck/hkMultiThreadCheck.h>
#include <Common/Base/Reflect/Version/hkReflectPatchRegistry.h>
#include <Common/Base/Fwd/hkcstdio.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>

using namespace std;

#if defined(HK_MEMORY_TRACKER_ENABLE)
#include <Common/Base/Memory/Tracker/Default/hkDefaultMemoryTracker.h>
#endif

hkMemorySystem* hkMemoryInitUtil::s_system = HK_NULL;
hkMemoryInitUtil::onQuitFunc hkMemoryInitUtil::s_onQuitFunc = HK_NULL;

#ifdef HK_PLATFORM_CTR
void HK_CALL hkMemoryInitUtil::outputDebugString(_In_z_ const char* s, _In_opt_ void* )
{
    nndbgDetailPrintf( "%s", s );
}
#elif defined(HK_PLATFORM_WIN32)
#include <Common/Base/Fwd/hkwindows.h>
#include <stdlib.h>
void HK_CALL hkMemoryInitUtil::outputDebugString(_In_z_ const char* s, _In_opt_ void* )
{
#ifndef HK_PLATFORM_WINRT
    OutputDebugStringA( s );
    printf( "%s", s ); // Also output to console for automated testing
    fflush( stdout );
#else
    // Unicode only
    // We don't know if the memory system is still running, so just use a local buffer
    wchar_t wideStr[2048];

    mbstowcs_s( HK_NULL, wideStr, HK_COUNT_OF( wideStr ), s, _TRUNCATE );
    OutputDebugString( wideStr );
#endif
}
#else
void HK_CALL hkMemoryInitUtil::outputDebugString(_In_z_  const char* s, _In_opt_ void* )
{
    using namespace std;
    printf( "%s", s );
    fflush( stdout );
}
#endif

hkResult HK_CALL hkMemoryInitUtil::quit()
{
    hkResult result = HK_SUCCESS;

    // Destroy the system
    HK_ASSERT_NO_MSG( 0x23432aa3, s_system != HK_NULL );
    if( s_system )
    {
        result = s_system->mainQuit();
        s_system->~hkMemorySystem();
        s_system = HK_NULL;

        // Ensure it's not set
        hkMemorySystem::replaceInstance( HK_NULL );
    }

    // Do any remaining destruction
    if (s_onQuitFunc && result.isSuccess())
    {
        result = (*s_onQuitFunc)();
        s_onQuitFunc = nullptr;
    }

    return result;
}

void HK_CALL hkMemoryInitUtil::refreshDebugSymbols()
{
    hkStackTracer tracer;
    tracer.refreshSymbols();
}

#if defined(HK_MEMORY_TRACKER_ENABLE)
void HK_CALL hkMemoryInitUtil::initMemoryTracker()
{
    if(hkMemorySystem::getInstancePtr())
    {
        // The memory tracker must be initialized before initializing the memory system
        HK_BREAKPOINT(0);
        return;
    }

    static hkMallocAllocator mallocAllocator;
    static hkUint8 s_tracker[sizeof(hkDefaultMemoryTracker)];
    hkDefaultMemoryTracker* tracker = new (s_tracker) hkDefaultMemoryTracker(&mallocAllocator);
    hkMemoryTracker::setInstance(tracker);
}

void HK_CALL hkMemoryInitUtil::quitMemoryTracker()
{
    if( hkMemoryTracker* tracker = hkMemoryTracker::getInstancePtr() )
    {
        tracker->~hkMemoryTracker();
        hkMemoryTracker::setInstance(HK_NULL);
    }
}
#endif

#ifndef HK_DYNAMIC_DLL

static hkBaseSystem::InitNode* findByName(_In_ hkBaseSystem::InitNode* list, _In_z_ const char* name)
{
    for(hkBaseSystem::InitNode* cur = list; cur; cur = cur->m_next)
    {
        if(cur->m_name
            && cur->m_name[0] == '$'
            && (hkString::strCmp(name, cur->m_name+1) == 0))
        {
            return cur;
        }
    }
    return HK_NULL;
}

static void s_moveToFront(_In_ hkBaseSystem::InitNode** oldHead, _In_ hkBaseSystem::InitNode* newHead)
{
    if(*oldHead == newHead)
    {
        return;
    }
    // find and unlink
    for(hkBaseSystem::InitNode* p = *oldHead; p; p = p->m_next)
    {
        if(p->m_next == newHead)
        {
            p->m_next = p->m_next->m_next;
            newHead->m_next = *oldHead;
            *oldHead = newHead;
            return;
        }
    }
}

static void populate(_In_ hkBaseSystem::InitNode** dstBaseAddr, _In_ const hkBaseSystem::InitNode* srcBase)
{
    hkStringMap<hkBaseSystem::InitNode*> dstNodeFromName;
    for(hkBaseSystem::InitNode* dstCur = *dstBaseAddr; dstCur != HK_NULL; dstCur = dstCur->m_next)
    {
        dstNodeFromName.insert(dstCur->m_name, dstCur);
    }
    // Loop over the source nodes becuase they are assumed to have been initialized in the correct
    // order. Remember this order in "shared" so that we can duplicate it in dst and get clean shutdown.
    hkArray<hkBaseSystem::InitNode*> shared;
    for(const hkBaseSystem::InitNode* srcCur = srcBase; srcCur != HK_NULL; srcCur = srcCur->m_next)
    {
        if(srcCur->m_name && srcCur->m_name[0] == '$') // it's a singleton
        {
            if(hkBaseSystem::InitNode* n = dstNodeFromName.getWithDefault(srcCur->m_name, HK_NULL))
            {
                typedef hkRefPtr<hkReferencedObject> RefPtr;
                RefPtr& srcp = *(RefPtr*)srcCur->m_arg;
                RefPtr& dstp = *(RefPtr*)n->m_arg;
                dstp = srcp;
                shared.pushBack(n);
            }
        }
    }

    for(int i = shared.getSize() - 1; i >= 0; --i)
    {
        s_moveToFront(dstBaseAddr, shared[i]);
    }
}


hkMemoryInitUtil::SyncInfo::SyncInfo()
{
}


extern HK_THREAD_LOCAL(int) hkThreadNumber;

void hkMemoryInitUtil::SyncInfo::getLocalInfo(SyncInfo& s)
{
    s.m_memoryRouter = hkMemoryRouter::getInstancePtr();
    s.m_fallBackMemoryRouter = hkMemoryRouter::s_fallbackRouter;
    s.m_systemInitNodes = hkBaseSystem::InitNode::s_listHead;
    s.m_memorySystem = hkMemorySystem::getInstancePtr();
    s.m_monitors = hkMonitorStream::getInstancePtr();
    s.m_mtCheckSection = hkMultiThreadCheck::s_criticalSection;
    s.m_stackTracerImpl = hkStackTracer::getImplementation();
    s.m_mtCheckStackTree = hkMultiThreadCheck::s_stackTree;
    s.m_patchRegistry = hkReflect::Version::PatchRegistry::getInstancePtr();
#ifdef HK_MEMORY_TRACKER_ENABLE
    s.m_memoryTracker = hkMemoryTracker::getInstancePtr();
#endif
    s.m_threadNumber = HK_THREAD_LOCAL_GET(hkThreadNumber);
}

void hkMemoryInitUtil::SyncInfo::syncLocalInitNodes(const SyncInfo& baseSystemInfo, ExcludeFlags exclude)
{
#ifdef HK_MEMORY_TRACKER_ENABLE
    // Must do this first of all so the tracker sees all subsequent allocations
    hkMemoryTracker::setInstance(baseSystemInfo.m_memoryTracker);
#endif
    hkMemoryRouter::s_fallbackRouter = baseSystemInfo.m_fallBackMemoryRouter;

    // Basic singletons (non hkSingleton).
    hkStackTracer::replaceImplementation(baseSystemInfo.m_stackTracerImpl);
    hkMemoryRouter::replaceInstance(baseSystemInfo.m_memoryRouter);
    hkMemorySystem::replaceInstance(baseSystemInfo.m_memorySystem);

    if (baseSystemInfo.m_systemInitNodes)
    {
        // Disable patch registry synchronization.
        hkBaseSystem::InitNode* verReg = findByName(baseSystemInfo.m_systemInitNodes, "hkReflect::Version::PatchRegistry");
        const char* restVerReg = HK_NULL;

        // We change the name of this node so it will not get synchronised (we will never find a matching name)
        if (verReg && exclude.anyIsSet(EXCLUDE_VERSION_REGISTRY))
        {
            restVerReg = verReg->m_name;
            verReg->m_name = "DONOTSYNC";
        }

        // Populate the singletons list.
        populate(&hkBaseSystem::InitNode::s_listHead, baseSystemInfo.m_systemInitNodes);
        if (restVerReg)
        {
            verReg->m_name = restVerReg;
        }
    }

    // Set thread number
    HK_THREAD_LOCAL_SET(hkThreadNumber, baseSystemInfo.m_threadNumber);
}


extern HK_THREAD_LOCAL( hkMonitorStream* ) hkMonitorStream__m_instance;
extern HK_THREAD_LOCAL( hkInt32 ) hkMonitorStream__m_lockCount;

void hkMemoryInitUtil::SyncInfo::syncDebugInfo(const SyncInfo& info)
{
    // Monitor stream.
    HK_THREAD_LOCAL_SET(hkMonitorStream__m_instance, info.m_monitors);
    HK_THREAD_LOCAL_SET(hkMonitorStream__m_lockCount, 0);

    // Multithreading singletons.
    HK_ON_DEBUG_MULTI_THREADING( hkMultiThreadCheck::s_criticalSection = info.m_mtCheckSection; )
    HK_ON_DEBUG_MULTI_THREADING( hkMultiThreadCheck::s_stackTree = info.m_mtCheckStackTree; )
}

#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
