// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Router/hkMemoryRouter.h>

namespace
{
    struct RouterSizeInfo
    {
        void* m_allocator;
        hkInt32 m_size;
        hkInt32 m_alignedOffset; // only used in aligned{Allocate,Deallocate}
        #if HK_POINTER_SIZE==4
            int pad16;
        #endif
    };
    HK_COMPILE_TIME_ASSERT(sizeof(RouterSizeInfo)==16);
}

HK_THREAD_LOCAL(hkMemoryRouter*) hkMemoryRouter::s_memoryRouter;
hkMemoryRouter* hkMemoryRouter::s_fallbackRouter = HK_NULL;

void HK_CALL hkMemoryRouter::replaceInstance(_In_ hkMemoryRouter* a)
{
    HK_THREAD_LOCAL_SET(s_memoryRouter, a);
}

void HK_CALL hkMemoryRouter::replaceFallbackInstance(_In_ hkMemoryRouter* m)
{
    s_fallbackRouter = m;
}

hkMemoryRouter::hkMemoryRouter()
: m_temp(HK_NULL)
, m_heap(HK_NULL)
, m_debug(HK_NULL)
, m_solver(HK_NULL)
{
}

_Ret_notnull_ _Post_writable_byte_size_(nbytes) void* HK_CALL hkMemoryRouter::alignedAlloc( Allocator& b, int nbytes, int alignment )
{
    // allocate enough to hold the nbytes, the size info and the alignment window
    char* unaligned = reinterpret_cast<char*>(  b.blockAlloc(alignment + nbytes + hkSizeOf(RouterSizeInfo)) );

    HK_ASSERT( 0x30a343ed, unaligned != HK_NULL, "Out of memory" );

    // the aligned memory is the nearest aligned block, taking into account that the
    // sizeinfo which is placed just before the returned pointer.
    char* aligned = reinterpret_cast<char*>( HK_NEXT_MULTIPLE_OF( alignment, hkUlong(unaligned+hkSizeOf(RouterSizeInfo))) );

    // store the sizeinfo just before the returned pointer
    {
        RouterSizeInfo* x = reinterpret_cast<RouterSizeInfo*>(aligned) - 1;
        x->m_allocator = &b;
        x->m_size = nbytes + alignment;
        x->m_alignedOffset = (int)(aligned - unaligned);
    }
    return static_cast<void*>(aligned);
}

void HK_CALL hkMemoryRouter::alignedFree( Allocator& b, _In_opt_ void* p )
{
    if(p)
    {
        RouterSizeInfo* x = static_cast<RouterSizeInfo*>(p) - 1;
        x->m_allocator = HK_NULL;
        char* unaligned = reinterpret_cast<char*>(p) - x->m_alignedOffset;
        b.blockFree( static_cast<void*>(unaligned), x->m_size + hkSizeOf(RouterSizeInfo) );
    }
}

void* HK_CALL hkMemoryRouter::alignedRealloc(Allocator& b, void* pold, int oldSize, int newSize, int alignment)
{
    void* pnew = alignedAlloc(b, newSize, alignment);
    if ( pnew )
    {
        hkMemUtil::memCpy(pnew, pold, hkMath::min2(oldSize, newSize));
    }
    alignedFree(b, pold);
    return pnew;
}

_Ret_notnull_ const void* HK_CALL hkMemoryRouter::getAlignedAllocStartAddress(Allocator& b, _In_ const void* p)
{
    const RouterSizeInfo* x = static_cast<const RouterSizeInfo*>(p) - 1;
    const char* unaligned = reinterpret_cast<const char*>(p) - x->m_alignedOffset;
    return unaligned;
}

int HK_CALL hkMemoryRouter::getAlignedAllocatedSize(Allocator& a, const void* ptr, int nbytes)
{
    const void* unaligned = hkMemoryRouter::getAlignedAllocStartAddress(a, ptr);
    return a.getAllocatedSize(unaligned, nbytes) - hkGetByteOffsetInt(unaligned, ptr);
}

_Ret_notnull_ _Post_writable_byte_size_(nbytes) void* HK_CALL hkMemoryRouter::easyAlloc( Allocator& b, int nbytes )
{
    RouterSizeInfo* x = static_cast<RouterSizeInfo*>( b.blockAlloc(nbytes + hkSizeOf(RouterSizeInfo)) );
    HK_ASSERT( 0x30a343ef, x != HK_NULL, "Out of memory" );
    x->m_allocator = &b;
    x->m_size = nbytes;
    return static_cast<void*>(x+1);
}

/* static */hk_size_t HK_CALL hkMemoryRouter::getEasyAllocSize(Allocator& b, _In_ const void* ptr)
{
    const RouterSizeInfo* x = ((const RouterSizeInfo*)ptr) - 1;
    return x->m_size;
}

/* static */_Ret_notnull_ const void* HK_CALL hkMemoryRouter::getEasyAllocStartAddress(Allocator& b, _In_ const void* ptr)
{
    const RouterSizeInfo* x = ((const RouterSizeInfo*)ptr) - 1;
    return (const void*)x;
}

void HK_CALL hkMemoryRouter::easyFree( Allocator& b, _In_opt_ void* p )
{
    if (p) // as we allowed by convention to deallocate NULL
    {
        RouterSizeInfo* x = static_cast<RouterSizeInfo*>(p) - 1;
        x->m_allocator = HK_NULL;
        b.blockFree( static_cast<void*>(x), x->m_size + hkSizeOf(RouterSizeInfo) );
    }
}

void hkMemoryRouter::resetPeakMemoryStatistics()
{
    m_debug->resetPeakMemoryStatistics();
    m_heap->resetPeakMemoryStatistics();
    m_solver->resetPeakMemoryStatistics();
    m_temp->resetPeakMemoryStatistics();
}

void hkMemoryRouter::advanceFrame()
{
    m_debug->advanceFrame();
    m_heap->advanceFrame();
    m_solver->advanceFrame();
    m_temp->advanceFrame();
}

void hkMemoryRouter::pushFrame()
{
    m_debug->pushFrame();
    m_heap->pushFrame();
    m_solver->pushFrame();
    m_temp->pushFrame();
}

void hkMemoryRouter::popFrame()
{
    m_debug->popFrame();
    m_heap->popFrame();
    m_solver->popFrame();
    m_temp->popFrame();
}

_Ret_notnull_ _Post_writable_byte_size_(nbytes) void* hkMemory::heapBlockAlloc(hk_size_t nbytes)
{
    return hkMemoryRouter::getInstance().heap().blockAlloc( int(nbytes) );
}

void hkMemory::heapBlockFree(_In_opt_bytecount_(nbytes) void* p, hk_size_t nbytes)
{
    hkMemoryRouter::getInstance().heap().blockFree(p, int(nbytes));
}

namespace
{
    struct MemHeapAllocator : public hkMemoryAllocator
    {
        virtual _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* blockAlloc(int numBytes) HK_OVERRIDE
        {
            return hkMemoryRouter::getInstance().heap().blockAlloc(numBytes);
        }
        virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE
        {
            hkMemoryRouter::getInstance().heap().blockFree(p, numBytes);
        }
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut) void* bufAlloc(int& reqNumBytesInOut) HK_OVERRIDE
        {
            return hkMemoryRouter::getInstance().heap().bufAlloc(reqNumBytesInOut);
        }
        virtual void bufFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE
        {
            hkMemoryRouter::getInstance().heap().bufFree(p, numBytes);
        }
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut) void* bufRealloc(_In_reads_bytes_(oldNumBytes) void* pold, int oldNumBytes, int& reqNumBytesInOut) HK_OVERRIDE
        {
            return hkMemoryRouter::getInstance().heap().bufRealloc(pold, oldNumBytes, reqNumBytesInOut);
        }
        virtual void getMemoryStatistics(MemoryStatistics& u) const HK_OVERRIDE
        {
            hkMemoryRouter::getInstance().heap().getMemoryStatistics(u);
        }
        virtual int getAllocatedSize(_In_bytecount_(nbytes) const void* obj, int nbytes) const HK_OVERRIDE
        {
            return hkMemoryRouter::getInstance().heap().getAllocatedSize(obj, nbytes);
        }
        static MemHeapAllocator s_alloc;
    };
    MemHeapAllocator MemHeapAllocator::s_alloc;
}

_Ret_notnull_ hkMemoryAllocator* HK_CALL hkMemHeapAllocator()
{
    return &MemHeapAllocator::s_alloc;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
