// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// trick to have hkDeallocate() work with a void* argument (sizeof(void) is undefined, but makes sense to be 1)
template <typename T> struct hkSizeOfTypeOrVoid { enum { val = sizeof(T) }; };
template <> struct hkSizeOfTypeOrVoid<void> { enum { val = 1 }; };
template <> struct hkSizeOfTypeOrVoid<const void> { enum { val = 1 }; };

class hkMemorySnapshot;
struct hkResult;

    /// Base class for all memory allocation implementations.
    /// A derived class may simply override blockAlloc, blockFree,
    /// getMemoryStatistics and getAllocatedSize with an alternative
    /// allocation system.
    /// If an implementation may supply more memory than requested
    /// or may be able to reallocate without copying, it may benefit from
    /// also overriding the buf* methods.
class HK_EXPORT_COMMON hkMemoryAllocator
{
    public:
        HK_DECLARE_PLACEMENT_ALLOCATOR();
        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true));

        typedef void (HK_CALL *MemoryWalkCallback)(_Inout_bytecount_(size) void* start, hk_size_t size, bool allocated, int pool, void* param);

            /// A struct holding memory allocation information.
            /// Not all allocators support all statistics. If a statistic is
            /// unavailable/unsupported, its value is -1.
        struct HK_EXPORT_COMMON MemoryStatistics
        {
                /// The default constructor initializes all values to -1.
            MemoryStatistics() : m_allocated(-1), m_inUse(-1), m_peakInUse(-1), m_available(-1), m_totalAvailable(-1), m_largestBlock(-1) {}

                /// Infinite as far as the memory system is concerned.
            enum { INFINITE_SIZE = -1 };

                /// Total amount of allocated memory. Allocated memory means that it is 'owned' by the hkMemory reporting it -
                /// but the actual memory inside may be available or used. Thus m_inUse + m_available <= m_allocated. The difference will
                /// be due to overhead for managing the memory.
            hkLong m_allocated;

                /// Total used memory
            hkLong m_inUse;

                /// The peak usage
            hkLong m_peakInUse;

                /// Total available from the _allocated_ memory. This is NOT the total amount of memory that is potentially available,
                /// only the amount that remains available from what has been allocated.
            hkLong m_available;

                /// The approx total amount of memory that is available. The value is the amount of available memory (m_available)
                /// plus the approx amount of memory that is potentially available but not yet allocated.
                /// Depending on the allocator being used, this value may be meaningless. For example for an allocator with virtual memory
                /// the 'total amount of available' is somewhat meaningless. In those scenarios, m_totalMemory will be INFINITE_SIZE
            hkLong m_totalAvailable;

                /// Largest block available. On some memory allocators this is meaningless (for example with virtual memory). In that
                /// case this will be set to INFINITE_SIZE
            hkLong m_largestBlock;
        };

            /// The ExtendedInterface is an optional interface that an implementation of an Allocator may implement.
            /// In implementing the interface - the allocator can be used to provide easily a full implementation of
            /// a memory system, using the hkFreeListMemorySystem. (NOTE - that the underlying memory allocator does not
            /// need to be a freelist to work within the hkFreeListMemorySystem)
        struct HK_EXPORT_COMMON ExtendedInterface
        {
            virtual ~ExtendedInterface() {}

                /// Try to coalesce freed blocks and return them to the parent allocator.
            virtual void garbageCollect() = 0;
                /// Do an incremental garbage collection. The parameter controls how much work will be performed in the
                /// collection. Passing 0 means the default amount. How the parameter controls the collection is dependent
                /// on the underlying implementation.
            virtual void incrementalGarbageCollect(int numBlocks) = 0;
                /// Set the soft memory limit
            virtual hkResult setMemorySoftLimit(hk_size_t maxMemory) = 0;
                /// Get the soft memory limit
            virtual hk_size_t getMemorySoftLimit() const = 0;
                /// Returns if an alloc of the size specified will succeed.
            virtual bool canAllocTotal( int numBytes ) = 0;
                /// Traverses all of the memory used by the subsystem (allocated and free). The order that allocations
                /// are returned is undefined. Not all memory allocators can walk memory - if an implementation cannot
                /// it will return HK_FAILURE from this method.
            virtual hkResult walkMemory(MemoryWalkCallback callback, void* param) = 0;
                /// Returns (can be somewhat approximate) the total memory allocated. The implementation of the method
                /// must be thread safe - and be so ideally without a critical section. If a critical section is used it
                /// can seriously undermine performance.
            virtual hk_size_t getApproxTotalAllocated() const = 0;
                /// Sets the scrub values used when allocating and deallocating memory from this allocator.
                /// Since not all allocators will perform scrubbing, this is part of the extended interface.
            virtual void setScrubValues(hkUint32 allocValue, hkUint32 freeValue) = 0;
                /// Add allocations to the given memory snapshot.
                /// Allocations requests which were made to a parent may be given with the parentId.
                /// Returns -1 if not supported.
            virtual int addToSnapshot(hkMemorySnapshot& snap, int /*hkMemoryWalk::ProviderId*/ parentId) = 0;
        };

            ///
        virtual ~hkMemoryAllocator();

        
            /// Allocate numBytes.
            /// Allocations of zero bytes may return either the null pointer or
            /// act as if numBytes was nonzero.
        _Ret_notnull_ _Post_writable_byte_size_(numBytes)
        virtual void* blockAlloc( int numBytes ) = 0;

            /// Free numBytes. This method needs to handle p being null, as well
            /// as matching the handling of zero byte allocations as described in blockAlloc.
        virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes ) = 0;

            /// Request a buffer of reqNumBytesInOut.
            /// The implementation may modify reqNumInOut to return a larger
            /// block than requested.
            /// Default implementation forwards to blockAlloc.
        _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut)
        virtual void* bufAlloc( int& reqNumBytesInOut );

            /// Free a buffer.
            /// Default implementation forwards to blockFree.
            /// \a numElem is the number of elements as returned by the bufAlloc in reqNumInOut
        virtual void bufFree(_In_opt_bytecount_(numBytes) void* p, int numBytes );

            /// Reallocate a buffer.
            /// Default implementation is blockAlloc, memCpy, blockFree.
        _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut)
        virtual void* bufRealloc(_In_reads_bytes_(oldNumBytes) void* pold, int oldNumBytes, int& reqNumBytesInOut );
        

            /// Allocate several blocks at the same time.
            /// May be more efficient than individual allocations if the allocator must be locked.
        virtual void blockAllocBatch(_Out_writes_all_(numPtrs) void** ptrsOut, int numPtrs, int blockSize);

            /// Free several blocks at the same time.
        virtual void blockFreeBatch(_In_reads_(numPtrs) void** ptrsIn, int numPtrs, int blockSize);

            /// Work out memory statistics. This function tries to work out fairly accurately details about memory usage -
            /// this functions performance may not be great depending on the underlying implementation.
            /// See hkMemoryStatistics for details of the information returned.
            /// NOTE! That if you have a memory limit set (via setMemoryLimit) the values returned will reflect that limit
            /// you must have the limit set to zero if you want to find out the 'true' memory statistics.
        virtual void getMemoryStatistics( MemoryStatistics& u ) const = 0;

            /// Return the padded size. Most memory system will round up the allocation size or add extra headers and footers.
            /// This function should return the (padded) size which potentially is usable by the user
            ///
            /// The size (and any size between the original allocated size, and the value returned from getAllocatedSize) can be passed
            /// to this method, and it will produce the same result. Also note the block/bufFree can take any value between the original
            /// allocated size, and the size returned from getAllocatedSize, and work correctly.
        virtual int getAllocatedSize(_In_bytecount_(nbytes) const void* obj, int nbytes) const = 0;

            /// Reset internal counts of mem usage.
        virtual void resetPeakMemoryStatistics() {}

        /// \name Allocation frames
        ///
        /// Allocation frames are used to indicate how long temp allocations can
        /// be kept. Specifically, any temp allocation must be freed in the same
        /// allocation frame it was allocated in.
        ///
        /// \{

        /// Ends the current (sub-)frame or and starts a new one.
        virtual void advanceFrame() {}

        /// Pushes the current frame onto the frame stack and starts a sequence
        /// of sub-frames. To end the sequence of sub-frames, call \ref popFrame.
        ///
        /// Frames pushed onto the stack remain active, which means that temp
        /// allocations which belong to this frame can remain active as well, or
        /// can even be freed while we're in a sub-frame.
        virtual void pushFrame() {}

        /// Ends the sequence of sub-frames and pops the parent frame from the stack.
        virtual void popFrame() {}

        /// \}

        void* blockAllocZero( int numBytes )
        {
            void* p = blockAlloc(numBytes);
            hkMemUtil::memSet(p, 0, numBytes);
            return p;
        }

            /// Optional methods on this allocator.
        virtual _Ret_maybenull_ ExtendedInterface* getExtendedInterface() { return HK_NULL; }

            /// Allocate a buffer which is a multiple of SIZE_ELEM.
        template<typename TYPE>
        inline _Ret_notnull_ _Post_writable_size_(reqNumInOut) TYPE* _bufAlloc(int& reqNumInOut)
        {
            void* p = bufAlloc2(hkSizeOfTypeOrVoid<TYPE>::val, reqNumInOut);
            return static_cast<TYPE*>(p);
        }

            /// Free a buffer which is a multiple of SIZE_ELEM.
        template<typename TYPE>
        inline void _bufFree(_In_opt_bytecount_(_Inexpressible_()) void* p, int numElem )
        {
            bufFree2(p, hkSizeOfTypeOrVoid<TYPE>::val, numElem);
        }

            /// Reallocate a buffer which is a multiple of SIZE_ELEM.
        template<typename TYPE>
        inline _Ret_notnull_ _Post_writable_size_(reqNumInOut) TYPE* _bufRealloc(
            _In_reads_bytes_(_Inexpressible_()) void* pold, int oldNum, int& reqNumInOut )
        {
            void* p = bufRealloc2(pold, hkSizeOfTypeOrVoid<TYPE>::val, oldNum, reqNumInOut);
            return static_cast<TYPE*>(p);
        }

            /// Allocate a block to hold n TYPEs
        template<typename TYPE>
        inline _Ret_notnull_ _Post_writable_size_(n) TYPE* _blockAlloc(int n)
        {
            return static_cast<TYPE*>(blockAlloc2(hkSizeOfTypeOrVoid<TYPE>::val, n));
        }
            /// Free a block of n TYPEs
        template<typename TYPE>
        inline void _blockFree(_In_opt_count_(n) TYPE* p, int n )
        {
            blockFree2(p, hkSizeOfTypeOrVoid<TYPE>::val, n);
        }

        template<typename TC>
        _Ret_notnull_ TC* create() { void* p = this->blockAlloc(sizeof(TC)); return new (p) TC; }
        template<typename TC, typename T0>
        _Ret_notnull_ TC* create(T0 t0) { void* p = this->blockAlloc(sizeof(TC)); return new (p) TC(t0); }
        template<typename TC, typename T0, typename T1>
        _Ret_notnull_ TC* create(T0 t0, T1 t1) { void* p = this->blockAlloc(sizeof(TC)); return new (p) TC(t0,t1); }
        template<typename TC, typename T0, typename T1, typename T2>
        _Ret_notnull_ TC* create(T0 t0, T1 t1, T2 t2) { void* p = this->blockAlloc((int)sizeof(TC)); return new (p) TC(t0,t1,t2); }
        template<typename TC, typename T0, typename T1, typename T2, typename T3>
        _Ret_notnull_ TC* create(T0 t0, T1 t1, T2 t2, T3 t3) { void* p = this->blockAlloc((int)sizeof(TC)); return new (p) TC(t0,t1,t2, t3); }
        template<typename TC, typename T0, typename T1, typename T2, typename T3, typename T4>
        _Ret_notnull_ TC* create(T0 t0, T1 t1, T2 t2, T3 t3, T4 t4) { void* p = this->blockAlloc((int)sizeof(TC)); return new (p) TC(t0,t1,t2, t3, t4); }
        template<typename TC, typename T0, typename T1, typename T2, typename T3, typename T4, typename T5>
        _Ret_notnull_ TC* create(T0 t0, T1 t1, T2 t2, T3 t3, T4 t4, T5 t5) { void* p = this->blockAlloc((int)sizeof(TC)); return new (p) TC(t0, t1, t2, t3, t4, t5); }
        template<typename TC, typename T0, typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        _Ret_notnull_ TC* create(T0 t0, T1 t1, T2 t2, T3 t3, T4 t4, T5 t5, T6 t6) { void* p = this->blockAlloc((int)sizeof(TC)); return new (p)TC(t0, t1, t2, t3, t4, t5, t6); }

        template<typename TD>
        void destroy(_Inout_opt_ TD* obj) { if (obj) { obj->~TD(); this->blockFree(obj, (int)sizeof(TD)); } }

    public:

        /// Status of the memory manager.
        enum MemoryState
        {
            MEMORY_STATE_OK,            ///< Memory OK
            MEMORY_STATE_OUT_OF_MEMORY  ///< Low on Memory
        };

    protected:
        _Analysis_noreturn_ void* outOfMemory(_In_opt_z_ const char* msg=nullptr);
            // 2-parameter versions which do a checked size*num as part of the allocations.
        _Ret_notnull_ _Post_writable_byte_size_(numInOut * size) void* bufAlloc2(int size, int& numInOut);
        void bufFree2(_In_opt_bytecount_(num * size) void* p, int size, int num);
        _Ret_notnull_ _Post_writable_byte_size_(reqNumInOut * size) void* bufRealloc2(_In_reads_bytes_(oldNum * size) void* pold, int size, int oldNum, int& reqNumInOut);
        _Ret_notnull_ _Post_writable_byte_size_(size * num) void* blockAlloc2(int size, int num);
        void blockFree2(_In_opt_bytecount_(num * size) void* p, int size, int num);
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
