// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/Transient/hkTransientAllocator.h>

void hkTransientAllocator::ensureBlockCanAllocate(int numBytes)
{
    // Large blocks go somewhere else
    if(numBytes >= m_blockSize)
    {
        return;
    }

    const bool blockListEmpty = m_blocks.getSize() == 0;
    // See if we can fit in the current block. If not, move along a block
    // If the list is full or empty, allocate a new block
    if(blockListEmpty || (m_currentBlockOffset + numBytes > m_blockSize))
    {
        // Need to move along a block
        if(blockListEmpty || (m_currentBlock + 1 >= m_blocks.getSize()))
        {
            m_blocks._expandOne(*m_chainedAllocator) = m_chainedAllocator->blockAlloc(m_blockSize);
        }
        m_currentBlock++;
        m_currentBlockOffset = 0;
    }
}

_Ret_notnull_ _Post_writable_byte_size_(numBytes) void* hkTransientAllocator::blockAlloc(int numBytes)
{
    numBytes = roundUpTo16(numBytes);
    m_currentUsed += numBytes;
    if(m_currentUsed > m_peakUse)
    {
        m_peakUse = m_currentUsed;
    }

    // Large blocks get allocated directly from the chained allocator
    // Blocks equal to the block size get forwarded to the large block
    // allocator, so that two hkTransientAllocators chained together do
    // not constantly trigger block reallocations
    if(numBytes >= m_blockSize)
    {
        void* ret = m_chainedAllocator->blockAlloc(numBytes);
        // Large blocks are not the common case, they are stored in an array
        m_largeBlocks._expandOne(*m_chainedAllocator).set(ret, numBytes);
        return ret;
    }
    ensureBlockCanAllocate(numBytes);
    void* ret = hkAddByteOffset(m_blocks[m_currentBlock], m_currentBlockOffset);
    m_currentBlockOffset += numBytes;

    // Scrub the allocated memory in debug
    // 0xfe for allocated memory
#if defined(HK_DEBUG)
    hkMemUtil::memSet(ret, 0xfe, numBytes);
#endif
    return ret;
}

void hkTransientAllocator::blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes)
{
    numBytes = roundUpTo16(numBytes);
    m_currentUsed -= numBytes;

    // Large blocks get freed, small ones don't
    if(numBytes >= m_blockSize)
    {
        // Deallocating a single large block is not a fast path -- it must look it up in the array
        for(int i=0;i<m_largeBlocks.getSize();i++)
        {
            if(m_largeBlocks[i].m_alloc == p)
            {
                m_largeBlocks.removeAt(i);
                // RemoteAt has put a new value in position i, we need to view it again
                return m_chainedAllocator->blockFree(p, numBytes);
            }
        }
        HK_ASSERT_NO_MSG(0xcc0807, 0); // The block was not in the list
    }
}

void hkTransientAllocator::getMemoryStatistics( MemoryStatistics& u ) const
{
    u.m_allocated = m_currentUsed;
    u.m_peakInUse = m_peakUse;
}

void hkTransientAllocator::resetPeakMemoryStatistics()
{
    m_peakUse = m_currentUsed;
}

int hkTransientAllocator::getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const
{
    return numBytes;
}

void hkTransientAllocator::clear()
{
    // -1 signals that no blocks are allocated
    m_currentBlock = (m_blocks.getSize() > 0 ? 0 : -1);
    m_currentBlockOffset = 0;
    for(int i=0;i<m_largeBlocks.getSize();i++)
    {
        m_chainedAllocator->blockFree(m_largeBlocks[i].m_alloc, m_largeBlocks[i].m_size);
    }
    m_largeBlocks._setSize(*m_chainedAllocator, 0);
}

void hkTransientAllocator::clearAndDeallocate()
{
    clear();
    for(int i=0;i<m_blocks.getSize();i++)
    {
#if defined(HK_DEBUG)
        // 0xdd for freed blocks
        hkString::memSet(m_blocks[i], 0xdd, m_blockSize);
#endif
        m_chainedAllocator->blockFree(m_blocks[i], m_blockSize);
    }
    m_blocks._clearAndDeallocate(*m_chainedAllocator);
    m_largeBlocks._clearAndDeallocate(*m_chainedAllocator);
    m_currentBlock = -1;
}

hkTransientAllocator::hkTransientAllocator(_In_ class hkMemoryAllocator* chainedAllocator, hkUint32 blockSize)
    : m_currentUsed(0)
    , m_peakUse(0)
    , m_blockSize(blockSize)
    , m_currentBlock(-1)
    , m_currentBlockOffset(0)
    , m_chainedAllocator(chainedAllocator)
{
}

hkTransientAllocator::~hkTransientAllocator()
{
    clearAndDeallocate();
    m_currentBlock = -1;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
