// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

    /// This class is a block cache used in conjunction with another allocator.
    ///
    /// Notes:
    ///    - It is highly optimized for handling memory for a single thread. It
    ///      is doing so by caching a limited number of blocks in a thread local
    ///      free list. There is a several free lists for different sizes of blocks.
    ///    - Each free list (for each size) can hold only a maximum number of items (m_maxNumElemsOnFreeList)
    ///    - If maxNumElemsOnFreeList==0 than the free list is not checked at all and the hkThreadMemory
    ///      calls the appropriate parent allocator functions
    ///    - If for a given size, there is no free block on the free list, some blocks are fetched by calling
    ///      parents blockAllocBatch() method
    ///    - If a block is freed, it is assumed that this block was allocated by this memory manager, its
    ///      corresponding free list is found (by rounding up the size of the block to the next suitable
    ///      block size) and the block added to the free list.
    ///      If the number of elements on this particular free list is exceeding m_maxNumElemsOnFreeList, some
    ///      blocks are forwarded to the parents blockFreeBatch() method.
    ///
    /// Notes:
    ///     - all allocations should be 16 byte aligned.
    ///     - Only if you use this class with the pool memory, you are allowed to set maxNumElemsOnFreeList != 0
class HK_EXPORT_COMMON hkThreadMemory : public hkMemoryAllocator
{
    public:

        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true));
        HK_DECLARE_PLACEMENT_ALLOCATOR();

        struct FreeList
        {
            HK_RECORD_ATTR( hk::MemoryTracker( opaque = true ) );

            FreeList() : m_head( HK_NULL ), m_numElem( 0 ) {}
            void put(_In_ void* p)
            {
                m_numElem += 1;
                FreeElem* fl = static_cast<FreeElem*>(p);
                fl->m_next = m_head;
                m_head = fl;
            }
            _Ret_maybenull_ void* get()
            {
                if( void* p = m_head )
                {
                    m_numElem -= 1;
                    m_head = m_head->m_next;
                    return p;
                }
                return HK_NULL;
            }

            struct FreeElem { FreeElem* m_next; };
            FreeElem* m_head;
            int m_numElem;
        };

        // hkMemoryAllocator interface
        virtual _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* blockAlloc(int numBytes) HK_OVERRIDE;
        virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE;
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* bufAlloc(int& reqNumInOut) HK_OVERRIDE;
        virtual void bufFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE;
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* bufRealloc(_In_opt_bytecount_(oldNum) void* p, int oldNum, int& reqNumInOut) HK_OVERRIDE;
        virtual void blockAllocBatch(_Out_writes_all_(numPtrs) void** ptrsOut, int numPtrs, int blockSize) HK_OVERRIDE;
        virtual void blockFreeBatch(_In_reads_(numPtrs) void** ptrsIn, int numPtrs, int blockSize) HK_OVERRIDE;
        virtual void getMemoryStatistics( MemoryStatistics& u ) const HK_OVERRIDE;
        virtual int getAllocatedSize( _In_bytecount_(numBytes) const void* obj, int nbytes ) const HK_OVERRIDE;

        hkThreadMemory();

            /// Creates a new instance of a thread memory.
        hkThreadMemory(_In_opt_ hkMemoryAllocator* mainMemoryManager);

            /// Calls releaseCachedMemory
        HK_INLINE ~hkThreadMemory() { releaseCachedMemory(); }

            ///
        void setMemory(_In_opt_ hkMemoryAllocator* mainMemoryManager, int maxNumElemsOnFreeList=8 );

            /// Gives all local memory back to the main memory manager
        virtual void releaseCachedMemory();

            /// Get simple memory statistics of cached blocks
        hk_size_t getCachedSizeUnchecked() const;

    public:

        enum
        {
#if defined(HK_REAL_IS_DOUBLE) || defined(HK_PLATFORM_PS4)
            MEMORY_SMALL_BLOCK_RSHIFT_BITS = 5,
            MEMORY_SMALL_BLOCK_ADD = 0x1f,          // Added before the shift
            MEMORY_MAX_SMALL_ROW = 12,              // The number of small chunk sizes
#else
            MEMORY_SMALL_BLOCK_RSHIFT_BITS = 4,
            MEMORY_SMALL_BLOCK_ADD = 0xf,           // Added before the shift
            MEMORY_MAX_SMALL_ROW = 13,              // The number of small chunk sizes
#endif

            // The number of small and large chunk sizes
            MEMORY_MAX_ALL_ROW = (MEMORY_MAX_SMALL_ROW + 4),

            // The largest small block we allocate from this pool - this is specially sized to match sizeof(hkpRigidBody)
            MEMORY_MAX_SIZE_SMALL_BLOCK = 640, // (512 + 32) aligned to 128 alignment

            // The largest large block we allocate from this pool
            MEMORY_MAX_SIZE_LARGE_BLOCK = 8192,

            // The low bits we ignore when indexing into the large arrays
            MEMORY_LARGE_BLOCK_RSHIFT_BITS = 10,
        };

    protected:

            // thread memory talks to main memory in batches of this size
        enum { BATCH_SIZE = 4 };

            // Called when an allocation from an empty row is requested.
        _Ret_notnull_ void* onRowEmpty(int row);

            // Called when an deallocation from a full row is requested. p is a ptr that will be added to that row
        void onRowFull(int row, _In_ void* p);

            // Clear all of the pointers on that row.
        void clearRow(int rowIndex);

        //
        //  Internal public section
        //

    public:

            // pointer to the main memory
        hkMemoryAllocator* m_memory;

            // the maximum number of elements on each free list before the parent hkMemory is called
        int m_maxNumElemsOnFreeList;

            // free list for blocks of each size
        FreeList m_free_list[MEMORY_MAX_ALL_ROW];

            // a lookup table of size of each block size
        static const int s_row_to_size_lut[MEMORY_MAX_ALL_ROW];

            // a lookup table of sizes to small block size
        static const char s_small_size_to_row_lut[(MEMORY_MAX_SIZE_SMALL_BLOCK >> MEMORY_SMALL_BLOCK_RSHIFT_BITS)+1];

            // a lookup table of sizes to large block size
        static const int s_large_size_to_row_lut[ (MEMORY_MAX_SIZE_LARGE_BLOCK >> MEMORY_LARGE_BLOCK_RSHIFT_BITS) ];
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
