// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/Thread/hkThreadMemory.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>


//////////////////////////////////////////////////////////////////////////
// Initialize static lookup tables
//////////////////////////////////////////////////////////////////////////

namespace
{
    template<int SIZE> struct SizeToRow
    {
    #if defined(HK_REAL_IS_DOUBLE) || defined(HK_PLATFORM_PS4)
        enum { Offset = -1 };
    #else
        enum { Offset = 0 };
    #endif

        enum { Row =
            // small blocks
    #if !defined(HK_REAL_IS_DOUBLE) && !defined(HK_PLATFORM_PS4)
            ( SIZE <= 16 )   ? 1 :
    #endif
            ( SIZE <= 32 )   ? 2 + Offset :
            ( SIZE <= 48 )   ? 3 + Offset :
            ( SIZE <= 64 )   ? 4 + Offset :
            ( SIZE <= 96 )   ? 5 + Offset :
            ( SIZE <= 128 )  ? 6 + Offset :
            ( SIZE <= 160 )  ? 7 + Offset :
            ( SIZE <= 192 )  ? 8 + Offset :
            ( SIZE <= 256 )  ? 9 + Offset :
            ( SIZE <= 320 )  ? 10 + Offset :
            ( SIZE <= 512 )  ? 11 + Offset :
            ( SIZE <= hkThreadMemory::MEMORY_MAX_SIZE_SMALL_BLOCK ) ? 12 + Offset :

            // large blocks
            ( SIZE <= 1024 ) ? 13 + Offset :
            ( SIZE <= 2048 ) ? 14 + Offset :
            ( SIZE <= 4096 ) ? 15 + Offset :
            ( SIZE <= 8192 ) ? 16 + Offset :

            // end
            -1
        };
    };

    template<int INDEX> struct SmallBlockTable
    {
        enum { Size = INDEX << hkThreadMemory::MEMORY_SMALL_BLOCK_RSHIFT_BITS };
        enum { Row = SizeToRow<Size>::Row };
    };
    template<int INDEX> struct LargeBlockTable
    {
        enum { Size = (INDEX + 1) << hkThreadMemory::MEMORY_LARGE_BLOCK_RSHIFT_BITS };
        enum { Row = SizeToRow<Size>::Row };
    };

    template< template<int INDEX> class TABLE, int ROW, int INDEX> struct MaxRowSize
    {
        enum { Size = ( (int)TABLE<INDEX>::Row == ROW ) ? ( (int)TABLE<INDEX>::Size ) : ( (int)MaxRowSize<TABLE, ROW, INDEX-1>::Size ) };
    };
    template< template<int INDEX> class TABLE, int ROW> struct MaxRowSize<TABLE, ROW, -1>
    {
        enum { Size = 0 };
    };
    template<int ROW> struct RowToMaxSize
    {
        enum { SmallSize = MaxRowSize< SmallBlockTable, ROW, HK_COUNT_OF(hkThreadMemory::s_small_size_to_row_lut)-1 >::Size };
        enum { LargeSize = MaxRowSize< LargeBlockTable, ROW, HK_COUNT_OF(hkThreadMemory::s_large_size_to_row_lut)-1 >::Size };
        enum { Size = ( (int)SmallSize > (int)LargeSize ) ? (int)SmallSize : (int)LargeSize };
    };
}

// Macros used in HK_PP_FOR() below.
// That calls our macro with "I" in descending order (from NUM to 0), so we reverse it to get an ascending order (from 0 to NUM).
#define SmallSizeToRow_Action(I) SmallBlockTable< HK_COUNT_OF(hkThreadMemory::s_small_size_to_row_lut)-I-1 >::Row HK_PP_COMMA_IF(I)
#define LargeSizeToRow_Action(I) LargeBlockTable< HK_COUNT_OF(hkThreadMemory::s_large_size_to_row_lut)-I-1 >::Row HK_PP_COMMA_IF(I)
#define RowToSize_Action(I) RowToMaxSize< HK_COUNT_OF(hkThreadMemory::s_row_to_size_lut)-I-1 >::Size HK_PP_COMMA_IF(I)

// Note, HK_PP_FOR() only accepts an integer as NUM - we can't use macros for that.
#if defined(HK_REAL_IS_DOUBLE) || defined(HK_PLATFORM_PS4)
    const int hkThreadMemory::s_row_to_size_lut[16] = { HK_PP_FOR( 16, RowToSize_Action ) };
    const char hkThreadMemory::s_small_size_to_row_lut[21] = { HK_PP_FOR( 21, SmallSizeToRow_Action ) };
#else
    const int hkThreadMemory::s_row_to_size_lut[17] = { HK_PP_FOR( 17, RowToSize_Action ) };
    const char hkThreadMemory::s_small_size_to_row_lut[41] = { HK_PP_FOR( 41, SmallSizeToRow_Action ) };
#endif
const int hkThreadMemory::s_large_size_to_row_lut[8] = { HK_PP_FOR( 8, LargeSizeToRow_Action ) };

//////////////////////////////////////////////////////////////////////////


HK_ON_DEBUG( hkCriticalSection s_debugLock( 0 ) );

_Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* hkThreadMemory::bufAlloc( int& reqNumInOut )
{
    return blockAlloc(reqNumInOut);
}
void hkThreadMemory::bufFree(_In_opt_bytecount_(numBytes) void* p, int numBytes)
{
    return blockFree(p, numBytes);
}
_Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* hkThreadMemory::bufRealloc(_In_opt_bytecount_(oldNum) void* pold, int oldNum, int& reqNumInOut)
{
    void* pnew = hkThreadMemory::blockAlloc(reqNumInOut);
    if( pnew )
    {
        hkMemUtil::memCpy( pnew, pold, hkMath::min2(reqNumInOut,oldNum));
    }
    hkThreadMemory::blockFree(pold, oldNum);
    return pnew;
}
void hkThreadMemory::blockAllocBatch(_Out_writes_all_(numPtrs) void** ptrsOut, int numPtrs, int blockSize)
{
    if( blockSize <= MEMORY_MAX_SIZE_LARGE_BLOCK)
    {
        for( int i = 0; i < numPtrs; ++i )
        {
            ptrsOut[i] = blockAlloc(blockSize);
        }
    }
    else
    {
        m_memory->blockAllocBatch(ptrsOut, numPtrs, blockSize );
    }
}
void hkThreadMemory::blockFreeBatch(_In_reads_(numPtrs) void** ptrsIn, int numPtrs, int blockSize)
{
    if( blockSize <= MEMORY_MAX_SIZE_LARGE_BLOCK)
    {
        for( int i = 0; i < numPtrs; ++i )
        {
            blockFree(ptrsIn[i], blockSize);
        }
    }
    else
    {
        m_memory->blockFreeBatch(ptrsIn, numPtrs, blockSize );
    }
}

hkThreadMemory::hkThreadMemory()
{
    setMemory(HK_NULL);
}

hkThreadMemory::hkThreadMemory(_In_opt_ hkMemoryAllocator* memoryInstance)
{
    setMemory(memoryInstance);
}


void hkThreadMemory::getMemoryStatistics( hkThreadMemory::MemoryStatistics& u ) const
{
    u.m_allocated = -1;
    u.m_inUse = -1;
    u.m_peakInUse = -1;
    hkLong available = 0;
    for( int i = 0; i < MEMORY_MAX_ALL_ROW; ++i )
    {
        const FreeList& f = m_free_list[i];
        available += f.m_numElem * s_row_to_size_lut[i];
    }
    u.m_available = available;
    u.m_totalAvailable = -1;
    u.m_largestBlock = -1;
}

hk_size_t hkThreadMemory::getCachedSizeUnchecked() const
{
    int sum = 0;
    for (int i =0; i < HK_COUNT_OF(s_row_to_size_lut) ;i++)
    {
        int size = s_row_to_size_lut[i];
        int numFree = m_free_list[i].m_numElem;
        sum += size * numFree;
    }
    return sum;
}

void hkThreadMemory::setMemory(_In_opt_ hkMemoryAllocator* memoryInstance, int maxNumElemsOnFreeList )
{
    if (memoryInstance)
    {
        m_memory = memoryInstance;
        m_maxNumElemsOnFreeList = maxNumElemsOnFreeList;
    }
    else
    {
        m_memory = HK_NULL;
    }
}


void hkThreadMemory::releaseCachedMemory()
{
    for(int rowIndex = MEMORY_MAX_ALL_ROW-1; rowIndex >= 0; --rowIndex )
    {
        if( m_free_list[rowIndex].m_numElem )
        {
            clearRow(rowIndex); //CLEAR_WHOLE_ROW
        }
    }
}

_Ret_notnull_ void* hkThreadMemory::onRowEmpty(int rowIndex)
{
    if (m_maxNumElemsOnFreeList==0)
    {
        // If no caching, just return the allocation
        return m_memory->blockAlloc( s_row_to_size_lut[rowIndex] );
    }
    else
    {
        HK_COMPILE_TIME_ASSERT( BATCH_SIZE >= 1 );
        void* ptrs[BATCH_SIZE];

        int size = m_maxNumElemsOnFreeList<BATCH_SIZE?m_maxNumElemsOnFreeList:BATCH_SIZE;

        m_memory->blockAllocBatch(ptrs, size, s_row_to_size_lut[rowIndex] );
        FreeList& flist = m_free_list[rowIndex];
        for( int i = 1; i < size; ++i )
        {
            flist.put(ptrs[i]);
        }
        return ptrs[0];
    }
}

void hkThreadMemory::onRowFull(int rowIndex, _In_ void* p)
{
    int blockSize = s_row_to_size_lut[rowIndex];

    if (m_maxNumElemsOnFreeList == 0)
    {
        /// No caching
        m_memory->blockFree(p,blockSize);
    }
    else
    {
        FreeList& flist = m_free_list[rowIndex];
        int numBlocks = flist.m_numElem;

        int rowNeeded =  m_maxNumElemsOnFreeList / 2;
        while( numBlocks > rowNeeded )
        {
            void* ptrs[BATCH_SIZE];
            int n = BATCH_SIZE  < numBlocks-rowNeeded ? BATCH_SIZE : numBlocks-rowNeeded;
            int i;
            for( i = 0; i < n; ++i )
            {
                ptrs[i] = flist.get();
            }
            numBlocks -= n;
            m_memory->blockFreeBatch(ptrs, n, blockSize);
        }

        /// Put p on the list
        m_free_list[rowIndex].put(p);
    }
}

void hkThreadMemory::clearRow(int rowIndex)
{
    int blockSize = s_row_to_size_lut[rowIndex];
    FreeList& flist = m_free_list[rowIndex];
    int numBlocks = flist.m_numElem;

    while( numBlocks > 0 )
    {
        void* ptrs[BATCH_SIZE];
        int n = BATCH_SIZE  < numBlocks ? BATCH_SIZE : numBlocks;
        int i;
        for( i = 0; i < n; ++i )
        {
            ptrs[i] = flist.get();
        }
        numBlocks -= n;
        m_memory->blockFreeBatch(ptrs, n, blockSize);
    }
}


static HK_INLINE int hkThreadMemory_getRow( int nbytes )
{
#ifdef HK_DEBUG
    if( nbytes > hkThreadMemory::MEMORY_MAX_SIZE_LARGE_BLOCK )
    {
        // Allocating more than the allowed maximum memory block size
        HK_BREAKPOINT( 0x3 );
    }
#endif

    return (nbytes <= hkThreadMemory::MEMORY_MAX_SIZE_SMALL_BLOCK)
        ? int( hkThreadMemory::s_small_size_to_row_lut[(nbytes + hkThreadMemory::MEMORY_SMALL_BLOCK_ADD) >> hkThreadMemory::MEMORY_SMALL_BLOCK_RSHIFT_BITS] )
        : hkThreadMemory::s_large_size_to_row_lut[(nbytes - 1) >> hkThreadMemory::MEMORY_LARGE_BLOCK_RSHIFT_BITS];
}

_Ret_notnull_ _Post_writable_byte_size_(nbytes) void* hkThreadMemory::blockAlloc(int nbytes)
{
    if ( nbytes <= MEMORY_MAX_SIZE_LARGE_BLOCK)
    {
        int row = hkThreadMemory_getRow(nbytes);
        if( void* p = m_free_list[row].get() )
        {
            return p;
        }
        return onRowEmpty( row );
    }
    else
    {
        return m_memory->blockAlloc( nbytes );
    }
}

void hkThreadMemory::blockFree(_In_opt_bytecount_(nbytes) void* p, int nbytes)
{
    if (p)
    {
        if ( nbytes <= MEMORY_MAX_SIZE_LARGE_BLOCK )
        {
            int row = hkThreadMemory_getRow(nbytes);
            if ( m_free_list[row].m_numElem >= m_maxNumElemsOnFreeList )
            {
                onRowFull(row,p);
            }
            else
            {
                m_free_list[row].put(p);
            }
        }
        else
        {
            m_memory->blockFree(p, nbytes );
        }
    }
}

int hkThreadMemory::getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const
{
    if ( numBytes <= MEMORY_MAX_SIZE_LARGE_BLOCK )
    {
        int row = hkThreadMemory_getRow(numBytes);
        numBytes = s_row_to_size_lut[row];
    }
    return m_memory->getAllocatedSize( obj, numBytes );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
