// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/TempDetect/hkTempDetectAllocator.h>
#include <Common/Base/Thread/Thread/hkThread.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>

template class hkMapBase<void*, int>;

static hkUlong foo1[8*(sizeof(hkUint64)+sizeof(hkTempDetectAllocator::AllocInfo))];
static hkUlong foo2[8*sizeof(int)];

hkTempDetectAllocator::hkTempDetectAllocator()
    : m_child(HK_NULL)
    , m_allocs(foo1, hkSizeOf(foo1))
    , m_freeFromAlloc(foo2, hkSizeOf(foo2))
    , m_outputFunc(HK_NULL)
    , m_outputFuncArg(HK_NULL)
{
}

void hkTempDetectAllocator::init(_In_ hkMemoryAllocator* child, OutputStringFunc output, _In_opt_ void* outputUserData)
{
    m_child = child;
    m_callTree.init(child);
    m_outputFunc = output;
    m_outputFuncArg = outputUserData;
}

static hkBool32 shouldIgnoreCallstack(const hkStackTracer& tracer, _In_ hkMemoryAllocator* allocator, _In_reads_(ntrace) const hkUlong* trace, int ntrace )
{
    struct HK_VISIBILITY_HIDDEN PrintArgs
    {
        hkArrayBase<char> array;
        hkMemoryAllocator* allocator;

        PrintArgs(hkMemoryAllocator* a) : allocator(a) {}
        ~PrintArgs() { array._clearAndDeallocate(*allocator); }

        static void appendToBuffer(_In_z_ const char* s, _In_ void* vp)
        {
            PrintArgs& args = *static_cast<PrintArgs*>(vp);
            args.array._append( *args.allocator, s, hkString::strLen(s) );
        }
    } ;

    PrintArgs args(allocator);
    tracer.dumpStackTrace(trace, ntrace, PrintArgs::appendToBuffer, &args);

    const char* ignoredStacks[] =
    {
        // all part of old versioning system which is going away
        "hkVersionUtil::",
        "hkBinaryPackfileReader::",
        "hkBinaryPackfileWriter::",
        "hkXmlPackfileReader::",
        "hkPackfileWriter::",
        "hkDynamicClassNameRegistry::merge",
        "hkSerializeUtil::",
    };
    for( int i = 0; i < (int)HK_COUNT_OF(ignoredStacks); ++i )
    {
        if( hkString::strStr(args.array.begin(), ignoredStacks[i]) != HK_NULL )
        {
            return true;
        }
    }
    return false;
}

void hkTempDetectAllocator::quit()
{
    // show leaks?
    if( 0 )
    {
        if( m_allocs.getSize() )
        {
            m_outputFunc("\n********************************************\n* LEAKS\n********************************************\n", m_outputFuncArg);
            for( hkMapBase<void*, AllocInfo>::Iterator it = m_allocs.getIterator();
                m_allocs.isValid(it); it = m_allocs.getNext(it) )
            {
                hkUlong trace[64];
                int ntrace = m_callTree.getCallStack( m_allocs.getValue(it).m_allocId, trace, HK_COUNT_OF(trace) );
                m_tracer.dumpStackTrace(trace, ntrace, m_outputFunc, m_outputFuncArg);
                m_outputFunc("-----------------------------------------------\n", m_outputFuncArg);
            }
        }
    }

    hkBool32 somethingPrinted = false;

    //HK_ASSERT_NO_MSG(0x70a041d9, m_allocs.getSize() == 0 );
    m_allocs.clearAndDeallocate(*m_child);
    for( hkMapBase<TraceId, TraceId>::Iterator it = m_freeFromAlloc.getIterator();
        m_freeFromAlloc.isValid(it); it = m_freeFromAlloc.getNext(it) )
    {
        if( m_freeFromAlloc.getValue(it) > 0 )
        {
            TraceId allocTraceId = m_freeFromAlloc.getKey(it);
            SizeInfo sinfo = {0,0,0,0};
            m_sizeFromAlloc.get(allocTraceId, &sinfo);
            if( sinfo.m_count < 5 )
            {
                continue;
            }
            hkUlong trace0[64] HK_ON_DEBUG(= {});
            int ntrace0 = m_callTree.getCallStack( allocTraceId, trace0, HK_COUNT_OF(trace0) );
            if( shouldIgnoreCallstack(m_tracer, m_child, trace0, ntrace0) == hkFalse32 )
            {
                hkUlong trace1[64] HK_ON_DEBUG(= {});
                int ntrace1 = m_callTree.getCallStack( m_freeFromAlloc.getValue(it), trace1, HK_COUNT_OF(trace1) );

                int prefix = 0; // find the common prefix of both callstacks
                while( trace0[ntrace0-1-prefix] == trace1[ntrace1-1-prefix] )
                {
                    ++prefix;
                }
                HK_ASSERT_NO_MSG(0x2363c570, prefix > 0);

                m_outputFunc("\n\n", m_outputFuncArg);
                m_outputFunc("\n\n********************************************\n* Possible temp allocation \n********************************************\n", m_outputFuncArg);
                char buf[128]; hkString::snPrintf(buf, HK_COUNT_OF(buf), HK_COUNT_OF(buf), "Hit %i times: min(%i), max(%i), avg(%i)\n", sinfo.m_count, sinfo.m_minSize, sinfo.m_maxSize, int(sinfo.m_total/sinfo.m_count) );
                m_outputFunc(buf, m_outputFuncArg);
                m_tracer.dumpStackTrace( &trace0[ntrace0-prefix-1], 1, m_outputFunc, m_outputFuncArg); // show probable alloc location first
                m_outputFunc("> alloc >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\n", m_outputFuncArg);
                m_tracer.dumpStackTrace(trace0, ntrace0-prefix, m_outputFunc, m_outputFuncArg);
                m_outputFunc("< free <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<\n", m_outputFuncArg);
                m_tracer.dumpStackTrace(trace1, ntrace1-prefix, m_outputFunc, m_outputFuncArg);
                m_outputFunc("- common prefix -------------------------------\n", m_outputFuncArg);
                m_tracer.dumpStackTrace( &trace0[ntrace0-prefix], prefix, m_outputFunc, m_outputFuncArg);
                m_outputFunc("\n\n", m_outputFuncArg);

                somethingPrinted = true;
            }
        }
    }
    if( somethingPrinted == hkFalse32 )
    {
        m_outputFunc("\n********************************************\n* No temp allocations detected\n********************************************\n", m_outputFuncArg);
    }

    m_freeFromAlloc.clearAndDeallocate(*m_child);
    m_sizeFromAlloc.clearAndDeallocate(*m_child);
    m_callTree.quit();
}

_Ret_notnull_ void* hkTempDetectAllocator::internalAlloc(_In_bytecount_(size) void* p, int size)
{
    HK_ASSERT_NO_MSG(0x46b722c2, size > 0);
    AllocInfo info;
    info.m_threadId = hkThread::getMyThreadId();
    info.m_allocId = m_callTree.insertCallStack(m_tracer);
    info.m_size = size;
    info.m_pushCount = 0;
    HK_ASSERT_NO_MSG(0x7994c3a4, m_allocs.hasKey(p) == false);
    m_allocs.insert(*m_child, p, info);
    return p;
}

void hkTempDetectAllocator::internalFree(_In_bytecount_(size) void* p, int size)
{
    HK_ASSERT_NO_MSG(0x25d51d15, size > 0);
    AllocInfo info = { hkUint64(-1), -1, -1};
    HK_ON_DEBUG( hkResult knownPtr = ) m_allocs.get(p, &info);
    HK_ASSERT_NO_MSG(0x59b82f9a, knownPtr.isSuccess() );
    TraceId thisFreeId = m_callTree.insertCallStack( m_tracer );
    TraceId prevFreeId = m_freeFromAlloc.getWithDefault(info.m_allocId, -1);
    if( prevFreeId == 0 ) // already given up on the allocation location
    {
        // skip
    }
    else if( prevFreeId == -1 ) // new free
    {
        m_freeFromAlloc.insert(*m_child, info.m_allocId, thisFreeId);
        SizeInfo sinfo = { size, size, unsigned(size), 1 };
        m_sizeFromAlloc.insert(*m_child, info.m_allocId, sinfo);
    }
    else if( prevFreeId != thisFreeId ||
        info.m_threadId != hkThread::getMyThreadId() ||
        info.m_pushCount != 0)
    {
        m_freeFromAlloc.insert(*m_child, info.m_allocId, 0); // give up on this one
        m_sizeFromAlloc.remove( info.m_allocId );
    }
    else // update size totals
    {
        hkMapBase<TraceId, SizeInfo>::Iterator it = m_sizeFromAlloc.findKey(info.m_allocId);
        HK_ASSERT_NO_MSG(0x59b82f9b, m_sizeFromAlloc.isValid(it));
        SizeInfo sinfo = m_sizeFromAlloc.getValue(it);
        sinfo.m_maxSize = hkMath::max2(sinfo.m_maxSize, size);
        sinfo.m_minSize = hkMath::max2(sinfo.m_minSize, size);
        sinfo.m_total += size;
        sinfo.m_count += 1;
        m_sizeFromAlloc.setValue(it, sinfo);
    }
    m_allocs.remove( p );
}

_Ret_notnull_ _Post_writable_byte_size_(size) void* hkTempDetectAllocator::blockAlloc( int size )
{
    if (size == 0)
    {
        // every allocation must correspond to a different pointer
        size = 1;
    }
    hkCriticalSectionLock lock(&m_lock);
    return internalAlloc(m_child->blockAlloc( size ), size);
}

void hkTempDetectAllocator::blockFree(_In_opt_bytecount_(size) void* p, int size)
{
    if( p != HK_NULL )
    {
        if (size == 0)
        {
            // use the correct size
            size = 1;
        }
        hkCriticalSectionLock lock(&m_lock);
        internalFree(p,size);
        m_child->blockFree(p, size);
    }
}

_Ret_notnull_ _Post_writable_byte_size_(size) void* hkTempDetectAllocator::bufAlloc(int& size)
{
    if (size == 0)
    {
        // every allocation must correspond to a different pointer
        size = 1;
    }
    hkCriticalSectionLock lock(&m_lock);
    void* p = m_child->bufAlloc(size);
    return internalAlloc( p, size);
}

void hkTempDetectAllocator::bufFree(_In_opt_bytecount_(size) void* p, int size)
{
    if( p != HK_NULL )
    {
        hkCriticalSectionLock lock(&m_lock);
        internalFree(p, size);
        m_child->bufFree(p, size);
    }
}

_Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* hkTempDetectAllocator::bufRealloc(_In_opt_bytecount_(oldNum) void* pold, int oldNum, int& reqNumInOut)
{
    hkCriticalSectionLock lock(&m_lock);

    if( pold != HK_NULL )
    {
        hkMapBase<void*, AllocInfo>::Iterator it = m_allocs.findKey(pold);
        HK_ASSERT_NO_MSG(0x67a1e3e7, m_allocs.isValid(it) );
        AllocInfo iold;
        iold = m_allocs.getValue(it);

        void* pnew = m_child->bufRealloc(pold, oldNum, reqNumInOut);
        if( pold != pnew )
        {
            m_allocs.remove( it );
            m_allocs.insert(*m_child, pnew, iold); // keep original alloc location
        }
        if( iold.m_threadId != hkThread::getMyThreadId() ||
            iold.m_pushCount != 0)
        {
            // Different thread, or different stack level (in the sense of
            // pushFrame/popFrame), so give up on this one.
            m_freeFromAlloc.insert(*m_child, iold.m_allocId, 0);
        }
        return pnew;
    }
    else
    {
        return bufAlloc(reqNumInOut);
    }
}

void hkTempDetectAllocator::pushFrame()
{
    hkCriticalSectionLock lock(&m_lock);

    hkUint64 curThreadId = hkThread::getMyThreadId();

    for (hkMapBase<void*, AllocInfo>::Iterator it = m_allocs.getIterator(); m_allocs.isValid(it); it = m_allocs.getNext(it))
    {
        AllocInfo& allocInfo = m_allocs.getValue(it);
        if (allocInfo.m_threadId == curThreadId)
        {
            allocInfo.m_pushCount++;
        }
    }
}

void hkTempDetectAllocator::popFrame()
{
    hkCriticalSectionLock lock(&m_lock);

    hkUint64 curThreadId = hkThread::getMyThreadId();

    for (hkMapBase<void*, AllocInfo>::Iterator it = m_allocs.getIterator(); m_allocs.isValid(it); it = m_allocs.getNext(it))
    {
        AllocInfo& allocInfo = m_allocs.getValue(it);
        if (allocInfo.m_threadId == curThreadId)
        {
            allocInfo.m_pushCount--;
        }
    }
}

void hkTempDetectAllocator::advanceFrame()
{
    hkCriticalSectionLock lock(&m_lock);

    hkUint64 curThreadId = hkThread::getMyThreadId();

    for( hkMapBase<void*, AllocInfo>::Iterator it = m_allocs.getIterator(); m_allocs.isValid(it); it = m_allocs.getNext(it) )
    {
        AllocInfo& allocInfo = m_allocs.getValue(it);

        if(allocInfo.m_threadId == curThreadId &&
            allocInfo.m_pushCount == 0)
        {
            // give up on all allocations lasting longer than a frame
            m_freeFromAlloc.insert(*m_child, allocInfo.m_allocId, 0);
        }
    }
}

void hkTempDetectAllocator::getMemoryStatistics( hkTempDetectAllocator::MemoryStatistics& u ) const
{
    HK_WARN( 0xf0345465, "hkTempDetectAllocator does not support getMemoryStatistics(), data will be wrong");
    m_child->getMemoryStatistics( u );
}

int hkTempDetectAllocator::getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const
{
    return m_child->getAllocatedSize( obj, numBytes );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
