// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

static hkMallocAllocator s_defaultMallocAllocator;
hkMallocAllocator* hkMallocAllocator::m_defaultMallocAllocator = &s_defaultMallocAllocator;

// If the platform has a builtin aligned allocation facility, we define
// aligned_malloc/aligned_free macros to be those functions. Otherwise, we define
// macros/functions for unaligned_malloc/unaligned free and manually align the
// allocations.
#if defined(HK_PLATFORM_WIN32) || (defined(HK_PLATFORM_NX) && defined(_WIN32))

    #include <Common/Base/Fwd/hkcstdlib.h>
    #define aligned_malloc _aligned_malloc
    #define aligned_free _aligned_free

#elif defined(HK_PLATFORM_LINUX) || defined( HK_PLATFORM_PSVITA ) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_TIZEN) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_NX)

    #include <Common/Base/Fwd/hkcmalloc.h> 
    #define aligned_malloc(size, align) memalign(align, size)
    #define aligned_free(p) free(p)

#elif defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_LRB_XENSIM) || defined(HK_PLATFORM_LRB)
    #include <Common/Base/Fwd/hkcmalloc.h> 
    #define unaligned_malloc(size) malloc(size) // native align only
    #define unaligned_free(p) free(p)

#elif defined(HK_PLATFORM_WIIU)
#   include <cafe/mem.h>
    #define aligned_malloc wiiuallocator_malloc
    #define aligned_free wiiudemoallocator_free

    static void* wiiuallocator_malloc(int size, int align)
    {
        return MEMAllocFromDefaultHeapEx(size, align);
    }

    static void wiiudemoallocator_free(void* p)
    {
        return MEMFreeToDefaultHeap(p);
    }

#elif defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_CTR)
    #include <Common/Base/Fwd/hkcmalloc.h> 
    #define unaligned_malloc(size) malloc(size) // native align only, which on arm is 8 bytes not 16
    #define unaligned_free(p) free(p)
#else

#   error no Malloc & Free defined for this platform.

#endif


#if !defined(aligned_malloc)
    // we synthesize an aligned malloc from an unaligned one
    static inline _Ret_maybenull_ void* aligned_malloc(int numBytes, int align)
    {
        // if numBytes == 0
        if (numBytes < 1)
        {
            return HK_NULL;
        }

        void* pvoid = unaligned_malloc(numBytes + sizeof(int) + (align>4? align : 4) ); // need extra int to store padding info, and pad for align
        if (pvoid == HK_NULL)
        {
            HK_WARN_ALWAYS(0x7da65776, "Failed to allocate " << numBytes << " bytes!");
            return HK_NULL;
        }

        HK_ASSERT(0x38c3fe7c,  (reinterpret_cast<hkUlong>(pvoid) & 0x3) == 0, "Pointer was not 4 byte aligned");
        char* p = reinterpret_cast<char*>(pvoid);
        char* aligned = reinterpret_cast<char*>(
            HK_NEXT_MULTIPLE_OF( align, reinterpret_cast<hkUlong>(p+1)) );
        reinterpret_cast<int*>(aligned)[-1] = (int)(aligned - p);
        hkMemUtil::memSet(aligned, 0xfe, numBytes);
        return aligned;
    }
    static inline void aligned_free(_In_opt_ void* p )
    {
        if (p)
        {
            int offset = reinterpret_cast<int*>(p)[-1];
            void* pvoid = static_cast<char*>(p) - offset;
            unaligned_free(pvoid);
        }
    }
#endif

_Ret_maybenull_ _Post_writable_byte_size_(numBytes) void* hkMallocAllocator::tryAlloc(int numBytes)
{
    if (void* p = aligned_malloc(numBytes, m_align))
    {
        hkUint32 cur = hkAtomic::exchangeAdd32(&m_currentUsed, numBytes);
        if(cur + numBytes > m_peakUse)
        {
            m_peakUse = m_currentUsed;
        }
        return p;
    }
    return nullptr;
}

_Ret_notnull_ _Post_writable_byte_size_(numBytes) void* hkMallocAllocator::blockAlloc(int numBytes)
{
    if (void* p = tryAlloc(numBytes))
    {
        return p;
    }

    if(numBytes)
    {
        return outOfMemory();
    }
    else
    {
        // Null is ok for a zero-sized allocation
        return nullptr;
    }
}

void hkMallocAllocator::blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes )
{
    hkAtomic::exchangeAdd32(&m_currentUsed,  -numBytes);

    aligned_free(p);
}

void hkMallocAllocator::getMemoryStatistics( MemoryStatistics& u ) const
{
    u.m_allocated = m_currentUsed;
    u.m_inUse = m_currentUsed;
    u.m_peakInUse = m_peakUse;
}

void hkMallocAllocator::resetPeakMemoryStatistics()
{
    m_peakUse = m_currentUsed;
}

int hkMallocAllocator::getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const
{
    return numBytes;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
