// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

    /// Specialized for LIFO or almost-LIFO pattern allocations.
    /// This allocator gets large blocks (slabs) from a parent allocator
    /// and slices them into smaller allocations. It is most efficient
    /// when the allocations are in pure LIFO order, but can handle a few
    /// out of order frees with only a small penalty.
    /// Requests larger than the slab size are forwarded to another allocator.
class HK_EXPORT_COMMON hkLifoAllocator : public hkMemoryAllocator
{
    public:

        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true));
        HK_DECLARE_PLACEMENT_ALLOCATOR();

        hkLifoAllocator();

        void init(_In_ hkMemoryAllocator* slabAllocator,
            _In_ hkMemoryAllocator* largeAllocator,
            _Inout_ hkMemoryAllocator* internalAllocator,
            int slabSize);

            /// The user can optionally retrieve a list of the allocators used during
            /// during operations: [slabAllocator, largeAllocator, internalAllocator]
        void quit(hkMemoryAllocator* allocators[3] = HK_NULL);

        virtual _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* blockAlloc(int numBytes) HK_OVERRIDE;

            /// Inline equivalent of blockAlloc
        HK_INLINE _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* fastBlockAlloc(int numBytes);

        virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes ) HK_OVERRIDE;

            /// Inline equivalent of blockFree
        HK_INLINE void fastBlockFree(_In_bytecount_(numBytes) void* p, int numBytes);

        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut) void* bufAlloc(int& reqNumBytesInOut) HK_OVERRIDE;
        virtual void bufFree( _In_opt_bytecount_(numBytes) void* p, int numBytes ) HK_OVERRIDE;
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumBytesInOut) void* bufRealloc( _In_reads_bytes_(oldNumBytes) void* pold, int oldNumBytes, int& reqNumBytesInOut ) HK_OVERRIDE;

        virtual void getMemoryStatistics( MemoryStatistics& u ) const HK_OVERRIDE {}
        virtual int getAllocatedSize(_In_bytecount_(nbytes) const void* obj, int nbytes) const HK_OVERRIDE { return nbytes; }

        bool isEmpty() const;

            /// number of large + slab allocations
        int numExternalAllocations() const;

    protected:

        _Ret_notnull_ void* allocateFromNewSlab(int nbytes);
        void popNonLifoFrees();
        void slowBlockFree(_In_opt_bytecount_(numBytesIn) void* p, int numBytesIn);
        void insertNonLifoFree(_In_bytecount_(nbytes) void* p, int nbytes);

    protected:

        class Implementation;

            /// Pointer to internal data.
        class Implementation* m_impl;

            /// The size of each slab.
        int m_slabSize;

            /// The start of the free space on the current slab.
        void* m_cur;
            /// The end of the current slab.
        void* m_end;
            /// The end of the last exception. After a free, when m_cur is equal
            /// to this value, we need to process some deferred frees.
        void* m_firstNonLifoEnd;

            /// We cache the last freed slab to avoid a performance spike when
            /// repeatedly crossing a slab boundary.
        void* m_cachedEmptySlab;

            /// For allocations of m_slabSize
        hkMemoryAllocator* m_slabAllocator;
            /// For blocks larger than the slab size
        hkMemoryAllocator* m_largeAllocator;
            /// For internal bookkeeping
        hkMemoryAllocator* m_internalAllocator;
};

#include <Common/Base/Memory/Allocator/Lifo/hkLifoAllocator.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
