// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Memory/System/Util/hkMemorySnapshot.h>

/// Fast fixed memory size allocations - used in the hkFreeListAllocator for small memory block allocations.
///
/// Can be used where ever you want fast same block size allocations, and you want to be able to free blocks
/// in any order when they are not in use.
///
/// This implementation is particularly fast at initial allocation - as it works just like a 'region'.
/// If memory is allocated and freed, the freelist uses a linked list of available blocks which whilst fast isn't
/// as fast as region style allocation. Doing a findGarbage will turn all memory that can be used as a region
/// into a region.
///
/// NOTE! Freeing freelist blocks does not free memory back to the allocator. Memory is only given back to the
/// hkFreeListAllocator when a garbageCollect or freeAllFreeBlocks is called.
class HK_EXPORT_COMMON hkFreeList
{
    public:

        HK_DECLARE_PLACEMENT_ALLOCATOR();

        typedef void (HK_CALL *MemoryWalkCallback)(_In_bytecount_(size) void* start, hk_size_t size, bool allocated, int pool,void* param);

            /// The size of an element, and the alignment all elements must have
        hkFreeList( hk_size_t elementSize, hk_size_t alignment, hk_size_t blockSize,
            _In_opt_ hkMemoryAllocator* elementAllocator = HK_NULL, _In_opt_ hkMemoryAllocator* blockAllocator = HK_NULL);

            /// Default Ctor - doesn't set up any allocation. Can be subsequently configured with a call to init'.
        hkFreeList();

            /// Dtor
        ~hkFreeList() { freeAllMemory(); }

        struct Element
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,Element);

            static HK_INLINE bool HK_CALL less(const Element& eA, const Element& eB)    {   return &eA < &eB;   }
            static HK_INLINE Element*& HK_CALL next(Element* e)                         {   return e->m_next;   }

            Element* m_next;
        };

            // The block is 16 byte aligned, as are allocations from the large block allocator. Which is nice - and a
            // requirement for the alignment.
        struct Block
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,Block);

            static HK_INLINE bool HK_CALL less(const Block& bA, const Block& bB)    {   return &bA < &bB;   }
            static HK_INLINE Block*& HK_CALL next(Block* b)                         {   return b->m_next;   }

            Block* m_next;                  ///< The next block in the list, or NULL if there isn't one
            hkUint8* m_elementsAlloc;       ///< The address of the allocation returned by the allocator. The size is the 'blockSize' (is HK_NULL if allocated with the block)
            hkUint8* m_elements;            ///< The start of the elements payload
            hk_size_t m_numElements;        ///< Number of elements in this block (needed as the alignment may mean not same as the freelist numBlockElements)
        };

            /// Initialize, can be called multiple time to reconfigure how a freelist is used.
            /// Will free all elements that are allocated on the freelist.
        void init(hk_size_t elementSize, hk_size_t alignment, hk_size_t blockSize,
            _In_opt_ hkMemoryAllocator* elementAllocator = HK_NULL, _In_opt_ hkMemoryAllocator* blockAllocator = HK_NULL);

            /// Returns true if alloc will succeed
        HK_INLINE hkBool canAlloc();

            /// Allocates a chunk of memory
        HK_INLINE _Ret_notnull_ void* alloc();

            /// Returns the chunk of memory to the pool. The chunk _must_ have been allocated from this pool
        HK_INLINE void free(_In_ void* data);

            /// Allocate a batch
        HK_INLINE void allocBatch(_Out_writes_all_(sizeIn) void** out, int sizeIn);

            /// Free a batch
        HK_INLINE void freeBatch(_In_reads_(sizeIn) void** in, int sizeIn);

            /// Frees all of the memory used by the memory pool
        void freeAllMemory();

            /// Makes all of the memory allocated available
        void freeAll();

            /// Get the element size
        hk_size_t getElementSize() const { return m_elementSize; }

            /// Does checks to see if this memory is ok
        hkBool isOk() const { return _checkFreeBlocks(); }

            /// Returns true if there is an element available (without extra allocation)
        hkBool isFreeElementAvailable() const { return m_free || m_top < m_blockEnd; }

            /// Simple memory stats
        void getMemoryStatistics(hkMemoryAllocator::MemoryStatistics& stats) const;

            /// Finds blocks which are not used, and puts them on the 'freeBlocks' list.
            /// Returns number of blocks found, or <0 if couldn't do the work
        int findGarbage();

            /// Returns the number of blocks examined.
            /// The free list holds state, so will only examine blocks after ones previously examined.
            /// Returns true when hits the end of all the blocks, will start on the first block when called again
        hkBool incrementalFindGarbage(int numBlocks, int& numBlocksOut);

            /// Collect
        inline void garbageCollect();

            /// Frees all of the free blocks
        int freeAllFreeBlocks();

            /// Returns true if there are freeblocks in the list
        hkBool hasFreeBlocks() const { return m_freeBlocks != HK_NULL; }

            /// Get the total amount of free elements that are available
        hk_size_t getNumFreeElements() const { return m_numFreeElements; }

            /// Get the total number of elements, free and used that have been allocated
        hk_size_t getTotalNumElements() const { return m_totalNumElements; }

            /// Get the total number of elements that have been used
        hk_size_t getNumUsedElements() const { return m_totalNumElements - m_numFreeElements; }

            /// Get the current allocation block size
        hk_size_t getBlockSize() const { return m_blockSize; }

            /// Add to the given snapshot with the given parentId.
            /// The allocator who owns this freelist will decide what is the usage of
            /// the elements contained in the freelist.
        int addToSnapshot(hkMemorySnapshot& snap, hkMemorySnapshot::Status usage, int parentId);

            /// Will callback for each free and allocated memory block
        void walkMemory(MemoryWalkCallback callback, int pool, _In_opt_ void* param);

            /// Get the active blocks
        _Ret_maybenull_ const Block* getActiveBlocks() const { return m_activeBlocks; }
            /// Get the free blocks
        _Ret_maybenull_ const Block* getFreeBlocks() const { return m_freeBlocks; }

            /// Blocksize for elements
        static hk_size_t HK_CALL bestBlockSize(hk_size_t elementSpace, hk_size_t align);
            /// Counts total blocks in list, with block passed in as the starting block
        static int HK_CALL calcNumBlocks(_In_opt_ const Block* blocks);

            // Internal method to get the block lists used for mem walking.
        int _getSortedBlockHeads(_Out_writes_all_(2) const Block** heads, _In_range_(==, 2) int numHeads );

    protected:

            // Make more space and return a new block from the space that was allocated. If not possible to allocate
            // return HK_NULL
        _Ret_notnull_ void* addSpace();

            // Returns the amount free elements
        hk_size_t _calcNumFreeElements() const;

            // Returns the total amount elements, free or used
        hk_size_t _calcTotalNumElements() const;

            // Returns the number of used elements
        hk_size_t _calcNumUsedElements() const { return _calcTotalNumElements() - _calcNumFreeElements(); }

            // Goes down the linked list working out how many elements there are
        static hk_size_t _calcTotalNumElements(_In_opt_ const Block* cur);

            // Frees a link list of blocks
        int _freeBlocks(_In_opt_ Block* cur);

            // Calculates the sum size of a linked list of blocks
        hk_size_t _calculateBlockAllocatedSize(_In_opt_ const Block* cur) const;

            // Sets up allocation - assumes all memory has been freed
        void _init(hk_size_t elementSize, hk_size_t align, hk_size_t blockSize,
            _In_opt_ hkMemoryAllocator* elementAllocator, _In_opt_ hkMemoryAllocator* blockAllocator);

        static HK_INLINE hkBool _compareBlocks(_In_opt_ const Block* a, _In_opt_ const Block* b ) { return a < b; }
        static HK_INLINE hkBool _compareElements(_In_opt_ const Element* a, _In_opt_ const Element* b ) { return a < b; }

            /// Add all the elements of block to the free list
        void _addBlockElements(_In_ Block* block)
        {
            m_top = block->m_elements;
            m_blockEnd = block->m_elements + block->m_numElements * m_elementSize;
        }

        void _walkMemoryBlockList(_In_opt_ Block* block, MemoryWalkCallback callback, int pool, _In_opt_ void* param);

        hkBool _checkFreeBlocks() const;

            /// Analyses the block to see if all of its elements are free, if so it moves the block to the m_freeBlocks list, and returns true.
            /// Also in that case removes all the elements from the m_free list.
            /// If they aren't all free, it will reorder the free blocks so they will be returned contiguously from memory
        hkBool32 _calcBlockFree(_In_ const Block* block);

            /// Move any elements which are in the "unpartitioned" space of the current block, to the free list (m_free).
        void _moveTopToFree();

            /// Adds to the snapshot the memory overhead relative to the given block.
        void _addBlockOverhead(_In_ const Block* block, hkMemorySnapshot& snap, int parentId) const;

    protected:

            /// Singly linked list of free elements. NULL terminated
        Element* m_free;

            /// The size of a single element
        hk_size_t m_elementSize;

            /// The last block incrementally garbage collected. HK_NULL if at start of list.
            /// Should only point to used blocks (i.e., blocks in the m_activeBlocks list)
        Block* m_lastIncrementalBlock;

            // The active blocks
        Block* m_activeBlocks;

            /// The free blocks (blocks which haven't seen use in current allocation)
        Block* m_freeBlocks;

        hk_size_t m_blockSize;
        hk_size_t m_align;
        hk_size_t m_numBlockElements;           ///< The number of elements in an aligned block
        //hk_size_t m_maxBlockSize;

            // The top of the current block, above it memory is free up until blockEnd is hit
        hkUint8* m_top;
        hkUint8* m_blockEnd;

        hkMemoryAllocator* m_elementAllocator;          ///< Underlying element allocator
        hkMemoryAllocator* m_blockAllocator;            ///< Underlying block allocator

            // A cached value of the total amount of elements
        hk_size_t m_totalNumElements;

    public:
            // A cache of the total amount of free elements
        hk_size_t m_numFreeElements;
};

#include <Common/Base/Memory/Allocator/FreeList/hkFreeList.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
