// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/Checking/hkPaddedAllocator.h>

#ifdef HK_PLATFORM_PS4
#   define MEM_ALIGNMENT        HK_DOUBLE_ALIGNMENT
#   define SIZE_OF_QUADREAL     (2 * sizeof(hkQuadReal))
#else
#   define MEM_ALIGNMENT        HK_REAL_ALIGNMENT
#   define SIZE_OF_QUADREAL     sizeof(hkQuadReal)
#endif

hkPaddedAllocator::hkPaddedAllocator()
    : m_next(HK_NULL)
{
}

void hkPaddedAllocator::init(_In_ hkMemoryAllocator* next, _In_opt_ const Cinfo* cinfoPtr )
{
    m_next = next;
    Cinfo defaultCinfo;
    m_cinfo = cinfoPtr ? *cinfoPtr : defaultCinfo;
    HK_ASSERT_NO_MSG(0x7f05fa8d, m_cinfo.m_numQuadsPad >= 1 );
}

void hkPaddedAllocator::quit()
{
    m_next = HK_NULL;
}

_Ret_notnull_ _Post_writable_byte_size_(numBytes) void* hkPaddedAllocator::blockAlloc(int numBytes)
{
    // example numBytes = 11, pad = 16:
    // - numBytesAligned = 16
    // - allocSize = 16 + 2*16 = 48
    // - numBodyWords = (11+4-1)/4 = 3
    // pad and scrub:
    // [XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX] -> [PPPP PPPP PPPP PPPP XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX]
    // [PPPP PPPP PPPP PPPP XXXX XXXX XXXX XXXX XXXX XXXX XXXX XXXX] -> [PPPP PPPP PPPP PPPP BBBB BBBB BBBB XXXX XXXX XXXX XXXX XXXX]
    // [PPPP PPPP PPPP PPPP BBBB BBBB BBBB XXXX XXXX XXXX XXXX XXXX] -> [PPPP PPPP PPPP PPPP BBBB BBBB BBBA AAAA XXXX XXXX XXXX XXXX]
    // [PPPP PPPP PPPP PPPP BBBB BBBB BBBA AAAA XXXX XXXX XXXX XXXX] -> [PPPP PPPP PPPP PPPP BBBB BBBB BBBA AAAA PPPP PPPP PPPP PPPP]
    int numBytesAligned = HK_NEXT_MULTIPLE_OF(MEM_ALIGNMENT, numBytes);
    unsigned pad = SIZE_OF_QUADREAL * m_cinfo.m_numQuadsPad; // one side
    int allocSize = numBytesAligned + 2*pad;
    m_allocated += allocSize;
    m_inUse += numBytes;
    void* p = m_next->blockAlloc( allocSize );

    if ( p )
    {
        int numBodyWords = (numBytes+sizeof(hkUint32)-1) / sizeof(hkUint32);
        // pad and scrub
        hkString::memSet4( p, m_cinfo.m_padPattern, pad/sizeof(hkUint32) );
        hkString::memSet4( hkAddByteOffset(p,pad), m_cinfo.m_bodyPattern, numBodyWords );
        hkString::memSet( hkAddByteOffset(p,pad+numBytes), m_cinfo.m_alignPattern, numBytesAligned-numBytes );
        hkString::memSet4( hkAddByteOffset(p,pad+numBytesAligned), m_cinfo.m_padPattern + 1, pad/sizeof(hkUint32) );
        return hkAddByteOffset(p,pad);
    }

    return outOfMemory();
}

void hkPaddedAllocator::blockFree(_In_opt_bytecount_(HK_NEXT_MULTIPLE_OF(MEM_ALIGNMENT, numBytes)) void* p, int numBytes)
{
    int pad = SIZE_OF_QUADREAL * m_cinfo.m_numQuadsPad;
    int numBytesAligned = HK_NEXT_MULTIPLE_OF(MEM_ALIGNMENT, numBytes);
    // check the aligned part
    hkUint32* ps4 = (hkUint32*)hkAddByteOffset(p, -pad);
    hkUint32* pe4 = (hkUint32*)hkAddByteOffset(p, numBytesAligned);
    {
        for( int i = 0; i < 4*m_cinfo.m_numQuadsPad; ++i )
        {
            if( ps4[i] != m_cinfo.m_padPattern ) HK_BREAKPOINT(0); 
            if( pe4[i] != m_cinfo.m_padPattern + 1 ) HK_BREAKPOINT(0);
        }
    }

    // check the unaligned part between the end and pad
    // they were filled with alignPattern, not padPattern or bodyPattern
    {
        const hkUint8* p8 = static_cast<const hkUint8*>(p);
        for( int i = numBytes; i < numBytesAligned; ++i )
        {
            if( p8[i] != m_cinfo.m_alignPattern )
            {
                HK_BREAKPOINT(0);
            }
        }
    }

    int freeSize = numBytesAligned + 2*pad;
    m_allocated -= freeSize;
    m_inUse -= numBytes;
    hkString::memSet4( ps4, m_cinfo.m_freePattern, 2*4*m_cinfo.m_numQuadsPad); // 2=pre+post 4=4 ints per quad
    m_next->blockFree( ps4, freeSize );
}

hkBool32 hkPaddedAllocator::isOk(_In_bytecount_(HK_NEXT_MULTIPLE_OF(MEM_ALIGNMENT, numBytes)) const void* p, int numBytes ) const
{
    
    
    

    int pad = SIZE_OF_QUADREAL * m_cinfo.m_numQuadsPad;
    int numBytesAligned = HK_NEXT_MULTIPLE_OF(MEM_ALIGNMENT, numBytes);
    // check the aligned part
    const hkUint32* ps4 = (const hkUint32*)hkAddByteOffsetConst(p, -pad);
    const hkUint32* pe4 = (const hkUint32*)hkAddByteOffsetConst(p, numBytesAligned);
    {
        for( int i = 0; i < 4*m_cinfo.m_numQuadsPad; ++i )
        {
            if( ps4[i] != m_cinfo.m_padPattern ) return false; 
            if( pe4[i] != m_cinfo.m_padPattern + 1 ) return false;
        }
    }
    // check the unaligned part between the end and pad
    // they were filled with alignPattern, not padPattern or bodyPattern because of alignment
    {
        const hkUint8* p8 = static_cast<const hkUint8*>(p);
        for( int i = numBytes; i < numBytesAligned; ++i )
        {
            if( p8[i] != m_cinfo.m_alignPattern )
            {
                return false;
            }
        }
    }
    return true;
}

void hkPaddedAllocator::getMemoryStatistics( hkMemoryAllocator::MemoryStatistics& u ) const
{
    if ( m_next )
    {
        m_next->getMemoryStatistics( u );
    }
    u.m_allocated = m_allocated;
    u.m_inUse = m_inUse;
    u.m_available = 0;
}

hkPaddedAllocator::Allocation hkPaddedAllocator::getUnderlyingAllocation(_In_bytecount_(numBytes) const void* obj, int numBytes) const
{
    int pad = SIZE_OF_QUADREAL * m_cinfo.m_numQuadsPad;
    int numBytes16 = HK_NEXT_MULTIPLE_OF(16, numBytes);
    Allocation ret;
    ret.address = hkAddByteOffsetConst(obj, -pad);
    ret.size = numBytes16 + 2*pad;
    return ret;
}

int hkPaddedAllocator::getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const
{
    int pad = SIZE_OF_QUADREAL * m_cinfo.m_numQuadsPad;
    int numBytesAligned = HK_NEXT_MULTIPLE_OF(MEM_ALIGNMENT, numBytes);
    const void* unpadded = hkAddByteOffsetConst(obj, -pad);
    return m_next->getAllocatedSize( unpadded, numBytesAligned + 2*pad ) - 2*pad;
}

void hkPaddedAllocator::setScrubValues(hkUint32 bodyValue, hkUint32 freeValue)
{
    m_cinfo.m_bodyPattern = bodyValue;
    m_cinfo.m_freePattern = freeValue;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
