// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/StackTracer/hkStackTracer.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Container/PointerMap/hkMap.h>

    /// Allocator for detecting memory leaks.
    /// This class is useful for detecting leaks in a single allocator.
    /// Mostly you'll want to use the hkCheckingMemorySystem which hooks
    /// all allocators in all threads.
class HK_EXPORT_COMMON hkLeakDetectAllocator : public hkMemoryAllocator
{
    public:
        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true));
        typedef void (HK_CALL *OutputStringFunc)(_In_z_ const char* s, _In_ void* userData);

            /// initialize.
            /// The \a debugAllocator will be used to allocate internal debug tracking information
        void init(_In_ hkMemoryAllocator* childAllocator, _Inout_ hkMemoryAllocator* debugAllocator, OutputStringFunc output, _In_ void* outputUserData);

        void quit();

        virtual _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* blockAlloc( int numBytes ) HK_OVERRIDE;
        virtual void blockFree(_In_opt_bytecount_(numBytes) void* p, int numBytes ) HK_OVERRIDE;
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* bufAlloc(int& reqNumInOut) HK_OVERRIDE;
        virtual void bufFree(_In_opt_bytecount_(numBytes) void* p, int numBytes) HK_OVERRIDE;
        virtual _Ret_notnull_ _Post_writable_byte_size_(reqNumInOut) void* bufRealloc(_In_reads_bytes_(oldNum) void* pold, int oldNum, int& reqNumInOut ) HK_OVERRIDE;

        struct AllocInfo
        {
            bool operator<(const AllocInfo& i) const { return time < i.time; }
            hkUint64 time;
            int traceId;
            int size;
            hkBool isBuf;
        };

        virtual void getMemoryStatistics( MemoryStatistics& out ) const HK_OVERRIDE;
        virtual int getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes) const HK_OVERRIDE;

            /// Output all the currently allocated pointers by allocation location, indented by tabs.
            /// Use tabview to view this file as a tree.
        void printAllocationsByCallTree(hkOstream& os);

    protected:

        _Ret_notnull_ _Post_writable_byte_size_(numBytes) void* checkedAlloc(hkBool32 isBuf, int numBytes);
        void checkedFree( hkBool32 isBuf, _In_opt_bytecount_(numBytes) void* p, int numBytes );
        void danger(_In_z_ const char* message, _In_ const void* ptr, const AllocInfo& info) const;

        hkStackTracer m_stackTracer;
        hkStackTracer::CallTree m_callTree;
        mutable hkCriticalSection m_criticalSection;

        hkMemoryAllocator* m_childAllocator;
        hkMemoryAllocator* m_debugAllocator;

        OutputStringFunc m_outputFunc;
        void* m_outputFuncArg;

        typedef hkMapBase<hkUlong,AllocInfo> MapType;

        struct Dynamic
        {
            // remove this struct when hkMap doesn't alloc in it ctor
            HK_DECLARE_PLACEMENT_ALLOCATOR();
            Dynamic(_In_opt_ const hkMemoryAllocator* a) {}
            void quit(_Inout_ hkMemoryAllocator* a)
            {
                m_activePointers.clearAndDeallocate(*a);
            }

            MapType m_activePointers;
        };
        Dynamic* m_dynamic;

        hk_size_t m_inUse;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
