// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : X64 OSINTERNAL_ARM64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>

#include <Common/Base/Memory/Allocator/Checking/hkGuardingAllocator.h>

#include <Windows.h>

#define DELAYED_FREE_QUEUE_SIZE 65536
#define NO_MANS_LAND_SIZE 32
#define MAGIC_MARKER 0x52574152
#define LEADING_PAGES 1
#define TRAILING_PAGES 1
#define ALIGNMENT 16

namespace {
    int getPageSize()
    {
        static int s_pageSize = 0;

        if (s_pageSize == 0)
        {
            SYSTEM_INFO info;
            GetSystemInfo(&info);
            s_pageSize = info.dwPageSize;
        }

        return s_pageSize;
    }

    int getPageGranularity()
    {
        static int s_pageGranularity = 0;

        if (s_pageGranularity == 0)
        {
            SYSTEM_INFO info;
            GetSystemInfo(&info);
            s_pageGranularity = info.dwAllocationGranularity;
        }

        return s_pageGranularity;
    }

    struct hkGuardingAllocatorData
    {
        HK_DECLARE_CLASS(hkGuardingAllocatorData, NewPlacement)

        int m_numBytes;
        hkUint32 m_noMansLand[NO_MANS_LAND_SIZE];

        hkGuardingAllocatorData()
        {
            for (int i = 0; i < NO_MANS_LAND_SIZE; i++)
            {
                m_noMansLand[i] = MAGIC_MARKER;
            }
        }

        void verify() const
        {
            for (int i = 0; i < NO_MANS_LAND_SIZE; i++)
            {
                if (m_noMansLand[i] != MAGIC_MARKER)
                {
                    // Memory in front of allocated block was modified
                    HK_BREAKPOINT(0);
                }
            }
        }
    };

    bool isGuardedAllocation(_In_ const void* ptr)
    {
        int pageSize = getPageSize();

        void* firstPage = reinterpret_cast<void*>(((reinterpret_cast<size_t>(ptr) - sizeof(hkGuardingAllocatorData)) / pageSize - LEADING_PAGES) * pageSize);

        // Our allocations are aligned at page allocation granularity (usually 64KB)
        if (reinterpret_cast<size_t>(firstPage) % getPageGranularity() != 0)
        {
            return false;
        }

        // The first LEADING_PAGES pages must be noaccess protected
        MEMORY_BASIC_INFORMATION memInfo;
        VirtualQuery(firstPage, &memInfo, sizeof(MEMORY_BASIC_INFORMATION));
        if (memInfo.AllocationProtect != PAGE_NOACCESS || memInfo.RegionSize != getPageSize() * LEADING_PAGES)
        {
            return false;
        }

        return true;
    }
}

hkGuardingAllocator::hkGuardingAllocator()
{
    m_numDelayedFrees = 0;
    m_lastDelayedFree = 0;
    m_delayedFrees = reinterpret_cast<void**>(VirtualAlloc(NULL, DELAYED_FREE_QUEUE_SIZE * sizeof(void*), MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE));
}

hkGuardingAllocator::~hkGuardingAllocator()
{
    int endIndex = (m_numDelayedFrees == DELAYED_FREE_QUEUE_SIZE ? m_numDelayedFrees : m_lastDelayedFree);
    for (int i = 0; i < endIndex; i++)
    {
        void* page = m_delayedFrees[i];
        HK_ON_DEBUG(BOOL freeSucceeded = ) VirtualFree(page, 0, MEM_RELEASE);
        HK_ASSERT(0x10e497d4, freeSucceeded, "Guarding allocator: Releasing a virtual page failed.");
    }
    VirtualFree(m_delayedFrees, DELAYED_FREE_QUEUE_SIZE * sizeof(void*), MEM_DECOMMIT | MEM_RELEASE);
}

_Ret_notnull_ _Post_writable_byte_size_(numBytes) void* hkGuardingAllocator::blockAlloc(int numBytes)
{
    if (numBytes == 0)
        numBytes = 1;

    // Mutable memory region, including allocation data. The RW size we actually rounded up to a page size.
    size_t rwReqSize = HK_NEXT_MULTIPLE_OF(ALIGNMENT, numBytes) + sizeof(hkGuardingAllocatorData);

    int pageSize = getPageSize();
    size_t rwActualSize = HK_NEXT_MULTIPLE_OF(pageSize, rwReqSize);

    // including the uncommitted guard pages
    size_t reservedSize = (LEADING_PAGES + TRAILING_PAGES) * pageSize + rwActualSize;

    // Allocate the whole memory region as access protected, but don't commit yet
    void* reservedMem = VirtualAlloc(NULL, reservedSize, MEM_RESERVE, PAGE_NOACCESS);
    if (reservedMem == nullptr)
    {
        return outOfMemory("Internal error. Guarding allocator virtual address space exhausted.");
    }

    // Commit and allow access for the memory region between the guard pages
    void* rwMem = VirtualAlloc( hkAddByteOffset(reservedMem, LEADING_PAGES * pageSize), rwActualSize, MEM_COMMIT, PAGE_READWRITE);
    if (rwMem == nullptr)
    {
        return outOfMemory("Internal error. Guarding allocator out of memory.");
    }

    // Arrange for the userMem to be "right-aligned" because overruns are more common than underruns.
    // We end up with (note nesting)
    // * LEADING_PAGES unmapped pages
    // * rwActualSize bytes of read-write pages
    //   * possibly some empty bytes due to right-alignment
    //   * hkGuardingAllocatorData (size+some internal no-mans-land padding)
    //   * ***user mem***
    //   * possibly some empty bytes due to ALIGNMENT
    // * TRAILING_PAGES unmapped pages
    void* allocDataMem = hkAddByteOffset(rwMem, rwActualSize - rwReqSize);
    void* userMem = hkAddByteOffset(allocDataMem, sizeof(hkGuardingAllocatorData));

    hkGuardingAllocatorData* allocData = new (allocDataMem) hkGuardingAllocatorData();
    allocData->m_numBytes = numBytes;

    hkMemUtil::memSet(userMem, 0xCD, numBytes);

    return userMem;
}

void hkGuardingAllocator::blockFree(_In_opt_bytecount_(numBytesUnused) void* userMem, int numBytesUnused)
{
    if (userMem == NULL)
        return;

    HK_ASSERT(0x31843cdf, isGuardedAllocation(userMem), "Guarding allocator: Pointer was not allocated by this allocator.");

    hkCriticalSectionLock lock(&m_criticalSection);

    if (m_numDelayedFrees == DELAYED_FREE_QUEUE_SIZE)
    {
        void* page = m_delayedFrees[m_lastDelayedFree];
        HK_ON_DEBUG(BOOL freeSucceeded =) VirtualFree(page, 0, MEM_RELEASE);
        HK_ASSERT(0x10e497d4, freeSucceeded, "Guarding allocator: Releasing a virtual page failed.");
        m_numDelayedFrees--;
    }

    hkGuardingAllocatorData* allocData = reinterpret_cast<hkGuardingAllocatorData*>(userMem) - 1;
    allocData->verify();

    size_t numBytes = allocData->m_numBytes;

    int pageSize = getPageSize();

    size_t rwReqSize = HK_NEXT_MULTIPLE_OF(ALIGNMENT, numBytes) + sizeof(hkGuardingAllocatorData);
    size_t rwActualSize = HK_NEXT_MULTIPLE_OF(pageSize, rwReqSize);

    void* rwMem = hkAddByteOffset(allocData, rwReqSize - rwActualSize);
    void* reservedMem = hkAddByteOffset(rwMem, -LEADING_PAGES * pageSize);

    // Don't release memory immediately, instead append to the queue and release later
    // to catch accesses of recently freed memory

    // warning C6250: "Calling <function> VirtualFree without the MEM_RELEASE flag may free memory but not address descriptors(VADs); results in address space leaks"
    // Which is exactly what we want...
    HK_DETAIL_DIAG_MSVC_PUSH();
    HK_DETAIL_DIAG_MSVC_OFF(6250);
    HK_ON_DEBUG(BOOL freeSucceeded =) VirtualFree(rwMem, rwActualSize, MEM_DECOMMIT);
    HK_ASSERT(0x27d179c9, freeSucceeded, "Guarding allocator: Decommitting a virtual page failed.");
    HK_DETAIL_DIAG_MSVC_POP();

    m_delayedFrees[m_lastDelayedFree] = reservedMem;
    m_lastDelayedFree = (m_lastDelayedFree + 1) % DELAYED_FREE_QUEUE_SIZE;
    m_numDelayedFrees++;
}

void hkGuardingAllocator::getMemoryStatistics(MemoryStatistics& u) const
{
}

int hkGuardingAllocator::getAllocatedSize(_In_bytecount_(nbytes) const void* obj, int nbytes) const
{
    if (obj == NULL)
        return 0;

    HK_ASSERT(0x10a0e2d4, isGuardedAllocation(obj), "Guarding allocator: Pointer was not allocated by this allocator.");

    const hkGuardingAllocatorData* allocData = reinterpret_cast<const hkGuardingAllocatorData*>(obj) - 1;
    allocData->verify();

    return allocData->m_numBytes;
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
