// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/Allocator/Checking/hkDelayedFreeAllocator.h>
#include <Common/Base/Memory/System/Util/hkMemorySnapshot.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>

#include <Common/Base/System/Io/Writer/Printf/hkPrintfStreamWriter.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>

// This detects double frees in hkReferencedObject
static const int s_scrubValue = 0x0000feee;

hkDelayedFreeAllocator::hkDelayedFreeAllocator(hkStackTracer::CallTree& callTree)
    : m_next(HK_NULL)
    , m_prng(0)
    , m_callTree(callTree)
{
}

void hkDelayedFreeAllocator::init(_Inout_ hkMemoryAllocator* next, _In_opt_ const Limits* limits)
{
    Limits defaultLimits;
    m_next = next;
    m_limits = limits ? *limits : defaultLimits;
    m_kept._reserveExactly( *next, m_limits.m_numAllocsKept );
    m_prng.setSeed( m_limits.m_randomSeed );
    m_curBytesKept = 0;
}

void hkDelayedFreeAllocator::quit()
{
    releaseDelayedFrees();
    m_kept._clearAndDeallocate(*m_next);
    m_next = HK_NULL;
}

void hkDelayedFreeAllocator::stealAllocTraceId(_In_bytecount_(_Inexpressible_()) const void* p, hkStackTracer::CallTree::TraceId traceId)
{
    HK_ASSERT_NO_MSG(0x7d10d171, !m_traceIds.hasKey(p));
    m_traceIds.insert(*m_next, p, traceId);
}

void hkDelayedFreeAllocator::releaseDelayedFrees()
{
    if (m_next)
    {
        for (MapType::Iterator it = m_traceIds.getIterator(); m_traceIds.isValid(it); it = m_traceIds.getNext(it))
        {
            m_callTree.releaseCallStack(m_traceIds.getValue(it));
        }
        m_traceIds.clearAndDeallocate(*m_next);

        for( int i = 0; i < m_kept.getSize(); ++i )
        {
            const Alloc& a = m_kept[i];
            m_next->blockFree(a.p, a.size);
        }
        m_kept.clear();
        m_curBytesKept = 0;
    }
}

_Ret_notnull_ _Post_writable_byte_size_(numBytes) void* hkDelayedFreeAllocator::blockAlloc( int numBytes )
{
    m_inUse += numBytes;
    return m_next->blockAlloc( numBytes );
}

static hkBool32 blockOk(_In_reads_bytes_(size) const void* p, int size )
{
    HK_COMPILE_TIME_ASSERT(sizeof(int)==4);
    for( int i = 0; i < size/4; ++i )
    {
        if (static_cast<const int*>(p)[i] != s_scrubValue)
        {
            return false;
        }
    }
    return true;
}

#if defined(HK_DELAYED_FREE_LIST_ALLOCATOR_ENABLE_SERIAL_NUMBER)
static int s_blockFree_serial = 0;
#endif

void hkDelayedFreeAllocator::releaseTraceId(_In_bytecount_(_Inexpressible_()) const void* p)
{
    MapType::Iterator it = m_traceIds.findKey(p);
    if (m_traceIds.isValid(it))
    {
        m_callTree.releaseCallStack(m_traceIds.getValue(it));
        m_traceIds.remove(it);
    }
}

void hkDelayedFreeAllocator::blockFree(_In_bytecount_(numBytes) void* pfree, int numBytes )
{
    m_inUse -= numBytes;

#if 0 && defined(HK_DEBUG)
    int total = 0;
    for( int i = 0; i < m_kept.getSize(); ++i )
    {
        total += m_kept[i].size;
    }
    HK_ASSERT_NO_MSG(0x1f377c7d, total == m_curBytesKept );
#endif

    if( numBytes > m_limits.m_blockTooBig )
    {
        m_next->blockFree( pfree, numBytes ); // keeping such a large block is probably a waste
        releaseTraceId(pfree);
        return;
    }

    
    while( numBytes + m_curBytesKept > m_limits.m_maxBytesKept || m_kept.getSize() >= m_limits.m_numAllocsKept )
    {
        int kill = m_prng.getRand32() % m_kept.getSize();
        Alloc a = m_kept[kill];
        m_kept.removeAt(kill);
        if( blockOk( a.p, a.size) == hkFalse32 )
        {
            // memory modified after being freed
            hkPrintfStreamWriter printfWriter;
            hkOstream ostr(&printfWriter);
            ostr << "Use after free in block allocated at:\n";
            m_callTree.dumpTrace(m_traceIds.getWithDefault(a.p, -1), ostr);

            HK_BREAKPOINT(0);
        }

        releaseTraceId(a.p);

        m_curBytesKept -= a.size;
        m_next->blockFree(a.p, a.size);
    }


#if defined(HK_DELAYED_FREE_LIST_ALLOCATOR_ENABLE_SERIAL_NUMBER)
    Alloc a = {pfree, numBytes, ++s_blockFree_serial};
    if ( s_blockFree_serial == 0xc38 )  // check for your serial number of you bad allocation
    {
        s_blockFree_serial = s_blockFree_serial;    // and set a breakpoint
    }
#else
    Alloc a = { pfree, numBytes };
#endif

    hkString::memSet4(pfree, s_scrubValue, numBytes / 4);
    m_curBytesKept += numBytes;
    m_kept.pushBackUnchecked( a);
}

hkBool32 hkDelayedFreeAllocator::isOk() const
{
    for( int i = 0; i < m_kept.getSize(); ++i )
    {
        if( blockOk( m_kept[i].p, m_kept[i].size ) == hkFalse32 )
        {
            return false;
        }
    }
    return true;
}

void hkDelayedFreeAllocator::getMemoryStatistics( hkMemoryAllocator::MemoryStatistics& u ) const
{
    u.m_allocated = m_inUse + m_curBytesKept + (m_kept.getCapacity() * hkSizeOf(Alloc));
    u.m_inUse = m_inUse;
    u.m_available = 0;
}

int hkDelayedFreeAllocator::getAllocatedSize(_In_bytecount_(numBytes) const void* obj, int numBytes ) const
{
    return m_next->getAllocatedSize( obj, numBytes );
}

int hkDelayedFreeAllocator::addToSnapshot(hkMemorySnapshot& snapshot, int parentId) const
{
    int thisId = snapshot.addProvider("hkDelayedFreeAllocator", parentId);
    if(m_kept.getCapacity() > 0)
    {
        snapshot.addOverhead( thisId, m_kept.begin(), m_kept.getCapacity()*sizeof(Alloc) );
        for( int i = 0; i < m_kept.getSize(); ++i )
        {
            // show the delayed frees as overhead
            snapshot.addOverhead( thisId, m_kept[i].p, m_kept[i].size );
        }
    }
    return thisId;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
