// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef HK_MATH_MATH_H
#define HK_MATH_MATH_H

#include <Common/Base/Types/hkBaseTypes.h>
#include <Common/Base/Config/hkConfigSimd.h>

#if defined(__HAVOK_PARSER__)
    // This is a cut down version of math.h sufficient for the clang parser to work (without requiring external includes)
    namespace std
    {
        double sqrt(double);
        float sqrtf(float);
        double fabs(double);
        float fabsf(float);
        double pow(double,double);
        float powf(float,float);
        double ceil(double);
        float ceilf(float);
        double floor(double);
        float floorf(float);
        double sin(double);
        float sinf(float);
        double cos(double);
        float cosf(float);
        double asin(double);
        float asinf(float);
        double acos(double);
        float acosf(float);
        double tab(double);
        float tan(float);
        double atan2(double,double);
        float atan2f(float,float);
        double log(double);
        float logf(float);
        float fmodf(float, float);
        double fmod(double, double);
    }
    #ifndef HK_STD_NAMESPACE
        #define HK_STD_NAMESPACE std
    #endif
#elif defined(HK_CRT_NOT_ALLOWED)
    // Prevent inadvertent math/float inclusion outside hkBase via header file includes
#elif defined(HK_STANDARD_HEADER_USE_C_PREFIX)
#   include <cmath>
#   include <cfloat>
#else
#   include <math.h>
#   include <float.h>
#endif


// Only allow FPU exception checking on certain configurations.
// This is because we have to do a little extra work in some operations to avoid division by zero.
#if defined(HK_DEBUG) && defined(HK_PLATFORM_WIN32) && !defined(HK_ARCH_X64) && !defined(HK_REAL_IS_DOUBLE) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
#   define HK_ALLOW_FPU_EXCEPTION_CHECKING
#endif

// Temporarily disable the alignment warning for AMD64
#if defined(HK_ARCH_X64)
    HK_DETAIL_DIAG_MSVC_OFF(4328)
#endif
#if !defined(HK_DISABLE_MATH_CONSTRUCTORS)
#   define HK_ENABLE_MATH_CTOR
#endif

// Function to check whether or not denormals are being flushed to zero.
// All Havok math code expects this mode to be set, and may not function correctly if it isn't.
// Typically operations on denormals are very slow, up to 100 times slower than normal numbers.

HK_EXPORT_COMMON void HK_CALL hkCheckFlushDenormals();
#if defined(HK_DEBUG)
#   define HK_CHECK_FLUSH_DENORMALS() hkCheckFlushDenormals()
#else
#   define HK_CHECK_FLUSH_DENORMALS()
#endif

// Aligned, aggregate-initializable type for math defaults.
template<typename FT> struct hkAlignedQuad;
template<> struct hkAlignedQuad<float>  { HK_ALIGN_FLOAT(float a); float b; float c; float d; };
template<> struct hkAlignedQuad<double> { HK_ALIGN_DOUBLE(double a); double b; double c; double d; };
template<> struct hkAlignedQuad<int>    { HK_ALIGN16(int a); int b; int c; int d; };

// The order of these includes is crucial.
#include <Common/Base/Math/Header/hkMathHeaderConstantDefinitions.h>
#include <Common/Base/Math/Header/hkMathHeaderEnums.h>
#include <Common/Base/Math/Header/hkMathHeaderTypes.h>

#include <Common/Base/Math/Header/hkMathHeaderFuncs.h>
#include <Common/Base/Math/Header/hkMathHeaderForwardDeclarations.h>

#include <Common/Base/Math/Vector/hkVector4fComparison.h>
#include <Common/Base/Math/Vector/hkVector4dComparison.h>

#include <Common/Base/Math/Vector/hkSimdFloat32.h>
#include <Common/Base/Math/Vector/hkSimdDouble64.h>

#include <Common/Base/Math/Vector/hkVector4f.h>
#include <Common/Base/Math/Vector/hkVector4d.h>

#include <Common/Base/Math/Quaternion/hkQuaternionf.h>


#include <Common/Base/Math/Matrix/hkMatrix3.h>
#include <Common/Base/Math/Matrix/hkRotation.h>
#include <Common/Base/Math/Matrix/hkMatrix4f.h>


#include <Common/Base/Math/Matrix/hkTransformf.h>
#include <Common/Base/Math/QsTransform/hkQsTransformf.h>
#include <Common/Base/Math/QTransform/hkQTransformf.h>

#include <Common/Base/Math/Quaternion/hkQuaterniond.h>

#include <Common/Base/Math/Matrix/hkTransformd.h>
#include <Common/Base/Math/QsTransform/hkQsTransformd.h>
#include <Common/Base/Math/QTransform/hkQTransformd.h>



HK_ON_REAL_IS_DOUBLE(typedef hkQTransformd hkQTransformType);
HK_ON_REAL_IS_FLOAT( typedef hkQTransformf hkQTransformType);

/// Reflected hkQTransform
struct hkQTransform : public hkQTransformType
{
    //+version(2)
    HK_DECLARE_CLASS(hkQTransform, New, Pod);
    HK_DETAIL_DECLARE_REFLECT_EXPORT( hkQTransform, HK_EXPORT_COMMON );

    /// Default constructor - all elements are uninitialized.
    HK_INLINE hkQTransform();

    /// Creates a new hkQTransformf using the rotation quaternion q and translation t.
#if defined(HK_REAL_IS_DOUBLE)
    HK_INLINE hkQTransform(hkQuaterniondParameter q, hkVector4dParameter t);
#else
    HK_INLINE hkQTransform(hkQuaternionfParameter q, hkVector4f_ t);
#endif

    /// Copy constructor
    HK_INLINE hkQTransform(const hkQTransformType& qt);

    /// Returns a global identity transform.
    HK_INLINE static const hkQTransform& HK_CALL getIdentity();
};

namespace hkMath
{
    HK_INLINE void transpose( hkVector4f& a, hkVector4f& b, hkVector4f& c ) {       HK_TRANSPOSE3f( a,b,c); }
    HK_INLINE void transpose( hkVector4d& a, hkVector4d& b, hkVector4d& c ) {       HK_TRANSPOSE3d( a,b,c); }
    HK_INLINE void transpose( hkVector4f& a, hkVector4f& b, hkVector4f& c, hkVector4f& d )  {       HK_TRANSPOSE4f( a,b,c,d);   }
    HK_INLINE void transpose( hkVector4d& a, hkVector4d& b, hkVector4d& c, hkVector4d& d )  {       HK_TRANSPOSE4d( a,b,c,d);   }
};


#include <Common/Base/Math/Header/hkMathHeaderPlatformImplementations.h>

#include <Common/Base/Math/Header/hkMathTypes2.h>



# if defined(HK_REAL_IS_DOUBLE)
    typedef hkDouble64 hkReal;
    typedef hkQuadDouble64 hkQuadReal;

    #define g_vectorConstants g_vectordConstants
    #define HK_QUADREAL_CONSTANT HK_QUADDOUBLE_CONSTANT
    #define HK_TRANSPOSE3 HK_TRANSPOSE3d
    #define HK_TRANSPOSE4 HK_TRANSPOSE4d
# else
    typedef hkFloat32 hkReal;
    typedef hkQuadFloat32 hkQuadReal;

#define g_vectorConstants g_vectorfConstants
#define HK_QUADREAL_CONSTANT HK_QUADFLOAT_CONSTANT
#define HK_TRANSPOSE3 HK_TRANSPOSE3f
#define HK_TRANSPOSE4 HK_TRANSPOSE4f
# endif


#if !defined(HK_DISABLE_IMPLICIT_SIMDREAL_FLOAT_CONVERSION)
#   include <Common/Base/Math/hkMathMixedOperators.h>
#endif

// The declaration of math typedefs change depending on the build config (float/double). Since generated reflection
// code must be the same in both configs we force the underlying type to be config-dependent
#define HK_REFLECT_MATH_TYPEDEF(TYPE) \
    HK_REFLECT_TYPEDEF(HK_ATTR(hk::ReflectAsTypedef(TYPE)) HK_EXPORT_COMMON, TYPE, TYPE ## _Tag)

HK_REFLECT_MATH_TYPEDEF(hkReal);
HK_REFLECT_MATH_TYPEDEF(hkSimdReal);
HK_REFLECT_MATH_TYPEDEF(hkVector4);
HK_REFLECT_MATH_TYPEDEF(hkQuaternion);
HK_REFLECT_MATH_TYPEDEF(hkTransform);
HK_REFLECT_MATH_TYPEDEF(hkQsTransform);

// Some math typedefs are a typedef of a typedef, so the underlying type in the reflected Type must be a Tag (e.g.
// the type of hkMatrix3 must inherit from hkMatrix3f_Tag instead of hkMatrix3f). We define <type>_Underlying
// conditionally to one of the Tags and override the underlying type in reflection.
#define HK_REFLECT_MATH_TYPEDEF_NESTED(TYPE) \
    namespace hkReflect { namespace Typedef { \
        HK_ON_REAL_IS_FLOAT(typedef TYPE ## f_Tag TYPE ## _Underlying); \
        HK_ON_REAL_IS_DOUBLE(typedef TYPE ## d_Tag TYPE ## _Underlying); \
    } } \
    HK_REFLECT_TYPEDEF(HK_ATTR(hk::ReflectAsTypedef(hkReflect::Typedef::TYPE ## _Underlying)) HK_EXPORT_COMMON, TYPE, TYPE ## _Tag)


HK_REFLECT_MATH_TYPEDEF_NESTED(hkMatrix3);
HK_REFLECT_MATH_TYPEDEF_NESTED(hkMatrix4);
HK_REFLECT_MATH_TYPEDEF_NESTED(hkRotation);

#undef HK_REFLECT_MATH_TYPEDEF
#undef HK_REFLECT_MATH_TYPEDEF_NESTED

#endif // HK_MATH_MATH_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
