// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#ifndef HK_MATH_MATH_H
#   error Please include Common/Base/hkBase.h instead of this file.
#endif

// Components that aren't loaded during hkVector4d::load() are initialized to this in Debug and Dev configurations.
#define HK_VECTOR4d_DEBUG_FILL_VALUE (g_vectordConstants[HK_QUADREAL_NAN])

#define hkVector4d_  hkVector4dParameter
#define hkSimdDouble64_  hkSimdDouble64Parameter
#define hkVector4dComparison_ hkVector4dComparisonParameter

#if defined(HK_REAL_IS_DOUBLE)
    typedef hkVector4d  hkVector4;
    typedef hkVector4d_ hkVector4_;
    typedef hkVector4d hkVector4;
    typedef hkVector4dParameter hkVector4Parameter;
#endif


/// \class hkVector4d
///
/// A four component vector. It holds four floating point numbers according to the
/// definition of double. There are optimized SIMD implementations
/// available for several platforms. They use platform specific data structures and
/// code to provide efficient processing. Thus to be portable, nothing should be assumed
/// about the internal layout of the four floating point numbers.
///
/// The vector has several methods which take a template parameter for the number of
/// components to work on, for example dot<3>. This guarantees to be able to use
/// optimized implementations for the method as well as using the vector as a 3-component
/// point or a 4-component homogeneous vector which can be verified at compile time.
///
/// Methods for component access or component modification are declared both taking
/// a template argument or an integer index (broadcast() for example). Using the template
/// parameter version is preferred because it allows for compile time address calculation.
/// Use the integer index version only if the index is not known at compile time.
///
/// For accessing the contents of a vector, it is strongly recommended to use the
/// get/setComponent methods as they operate with hkSimdDouble64 which in turn has
/// efficient methods for performing arithmetic and other operations which avoid
/// leaving the SIMD processing pipeline. Using the operator() is only useful for
/// debugging or explicit storage purposes.
///
/// The methods which take a matrix and a vector to transform are available both
/// as forced inline (with _ prefix) or as out-of-line calls. The inline versions produce
/// faster code because of better interleaving of instructions but can bloat the
/// code considerably. In situations where code size matters, use the out-of-line versions.
///
/// Some floating point operations which involve division or square root calculations can be quite
/// slow. The implementations in this class offer improved performance for these operations
/// at the expense of accuracy and handling of denormalized floating point numbers.
/// We have balanced performance with behavior one expects from a standard math operator,
/// but these methods will in general not behave IEEE compliant like the system math library
/// calls. See the comments on setDiv(), setReciprocal() and similar methods on how they behave.
///
/// If one needs to control the denormals handling, can do with even lower floating point
/// accuracy, or your algorithm requires full IEEE compliant behavior, the vector offers an
/// advanced interface in the form of member template methods for the division and square
/// root routines. Each has explicit flags for controlling accuracy and denormal handling.
/// The given bit width is the calculated accuracy in the floating point mantissa.
/// The 23Bit versions are almost as precise as the full version (24Bit) but offer
/// a substantial speedup on most platforms because of less strict denorms handling.
/// The 12Bit versions again offer faster execution with even less precision. Their
/// usefulness depends on the algorithm at hand. When writing new code, it is
/// recommended to start with the full precision version and then check whether a
/// lower precision is acceptable. The template flags for controlling handling of
/// division-by-zero or negative square root arguments offer easy writing of checked
/// fail-safe math code.
///
/// The load() and store() methods take as a template argument the number of components to load/store.
/// The advanced interface in addition has template methods which take a flag for the
/// alignment of the memory pointer and for the rounding mode to apply in case of storing
/// to a lower precision floating point format.
///
/// \sa hkSimdDouble64 hkVector4dComparison hkHalf16 hkFloat16 hkMathAccuracyMode hkMathDivByZeroMode hkMathNegSqrtMode hkMathIoMode hkMathRoundingMode
class HK_EXPORT_COMMON hkVector4d
{
    public:

        HK_DECLARE_CLASS(hkVector4d, New, Reflect, Pod);
        HK_REFLECT_AS_ARRAY_FIXED(4, double);
        HK_RECORD_ATTR(hk::TypeLayout(hkMath::computeTypeSizeAlign));
        HK_RECORD_ATTR(hk::ReflectDetails(fields = false));
        HK_RECORD_ATTR( hk::Default(0,0,0,0) );
        HK_RECORD_ATTR( hk::DiffAtomic );
        HK_RECORD_ATTR(hk::IncludeInMgd(false));
        typedef hkAlignedQuad<double> ReflectDefaultType;

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
        /// Empty constructor.
        HK_INLINE hkVector4d() { }

        /// Creates a new hkVector4d from the specified three or four values, with the obvious definition. ( self = xyzw )
        HK_INLINE hkVector4d(double x, double y, double z, double w=double(0));

        explicit HK_ALWAYS_INLINE hkVector4d(double x) { set(x,x,x,x); }

        /// Creates a new hkVector4d, copying all 4 values from \a v. ( self.xyzw = v.xyzw ).
        /// \warning This constructor causes the compiler NOT to pass const hkVector4d's by register.
        HK_INLINE hkVector4d( const hkVector4d& v);
#endif


        /// \name Vector initialization
        ///@{

        HK_ALWAYS_INLINE void operator= ( hkVector4d_ v );      ///< self = v
        HK_ALWAYS_INLINE void operator= ( hkVector4f_ v ){ this->load<4>(&v(0)); }  ///< self = (double)v

        HK_ALWAYS_INLINE void set( hkSimdDouble64_ a, hkSimdDouble64_ b, hkSimdDouble64_ c, hkSimdDouble64_ d );    ///< Sets the components to xyzw
        HK_INLINE void set(double x, double y, double z, double w=double(0));           ///< Sets the components to xyzw

        HK_ALWAYS_INLINE void setAll( hkSimdDouble64_ x);       ///< Sets all components of self to the same value \a x.
        HK_INLINE void setAll(const double& x);             ///< Sets all components of self to the same value \a x.
        HK_INLINE void setZero();               ///< Sets all components to zero.

        HK_INLINE void set( hkVector4f_ a){ *this = a; }
        HK_INLINE void set( hkVector4d_ a){ *this = a; }
        HK_INLINE void set(const hkQuadDouble64& q);

        HK_INLINE void zeroComponent(const int i);///< Set component \a i to zero. ( self[i] = 0 ).
        template <int I>
        HK_ALWAYS_INLINE void zeroComponent();  ///< This is the faster version of zeroComponent(int i).

        HK_ALWAYS_INLINE static hkVector4d HK_CALL ctor( hkSimdDouble64_ a){ hkVector4d res; res.setAll(a); return res; }
        HK_ALWAYS_INLINE static hkVector4d HK_CALL ctor( float a ) { hkVector4d res; res.setAll( a ); return res; }
        HK_ALWAYS_INLINE static hkVector4d HK_CALL ctor( double a ) { hkVector4d res; res.setAll( a ); return res; }

        ///@}

        /// \name Vector calc methods
        ///@{

        HK_ALWAYS_INLINE void add( hkVector4d_   a )    {   setAdd( *this, a );     }   ///< this += a
        HK_ALWAYS_INLINE void sub( hkVector4d_   a )    {   setSub( *this, a );     }   ///< this -= a
        HK_ALWAYS_INLINE void mul( hkVector4d_   a )    {   setMul( *this, a );     }   ///< this *= a
        HK_ALWAYS_INLINE void mul( hkSimdDouble64_ a )  {   setMul( *this, a );     }   ///< this *= a
        HK_INLINE        void div( hkVector4d_   a )    {   setDiv( *this, a );     }   ///< this /= a, 23bit accuracy, no division by zero checks

        HK_ALWAYS_INLINE void operator+=( hkVector4d_   a ) {   setAdd( *this, a ); }   ///< this += a
        HK_ALWAYS_INLINE void operator-=( hkVector4d_   a ) {   setSub( *this, a ); }   ///< this -= a
        HK_ALWAYS_INLINE void operator*=( hkVector4d_   a ) {   setMul( *this, a ); }   ///< this *= a
        HK_ALWAYS_INLINE void operator*=( hkSimdDouble64_ a )   {   setMul( *this, a ); }   ///< this *= a
        HK_INLINE        void operator/=( hkVector4d_   a ) {   setDiv( *this, a ); }   ///< this /= a, 23bit accuracy, no division by zero checks
        HK_INLINE        void operator/=( hkSimdDouble64_ a )   {   setDiv( *this, a ); }   ///< this /= a, 23bit accuracy, no division by zero checks

        HK_ALWAYS_INLINE hkVector4d operator+( hkVector4d_   a ) const { hkVector4d r; r.setAdd( *this, a );    return r; } ///< this + a
        HK_ALWAYS_INLINE hkVector4d operator-( hkVector4d_   a ) const { hkVector4d r; r.setSub( *this, a );    return r; } ///< this - a
        HK_ALWAYS_INLINE hkVector4d operator-(            )   const { hkVector4d r; r.setSub( getZero(), *this );   return r; } ///< -this
        HK_ALWAYS_INLINE hkVector4d operator*( hkVector4d_   a ) const { hkVector4d r; r.setMul( *this, a );    return r; } ///< this *= a
        HK_ALWAYS_INLINE hkVector4d operator*( hkSimdDouble64_ a ) const { hkVector4d r; r.setMul( *this, a );  return r; } ///< this *= a
        HK_INLINE        hkVector4d operator/( hkVector4d_   a ) const { hkVector4d r; r.setDiv( *this, a );    return r; } ///< this /= a, 23bit accuracy, no division by zero checks
        HK_INLINE        hkVector4d operator/( hkSimdDouble64_ a ) const { hkVector4d r; r.setDiv( *this, a );  return r; } ///< this /= a, 23bit accuracy, no division by zero checks

        HK_ALWAYS_INLINE void setAdd( hkVector4d_ a, hkVector4d_   b );     ///<    self = a+b
        HK_ALWAYS_INLINE void setAdd( hkVector4d_ a, hkSimdDouble64_ b );       ///<    self = a+b, b is splatted to all 4 components
        HK_ALWAYS_INLINE void setSub( hkVector4d_ a, hkVector4d_   b );     ///<    self = a-b
        HK_ALWAYS_INLINE void setSub( hkVector4d_ a, hkSimdDouble64_ b );       ///<    self = a+b, b is splatted to all 4 components
        HK_ALWAYS_INLINE void setMul( hkVector4d_ a, hkVector4d_   b );     ///<    self = a*b
        HK_ALWAYS_INLINE void setMul( hkVector4d_ a, hkSimdDouble64_ b );       ///<    self = a*b
        HK_ALWAYS_INLINE void setDiv( hkVector4d_ a, hkVector4d_   b );     ///<    self = a/b  23bit accuracy, no division by zero checks
        HK_ALWAYS_INLINE void setDiv( hkVector4d_ a, hkSimdDouble64_ b ){ hkVector4d d; d.setAll(b); setDiv(a,d); };        ///<    self = a/b  23bit accuracy, no division by zero checks

        HK_ALWAYS_INLINE void setMul( hkSimdDouble64_ a, hkVector4d_ b ){ setMul(b,a); };   ///<    self = a*b

        HK_INLINE        void setReciprocal( hkVector4d_ v);        ///<    self = 1/b  23bit accuracy, no division by zero checks
        HK_INLINE        void setSqrt( hkVector4d_ a );         ///<    self = sqrt(a )  23bit accuracy, if (b<=0), zero is returned.
        HK_INLINE        void setSqrtInverse( hkVector4d_ a );      ///<    self = 1.0f/sqrt(a )  23bit accuracy, if (b<=0), zero is returned.

        HK_ALWAYS_INLINE void setAddMul( hkVector4d_ a, hkVector4d_   b, hkVector4d_   c);      ///< self = a + b*c
        HK_ALWAYS_INLINE void setAddMul( hkVector4d_ a, hkVector4d_   b, hkSimdDouble64_ c);        ///< self = a + b*c
        HK_ALWAYS_INLINE void setSubMul( hkVector4d_ a, hkVector4d_   b, hkVector4d_   c);      ///< self = a - b*c
        HK_ALWAYS_INLINE void setSubMul( hkVector4d_ a, hkVector4d_   b, hkSimdDouble64_ c);        ///< self = a - b*c

        HK_ALWAYS_INLINE void addMul    ( hkVector4d_   a, hkVector4d_   b ){ setAddMul(*this, a, b );  }   ///< self += b*c
        HK_ALWAYS_INLINE void addMul    ( hkVector4d_   a, hkSimdDouble64_ b ){ setAddMul(*this, a, b );    }   ///< self += b*c
        HK_ALWAYS_INLINE void addMul    ( hkSimdDouble64_ a, hkVector4d_   b ){ setAddMul(*this, b, a );    }   ///< self += b*c
        HK_ALWAYS_INLINE void subMul    ( hkVector4d_   a, hkVector4d_   b ){ setSubMul(*this, a, b );  }   ///< self -= b*c
        HK_ALWAYS_INLINE void subMul    ( hkVector4d_   a, hkSimdDouble64_ b ){ setSubMul(*this, a, b );    }   ///< self -= b*c
        HK_ALWAYS_INLINE void subMul    ( hkSimdDouble64_ a, hkVector4d_   b ){ setSubMul(*this, b, a );    }   ///< self -= b*c

        HK_ALWAYS_INLINE void setCross( hkVector4d_ a, hkVector4d_   b );                   ///< self = a cross b
        HK_ALWAYS_INLINE hkVector4d cross( hkVector4d_ a) const { hkVector4d r; r.setCross(*this,a); return r; };       ///< return a cross b

        template <int N>
        HK_ALWAYS_INLINE const hkSimdDouble64 dot( hkVector4d_ a ) const;   ///< return self dot<N> a
        HK_ALWAYS_INLINE hkSimdDouble64 dot3( hkVector4d_ a)  const;        ///< return self dot<3> a
        HK_ALWAYS_INLINE hkSimdDouble64 dot4( hkVector4d_ a ) const;        ///< return self dot<4> a

        HK_INLINE        void setInterpolate( hkVector4d_ a, hkVector4d_   b, hkSimdDouble64_ t );  ///< self = (1-t)*a + t*b


        ///@}

        /// \name Comparisons by-value and selection
        ///@{

        HK_ALWAYS_INLINE const hkVector4dComparison less( hkVector4d_ a ) const;            ///< result[i] = this[i] <  a[i]
        HK_ALWAYS_INLINE const hkVector4dComparison lessEqual( hkVector4d_ a ) const;       ///< result[i] = this[i] <= a[i]
        HK_ALWAYS_INLINE const hkVector4dComparison greater( hkVector4d_ a ) const;         ///< result[i] = this[i] >  a[i]
        HK_ALWAYS_INLINE const hkVector4dComparison greaterEqual( hkVector4d_ a ) const;    ///< result[i] = this[i] >= a[i]
        HK_ALWAYS_INLINE const hkVector4dComparison equal( hkVector4d_ a ) const;           ///< result[i] = this[i] == a[i]
        HK_ALWAYS_INLINE const hkVector4dComparison notEqual( hkVector4d_ a ) const;        ///< result[i] = this[i] != a[i]

        HK_ALWAYS_INLINE const hkVector4dComparison lessZero() const;                   ///< result[i] = this[i] < 0
        HK_ALWAYS_INLINE const hkVector4dComparison lessEqualZero() const;              ///< result[i] = this[i] <=0
        HK_ALWAYS_INLINE const hkVector4dComparison greaterZero() const;                    ///< result[i] = this[i] >0
        HK_ALWAYS_INLINE const hkVector4dComparison greaterEqualZero() const;           ///< result[i] = this[i] >=0
        HK_ALWAYS_INLINE const hkVector4dComparison equalZero() const;                  ///< result[i] = this[i] ==0
        HK_ALWAYS_INLINE const hkVector4dComparison notEqualZero() const;               ///< result[i] = this[i] !=0

        HK_ALWAYS_INLINE hkVector4dComparison operator<  ( hkVector4d_ a) const{ return this->less(a); }            ///< result[i] = this[i] <  a[i]
        HK_ALWAYS_INLINE hkVector4dComparison operator<= ( hkVector4d_ a) const{ return this->lessEqual(a); }       ///< result[i] = this[i] <= a[i]
        HK_ALWAYS_INLINE hkVector4dComparison operator>  ( hkVector4d_ a) const{ return this->greater(a); }         ///< result[i] = this[i] >  a[i]
        HK_ALWAYS_INLINE hkVector4dComparison operator>= ( hkVector4d_ a) const{ return this->greaterEqual(a); }    ///< result[i] = this[i] >= a[i]
        HK_ALWAYS_INLINE hkVector4dComparison operator== ( hkVector4d_ a) const{ return this->equal(a); }           ///< result[i] = this[i] == a[i]
        HK_ALWAYS_INLINE hkVector4dComparison operator!= ( hkVector4d_ a) const{ return this->notEqual(a); }        ///< result[i] = this[i] != a[i]

        HK_ALWAYS_INLINE hkVector4dComparison operator<  ( hkSimdDouble64_ a) const{ return this->less(hkVector4d::ctor(a)); }          ///< result[i] = this[i] <  a
        HK_ALWAYS_INLINE hkVector4dComparison operator<= ( hkSimdDouble64_ a) const{ return this->lessEqual(hkVector4d::ctor(a)); }     ///< result[i] = this[i] <= a
        HK_ALWAYS_INLINE hkVector4dComparison operator>  ( hkSimdDouble64_ a) const{ return this->greater(hkVector4d::ctor(a)); }           ///< result[i] = this[i] >  a
        HK_ALWAYS_INLINE hkVector4dComparison operator>= ( hkSimdDouble64_ a) const{ return this->greaterEqual(hkVector4d::ctor(a)); }  ///< result[i] = this[i] >= a
        HK_ALWAYS_INLINE hkVector4dComparison operator== ( hkSimdDouble64_ a) const{ return this->equal(hkVector4d::ctor(a)); }         ///< result[i] = this[i] == a
        HK_ALWAYS_INLINE hkVector4dComparison operator!= ( hkSimdDouble64_ a) const{ return this->notEqual(hkVector4d::ctor(a)); }      ///< result[i] = this[i] != a

        /// Compares the first N components of self to \a v and returns true if all the components are the same respective value. ( self == v )
        template <int N> HK_INLINE hkBool32 allExactlyEqual(hkVector4d_ v) const;
        template <int N> HK_ALWAYS_INLINE hkBool32 allExactlyEqual( hkSimdDouble64_ v ) const { return allExactlyEqual<N>( hkVector4d::ctor(v)); }

        /// Compares the first N components of self to zero and returns true if all the components are zero. ( self == 0 )
        template <int N> HK_INLINE hkBool32 allExactlyEqualZero() const;

        /// Compares the first N components of self to \a v and returns true if all the differences of all components are within \a epsilon range (inclusive). ( self >= v-epsilon && self <= v+epsilon ).
        /// \remark This is not the Euclidean epsilon distance.
        template <int N> HK_INLINE hkBool32 allEqual(hkVector4d_ v, hkSimdDouble64_ epsilon) const;

        /// Compares the first N components of self to zero and returns true if all the differences of all components are within \a epsilon range (inclusive). ( self >= -epsilon && self <= +epsilon ).
        /// \remark This is not the Euclidean epsilon distance.
        template <int N> HK_INLINE hkBool32 allEqualZero(hkSimdDouble64_ epsilon) const;

        /// Compares the first N components and returns true if all the components of self are less than the components of \a a. ( self < a )
        template <int N> HK_INLINE hkBool32 allLess(hkVector4d_ a) const;

        /// Compares the first N components and returns true if all the components of self are less than zero. ( self < 0 )
        template <int N> HK_INLINE hkBool32 allLessZero() const;

        /// Component-wise select values from \a trueValue or \a falseValue depending on whether
        /// the component is marked true or false in the \a compareMask. ( self = mask ? trueValue : falseValue )
        HK_INLINE void setSelect( hkVector4dComparison_ compareMask, hkVector4d_ trueValue, hkVector4d_ falseValue );

        /// Set every component to zero if the corresponding \a compareMask component is false. (this = compareMask ? this : zero)
        HK_INLINE void zeroIfFalse( hkVector4dComparison_ compareMask );

        /// Set every component to zero if the corresponding \a compareMask component is true. (this = compareMask ? zero : this)
        HK_INLINE void zeroIfTrue( hkVector4dComparison_ compareMask );

        /// Component-wise select values from \a trueValue or \a falseValue depending on whether
        /// the component is marked true or false in the template mask. ( self = mask ? trueValue : falseValue )
        template<hkVector4ComparisonMask::Mask M> HK_INLINE void setSelect( hkVector4d_ trueValue, hkVector4d_ falseValue );

        ///@}

        /// \name Sign and clamping operations
        ///@{

        /// Store \a v with self while flipping the signs of the first N components. ( self = N ? -v : v )
        template <int N> HK_INLINE void setNeg(hkVector4d_ v);

        /// Component-wise copy values from \a v while flipping the sign of the value if the corresponding flag in the \a mask is true. ( self = mask ? -v : v )
        HK_INLINE void setFlipSign(hkVector4d_ v, hkVector4dComparison_ mask);

        /// Component-wise copy values from \a v while flipping the sign of the value if the corresponding component in \a vSign is negative. ( self = (vSign < 0) ? -v : v )
        HK_INLINE void setFlipSign(hkVector4d_ v, hkVector4d_ vSign);

        /// Component-wise copy values from \a v while flipping the sign of each value if the scalar value of \a sSign is negative. ( self = (sSign < 0) ? -v : v )
        HK_INLINE void setFlipSign(hkVector4d_ v, hkSimdDouble64_ sSign);

        /// Sets the components of self to the absolute value of the components in \a v. ( self = abs(v) )
        HK_INLINE void setAbs(hkVector4d_ v);

        /// Component-wise compare \a a and \a b and store the smaller value to self. ( self = (a < b) ? a : b )
        HK_INLINE void setMin(hkVector4d_ a, hkVector4d_ b);

        /// Component-wise compare \a a and \a b and store the larger value to self. ( self = (a > b) ? a : b )
        HK_INLINE void setMax(hkVector4d_ a, hkVector4d_ b);

        /// setMax() but using integer compare if available, else using floating point compare
        HK_ALWAYS_INLINE void setMaxI( hkVector4d_ a, hkVector4d_ b );

        /// setMin() but using integer compare if available, else using floating point compare
        HK_ALWAYS_INLINE void setMinI( hkVector4d_ a, hkVector4d_ b );

        /// Component-wise clamp \a a between \a minVal and \a maxVal and store to self. ( self = min( maxVal, max(a, minVal) ) ).
        /// Note that if \a a is NaN, the result will be \a maxVal.
        HK_INLINE void setClamped( hkVector4d_ a, hkVector4d_ minVal, hkVector4d_ maxVal );

        /// Component-wise clamp \a a between zero and one and store to self. ( self = min( 1, max(a, 0) ) ).
        /// Note that if \a a is NAN, the result will be one.
        HK_INLINE void setClampedZeroOne( hkVector4d_ a );

        /// Sets self to a copy of \a vSrc that is rescaled to have a maximum length of \a maxLen.
        /// If \a vSrc is shorter than \a maxLen, no rescaling is performed.
        HK_INLINE void setClampedToMaxLength(hkVector4d_ vSrc, hkSimdDouble64_ maxLen);

        ///@}

        /// \name Out-of-line matrix operations
        ///@{

        /// Sets self to the vector \a b rotated by matrix \a a.
        void setRotatedDir(const hkMatrix3d& a, hkVector4d_ b);

        /// Sets self to the vector \a b rotated by the inverse matrix of \a a.
        void setRotatedInverseDir(const hkMatrix3d& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by matrix \a a.
        void setTransformedPos(const hkTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by the inverse matrix of \a a.
        void setTransformedInversePos(const hkTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a direction rotated by the quaternion \a quat.
        void setRotatedDir(hkQuaterniondParameter quat, hkVector4d_ direction);

        /// Sets self to the vector \a direction rotated by the inverse of quaternion \a quat.
        void setRotatedInverseDir(hkQuaterniondParameter quat, hkVector4d_ direction);

        /// Sets self to the vector \a b transformed by matrix \a a.
        /// \remark You need to use this method if scale is present.
        void setTransformedPos(const hkQsTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by the inverse matrix of \a a.
        /// \remark You need to use this method if scale is present.
        void setTransformedInversePos(const hkQsTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by matrix \a a.
        void setTransformedPos(const hkQTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by the inverse matrix of \a a.
        void setTransformedInversePos(const hkQTransformd& a, hkVector4d_ b);

        ///@}

        /// \name Forced in-line matrix operations
        ///@{

        /// Sets self to the vector \a b rotated by matrix \a a.
        /// This method is enforced inline.
        HK_INLINE void _setRotatedDir(const hkMatrix3d& a, hkVector4d_ b);

        /// Sets self to the vector \a b rotated by the inverse matrix of \a a.
        /// This method is enforced inline.
        HK_INLINE void _setRotatedInverseDir(const hkMatrix3d& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by matrix \a a.
        /// This method is enforced inline.
        HK_INLINE void _setTransformedPos(const hkTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by the inverse matrix of \a a.
        /// This method is enforced inline.
        HK_INLINE void _setTransformedInversePos(const hkTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by matrix \a a.
        /// This method is enforced inline.
        /// \remark You need to use this method if scale is present.
        HK_INLINE void _setTransformedPos(const hkQsTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by the inverse matrix of \a a.
        /// This method is enforced inline.
        /// \remark You need to use this method if scale is present.
        HK_INLINE void _setTransformedInversePos(const hkQsTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by matrix \a a.
        /// This method is enforced inline.
        HK_INLINE void _setTransformedPos(const hkQTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a b transformed by the inverse matrix of \a a.
        /// This method is enforced inline.
        HK_INLINE void _setTransformedInversePos(const hkQTransformd& a, hkVector4d_ b);

        /// Sets self to the vector \a direction rotated by the quaternion \a quat.
        /// This method is enforced inline.
        HK_INLINE void _setRotatedDir(hkQuaterniondParameter quat, hkVector4d_ direction);

        /// Sets self to the vector \a direction rotated by the inverse of quaternion \a quat.
        /// This method is enforced inline.
        HK_INLINE void _setRotatedInverseDir(hkQuaterniondParameter quat, hkVector4d_ direction);

        ///@}

        /// \name Length and normalization
        ///@{

        /// Sets all components of self to the N component dot product of \a a and \a b. ( self = a dot b )
        template <int N> HK_INLINE void setDot(hkVector4d_ a, hkVector4d_ b);

        /// Sums up N components. ( return x+y+z+w )
        template <int N> HK_INLINE const hkSimdDouble64 horizontalAdd() const;

        /// Sums up N components of \a v and stores to all components of self. ( self = v(x+y+z+w) )
        template <int N> HK_INLINE void setHorizontalAdd(hkVector4d_ v);

        /// Product of N components. ( return x*y*z*w )
        template <int N> HK_INLINE const hkSimdDouble64 horizontalMul() const;

        /// Product of N components of \a v and stores to all components of self. ( self = v(x*y*z*w) )
        template <int N> HK_INLINE void setHorizontalMul(hkVector4d_ v);

        /// Returns the maximum value occurring in N components. ( return max(x,y,z,w) )
        template <int N> HK_INLINE const hkSimdDouble64 horizontalMax() const;

        /// horizontalMax() but using integer compare if available, else using floating point compare
        template <int N> HK_INLINE const hkSimdDouble64 horizontalMaxI() const;

        /// Calculates the maximum value occurring in N components of \a v and stores to all components of self. ( self = v.max(x,y,z,w) )
        template <int N> HK_INLINE void setHorizontalMax(hkVector4d_ v);

        /// setHorizontalMax() but using integer compare if available, else using floating point compare
        template <int N> HK_INLINE void setHorizontalMaxI(hkVector4d_ v);

        /// Returns the minimum value occurring in N components. ( return min(x,y,z,w) )
        template <int N> HK_INLINE const hkSimdDouble64 horizontalMin() const;

        /// horizontalMin() but using integer compare if available, else using floating point compare
        template <int N> HK_INLINE const hkSimdDouble64 horizontalMinI() const;

        /// Calculates the minimum value occurring in N components of \a v and stores to all components of self. ( self = v.min(x,y,z,w) )
        template <int N> HK_INLINE void setHorizontalMin(hkVector4d_ v);

        /// setHorizontalMin() but using integer compare if available, else using floating point compare
        template <int N> HK_INLINE void setHorizontalMinI(hkVector4d_ v);

        /// this = (a.x+a.y),(a.z+a.w), (b.x+b.y),(b.z+b.w), this maps to a nice intrinsic on SSE3.0 or ARM neon
        HK_ALWAYS_INLINE void setPairedAdd(hkVector4d_ a, hkVector4d_ b);


        /// Returns the length of the vector represented by N of its components. ( return sqrt(self dot self) ).
        /// Accuracy: 23 bit, negative values checked and set to zero
        template <int N> HK_INLINE const hkSimdDouble64 length() const;

        /// Returns the squared length of the vector represented by N of its components. ( return (self dot self) )
        template <int N> HK_INLINE const hkSimdDouble64 lengthSquared() const;

        /// Returns the inverse length of the vector represented by N of its components. ( return 1 / sqrt(self dot self) ).
        /// Accuracy: 23 bit, negative sqrt values checked and set to zero, divide-by-0 checked and set to zero
        template <int N> HK_INLINE const hkSimdDouble64 lengthInverse() const;


        /// Normalizes self as an N-component vector. Unused components in self are undefined afterwards. ( self = |self| ).
        /// Accuracy: 23 bit, negative sqrt values checked and set to zero, divide-by-0 checked and set to zero
        template <int N> HK_INLINE void normalize();

        /// Normalizes self as an N-component vector and returns the length of self before normalization. Unused components in self are undefined afterwards. ( return self = |self| ).
        /// Accuracy: 23 bit, negative sqrt values checked and set to zero, divide-by-0 checked and set to zero
        template <int N> HK_INLINE const hkSimdDouble64 normalizeWithLength();

        /// Normalizes self as an N-component vector. Unused components in self are undefined afterwards. ( self = |self| ).
        /// If self is the zero vector, no normalization will occur and false is returned, else true.
        /// Accuracy: 23 bit, negative sqrt values checked and set to zero, divide-by-0 checked and set to zero
        template <int N> HK_INLINE hkBool32 normalizeIfNotZero();

        /// Normalizes \a v as an N-component vector, stores with self and returns the length of \a v before normalization. ( return self = |v| ).
        /// If \a v is zero, self is set to the default value (default: xyzw=(1,0,0,0) ).
        template <int N> HK_ALWAYS_INLINE const hkSimdDouble64 setNormalizedEnsureUnitLength( hkVector4d_ v, hkVector4d_ defaultValue = *(const hkVector4d*)(g_vectordConstants + HK_QUADREAL_1000) );

        ///@}

        /// \name Special functionality
        ///@{

        /// Set the w component of self to the negative 3-component dot product of \a aPointOnPlane with self. The xyz components of self are unchanged.
        /// ( self.w = -p.dot<3>(self) )
        HK_INLINE void setPlaneConstant(hkVector4d_ aPointOnPlane );

        /// Returns the dot product of self and \a a with the w-component of \a a replaced by 1. ( return self.xyzw dot a.xyz1 )
        HK_INLINE const hkSimdDouble64 dot4xyz1(hkVector4d_ a) const;

        /// Returns the Euclidean distance between self as a point and the point \a p. ( return len(self - p) ).
        /// Accuracy: 23 bit, negative sqrt values checked and set to zero, divide-by-0 checked and set to zero
        HK_INLINE const hkSimdDouble64 distanceTo( hkVector4d_ p ) const;

        /// Returns the squared Euclidean distance between self as a point and the point \a p. ( return len*len(self - p) )
        HK_INLINE const hkSimdDouble64 distanceToSquared( hkVector4d_ p ) const;

        ///@}

        /// \name Component access
        ///@{

        /// Set the xyz components of self from \a xyz. Set the w component of self from \a w.
        HK_INLINE void setXYZ_W(hkVector4d_ xyz, hkVector4d_ w);

        /// Set the xyz components of self from \a xyz. Set the w component of self from \a w.
        HK_INLINE void setXYZ_W(hkVector4d_ xyz, hkSimdDouble64_ w);

        /// Set the w component of self from \a v. The xyz components are unchanged.
        HK_INLINE void setW(hkVector4d_ v);

        /// Set the w component of self from \a w. The xyz components are unchanged.
        HK_INLINE void setW(hkSimdDouble64_ w);

        /// Set the xyz components of self from \a v. The w component is unchanged.
        HK_INLINE void setXYZ(hkVector4d_ v);

        /// Set the xyz components of self all to the same value \a v. The w component is unchanged.
        HK_INLINE void setXYZ(double v);

        /// Set the xyz components of self all to the same value \a v. The w component is unchanged.
        HK_INLINE void setXYZ(hkSimdDouble64_ v);

        /// Set the xyz components of self from \a xyz. Set the w component of self to zero.
        HK_INLINE void setXYZ_0(hkVector4d_ xyz);

        /// Stores a 24 bit integer value in the w component of self. The integer can be retrieved using the getInt24W() method.
        /// Note for storing negative integers, the sign handling needs to be done prior to using this method.
        /// \warning Using the w component in floating point calculations after this can destroy the integer value and
        ///          will most likely produce denormalized floating point numbers.
        HK_INLINE void setInt24W( int value );

        /// Returns a 24 bit integer stored in the w component of self. The integer value must have been set with
        /// setInt24W() before.
        HK_INLINE int getInt24W( ) const ;

        /// Returns a 16 bit integer stored in the w component of self. The integer value must have been set with
        /// setInt24W() before.
        HK_INLINE int getInt16W( ) const ;


        /// Gives read/write access to element i.  (0,1,2,3) correspond to the (x,y,z,w) components respectively.
        /// \remark Use this method only for storage purposes or immediate manipulation when the component index is not a compile time constant.
        HK_INLINE double& operator() (int i);

        /// Gives read only access to element i.  (0,1,2,3) correspond to the (x,y,z,w) components respectively.
        /// \remark Use this method only for storage purposes when the component index is not a compile time constant.
        HK_INLINE const double& operator() (int i) const;

        /// Return component I.  (0,1,2,3) correspond to the (x,y,z,w) components respectively.
        template <int I> HK_INLINE const hkSimdDouble64 getComponent() const;

        /// Return the w component, short for getComponent<3>()
        HK_ALWAYS_INLINE const hkSimdDouble64 getX() const;
        HK_ALWAYS_INLINE const hkSimdDouble64 getY() const;
        HK_ALWAYS_INLINE const hkSimdDouble64 getZ() const;
        HK_ALWAYS_INLINE const hkSimdDouble64 getW() const;

        /// Return component \a i.  (0,1,2,3) correspond to the (x,y,z,w) components respectively.
        /// \remark Use this method only when the component index is not a compile time constant.
        HK_INLINE const hkSimdDouble64 getComponent(const int i) const;

        /// Sets value of component I.  (0,1,2,3) correspond to the (x,y,z,w) components respectively.
        template <int I> HK_INLINE void setComponent(hkSimdDouble64_ val);

        /// Sets value of component \a i.  (0,1,2,3) correspond to the (x,y,z,w) components respectively.
        /// \remark Use this method only when the component index is not a compile time constant.
        HK_INLINE void setComponent(const int i, hkSimdDouble64_ val);

        /// Returns the index of the component for the first occurrence of the exact given \a value among the first N components.
        /// Returns -1 if no component matches. ( return self ?= component )
        template <int N> HK_INLINE int findComponent(hkSimdDouble64_ value) const;

        /// Set self to a component permutation of \a v. See the symbols in hkVectorPermutation.
        template <hkVectorPermutation::Permutation P> HK_INLINE void setPermutation(hkVector4d_ v);

        /// Set the value of component I of self on all components of self.
        template <int I> HK_INLINE void broadcast();

        /// Set the value of component \a i of self on all components of self.
        /// \remark Use this method only when the component index is not a compile time constant.
        HK_INLINE void broadcast(int i);

        /// Set the value of component I of \a v on all components of self.
        template <int I> HK_INLINE void setBroadcast(hkVector4d_ v);

        /// Set the value of component \a i of \a v on all components of self.
        /// \remark Use this method only when the component index is not a compile time constant.
        HK_INLINE void setBroadcast(const int i, hkVector4d_ v);

        /// Returns the index of the component with the largest absolute value among the first N components.
        /// In case of equality, returns the last component index given X,Y,Z,W ordering.
        template <int N> HK_INLINE int getIndexOfMaxAbsComponent() const;

        /// Returns the index of the component with the largest signed value among the first N components.
        /// In case of equality, returns the last component index given X,Y,Z,W ordering.
        template <int N> HK_INLINE int getIndexOfMaxComponent() const;

        /// same as getIndexOfMaxComponent, but uses integer compare, so this function works as long
        /// as the max component is positive, slightly faster than getIndexOfMaxComponent;
        template <int N> HK_INLINE int getIndexOfMaxIComponent() const;

        /// Returns the index of the component with the smallest absolute value among the first N components.
        /// In case of equality, returns the first component index given X,Y,Z,W ordering.
        template <int N> HK_INLINE int getIndexOfMinAbsComponent() const;

        /// Returns the index of the component with the smallest signed value among the first N components.
        /// In case of equality, returns the first component index given X,Y,Z,W ordering.
        template <int N> HK_INLINE int getIndexOfMinComponent() const;

        ///@}

        /// \name Checking, loading and storing
        ///@{

        /// Returns true if the first N components are valid finite floating point numbers.
        template <int N> HK_INLINE hkBool32 isOk() const;

        /// Returns true if the length of self is one within the given \a epsilon bounds.
        template <int N> HK_INLINE bool isNormalized(double epsilon = double(1e-4f)) const;

        /// Get a constant vector with all components zero.
        HK_INLINE static const hkVector4d& HK_CALL getZero();

        /// Get a constant vector. See the symbols in hkVectorConstant.
        template<int vectorConstant>
        HK_INLINE static const hkVector4d& HK_CALL getConstant();

        /// Get a constant vector. See the symbols in hkVectorConstant.
        /// \remark Use this method only when the constant is not known at compile time.
        HK_INLINE static const hkVector4d& HK_CALL getConstant(hkVectorConstant constant);

        /// Set this vector to a constant.
        template<int vectorConstant>
        HK_INLINE void setConstant();

        /// Load double precision floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void load(_In_reads_(N) const double* p);

        /// Load single precision floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void load(_In_reads_(N) const hkFloat32* p);

        /// Load single precision floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void load(_In_reads_(N) const hkHalf32* p);

        /// Load and unpack floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void load(_In_reads_(N) const hkHalf16* p);

        /// Load and unpack floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void load(_In_reads_(N) const hkFloat16* p);

        /// Store double precision floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void store(_Out_writes_all_(N) double* p) const;

        /// Store single precision floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void store(_Out_writes_all_(N) hkFloat32* p) const;

        /// Store single precision floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void store(_Out_writes_all_(N) hkHalf32* p) const;

        /// Pack and store floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void store(_Out_writes_all_(N) hkHalf16* p) const;

        /// Pack and store floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// The pointer \a p must be aligned for SIMD operations.
        template <int N> HK_INLINE void store(_Out_writes_all_(N) hkFloat16* p) const;

        ///@}


        /// \name Advanced interface
        ///@{

        /// Load double precision floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void load(_In_reads_(N) const double* p);

        /// Load single precision floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void load(_In_reads_(N) const hkFloat32* p);

        /// Load single precision floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void load(_In_reads_(N) const hkHalf32* p);

        /// Load and unpack floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void load(_In_reads_(N) const hkHalf16* p);

        /// Load and unpack floating point values for N components from linear addresses at \a p. Not loaded components are undefined.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void load(_In_reads_(N) const hkFloat16* p);

        /// Store double precision floating point values of N components to linear addresses at \a p.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A, hkMathRoundingMode R> HK_INLINE void store(_Out_writes_all_(N) double* p) const;

        /// Store single precision floating point values of N components to linear addresses at \a p.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A, hkMathRoundingMode R> HK_INLINE void store(_Out_writes_all_(N) float* p) const;

        /// Store single precision floating point values of N components to linear addresses at \a p.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A, hkMathRoundingMode R> HK_INLINE void store(_Out_writes_all_(N) hkHalf32* p) const;

        /// Pack and store floating point values of N components to linear addresses at \a p.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A, hkMathRoundingMode R> HK_INLINE void store(_Out_writes_all_(N) hkHalf16* p) const;

        /// Pack and store floating point values of N components to linear addresses at \a p.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A, hkMathRoundingMode R> HK_INLINE void store(_Out_writes_all_(N) hkFloat16* p) const;

        /// Store double precision floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void store(_Out_writes_all_(N) double* p) const;

        /// Store single precision floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void store(_Out_writes_all_(N) float* p) const;

        /// Store single precision floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void store(_Out_writes_all_(N) hkHalf32* p) const;

        /// Pack and store floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void store(_Out_writes_all_(N) hkHalf16* p) const;

        /// Pack and store floating point values of N components to linear addresses at \a p.
        /// The rounding is the system default.
        /// See the documentation at the template values for the requested IO mode.
        template <int N, hkMathIoMode A> HK_INLINE void store(_Out_writes_all_(N) hkFloat16* p) const;

        /// Set self to the component-wise reciprocal of \a v. ( self = 1/v ).
        /// See the documentation at the template values for the requested behavior.
        template <hkMathAccuracyMode A, hkMathDivByZeroMode D> HK_INLINE void setReciprocal(hkVector4d_ v);

        /// Set self to the component-wise quotient of \a a over \a b. ( self = a/b ).
        /// See the documentation at the template values for the requested behavior.
        template <hkMathAccuracyMode A, hkMathDivByZeroMode D> HK_INLINE void setDiv(hkVector4d_ a, hkVector4d_ b);

        /// Component-wise division by \a a. ( self /= a ).
        /// See the documentation at the template values for the requested behavior.
        template <hkMathAccuracyMode A, hkMathDivByZeroMode D> HK_INLINE void div(hkVector4d_ a);

        /// Set self to the component-wise square root of \a a. ( self = sqrt(a) ).
        /// See the documentation at the template values for the requested behavior.
        template <hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE void setSqrt(hkVector4d_ a);

        /// Set self component-wise to one over square root of \a a. ( self = 1/sqrt(a) ).
        /// See the documentation at the template values for the requested behavior.
        template <hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE void setSqrtInverse(hkVector4d_ a);

        /// Returns the length of the vector represented by N of its components. ( return sqrt(self dot self) ).
        /// See the documentation at the template values for the requested behavior.
        template <int N, hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE const hkSimdDouble64 length() const;

        /// Returns the inverse length of the vector represented by N of its components. ( return 1 / sqrt(self dot self) ).
        /// See the documentation at the template values for the requested behavior.
        template <int N, hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE const hkSimdDouble64 lengthInverse() const;

        /// Normalizes self as an N-component vector. Unused components in self are undefined afterwards. ( self = |self| ).
        /// See the documentation at the template values for the requested behavior.
        template <int N, hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE void normalize();

        /// Normalizes self as an N-component vector and returns the length of self before normalization. Unused components in self are undefined afterwards. ( return self = |self| ).
        /// See the documentation at the template values for the requested behavior.
        template <int N, hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE const hkSimdDouble64 normalizeWithLength();

        /// Normalizes self as an N-component vector. Unused components in self are undefined afterwards. ( self = |self| ).
        /// If self is the zero vector, no normalization will occur and false is returned, else true.
        /// See the documentation at the template values for the requested behavior.
        template <int N, hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE hkBool32 normalizeIfNotZero();

        /// Returns the Euclidean distance between self as a point and the point \a p. ( return len(self - p) ).
        /// See the documentation at the template values for the requested behavior.
        template <hkMathAccuracyMode A, hkMathNegSqrtMode S> HK_INLINE const hkSimdDouble64 distanceTo( hkVector4d_ p ) const;

        /// Normalizes \a v as an N-component vector, stores with self and returns the length of \a v before normalization. ( return self = |v| ).
        /// If \a v is zero (within machine epsilon), self is set to the unit vector (1,0,0,0).
        /// See the documentation at the template values for the requested behavior.
        template <int N, hkMathAccuracyMode A> HK_INLINE const hkSimdDouble64 setNormalizedEnsureUnitLength(hkVector4d_ v, hkVector4d_ defaultValue = hkVector4d::getConstant(HK_QUADREAL_1000));

        /// Sets self to a copy of \a vSrc that is rescaled to have a maximum length of \a maxLen.
        /// If \a vSrc is shorter than \a maxLen, no rescaling is performed.
        template <int N, hkMathAccuracyMode A>
        HK_INLINE void setClampedToMaxLength(hkVector4d_ vSrc, hkSimdDouble64_ maxLen);

        ///@}


        /// Internal data storage of the vector components (platform dependent).
        /// For writing portable code, nothing can be assumed about the internal layout of the values.
        HK_ALIGN_DOUBLE( hkQuadDouble64 m_quad );   


#ifndef HK_DISABLE_OLD_VECTOR4_INTERFACE

        //
        // old interface
        //

        HK_INLINE hkQuadDouble64& getQuad();
        HK_INLINE const hkQuadDouble64& getQuad() const;
        HK_INLINE void operator= ( const hkQuadDouble64& v );
        HK_INLINE void add4(hkVector4d_ v);
        HK_INLINE void sub4(hkVector4d_ v);
        HK_INLINE void mul4(hkVector4d_ a);
        HK_INLINE void mul4(hkSimdDouble64_ a);
        HK_INLINE void div4(hkVector4d_ a);
        HK_INLINE void div4fast(hkVector4d_ a);
        HK_INLINE double dot3fpu(hkVector4d_ a) const;
        HK_INLINE void setMul4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setMul4(hkSimdDouble64_ a, hkVector4d_ b);
        HK_INLINE void _setMul3(const hkMatrix3d& m, hkVector4d_ v);
        HK_INLINE void setMul3(const hkMatrix3d& a, hkVector4d_ b );
        HK_INLINE void subMul4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void subMul4(hkSimdDouble64_ a, hkVector4d_ b);
        HK_INLINE void setSubMul4(hkVector4d_ a, hkVector4d_ x, hkVector4d_ y);
        HK_INLINE void setSubMul4(hkVector4d_ a, hkVector4d_ x, hkSimdDouble64_ y);
        HK_INLINE void setDot3(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setDot4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setSelect4( hkVector4dComparison_ comp, hkVector4d_ trueValue, hkVector4d_ falseValue);
        HK_INLINE void select32( hkVector4d_ falseValue, hkVector4d_ trueValue, hkVector4dComparison_ comp);
        HK_INLINE void setBroadcast(hkVector4d_ v, int i);
        HK_INLINE void setBroadcast3clobberW(hkVector4d_ v, int i);
        HK_INLINE void setXYZW(hkVector4d_ xyz, hkVector4d_ w);
        HK_INLINE void setXYZW(hkVector4d_ xyz, hkSimdDouble64_ w);
        HK_INLINE void setXYZ0(hkVector4d_ xyz);
        HK_INLINE void addMul4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void addMul4(hkSimdDouble64_ a, hkVector4d_ b);
        HK_INLINE void setZero4();
        HK_INLINE void zeroElement( int i );
        HK_INLINE void setAll3(double x);
        HK_INLINE void setSwapXY(const hkVector4d& x);
        HK_INLINE void setNeg3(hkVector4d_ v);
        HK_INLINE void setNeg4(hkVector4d_ v);
        HK_INLINE void setNegMask4(hkVector4d_ v, int mask);
        HK_INLINE void setDiv4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setDiv4fast(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE hkSimdDouble64 getSimdAt(int i) const;
        HK_INLINE void normalize3();
        HK_INLINE void normalize4();
        HK_INLINE void fastNormalize3();
        HK_INLINE void setFastNormalize3NonZero(hkVector4d_ other);
        HK_INLINE hkSimdDouble64 normalizeWithLength3();
        HK_INLINE hkSimdDouble64 normalizeWithLength4();
        HK_INLINE hkSimdDouble64 fastNormalizeWithLength3();
        HK_INLINE void fastNormalize3NonZero();
        HK_INLINE hkResult normalize3IfNotZero ();
        HK_INLINE hkBool isNormalized3(double eps = 1e-4f) const;
        HK_INLINE hkBool isNormalized4(double eps = 1e-4f) const;
        HK_INLINE hkSimdDouble64 length3() const;
        HK_INLINE hkSimdDouble64 length4() const;
        HK_INLINE hkSimdDouble64 lengthSquared3() const;
        HK_INLINE hkSimdDouble64 lengthSquared4() const;
        HK_INLINE hkSimdDouble64 lengthInverse3() const;
        HK_INLINE hkSimdDouble64 lengthInverse4() const;
        HK_INLINE void setMulSigns4(hkVector4d_ a, hkVector4d_ signs);
        HK_INLINE void setMulSigns4(hkVector4d_ a, hkSimdDouble64_ sharedSign);
        HK_INLINE void setAdd4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setSub4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setSqrtInverse4(hkVector4d_ v);
        HK_INLINE void setSqrtInverse4_7BitAccuracy(hkVector4d_ v);
        HK_INLINE void setReciprocal3(hkVector4d_ v);
        HK_INLINE void setReciprocal4(hkVector4d_ v);
        HK_INLINE void setAddMul4(hkVector4d_ a, hkVector4d_ x, hkVector4d_ y);
        HK_INLINE void setAddMul4(hkVector4d_ a, hkVector4d_ b, hkSimdDouble64_ r);
        HK_INLINE void setAbs4(hkVector4d_ v);
        HK_INLINE void setMin4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE void setMax4(hkVector4d_ a, hkVector4d_ b);
        HK_INLINE hkBool32 equals3(const hkVector4d &a, double epsilon = 1e-3f ) const;
        HK_INLINE hkBool32 equals4(const hkVector4d &a, double epsilon = 1e-3f ) const;
        HK_INLINE hkVector4dComparison compareEqual4(hkVector4d_ a) const;
        HK_INLINE hkVector4dComparison compareLessThan4(hkVector4d_ a) const;
        HK_INLINE hkVector4dComparison compareLessThanEqual4(hkVector4d_ a) const;
        HK_INLINE hkVector4dComparison compareGreaterThan4(hkVector4d_ a) const;
        HK_INLINE hkVector4dComparison compareGreaterThanEqual4(hkVector4d_ a) const;
        HK_INLINE hkVector4dComparison compareLessThanZero4() const;
        HK_INLINE hkBool32 allLessThan3(hkVector4d_ a) const;
        HK_INLINE hkBool32 allLessThan4(hkVector4d_ a) const;
        HK_INLINE void setInterpolate4( hkVector4d_ a, hkVector4d_ b, hkSimdDouble64_ t );
        HK_INLINE hkSimdDouble64 distanceTo3( hkVector4d_ p ) const;
        HK_INLINE hkSimdDouble64 distanceToSquared3( hkVector4d_ p ) const;
        HK_INLINE hkSimdDouble64 horizontalAdd3() const;
        HK_INLINE void setHorizontalMax4( hkVector4d_ p);
        HK_INLINE hkSimdDouble64 getHorizontalMin3() const;
        HK_INLINE hkSimdDouble64 getHorizontalMax3() const;
        HK_INLINE void _setMul4xyz1(const hkTransformd& a, hkVector4d_ b );
        template <int S> HK_INLINE void setShuffle(hkVector4d_ v);
        HK_INLINE hkBool isOk3() const;
        HK_INLINE hkBool isOk4() const;
        HK_INLINE const hkVector4dComparison isNegative() const;
        HK_INLINE const hkVector4dComparison isPositive() const;
        HK_INLINE void load3(_In_reads_(3) const hkFloat32* p);
        HK_INLINE void load4(_In_reads_(4) const hkFloat32* p);
        HK_INLINE void load4a(_In_reads_(4) const hkFloat32* p);
        HK_INLINE void store3(_Out_writes_all_(3) hkFloat32* p) const;
        HK_INLINE void store4(_Out_writes_all_(4) hkFloat32* p) const;
        HK_INLINE void store4a(_Out_writes_all_(4) hkFloat32* p) const;
        HK_INLINE void storeX(_Out_writes_all_(1) hkFloat32* dest) const;
        HK_INLINE void load3(_In_reads_(3) const double* p);
        HK_INLINE void load4(_In_reads_(4) const double* p);
        HK_INLINE void load4a(_In_reads_(4) const double* p);
        HK_INLINE void store3(_Out_writes_all_(3) double* p) const;
        HK_INLINE void store4(_Out_writes_all_(4) double* p) const;
        HK_INLINE void store4a(_Out_writes_all_(4) double* p) const;
        HK_INLINE void storeX(_Out_writes_all_(1) double* dest) const;
        HK_INLINE hkSimdDouble64 horizontalAdd4() const;
        HK_INLINE void setClamped(hkVector4d_ vSrc, const hkSimdDouble64& maxLen);
        HK_INLINE   int getMaxElementIndex4() const;
        HK_INLINE   int getMajorAxis3() const;
        HK_INLINE void storeUncached( void* dest) const;
        HK_INLINE void _setMul4(const hkMatrix3d& a, hkVector4d_ b );
        HK_INLINE void add3clobberW( hkVector4d_ a ){ setAdd(*this,a); }
        HK_INLINE void sub3clobberW( hkVector4d_ a ){ setSub(*this,a); }
        HK_INLINE void setAnd( hkVector4d_ v0, hkVector4d_ v1 );

#endif
};

HK_ALWAYS_INLINE hkVector4d HK_CALL hkSelect( hkVector4dComparison_ sel, hkVector4d_ a, hkVector4d_ b){ hkVector4d r; r.setSelect(sel,a,b); return r; }

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
