// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#ifndef HK_MATH_VECTOR4_UTIL_H
#define HK_MATH_VECTOR4_UTIL_H

// this: #include <Common/Base/Math/Vector/hkVector4Util.h>

#include <Common/Base/Math/Vector/hkIntVector.h>

class hkStringBuf;

template <typename FT>
class hkVector4UtilImpl
{
public:
    typedef hkVector4UtilImpl<FT> ThisType;

    typedef typename hkFloatTypes<FT>::Vec4 Vec4;
    typedef typename hkFloatTypes<FT>::Vec4_ Vec4_;

    typedef typename hkFloatTypes<FT>::Scalar Scalar;
    typedef typename hkFloatTypes<FT>::Scalar_ Scalar_;

    typedef typename hkFloatTypes<FT>::Transform Transform;
    typedef typename hkFloatTypes<FT>::Matrix3 Matrix3;
    typedef typename hkFloatTypes<FT>::QTransform QTransform;
    typedef typename hkFloatTypes<FT>::QsTransform QsTransform;
    typedef typename hkFloatTypes<FT>::Quaternion Quaternion;
    typedef typename hkFloatTypes<FT>::Rotation Rotation;
    typedef typename hkFloatTypes<FT>::Comparison Comparison;

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MATH, ThisType);

    /// Sets the calling vector to be the normal to the 2 vectors (b-a) and (c-a).
    ///
    /// NOTE: The calculated result is not explicitly normalized.
    static HK_INLINE void HK_CALL setNormalOfTriangle( Vec4& result, Vec4_ a, Vec4_ b, Vec4_ c);

    /// Compute the affine transformation that maps the triangle S to the triangle D.
    static bool HK_CALL computeAffineTransformFromTriangles(    Vec4_ sa, Vec4_ sb, Vec4_ sc,
                                                                Vec4_ da, Vec4_ db, Vec4_ dc,
                                                                Transform& transformOut);

    /// Returns atan2(y,x)
    static HK_INLINE Scalar HK_CALL atan2(Scalar_ y, Scalar_ x);

    /// Componentwise: result.xyzw = atan2(y(0),x(0)), atan2(y(1),x(1)), atan2(y(2),x(2)), atan2(y(3),x(3))
    static HK_INLINE void HK_CALL atan2(Vec4_ y, Vec4_ x, Vec4& result);

    /// Convenience returns atan2(v(1), v(0))
    static HK_INLINE Scalar HK_CALL atan2(Vec4_ v);

    static HK_INLINE Scalar HK_CALL linearAtan2Approximation(Scalar_ y, Scalar_ x);
    static HK_INLINE void   HK_CALL linearAtan2Approximation(Vec4_ y, Vec4_ x, Vec4& result);
    static HK_INLINE Scalar HK_CALL linearAtan2ApproximationRough(Scalar_ y, Scalar_ x);
    static HK_INLINE void   HK_CALL linearAtan2ApproximationRough(Vec4_ y, Vec4_ x, Vec4& result);
    static HK_INLINE void   HK_CALL logApproximation(Vec4_ v, Vec4& result);

    static HK_INLINE void HK_CALL convertQuaternionToRotation( const Quaternion& qi, Rotation& rotationOut );
    static void HK_CALL convertQuaternionToRotationNoInline( const Quaternion& qi, Rotation& rotationOut );

    /// convert 4 quaternions to 4 rotations, much faster on Xbox 360/PlayStation(R)3, no speedup on PC
    static void HK_CALL convert4QuaternionsToRotations(_In_reads_(4) const Quaternion* quats, _Inout_ Rotation* r0Out, _Inout_ Rotation* r1Out, _Inout_ Rotation* r2Out, _Inout_ Rotation* r3Out);


    /// Converts a vector to four 16-bit unsigned integer values, clamped between min and max.
    static HK_INLINE void HK_CALL convertToUint16WithClip( Vec4_ in, Vec4_ offset, Vec4_ scale, Vec4_ min, Vec4_ max, hkIntUnion64& out);

    /// calculates a value x so that convertToUint16WithClip( out, in + x/scale, ... ) == out = int(floor( (in+offset)*scale
    static void HK_CALL getFloatToInt16FloorCorrection(typename hkFloatTypes<FT>::Float& out);

    ///
    static HK_INLINE void HK_CALL convertToUint32(Vec4_ in, Vec4_ offset, Vec4_ scale, _Out_writes_all_(4) hkUint32* out);

    ///
    static HK_INLINE void HK_CALL convertToUint32WithClip( Vec4_ in, Vec4_ offset, Vec4_ scale, Vec4_ min, Vec4_ max, _Out_writes_all_(4) hkUint32* out);

    /// calculates a value x so that convertToUint32WithClip( out, in + x/scale, ... ) == out = int(floor( (in+offset)*scale
    static void HK_CALL getFloatToInt32FloorCorrection(typename hkFloatTypes<FT>::Float& out);

    static HK_INLINE void HK_CALL convertComparison(const Comparison& cin, hkVector4dComparison& cout);
    static HK_INLINE void HK_CALL convertComparison(const Comparison& cin, hkVector4fComparison& cout);

    static HK_INLINE void HK_CALL convertVector(Vec4_ vin, hkVector4d& vout);
    static HK_INLINE void HK_CALL convertVector(Vec4_ vin, hkVector4f& vout);

    /// Finds a vector that is perpendicular to a line segment.
    ///
    /// Achieved by constructing a vector from the segment vector with the following properties
    /// (segment vector is any vector parallel to the line segment):
    /// 1. Component with the smallest index is set to 0.
    /// 2. The remaining two component are copied into the new vector but are swapped in position.
    /// 3. One of the copied components is multiplied by -1.0 (has its sign flipped!).
    ///
    /// leaving (for example):
    /// segmentVector = (x, y, z), with let's say y as the smallest component.
    /// segmentNormal = (-z, 0 , x) or (z, 0, -x), either will do.
    ///
    /// This will in fact be orthogonal as (in the example) the dot product will be zero.
    /// I.e., x*-z + y*0 + z*x = 0
    ///
    static HK_INLINE void HK_CALL calculatePerpendicularVector(Vec4_ vectorIn, Vec4& biVectorOut);

    /// same as calculatePerpendicularVector:
    ///  - if INVERSE2= false: bi2VectorOut = vectorIn.cross(biVectorOutbut)
    ///  - if INVERSE2= true: bi2VectorOut  = biVectorOutbut.cross(vectorIn)
    template<bool INVERSE2>
    static HK_INLINE void HK_CALL calculatePerpendicularNormalizedVectors(Vec4_ vectorIn, Vec4& biVectorOut, Vec4& bi2VectorOut);

    static HK_INLINE void HK_CALL transpose(Vec4& v0, Vec4& v1, Vec4& v2);
    static HK_INLINE void HK_CALL transpose(Vec4& v0, Vec4& v1, Vec4& v2, Vec4& v3);

    /// Transforms an array of points.
    static HK_INLINE void HK_CALL transformPoints(const Transform& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut);
    static HK_INLINE void HK_CALL transformPoints( const QTransform& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut );
    static HK_INLINE void HK_CALL transformPoints( const typename hkFloatTypes<FT>::QsTransform& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut );

    /// Transforms an array of points including the .w component
    static HK_INLINE void HK_CALL mul4xyz1Points(const Transform& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut);

    /// Transforms an array of spheres (keeps the .w component as the radius)
    static HK_INLINE void HK_CALL transformSpheres(const Transform& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut);

    /// Transforms an array of spheres (keeps the .w component as the radius)
    static HK_INLINE void HK_CALL transformSpheres( const typename hkFloatTypes<FT>::QsTransform& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut );

    /// Transforms a plane equation
    static HK_INLINE void HK_CALL transformPlaneEquation( const Transform& t, Vec4_ planeIn, Vec4& planeOut );

    /// Inverse transform a plane equation
    static HK_INLINE void HK_CALL transformInversePlaneEquation( const Transform& t, Vec4_ planeIn, Vec4& planeOut );

    /// Transforms an array of plane equations
    static HK_INLINE void HK_CALL transformPlaneEquations(const Transform& t, _In_reads_(numPlanes) const Vec4* planesIn, int numPlanes, _Inout_updates_all_(numPlanes) Vec4* planesOuts);

    /// Transform a plane equation with non-uniform scaling
    static HK_INLINE void HK_CALL transformAndScalePlane( const Transform& transform, const Vec4& scale, Vec4& planeInOut );

    /// Scale a plane equation with non-uniform scaling
    static HK_INLINE void HK_CALL scalePlane(const Vec4& scale, Vec4& planeInOut );

    /// Transform a plane equation with non-uniform scaling
    static HK_INLINE void HK_CALL translatePlane( const Vec4& translation, Vec4& planeInOut );

    /// Rotate an array of points.
    static HK_INLINE void HK_CALL rotatePoints(const Matrix3& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut);

    /// Invert rotate an array of points.
    static HK_INLINE void HK_CALL rotateInversePoints(const Rotation& t, _In_reads_(numVectors) const Vec4* vectorsIn, int numVectors, _Inout_updates_all_(numVectors) Vec4* vectorsOut);

    /// set aTcOut = aTb * bTc
    static HK_INLINE void HK_CALL setMul( const Matrix3& aTb, const Matrix3& bTc, Matrix3& aTcOut );

    /// set aTcOut = bTa^1 * bTc
    static HK_INLINE void HK_CALL setInverseMul( const Rotation& bTa, const Matrix3& bTc, Matrix3& aTcOut );

    /// Fully unrolled version of Transform::_setMulInverseMul(), more code, faster
    static HK_INLINE void HK_CALL _setMulInverseMul(const Transform& bTa, const Transform &bTc, _Inout_ Transform* tOut);

    /// Invert a 2x2 matrix stored in a vector containing col-major [r0c0, r1c0, r0c1, r1c1].
    /// Returns HK_FAILURE if determinant is less than or equal to \a tolerance.
    static HK_INLINE hkResult HK_CALL invert2x2Matrix(Vec4_ m, Scalar_ tolerance, Vec4& out);

    /// Compute sin and cos at the same time sines.xyzw = sin(r.x)sin(r.y)sin(r.z)sin(r.w) cosines.xyzw = cos(r.x)cos(r.y)cos(r.z)cos(r.w)
    static HK_INLINE void HK_CALL sinCos(Vec4_ r, Vec4& sines, Vec4& cosines);

    /// Compute sin and cos in alternating components of sc.xyzw = sin(r.x) cos(r.y) sin(r.z) cos(r.w)
    static HK_INLINE void HK_CALL sinCos(Vec4_ r, Vec4& sc);

    /// Compute sin and cos from value \a r to sc.xyzw = sin(r) cos(r) sin(r) cos(r)
    static HK_INLINE void HK_CALL sinCos(Scalar_ r, Vec4& sc);

    /// Compute sin \a s and cos \a c from value \a r
    static HK_INLINE void HK_CALL sinCos(Scalar_ r, Scalar& s, Scalar& c);

    /// Compute sin and cos in alternating components of sc.xyzw = sin(r.x) cos(r.y) sin(r.z) cos(r.w)
    /// using an approximation
    static HK_INLINE void HK_CALL sinCosApproximation(Vec4_ r, Vec4& sc);

    /// Compute sin and cos from value \a r to sc.xyzw = sin(r) cos(r) sin(r) cos(r)
    /// using an approximation
    static HK_INLINE void HK_CALL sinCosApproximation(Scalar_ r, Vec4& sc);

    /// Compute sin \a s and cos \a c from value \a r using an approximation
    static HK_INLINE void HK_CALL sinCosApproximation(Scalar_ r, Scalar& s, Scalar& c);

    /// Compute asin in all components of sc.xyzw = asin(r.x) asin(r.y) asin(r.z) asin(r.w)
    static HK_INLINE void HK_CALL aSin(Vec4_ r, Vec4& sc);

    /// Compute acos in all components of sc.xyzw = acos(r.x) acos(r.y) acos(r.z) acos(r.w)
    static HK_INLINE void HK_CALL aCos(Vec4_ r, Vec4& sc);

    /// Return asin from value \a r
    static HK_INLINE Scalar HK_CALL aSin(Scalar_ r);

    /// Return acos from value \a r
    static HK_INLINE Scalar HK_CALL aCos(Scalar_ r);

    /// Compute asin and acos in alternating components of sc.xyzw = asin(r.x) acos(r.y) asin(r.z) acos(r.w)
    static HK_INLINE void HK_CALL aSinAcos(Vec4_ r, Vec4& sc);

    /// Compute asin and acos from value \a r to sc.xyzw = asin(r) acos(r) asin(r) acos(r)
    static HK_INLINE void HK_CALL aSinAcos(Scalar_ r, Vec4& sc);

    /// Compute asin \a s and acos \a c from value \a r
    static HK_INLINE void HK_CALL aSinAcos(Scalar_ r, Scalar& s, Scalar& c);

    /// Compute from a value v another value cv in the range [-Radius,+Radius] such that cv = v-k*2*Radius with k integer.
    static HK_INLINE const Scalar HK_CALL toRange(Scalar_ v, Scalar_ radius, Scalar& k);
    /// Compute from a value v another value cv in the range [-Radius,+Radius] such that cv = v-k*2*Radius with k integer.
    /// Passing the diameter allows pre-computing it if wanted.
    static HK_INLINE const Scalar HK_CALL toRange(Scalar_ v, Scalar_ radius, Scalar_ diameter, Scalar& k);
    /// Compute from a value v another value cv in the range [-Radius,+Radius] such that cv = v-k*2*Radius with k integer.
    /// Passing the diameter and its reciprocal allows pre-computing them if wanted.
    static HK_INLINE const Scalar HK_CALL toRange(Scalar_ v, Scalar_ radius, Scalar_ diameter, Scalar_ diameterReciprocal, Scalar& k);

    /// Sets this vector components: this(0) = a0.dot<3>(b0), this(1) = a1.dot<3>(b1) ...
    static HK_INLINE void HK_CALL dot3_3vs3( Vec4_ a0, Vec4_ b0, Vec4_ a1, Vec4_ b1, Vec4_ a2, Vec4_ b2, Vec4& dotsOut);

    /// Sets this vector components: this(0) = a0.dot<3>(b0) ... this(3) = a3.dot<3>(b3)
    static HK_INLINE void HK_CALL dot3_4vs4( Vec4_ a0, Vec4_ b0, Vec4_ a1, Vec4_ b1, Vec4_ a2, Vec4_ b2, Vec4_ a3, Vec4_ b3, Vec4& dotsOut);

    /// Sets this vector components: this(0) = a0.dot<4>(b0), this(1) = a1.dot<4>(b1) ...
    static HK_INLINE void HK_CALL dot4_3vs3(Vec4_ a0, Vec4_ b0, Vec4_ a1, Vec4_ b1, Vec4_ a2, Vec4_ b2, Vec4& dotsOut);

    /// Sets this vector components: this(0) = a0.dot<4>(b0) ... this(3) = a3.dot<4>(b3)
    static HK_INLINE void HK_CALL dot4_4vs4( Vec4_ a0, Vec4_ b0, Vec4_ a1, Vec4_ b1, Vec4_ a2, Vec4_ b2, Vec4_ a3, Vec4_ b3, Vec4& dotsOut);

    /// Sets this vector components: this(i) = vector.dot<3>(AI) for i=0..2
    static HK_INLINE void HK_CALL dot3_1vs3( Vec4_ vectorIn, Vec4_ a0, Vec4_ a1, Vec4_ a2, Vec4& dotsOut);

    /// Sets this vector components: this(i) = vector.dot<3>(AI) for i=0..3
    static HK_INLINE void HK_CALL dot3_1vs4( Vec4_ vectorIn, Vec4_ a0, Vec4_ a1, Vec4_ a2, Vec4_ a3, Vec4& dotsOut);

    /// Sets this vector components: this(i) = vector.dot<4>(AI) for i=0..3
    static HK_INLINE void HK_CALL dot4_1vs4( Vec4_ vectorIn, Vec4_ a0, Vec4_ a1, Vec4_ a2, Vec4_ a3, Vec4& dotsOut);

    /// Sets this vector components: this(i) = vector.dot<4>(AI) for i=0..2
    static HK_INLINE void HK_CALL dot4_1vs3( Vec4_ vectorIn, Vec4_ a0, Vec4_ a1, Vec4_ a2, Vec4& dotsOut);

    /// Sets this vector components: this(i) = vector.dot4xyz1(AI) for i=0..3
    static HK_INLINE void HK_CALL dot4xyz1_1vs4( Vec4_ vectorIn, Vec4_ a0, Vec4_ a1, Vec4_ a2, Vec4_ a3, Vec4& dotsOut);

    /// Sets this vector components: this(i+j) = AI.dot<3>(bj)
    static HK_INLINE void HK_CALL dot3_2vs2( Vec4_ a0, Vec4_ a2, Vec4_ b0, Vec4_ b1, Vec4& dotsOut);

    /// Sets this vector components: this(i+j) = AI.dot4xyz1(bj)
    static HK_INLINE void HK_CALL dot4xyz1_2vs2( Vec4_ a0, Vec4_ a2, Vec4_ b0, Vec4_ b1, Vec4& dotsOut);

    /// returns (a0*a1+b0*b1+c0*c1).horizontalAdd<X>()
    template<int X>
    static HK_INLINE Scalar HK_CALL dot3X( Vec4_ a0, Vec4_ a1, Vec4_ b0, Vec4_ b1, Vec4_ c0, Vec4_ c1);

    /// returns (a0*a1+b0*b1).horizontalAdd<X>()
    template<int X>
    static HK_INLINE Scalar HK_CALL dot2X( Vec4_ a0, Vec4_ a1, Vec4_ b0, Vec4_ b1);

    /// Computes an = (a x n), bn = (b x n), cn = (c x n)
    static HK_INLINE void HK_CALL cross_3vs1(   Vec4_ a, Vec4_ b, Vec4_ c, Vec4_ n,
                                                    Vec4& an, Vec4& bn, Vec4& cn);

    /// Computes an = (a x n), bn = (b x n), cn = (c x n), dn = (d x n)
    static HK_INLINE void HK_CALL cross_4vs1(   Vec4_ a, Vec4_ b, Vec4_ c, Vec4_ d, Vec4_ n,
                                                    Vec4& an, Vec4& bn, Vec4& cn, Vec4& dn);

    /// Computes the cross products: (vA, vB), (vB, vC), (vC, vA)
    static HK_INLINE void computeCyclicCrossProducts(   Vec4_ vA, Vec4_ vB, Vec4_ vC,
                                                            Vec4& vAB, Vec4& vBC, Vec4& vCA);

    /// Compute the normalized planes of the 4 edges of a quad.
    /// Note: if you only care about a triangle, just set vD = vA;
    static HK_INLINE void computeQuadEdgePlanes( Vec4_ vA, Vec4_ vB, Vec4_ vC, Vec4_ vD, Vec4* normalOut, Transform* edgesOut );


    /// build an orthonormal matrix, where the first column matches the parameter dir.
    /// Note: Dir must be normalized
    static HK_INLINE void HK_CALL buildOrthonormal( Vec4_ dir, Matrix3& out );

    /// build an orthonormal matrix, where the first column matches the parameter dir and the second
    /// column matches dir2 as close as possible.
    /// Note: Dir must be normalized
    static HK_INLINE void HK_CALL buildOrthonormal( Vec4_ dir, Vec4_ dir2, Matrix3& out );

    /// Returns the squared distance from p to the line segment ab
    static HK_INLINE const Scalar HK_CALL distToLineSquared( Vec4_ a, Vec4_ b, Vec4_ p );

    /// Set the of 'out' to the maximum of 'a','b','c' and 'd'
    static HK_INLINE void HK_CALL setMax44(Vec4_ a, Vec4_ b, Vec4_ c, Vec4_ d, Vec4& out);

    /// Set the of 'out' to the minimum of 'a','b','c' and 'd'
    static HK_INLINE void HK_CALL setMin44(Vec4_ a, Vec4_ b, Vec4_ c, Vec4_ d, Vec4& out);

    /// returns != 0  if any x y or z of a or b is not 0
    static HK_INLINE hkBool32 HK_CALL anyNonZero3(Vec4_ a, Vec4_ b);

    /// returns != 0  if any x y or z of a b c or d is not 0
    static HK_INLINE hkBool32 HK_CALL anyNonZero3(Vec4_ a, Vec4_ b, Vec4_ c, Vec4_ d);

    /// Optimized implementation of the following code:
    ///  - int index = check.getIndexOfMaxComponent<4>();
    ///  - iVinOut.setAll( iVinOut.getU32(index) );
    ///  - v0InOut.setAll( v0InOut(index) );
    ///  - v1InOut.setAll( v1InOut(index) );
    static HK_INLINE void HK_CALL setXAtVectorMax( Vec4_ check, hkIntVector& iVinOut, Vec4& v0InOut, Vec4& v1InOut);

    /// set the length of a vector
    static HK_INLINE void HK_CALL setLength( Scalar_ desiredLength, Vec4& vectorInOut );

    /// set the length of a vector so that its projected length matches desiredLength.
    static HK_INLINE void HK_CALL setProjectedLength( Scalar_ desiredLength, Vec4_ scaleMeasurePlane, Scalar_ maxScale, Vec4& vectorInOut );

    /// Helper function for use with hkAlgorithm::findMinimumIndex, etc.
    /// Sample usage (finding the index of the element of "vectors" that is closest to "point"):
    /// \code int minIndex = hkAlgorithm::findMinimumIndex( vectors.begin(), vectors.getSize(), hkVector4Util::DistanceToPoint( point ) ); \endcode
    struct DistanceToPoint
    {
        typedef typename hkVector4UtilImpl<FT>::DistanceToPoint DistType;
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MATH, DistType);

        HK_INLINE DistanceToPoint( Vec4_ p ) : m_p(p) { }
        HK_INLINE typename hkFloatTypes<FT>::Float operator() (Vec4_ p) const { return m_p.distanceToSquared(p).getReal(); }

        Vec4 m_p;
    };

    //
    //  conversions
    //

    /// convert an Vec4 to a string representing the x,y,z components in that order.
    static const char* HK_CALL toString3(Vec4_ x, hkStringBuf& s, char separator=',');

    /// convert an Vec4 to a string representing the x,y,z,w components in that order.
    static const char* HK_CALL toString4(Vec4_ x, hkStringBuf& s, char separator=',');

    //
    //  compression
    //

    // packs a normalized quaternion into a single 4*8 bit integer. The error is roughly 0.01f per component
    static hkUint32 HK_CALL packQuaternionIntoInt32( Vec4_ qin);

    // unpack an 32 bit integer into quaternion. Note: The resulting quaternion is not normalized ( |q.length<4>()-1.0f| < 0.04f )
    static HK_INLINE void HK_CALL unPackInt32IntoQuaternion( hkUint32 ivalue, Vec4& qout );

    /// Fill each component of 'out' from components of 'a' and 'b'. i, j, k, and l are used as 0-based indices into
    /// the 8-element concatenated vector [ax, ay, az, aw, bx, by, bz, bw].
    template<unsigned int i, unsigned int j, unsigned int k, unsigned int l>
    static HK_INLINE void HK_CALL setPermutation2(hkVector4f_ a, hkVector4f_ b, hkVector4f & out);

    template<unsigned int i, unsigned int j, unsigned int k, unsigned int l>
    static HK_INLINE void HK_CALL setPermutation2(hkVector4dParameter a, hkVector4dParameter b, hkVector4d & out);

    /// Sort the components of the vector in-place, such that the elements are put in increasing order
    static HK_INLINE void HK_CALL sortVectorIncreasing(Vec4& v);

    /// Sort the components of the vector in-place, such that the elements are put in decreasing order
    static HK_INLINE void HK_CALL sortVectorDecreasing(Vec4& v);


    static HK_INLINE void HK_CALL streamStore1( Scalar_ v, FT* dest );
    static HK_INLINE void HK_CALL streamStore4( Vec4_ v,    FT* dest );
};

#define hkVector4fParameter hkVector4f_

#if HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED
#   if defined(HK_COMPILER_HAS_INTRINSICS_IA32)
#           include <Common/Base/Math/Vector/Sse/hkSseVector4Util.inl>
#           include <Common/Base/Math/Vector/Sse/hkSseVector4Util_D.inl>
#   elif defined(HK_COMPILER_HAS_INTRINSICS_NEON)
#           include <Common/Base/Math/Vector/Neon/hkNeonVector4Util.inl>
#   endif
#endif // HK_CONFIG_SIMD

#include <Common/Base/Math/Vector/hkVector4Util.inl>
#undef hkVector4fParameter

typedef hkVector4UtilImpl<hkFloat32>  hkVector4fUtil;
typedef hkVector4UtilImpl<hkDouble64> hkVector4dUtil;

HK_ON_REAL_IS_DOUBLE(typedef hkVector4dUtil hkVector4Util);
HK_ON_REAL_IS_FLOAT(typedef hkVector4fUtil hkVector4Util);


#endif // HK_MATH_VECTOR4_UTIL_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
