// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Math/Vector/hkIntVector.h>



/// Compressed 8 byte sized float vector3 with 1 sign bit, 15 bit mantissa and a shared 8 bit exponent.
///
/// This class is great to store local space positions
/// (e.g., a position which is 100meters off the origin has a precision of +-3mm.
/// pack() takes a little time (~10 - 20 cycles), unpack is pretty fast on SIMD platforms (4 assembly instructions).
/// See also hkPackedUnitVector and hkVectorPackUtil.
/// unpack() = punpcklwd   xmm2,xmm0;  cvtdq2ps    xmm1, xmm2; pshufd      xmm0, xmm2, 0FFh; mulps       xmm1, xmm0
struct HK_EXPORT_COMMON hkPackedVector3
{
    public:
        HK_DECLARE_CLASS(hkPackedVector3, New, Reflect);

        void pack( hkVector4fParameter v ); ///< pack a vector4 into this
        void pack( hkVector4dParameter v ); ///< pack a vector4 into this

        /// inline version of pack, you need to include #include <Common/Base/Math/Vector/hkPackedVector3.inl> to use this
        HK_INLINE void _pack( hkVector4fParameter v );  ///< pack a vector4 into this
        HK_INLINE void _pack( hkVector4dParameter v );  ///< pack a vector4 into this

        HK_INLINE void unpack( hkVector4f& vOut ) const;            ///< unpack
        HK_INLINE void unpack( hkVector4d& vOut ) const {hkVector4f v;  unpack(v); vOut.set(v); }   ///< unpack

        HK_INLINE void unpackAligned16( hkVector4f& vOut ) const; ///< unpack if this is aligned to 8 byte
        HK_INLINE void unpackAligned16( hkVector4d& vOut ) const; ///< unpack if this is aligned to 8 byte

    public:
        HK_ALIGN( hkInt16 m_values[4], 8 );
};

/// Compressed 4 byte sized float vector3 with 1 sign bit, 7 bit mantissa and a shared 8 bit exponent.
///
/// This class is great to store velocities in a network environment.
/// pack() takes a little time (~20 cycles), unpack is pretty fast on SIMD platforms (6 assembly instructions).
/// See also hkPackedUnitVector and hkVectorPackUtil.
HK_CLASSALIGN(struct,4) HK_EXPORT_COMMON hkPackedVector8_3
{
    HK_DECLARE_CLASS(hkPackedVector8_3, New, Reflect);
    HK_RECORD_ATTR(hk::IncludeInMgd(false));
    HK_DECLARE_POD_TYPE();

    // -------------------------------------------------------------------------------------------------------------
    // Functions
    // -------------------------------------------------------------------------------------------------------------

    void pack( hkVector4fParameter v ); ///< pack a vector4 into this
    void pack( hkVector4dParameter v ); ///< pack a vector4 into this

    // inline version of pack, you need to include #include <Common/Base/Math/Vector/hkPackedVector3.inl> to use this
    HK_INLINE void _pack( hkVector4fParameter v );  ///< pack a vector4 into this, inlined.
    HK_INLINE void _pack( hkVector4dParameter v );  ///< pack a vector4 into this, inlined.

    /// unpack
    HK_INLINE void unpack( hkVector4f& vOut ) const;        ///< unpack
    HK_INLINE void unpack( hkVector4d& vOut ) const;    ///< unpack
    HK_INLINE void setZero()    { m_values.m_u32 = 0; } ///< setZero

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------
    union Union
    {
        hkInt8   m_u8[4];
        hkUint32 m_u32;
    };
    Union m_values
        HK_ATTR(hk::Type(hkInt8[4]));
};


/// Store a hkVector4 unit values (normals) using 8 bit accuracy
struct hkPackedUnitVector8
{
    // -------------------------------------------------------------------------------------------------------------
    // Functions
    // -------------------------------------------------------------------------------------------------------------
    HK_INLINE void setZero();                       ///< set to zero
    HK_INLINE bool isZero() const;                  ///< check for zero
    HK_INLINE void unpack(_Inout_ hkVector4* v ) const; ///< unpack
    HK_INLINE void pack( hkVector4Parameter v );    ///< pack
    template<hkMathRoundingMode R>
    HK_INLINE void pack( hkVector4Parameter v );    ///< pack

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------

    /// The packed data.
    /// Note that the memory layout matches the order in memory not the order of bits in a register,
    /// so to get the x component you have 2 choices:
    ///    - ((hkUint8*)&m_packed)[0] or 
    ///    - [HK_ENDIA_BIG, e.g. Intel]:  (m_packed & 0xff)
    ///    - [HK_ENDIAN_SMALL, e.g. PPC]: (m_packed>>24) 
    hkUint32 m_packed;
};


/// Store a hkVector4.xyz unit values (normals) using 8 bit accuracy.
/// Note that m_packed needs to be aligned to a 4 byte boundary by the user
struct hkPackedUnitVector8_3
{
    // -------------------------------------------------------------------------------------------------------------
    // Functions
    // -------------------------------------------------------------------------------------------------------------
    HK_INLINE void setZero();                       ///< set to zero
    HK_INLINE bool isZero() const;                  ///< check for zero
    HK_INLINE void unpack(_Inout_ hkVector4* v ) const; ///< unpack
    HK_INLINE void pack( hkVector4Parameter v );    ///< pack

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------
    hkUint8 m_packed[3];    // must be followed by a another byte
};


/// Store a set of unit values (normals/quaternions (1-4 floats)) using 15 bit accuracy
template<int NUM_ELEMS>
struct hkPackedUnitVector
{
    // -------------------------------------------------------------------------------------------------------------
    // Functions
    // -------------------------------------------------------------------------------------------------------------

    HK_INLINE void setZero();

    HK_INLINE void pack( hkVector4f_ vIn );         ///< pack the vector
    HK_INLINE void pack( hkVector4d_ vIn );         ///< pack the vector

    HK_INLINE void unpack(_Inout_ hkVector4f* HK_RESTRICT vecOut ) const;       ///< unpack the vector
    HK_INLINE void unpack(_Inout_ hkVector4d* HK_RESTRICT vecOut ) const;       ///< unpack the vector

    HK_INLINE void unpackAligned16(_Inout_ hkVector4f* HK_RESTRICT vecOut ) const; ///< unpack the vector assuming this is aligned on a 16byte boundary
    HK_INLINE void unpackAligned16(_Inout_ hkVector4d* HK_RESTRICT vecOut ) const;  ///< unpack the vector assuming this is aligned on a 16byte boundary

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------
    hkUint16 m_vec[NUM_ELEMS];
};


extern HK_EXPORT_COMMON const hkUint32 hkPackedUnitVector_m_offset[4];
extern HK_EXPORT_COMMON HK_ALIGN16(const hkUint32 hkPackedVector4_6_v0Mask[4]);

#ifdef HK_ARCH_ARM
#   define HK_VECTOR4f_ALIGN_CHECK 0x7
#else
#   define HK_VECTOR4f_ALIGN_CHECK (HK_FLOAT_ALIGNMENT-1)
#endif

template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::setZero()
{
    m_vec[0] = 0x8000;
    if ( NUM_ELEMS >=2 ) m_vec[1%NUM_ELEMS] = 0x8000;
    if ( NUM_ELEMS >=3 ) m_vec[2%NUM_ELEMS] = 0x8000;
    if ( NUM_ELEMS >=4 ) m_vec[3%NUM_ELEMS] = 0x8000;
}

/// pack the vector
template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::pack( hkVector4f_ vIn )
{
    hkVector4f v; v.setMul( hkVector4f::getConstant<HK_QUADREAL_PACK16_UNIT_VEC>(), vIn );
    hkIntVector iv32; iv32.setConvertF32toS32Unchecked( v );
    hkIntVector packedOffset; packedOffset.load<4>(hkPackedUnitVector_m_offset);
    iv32.setAddU32(iv32, packedOffset);
#if (HK_ENDIAN_BIG)
    const int hoffset = 0;  // extract the high word
#else
    const int hoffset = 1;
#endif
    m_vec[0] = iv32.getU16<hoffset>();
    if ( NUM_ELEMS >=2 ) m_vec[1%NUM_ELEMS] = iv32.getU16<2+hoffset>();
    if ( NUM_ELEMS >=3 ) m_vec[2%NUM_ELEMS] = iv32.getU16<4+hoffset>();
    if ( NUM_ELEMS >=4 ) m_vec[3%NUM_ELEMS] = iv32.getU16<6+hoffset>();
}

/// pack the vector
template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::pack( hkVector4d_ vIn )
{
    hkVector4d v; v.setMul( hkVector4d::getConstant<HK_QUADREAL_PACK16_UNIT_VEC>(), vIn );
    hkIntVector iv32; iv32.setConvertF32toS32Unchecked( v );
    hkIntVector packedOffset; packedOffset.load<4>(hkPackedUnitVector_m_offset);
    iv32.setAddU32(iv32, packedOffset);
#if (HK_ENDIAN_BIG)
    const int hoffset = 0;  // extract the high word
#else
    const int hoffset = 1;
#endif
    m_vec[0] = iv32.getU16<hoffset>();
    if ( NUM_ELEMS >=2 ) m_vec[1%NUM_ELEMS] = iv32.getU16<2+hoffset>();
    if ( NUM_ELEMS >=3 ) m_vec[2%NUM_ELEMS] = iv32.getU16<4+hoffset>();
    if ( NUM_ELEMS >=4 ) m_vec[3%NUM_ELEMS] = iv32.getU16<6+hoffset>();
}


// unpack
template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::unpack(_Inout_ hkVector4f* HK_RESTRICT vecOut ) const
{
    HK_ASSERT_NO_MSG( 0xf04523ef, (hkUlong(vecOut) & HK_VECTOR4f_ALIGN_CHECK) == 0 );
    hkIntVector zero; zero.setZero();
    hkIntVector iv16; iv16.loadNotAligned<(NUM_ELEMS+1)/2>((const hkUint32*)m_vec);
    hkIntVector iv32; iv32.setCombineHead16To32( iv16, zero );
    hkIntVector packedOffset; packedOffset.load<4>(hkPackedUnitVector_m_offset);
    iv32.setAddU32( iv32, packedOffset);
    hkVector4f v; iv32.convertS32ToF32( v );
    vecOut->setMul( hkVector4f::getConstant<HK_QUADREAL_UNPACK16_UNIT_VEC>(), v );
}

// unpack
template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::unpack(_Inout_ hkVector4d* HK_RESTRICT vecOut ) const
{
    hkVector4f v; unpack(&v);
    vecOut->load<4>(&v(0));
}

/// Gets the vector assuming that this instance is aligned on a 16 byte boundary
template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::unpackAligned16(_Inout_ hkVector4f* HK_RESTRICT vecOut ) const
{
    HK_COMPILE_TIME_ASSERT( NUM_ELEMS <= 4 );
    HK_ASSERT_NO_MSG( 0xf04523ed, (hkUlong(this) & 0xf) == 0 );
    HK_ASSERT_NO_MSG( 0xf04523ef, (hkUlong(vecOut) & HK_VECTOR4f_ALIGN_CHECK) == 0 );
    hkIntVector iv16; iv16.loadNotAligned<(NUM_ELEMS+1)/2>((const hkUint32*)m_vec);
    hkIntVector zero; zero.setZero();
    hkIntVector iv32; iv32.setCombineHead16To32( iv16, zero );
    hkIntVector packedOffset; packedOffset.load<4>(hkPackedUnitVector_m_offset);
    iv32.setAddU32( iv32, packedOffset);
    hkVector4f v; iv32.convertS32ToF32( v );
    vecOut->setMul( hkVector4f::getConstant<HK_QUADREAL_UNPACK16_UNIT_VEC>(), v );
}

/// Gets the vector assuming that this instance is aligned on a 16 byte boundary
template<int NUM_ELEMS>
void hkPackedUnitVector<NUM_ELEMS>::unpackAligned16(_Inout_ hkVector4d* HK_RESTRICT vecOut ) const
{
    hkVector4f v; unpack(&v);
    vecOut->load<4>(&v(0));
}
#undef HK_VECTOR4f_ALIGN_CHECK



/// Packed 4-byte-sized float pair of vector3s (vector3,vector3) with 1 sign bit, 3 bit mantissa
/// per-component and a shared 5/3 bit exponent (5 bits exp for v0, 3 bits for v1).
///
/// For pack5/unpack5:
///   The exponent for v0 ranges between -15 and 16, which gives
///   a range of 1.346e-5f to 114688f for v0 at full 3 bit precision.
///   Since it can store denormals, the smallest non zero number for v0 is: 1.923e-6.
///   The exponent for v1 is relative to v0 and has a relative range of
///   -5 and 2.
///   Should v1 be much larger than v0 so that we cannot store exponent1,
///   the exponent0 will be increased at the cost of precision loss for v0.
///   Example: should v1 be 8 times larger than v0, v0 will only get a 2 bit mantissa.
HK_CLASSALIGN(struct,4) HK_EXPORT_COMMON hkPackedVector4_6
{
    HK_DECLARE_CLASS(hkPackedVector4_6, New, Reflect);
    HK_DECLARE_POD_TYPE();

    enum { EXP1_MAX_INCREMENT = 2 };

    // -------------------------------------------------------------------------------------------------------------
    // Functions
    // -------------------------------------------------------------------------------------------------------------

    void pack5( hkVector4f_ v0, hkVector4f_ v1 ); ///< Packs 2 vector4s (w component ignored) into this instance, assuming a 5 bit exponent for v0 and a 3 bit exponent for v1
    void pack5( hkVector4d_ v0, hkVector4d_ v1 );

    // inline version of pack, too lengthy to be available, use pack5 instead
    protected:
    template <int numBitsExp0> HK_INLINE void _pack( hkVector4f_ v0, hkVector4f_ v1 );  ///< pack a vector4 into this, inlined.

    /// unpack
    public:
    template <int numBitsExp0> HK_INLINE void unpack( hkVector4f& v0Out, hkVector4f& v1Out ) const; ///< unpack

    HK_INLINE void unpack5(hkVector4f& v0Out, hkVector4f& v1Out) const { unpack<5>(v0Out, v1Out); }
    void unpack5(hkVector4d& v0Out, hkVector4d& v1Out) const;

    HK_INLINE void setZero() { m_values.m_u32 = 0; } ///< setZero

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------
    union Union
    {
        hkUint8  m_u8[4];
        hkUint32 m_u32;
    };
    Union m_values
        HK_ATTR(hk::Type(hkUint8[4]));
};


/// unpack
template <int numBitsExp0>
void hkPackedVector4_6::unpack( hkVector4f& vOut0, hkVector4f& vOut1 ) const
{
    // bring m_values into the high bits
    hkIntVector iv;
    {
        iv.load<1>( &m_values.m_u32 ); // loads 1*uint32 = 4*uint8
        hkIntVector zero; zero.setZero();
        iv.setCombineHead8To16 ( iv, zero );
        iv.setCombineHead16To32( iv, zero );
    }

    // split into v0 and v1
    hkIntVector iv0; iv0.setAnd(iv, *(hkIntVector*)hkPackedVector4_6_v0Mask ); // lower up
    hkIntVector iv1; iv1.setShiftLeft32<4>(iv); // lower up

    // convert to float
    hkVector4f v0;  iv0.convertS32ToF32( v0 );
    hkVector4f v1;  iv1.convertS32ToF32( v1 );

    int exp = m_values.m_u8[3]; // do not use int vector, slower on ARM

    const int maxExp0 = (1<<(numBitsExp0-1));
    const int minExp0 = maxExp0-((1<<numBitsExp0)-1);

    const int numBitsExp1 = 8- numBitsExp0;
    const int exp0Base = 0x7f-0x1e + minExp0;
    const int exp1Mask = 0xff >> numBitsExp0;

    const int EXP1_MAX_DECREMENT = (1<<numBitsExp1) - (1 + EXP1_MAX_INCREMENT);

    int exp0 = (exp>>numBitsExp1) + exp0Base;   // use the highest bits in the exponent
    int exp1 = (exp0 + (exp & exp1Mask)) - EXP1_MAX_DECREMENT;
    exp0 <<= 23;    // shift to the correct place
    exp1 <<= 23;

    // calculate the exp correction
    hkIntVector iexp0; iexp0.setAll(exp0);
    hkIntVector iexp1; iexp1.setAll(exp1);

    hkVector4f vexp0; iexp0.storeAsFloat32BitRepresentation( vexp0 );
    hkVector4f vexp1; iexp1.storeAsFloat32BitRepresentation( vexp1 );

    vOut0.setMul( v0, vexp0 );
    vOut1.setMul( v1, vexp1 );
}

/// unpack
HK_INLINE void hkPackedVector8_3::unpack( hkVector4f& vOut ) const
{
    // bring m_values into the high bits
    hkIntVector iv;
    {
        iv.load<1>( &m_values.m_u32 ); // loads 1*uint32 = 4*uint8
        hkIntVector zero; zero.setZero();
        iv.setCombineHead8To16( iv, zero );
        iv.setCombineHead16To32( iv, zero );
    }

    // convert to float
    hkVector4f v;   iv.convertS32ToF32( v );

    // calculate the exp correction
    hkIntVector iexp; iexp.setBroadcast<3>(iv);
    iexp.setShiftRight32<1>(iexp); // shift the exponent to the right place
    hkVector4f vexp; iexp.storeAsFloat32BitRepresentation( vexp );
    vOut.setMul( v, vexp );
}

HK_INLINE void hkPackedVector8_3::unpack( hkVector4d& vOut ) const
{
    hkVector4f v;   unpack(v);
    vOut.load<4>(&v(0) );
}
/// unpack assuming this is aligned to a 8 byte boundary
HK_INLINE void hkPackedVector3::unpackAligned16( hkVector4f& vOut ) const
{
    // bring m_values into the high bits
    hkIntVector iv;
    {
        iv.load<2>( (const hkUint32*)m_values ); // loads 2*uint32 = 4*uint16
        hkIntVector zero; zero.setZero();
        iv.setCombineHead16To32( iv, zero );
    }

    // convert to integer
    hkVector4f v;   iv.convertS32ToF32( v );

    // calculate the exp correction
    hkIntVector iexp; iexp.setBroadcast<3>(iv);
    hkVector4f vexp; iexp.storeAsFloat32BitRepresentation( vexp );
    v.mul( vexp );

    vOut = v;
}

/// unpack assuming this is aligned to a 8 byte boundary
HK_INLINE void hkPackedVector3::unpackAligned16( hkVector4d& vOut ) const
{
    hkVector4f v;   unpackAligned16(v);
    vOut.load<4>(&v(0) );
}

HK_INLINE void hkPackedVector3::unpack( hkVector4f& vOut ) const
{
    // bring m_values into the high bits
    hkIntVector iv;
    {
        iv.loadNotAligned<2>( (const hkUint32*)m_values ); // loads 2*uint32 = 4*uint16
        hkIntVector zero; zero.setZero();
        iv.setCombineHead16To32( iv, zero );
    }

    // convert to integer
    hkVector4f v;   iv.convertS32ToF32( v );

    // calculate the exp correction
    hkIntVector iexp; iexp.setBroadcast<3>(iv);
    hkVector4f  vexp; iexp.storeAsFloat32BitRepresentation( vexp );
    vOut.setMul( v, vexp);
}

// hkPackedVector3.inl is not inlined by default (!?) so have the inlined versions here.
extern HK_EXPORT_COMMON HK_ALIGN16(const float) hkPackedUnitVector8_offset[4];
extern HK_EXPORT_COMMON HK_ALIGN16(const float) hkPackedUnitVector8_packFactor[4];
extern HK_EXPORT_COMMON HK_ALIGN16(const float) hkPackedUnitVector8_unpackFactor[4];


HK_INLINE void hkPackedUnitVector8::unpack(_Inout_ hkVector4* v) const
{
    hkIntVector iv;           iv.load<1>( &m_packed );
    hkIntVector zeroOffset;   zeroOffset.setAll( 0x7f );

    iv.setSplit8To32( iv );
    iv.setSubS32( iv, zeroOffset );

    hkVector4f w;   iv.convertS32ToF32( w );
    w.setMul( w, *(const hkSimdFloat32*)&hkPackedUnitVector8_unpackFactor[0] );
    v->set(w);  // might do float/double conversion
}

HK_INLINE void hkPackedUnitVector8::pack( hkVector4Parameter vin )
{
    pack<HK_ROUND_DEFAULT>( vin );
}

template<hkMathRoundingMode R>
HK_INLINE void hkPackedUnitVector8::pack( hkVector4Parameter vin )
{
    hkVector4f v; v.set(vin);
    hkVector4f w;
    if ( R == HK_ROUND_NEAREST )
    {
        w.setAddMul( (const hkVector4f&)hkPackedUnitVector8_offset, v, (const hkVector4f&)hkPackedUnitVector8_packFactor );
    }
    else
    {
        w.setMul( v, (const hkVector4f&)hkPackedUnitVector8_packFactor );
    }

    hkIntVector iv; iv.setConvertF32toS32Unchecked( w );

    if ( R != HK_ROUND_NEAREST )
    {
        hkIntVector zeroOffset;   zeroOffset.setAll( 0x7f );
        iv.setAddS32( iv, zeroOffset );
    }

    iv.setConvertU32ToU16( iv, iv );
    iv.setConvertSaturateS16ToU8( iv,iv );

    iv.store<1>( &m_packed );
}

HK_INLINE void hkPackedUnitVector8::setZero()
{
    m_packed = 0x7f7f7f7f;
}

HK_INLINE bool hkPackedUnitVector8::isZero() const
{
    return m_packed == 0x7f7f7f7f;
}


HK_INLINE void hkPackedUnitVector8_3::unpack(_Inout_ hkVector4* v ) const
{
    HK_DETAIL_DIAG_MSVC_SUPPRESS(26000)
    hkIntVector iv;           iv.load<1>( (hkUint32*)&m_packed[0] );
    hkIntVector zeroOffset;   zeroOffset.setAll( 0x7f );

    iv.setSplit8To32( iv );
    iv.setSubS32( iv, zeroOffset );

    hkVector4f w;   iv.convertS32ToF32( w );
    w.setMul( w, *(const hkSimdFloat32*)&hkPackedUnitVector8_unpackFactor[0] );
    v->set(w);  // might do float/double conversion
}

HK_INLINE void hkPackedUnitVector8_3::pack( hkVector4Parameter vin )
{
    HK_ASSERT_NO_MSG( 0xf0434565, 0 == (0x3& hkUlong(&m_packed[0]) ) );
    hkVector4f v; v.set(vin);
    hkVector4f w; w.setMul(v, (const hkVector4f&)hkPackedUnitVector8_packFactor );

    hkIntVector iv; iv.setConvertF32toS32Unchecked( w );

    hkIntVector zeroOffset;   zeroOffset.setAll( 0x7f );
    iv.setAddS32( iv, zeroOffset );

    iv.setConvertU32ToU16( iv, iv );
    iv.setConvertSaturateS16ToU8( iv,iv );

    // Set as a 4 byte int, maintaining whatever was in the last byte
    HK_DETAIL_DIAG_MSVC_SUPPRESS(26000)
    const hkUint8 oldW = *(&m_packed[0] + 3);       // avoid compiler warnings about reading out of bounds
    HK_DETAIL_DIAG_MSVC_SUPPRESS(26000)
    iv.store<1>( (hkUint32*)&m_packed[0] );
    HK_DETAIL_DIAG_MSVC_SUPPRESS(26000)
    *(&m_packed[0] + 3) = oldW;
}

HK_INLINE void hkPackedUnitVector8_3::setZero()
{
    m_packed[0] = 0x7f;
    m_packed[1] = 0x7f;
    m_packed[2] = 0x7f;
}

HK_INLINE bool hkPackedUnitVector8_3::isZero() const
{
    HK_DETAIL_DIAG_MSVC_SUPPRESS(26000)
    hkUint32 packed = *(hkUint32*)m_packed;
#if defined( HK_ENDIAN_BIG )
    // intel
    return (packed& 0xffffff) == 0x7f7f7f;
#else
    return (packed& 0xffffff00) == 0x7f7f7f00;
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
