// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Math/Vector/hkFourTransposedPoints.h>

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

class hkcdVertex;

/// Represents 4 3D vertices bundled together. The vertices are transposed, i.e. (xxxx, yyyy, zzzz)
template <typename FT>
class hkFourTransposedPointsImpl
{
    public:
    typedef hkFourTransposedPointsImpl<FT> ThisType;

    typedef typename hkFloatTypes<FT>::Vec4 Vec4;
    typedef typename hkFloatTypes<FT>::Vec4_ Vec4_;

    typedef typename hkFloatTypes<FT>::Scalar Scalar;
    typedef typename hkFloatTypes<FT>::Scalar_ Scalar_;

    typedef typename hkFloatTypes<FT>::Transform Transform;
    typedef typename hkFloatTypes<FT>::Matrix3 Matrix3;
    typedef typename hkFloatTypes<FT>::QTransform QTransform;
    typedef typename hkFloatTypes<FT>::QsTransform QsTransform;
    typedef typename hkFloatTypes<FT>::Quaternion Quaternion;
    typedef typename hkFloatTypes<FT>::Rotation Rotation;
    typedef typename hkFloatTypes<FT>::Comparison Comparison;

    HK_DECLARE_CLASS( hkFourTransposedPointsImpl, Pod, New );
    HK_DETAIL_DECLARE_REFLECT_EXPORT( ThisType, HK_EXPORT_COMMON );

    /// Sets the vertex bundle to the given vertices
    HK_INLINE void set( Vec4_ vA, Vec4_ vB, Vec4_ vC, Vec4_ vD );

    /// Extracts the vectors from the vertex bundle
    HK_INLINE void extract( Vec4& vA, Vec4& vB, Vec4& vC, Vec4& vD ) const;

    /// Extracts the vectors from the vertex bundle, with w components inserted from wABCD.
    HK_INLINE void extractWithW( Vec4_ wABCD, Vec4& vA, Vec4& vB, Vec4& vC, Vec4& vD ) const;

    /// Extracts the vector at index from the bundle
    HK_INLINE void extract( int index, Vec4& vI ) const;

    /// Sets the vertex bundle to the same vertex
    HK_INLINE void setAll( Vec4_ v );

    /// Sets this = v - a
    HK_INLINE void setSub( const hkFourTransposedPointsImpl<FT>& v, Vec4_ a );

    /// Sets this = a - v
    HK_INLINE void setSub( Vec4_ a, const hkFourTransposedPointsImpl<FT>& v );

    /// Sets this = v - a
    HK_INLINE void setSub( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& a );

    /// Sets this = v + a
    HK_INLINE void setAdd( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& a );

    HK_INLINE void setAdd( const hkFourTransposedPointsImpl<FT>& v, Vec4_ a );

    /// Sets this = a / b
    template <hkMathAccuracyMode A, hkMathDivByZeroMode D>
    HK_INLINE void setDiv( const hkFourTransposedPointsImpl<FT>& a, const hkFourTransposedPointsImpl<FT>& b );

    /// Sets x = vx * a.x, y = vy * a.y, ...
    HK_INLINE void setMulC( const hkFourTransposedPointsImpl<FT>& v, Vec4_ a );

    /// Sets x(i) = v.x(i) * a(i), y(i) = v.y(i) * a(i), ...         i = 1..4
    HK_INLINE void setMulT( const hkFourTransposedPointsImpl<FT>& v, Vec4_ a );

    /// Sets this = v * a
    HK_INLINE void setMul( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& a );

    /// Sets this = v * a
    HK_INLINE void setMul( const hkFourTransposedPointsImpl<FT>& a, Scalar_ b );

    /// Sets this = min(v,a)
    HK_INLINE void setMin( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& a );

    /// Sets this = max(v,a)
    HK_INLINE void setMax( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& a );

    /// Sets minOut = min( v0, v1, v2 )
    HK_INLINE void horizontalMin( Vec4& minOut ) const;

    /// Sets maxOut = max( v0, v1, v2 )
    HK_INLINE void horizontalMax( Vec4& maxOut ) const;

    /// Sets v0 = b*c0, v1 = b*c1, etc.
    HK_INLINE void setOuterProduct( Vec4_ b, Vec4_ c );

    /// Sets v0 += b*c0, v1 = b*c1, etc.
    HK_INLINE void addOuterProduct( Vec4_ b, Vec4_ c );

    /// Sets v0 += b*c0, v1 = b*c1, etc.
    HK_INLINE void subOuterProduct( Vec4_ b, Vec4_ c );

    /// Returns the dot3 of this and a
    HK_INLINE void dot3( Vec4_ a, Vec4& dotOut ) const;

    /// Returns the dot3 of this and a
    HK_INLINE void dot3( const hkFourTransposedPointsImpl<FT>& a, Vec4& dotOut ) const;

    /// Translates the points along direction dir by distances(i), i.e. x(i) = v.x(i) + dir.x * distances(i), etc..
    HK_INLINE void setAddMul( const hkFourTransposedPointsImpl<FT>& v, Vec4_ dir, Vec4_ distances );

    /// Sets this = projection of each point on the given Aabb
    HK_INLINE void setAabbProjection( const hkFourTransposedPointsImpl<FT>& v, const hkAabb& aabb );

    /// Computes the distance from all vertices to the given Aaabb
    HK_EXPORT_COMMON void computeAabbDistanceSq( const hkAabb& aabb, Vec4& distancesOut ) const;

    /// Translates the points along direction -dir by distances(i), i.e. x(i) = v.x(i) - dir.x * distances(i), etc..
    HK_INLINE void setSubMul( const hkFourTransposedPointsImpl<FT>& v, Vec4_ dir, Vec4_ distances );

    /// Returns the dot4 of this and a, with the w-coordinates of this assumed to be 1
    HK_INLINE void dot4xyz1( Vec4_ a, Vec4& dotOut ) const;

    /// Sets v0 += a, v1 += a, etc.
    HK_INLINE void add( Vec4_ a );

    /// Sets v0 -= a, v1 -= a, etc.
    HK_INLINE void sub( Vec4_ a );

    /// Multiply all components by a
    HK_INLINE void mul( Scalar_ a );

    /// Sets this = this - v
    HK_INLINE void sub( const hkFourTransposedPointsImpl<FT>& v );

    /// Sets this = this + v
    HK_INLINE void add( const hkFourTransposedPointsImpl<FT>& v );

    /// Sets this = this * v
    HK_INLINE void mul( const hkFourTransposedPointsImpl<FT>& v );

    /// Sets x(i) = v.x(i) + u.x(i) * a(i), y(i) = v.y(i) + u.y(i) * a(i), ...         i = 1..4
    HK_INLINE void setAddMulT( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& u, Vec4_ a );

    /// Sets x(i) = v.x(i) - u.x(i) * a(i), y(i) = v.y(i) - u.y(i) * a(i), ...         i = 1..4
    HK_INLINE void setSubMulT( const hkFourTransposedPointsImpl<FT>& v, const hkFourTransposedPointsImpl<FT>& u, Vec4_ a );

    /// Sets x(i) += u.x(i) * a(i), y(i) += u.y(i) * a(i), ...         i = 1..4
    HK_INLINE void addMulT( const hkFourTransposedPointsImpl<FT>& u, Vec4_ a );

    /// Sets x(i) -= u.x(i) * a(i), y(i) -= u.y(i) * a(i), ...         i = 1..4
    HK_INLINE void subMulT( const hkFourTransposedPointsImpl<FT>& u, Vec4_ a );

    /// Sets x(i) *= a(i),   y(i) *= a(i), ...         i = 1..4
    HK_INLINE void mulT( Vec4_ a );

    /// Sets this = cross(n, v)
    HK_INLINE void setCross( Vec4_ n, const hkFourTransposedPointsImpl<FT>& v );
    HK_INLINE void setCross( const hkFourTransposedPointsImpl<FT>& v, Vec4_ n );

    /// Flips the signs, i.e. v0 = sign(ax) * v0, ..., v3 = sign(aw) * v3
    HK_INLINE void flipSigns( Vec4_ a );

    /// Selects a set of vectors i.e. vi = mask(i) ? ai : bi, for i = 0..3
    HK_INLINE void setSelect( const Comparison& mask, const hkFourTransposedPointsImpl<FT>& trueVecs, const hkFourTransposedPointsImpl<FT>& falseVecs );

    /// Sets this = Transpose(m) * v
    HK_EXPORT_COMMON void setTransformedInverseDir( const Matrix3& m, const hkFourTransposedPointsImpl<FT>& v );

    /// Inlined. Sets this = Transpose(m) * v
    HK_INLINE void _setTransformedInverseDir( const Matrix3& m, const hkFourTransposedPointsImpl<FT>& v );

    /// Sets this = Transpose(Rotation(m)) * (v - Translation(m))
    HK_EXPORT_COMMON void setTransformedInversePos( const Transform& m, const hkFourTransposedPointsImpl<FT>& v );

    /// Inlined. Sets this = Transpose(Rotation(m)) * (v - Translation(m))
    HK_INLINE void _setTransformedInversePos( const Transform& m, const hkFourTransposedPointsImpl<FT>& v );

    /// Sets this = m * v
    HK_EXPORT_COMMON void setRotatedDir( const Matrix3& m, const hkFourTransposedPointsImpl<FT>& v );

    /// Inlined. Sets this = m * v
    HK_INLINE void _setRotatedDir( const Matrix3& m, const hkFourTransposedPointsImpl<FT>& v );

    /// Normalizes the points
    HK_INLINE void normalize();

    /// Copies the points from a hkVector4f array into a hkFourTransposedPointsImpl array
    HK_EXPORT_COMMON static void HK_CALL copyVertexData( _In_reads_( numVerts ) const Vec4* HK_RESTRICT sourceVerts, const int numVerts,
        hkArray<hkFourTransposedPointsImpl<FT> >& pointsOut );

    /// The output array must have space for a multiple of 4 vertices = (numVertices|3) & ~3.
    HK_EXPORT_COMMON static void HK_CALL getOriginalVertices(
        _In_reads_( HK_NEXT_MULTIPLE_OF( 4, numVertices ) / 4 ) const hkFourTransposedPointsImpl* verts4in, int numVertices,
        _Inout_updates_all_( HK_NEXT_MULTIPLE_OF( 4, numVertices ) ) hkcdVertex* verticesOut );

    /// Permutes the original vectors, NOT their components!
    template <hkVectorPermutation::Permutation perm>
    HK_INLINE void setPermutation( const hkFourTransposedPointsImpl<FT>& pts );

    /// Computes the average of the four points
    HK_INLINE void computeAverage( Vec4& avgOut ) const;

    public:

    /// The vertices
    Vec4 m_vertices[3];
};

#if 0 && defined( HK_DYNAMIC_DLL )
/*HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION*/ template class /*HK_EXPORT_COMMON*/ hkFourTransposedPointsImpl<float>;
/*HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION*/ template class /*HK_EXPORT_COMMON*/ hkFourTransposedPointsImpl<double>;
#endif

typedef hkFourTransposedPointsImpl<hkFloat32>  hkFourTransposedPointsf;
typedef hkFourTransposedPointsImpl<hkDouble64> hkFourTransposedPointsd;

HK_ON_REAL_IS_DOUBLE( typedef hkFourTransposedPointsd hkFourTransposedPoints );
HK_ON_REAL_IS_FLOAT( typedef hkFourTransposedPointsf hkFourTransposedPoints );


#if defined(HK_COMPILER_ARMCC) || defined(HK_COMPILER_GHS)
template<> hkReflect::Detail::TypeData hkFourTransposedPointsImpl<float>::typeData;
template<> hkReflect::Detail::TypeData hkFourTransposedPointsImpl<double>::typeData;
#endif

#include <Common/Base/_Auto/TemplateTypes/hkFourTransposedPoints_Types.inl>

HK_REFLECT_TYPEDEF( HK_ATTR(hk::Version(1)) HK_EXPORT_COMMON, hkFourTransposedPointsf, hkFourTransposedPointsf_Tag );
HK_REFLECT_TYPEDEF( HK_ATTR(hk::Version(1)) HK_EXPORT_COMMON, hkFourTransposedPointsd, hkFourTransposedPointsd_Tag );

#include <Common/Base/Math/Vector/hkFourTransposedPoints.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
