// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Math/Util/hkVector2Util.h>
#include <Common/Base/Math/Vector/hkVector2f.h>
#include <Common/Base/Math/Vector/hkVector2d.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

// This namespace's functions are the template versions of the algorithms
// hkVector2Util will call these functions with proper types depending on the overload
namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    template<typename T>
    struct IndexedLess
    {
        IndexedLess(const hkArrayBase<T>& a) : m_a(a) {  }
        hkBool32 operator()(int i, int j)
        {
            const T& pi = m_a[i];
            const T& pj = m_a[j];
            return pi.x < pj.x || (pi.x == pj.x && pi.y < pj.y);
        }
        const hkArrayBase<T>& m_a;
    };

    template< typename T >
    struct Deque
    {
        // invariant: m_bot is the index of the first valid item
        // m_top index of one past the end
        // m_top-m_bot == number of elements
        Deque(int n) : m_vals(n), m_bot(1 + n / 2), m_top(m_bot)
        {
            HK_ON_DEBUG(hkString::memSet(m_vals.begin(), 0xff, n * sizeof(T)));
        }

        void pushBot(T t)
        {
            m_vals[--m_bot] = t;
        }
        void pushTop(T t)
        {
            m_vals[m_top++] = t;
        }
        void popBot()
        {
            HK_ON_DEBUG(m_vals[m_bot] = -1);
            ++m_bot;
            HK_ASSERT_NO_MSG(0x4d4c4a63, m_bot <= m_top);
        }
        void popTop()
        {
            HK_ON_DEBUG(m_vals[m_top - 1] = -1);
            --m_top;
            HK_ASSERT_NO_MSG(0x4d4c4a63, m_bot <= m_top);
        }
        T bot(int i)
        {
            return m_vals[m_bot + i];
        }
        T top(int i)
        {
            return m_vals[m_top - 1 - i];
        }
        void get(hkArray<T>& out)
        {
            int n = m_top - m_bot;
            hkArray<T> active(m_vals.begin() + m_bot, n, n); // noncopying ctor
            out = active;
        }

        hkLocalBuffer<T> m_vals;
        int m_bot;
        int m_top;
    };

    template<typename TVector2>
    HK_INLINE void HK_CALL convexHullSimplePolylineT(const hkArrayBase<TVector2>& line, hkArray<int>& indicesOut)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        const int NPOINT = line.getSize();
        Deque<int> deque(2 * (NPOINT - 1));

        if (line[2].leftOfLine(line[0], line[1]))
        {
            deque.pushBot(0);
            deque.pushTop(1);
        }
        else
        {
            deque.pushBot(1);
            deque.pushTop(0);
        }
        deque.pushBot(2);
        deque.pushTop(2);

        TVector2 bot0 = line[deque.bot(0)];
        TVector2 bot1 = line[deque.bot(1)];
        TVector2 top0 = line[deque.top(0)];
        TVector2 top1 = line[deque.top(1)];

        for (int i = 3; i < NPOINT; ++i)
        {
            if (line[i].leftOfLine(bot0, bot1) && line[i].leftOfLine(top1, top0))
            {
                continue;
            }
            while (!line[i].leftOfLine(bot0, bot1))
            {
                deque.popBot();
                bot0 = bot1;
                bot1 = line[deque.bot(1)];
            }
            deque.pushBot(i);
            bot1 = bot0;
            bot0 = line[deque.bot(0)];

            while (!line[i].leftOfLine(top1, top0))
            {
                deque.popTop();
                top0 = top1;
                top1 = line[deque.top(1)];
            }
            deque.pushTop(i);
            top1 = top0;
            top0 = line[deque.top(0)];
        }

        deque.get(indicesOut);
    }

    // http://geometryalgorithms.com/Archive/algorithm_0109/algorithm_0109.htm#Monotone%20Chain
    template<typename TVector2, typename TScalar>
    HK_INLINE hkResult HK_CALL convexHullIndicesT(const hkArrayBase<TVector2>& originalPoints, hkArrayBase<int>& indicesOut)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        
        // Can we save mem here: Overwrite 'orignalPoints' array
        int NPOINT = originalPoints.getSize();
        hkLocalBuffer<TVector2> points(NPOINT);
        hkLocalBuffer<int> originalIndices(NPOINT);
        if (points.begin() == HK_NULL || originalIndices.begin() == HK_NULL)
        {
            return HK_FAILURE;
        }

        {
            for (int i = 0; i < NPOINT; ++i)
            {
                originalIndices[i] = i;
            }
            hkSort(originalIndices.begin(), NPOINT, IndexedLess<TVector2>(originalPoints));
            int di = 0;
            points[0] = originalPoints[originalIndices[0]];
            for (int si = 1; si < NPOINT; ++si)
            {
                const TVector2& p = originalPoints[originalIndices[si]];
                if (points[di].x != p.x || points[di].y != p.y)
                {
                    di += 1; // remove duplicates
                    points[di] = p;
                    originalIndices[di] = originalIndices[si];
                }
            }
            NPOINT = di + 1;
            //indicesOut.reserve( NPOINT+1 );
            HK_ASSERT(0x52c91212, indicesOut.getCapacity() >= NPOINT + 1, "You must set the capacity of indicesOut before calling this method.");
        }

        
        // Can we get rid of this array?
        hkLocalArray<TVector2> pointsOut(NPOINT);
        if (pointsOut.begin() == HK_NULL)
        {
            return HK_FAILURE;
        }

        int minXminY = 0;
        int minXmaxY;
        int maxXminY;
        int maxXmaxY = NPOINT - 1;

        // get start and end walking indices
        {
            {
                TScalar minXpos = points[0].x;
                for (minXmaxY = 1; minXmaxY < NPOINT && points[minXmaxY].x == minXpos; ++minXmaxY)
                {
                }
            }
            if (minXmaxY == NPOINT) // degenerate - all x == xmin
            {
                indicesOut.pushBackUnchecked(0);
                if (points[NPOINT - 1].y != points[0].y) // segment or simple point?
                {
                    indicesOut.pushBackUnchecked(NPOINT - 1);
                    indicesOut.pushBackUnchecked(0);
                }
                return HK_SUCCESS;
            }
            minXmaxY -= 1;
            {
                TScalar maxXpos = points[NPOINT - 1].x;
                int i;
                for (i = NPOINT - 2; i >= 0 && points[i].x == maxXpos; --i)
                {
                }
                maxXminY = i + 1;
            }
        }

        // walk forwards along along bottom
        {
            TVector2 minXminYpoint = points[minXminY];
            TVector2 maxXminYpoint = points[maxXminY];

            indicesOut.pushBackUnchecked(originalIndices[minXminY]);
            pointsOut.pushBackUnchecked(minXminYpoint);

            for (int i = minXmaxY; i <= maxXminY; ++i)
            {
                TVector2 p = points[i];
                if (!p.rightOfLine(minXminYpoint, maxXminYpoint) && i < maxXminY)
                {
                    continue;
                }
                while (pointsOut.getSize() >= 2)
                {
                    int n = pointsOut.getSize();
                    if (p.leftOfLine(pointsOut[n - 2], pointsOut[n - 1]))
                    {
                        break;
                    }
                    indicesOut.popBack();
                    pointsOut.popBack();
                }
                indicesOut.pushBackUnchecked(originalIndices[i]);
                pointsOut.pushBackUnchecked(p);
            }
        }

        // walk backwards along top.
        {
            pointsOut.clear();
            if (maxXmaxY != maxXminY)
            {
                indicesOut.pushBackUnchecked(originalIndices[maxXmaxY]);
            }
            pointsOut.pushBackUnchecked(points[NPOINT - 1]);

            TVector2 maxXmaxYpoint = points[NPOINT - 1];
            TVector2 minXmaxYpoint = points[minXmaxY];

            for (int i = maxXminY - 1; i >= minXmaxY; --i)
            {
                TVector2 p = points[i];
                if (!p.rightOfLine(maxXmaxYpoint, minXmaxYpoint) && i > minXmaxY)
                {
                    continue;
                }
                while (pointsOut.getSize() >= 2)
                {
                    int n = pointsOut.getSize();
                    if (p.leftOfLine(pointsOut[n - 2], pointsOut[n - 1]))
                    {
                        break;
                    }
                    indicesOut.popBack();
                    pointsOut.popBack();
                }
                indicesOut.pushBackUnchecked(originalIndices[i]);
                pointsOut.pushBackUnchecked(p);
            }

            if (minXmaxY != minXminY)
            {
                indicesOut.pushBackUnchecked(originalIndices[minXminY]);
            }
        }
        //for( int i = 1; i < indicesOut.getSize(); ++i )   { HK_ASSERT_NO_MSG(0x5c35248e, indicesOut[i] != indicesOut[i-1] ); }
        return HK_SUCCESS;
    }

    template<typename TVector2, typename TScalar>
    HK_INLINE void HK_CALL convexHullVerticesT(const hkArrayBase<TVector2>& points, hkArray<TVector2>& hullOut)
    {
        HK_ASSERT_NO_MSG(0x1a776041, points.getSize());
        hkArray<int>::Temp indices; indices.setSize(points.getSize());
        convexHullIndicesT<TVector2, TScalar>(points, indices);
        hullOut.reserveExactly(indices.getSize());
        for (int i = 0; i < indices.getSize(); ++i)
        {
            hullOut.pushBackUnchecked(points[indices[i]]);
        }
    }

    template<typename TVector2, typename TScalar>
    HK_INLINE hkBool HK_CALL edgesIntersectT(const TVector2& a, const TVector2& b, const TVector2& c, const TVector2& d)
    {
        // Solving for a + r(b-a) = c + s(d-c)
        TScalar denom = (b.x - a.x) * (d.y - c.y) - (b.y - a.y) * (d.x - c.x);
        TScalar rNum = (a.y - c.y) * (d.x - c.x) - (a.x - c.x) * (d.y - c.y);
        TScalar sNum = (a.y - c.y) * (b.x - a.x) - (a.x - c.x) * (b.y - a.y);

        // I don't need to worry about a divide by 0, because
        // if denom is 0, the equality below requires rNum > 0.0 && rNum < 0.0... which can never happen

        /* If 0<=r<=1 & 0<=s<=1, intersection exists */
        if (denom < TScalar(0))
        {
            denom = -denom;
            rNum = -rNum;
            sNum = -sNum;
        }

        return (rNum > TScalar(0) && sNum > TScalar(0) && rNum < denom && sNum < denom);
    }

    
    template<typename TVector2, typename TScalar>
    HK_INLINE TScalar computeObbT(const hkArrayBase<TVector2>& points, const TScalar maxScalarValue, TVector2& centerOut, TVector2& e0Out, TVector2& e1Out, hkResult& resOut)
    {
        hkArray<int> hullIndices;
        hkResult hullIndicesRes = hullIndices.reserve(points.getSize() + 1);
        if (hullIndicesRes.isFailure())
        {
            resOut = HK_FAILURE;
            return TScalar(0);
        }

        hkResult cvxHullRes = convexHullIndicesT<TVector2, TScalar>(points, hullIndices);
        if (cvxHullRes.isFailure())
        {
            resOut = HK_FAILURE;
            return TScalar(0);
        }

        // convexHullIndices duplicates the first and last vertices. Remove the last one to avoid extra work.
        HK_ASSERT_NO_MSG(0x3984fef6, hullIndices[0] == hullIndices.back());
        hullIndices.popBack();

        TScalar minArea = maxScalarValue;

        for (int i = 0, j = hullIndices.getSize() - 1; i < hullIndices.getSize(); j = i, i++)
        {
            TVector2 e0; e0.setSub(points[hullIndices[i]], points[hullIndices[j]]);
            if (e0.normalizeWithLength() == TScalar(0))
                continue;

            TVector2 e1; e1.setPerp(e0);

            TScalar min0 = TScalar(0), min1 = TScalar(0), max0 = TScalar(0), max1 = TScalar(0);
            for (int k = 0; k < hullIndices.getSize(); k++)
            {
                TVector2 d; d.setSub(points[hullIndices[k]], points[hullIndices[j]]);
                TScalar dot = d.dot(e0);
                if (dot < min0) min0 = dot;
                if (dot > max0) max0 = dot;
                dot = d.dot(e1);
                if (dot < min1) min1 = dot;
                if (dot > max1) max1 = dot;
            }
            TScalar area = (max0 - min0) * (max1 - min1);

            if (area < minArea)
            {
                minArea = area;
                TVector2 v;
                v.x = (min0 + max0)*e0.x + (min1 + max1)*e1.x;
                v.y = (min0 + max0)*e0.y + (min1 + max1)*e1.y;
                centerOut.setAddMul(points[hullIndices[j]], v, TScalar(0.5));
                e0Out.setMul(e0, (max0 - min0)*TScalar(0.5));
                e1Out.setMul(e1, (max1 - min1)*TScalar(0.5));
            }
        }

        resOut = HK_SUCCESS;
        return minArea;
    }

} // HK_UNITY_ANONYMOUS_NAMESPACE

namespace hkVector2Util
{
    //
    // edgesIntersect (float ant double overloads)
    //
    hkBool HK_CALL edgesIntersect(const hkVector2f& a, const hkVector2f& b, const hkVector2f& c, const hkVector2f& d)
    {
        return HK_UNITY_ANONYMOUS_NAMESPACE::edgesIntersectT<hkVector2f, hkFloat32>(a, b, c, d);
    }

    hkBool HK_CALL edgesIntersect(const hkVector2d& a, const hkVector2d& b, const hkVector2d& c, const hkVector2d& d)
    {
        return HK_UNITY_ANONYMOUS_NAMESPACE::edgesIntersectT<hkVector2d, hkDouble64>(a, b, c, d);
    }

    //
    // convexHullSimplePolyline (float ant double overloads)
    //
    void HK_CALL convexHullSimplePolyline( const hkArrayBase<hkVector2f>& line, hkArray<int>& indicesOut )
    {
        HK_UNITY_ANONYMOUS_NAMESPACE::convexHullSimplePolylineT<hkVector2f>(line, indicesOut);
    }

    void HK_CALL convexHullSimplePolyline(const hkArrayBase<hkVector2d>& line, hkArray<int>& indicesOut)
    {
        HK_UNITY_ANONYMOUS_NAMESPACE::convexHullSimplePolylineT<hkVector2d>(line, indicesOut);
    }

    //
    // convexHullVertices (float ant double overloads)
    //
    void HK_CALL convexHullVertices(const hkArrayBase<hkVector2f>& points, hkArray<hkVector2f>& hullOut)
    {
        HK_UNITY_ANONYMOUS_NAMESPACE::convexHullVerticesT<hkVector2f, hkFloat32>(points, hullOut);
    }

    void HK_CALL convexHullVertices(const hkArrayBase<hkVector2d>& points, hkArray<hkVector2d>& hullOut)
    {
        HK_UNITY_ANONYMOUS_NAMESPACE::convexHullVerticesT<hkVector2d, hkDouble64>(points, hullOut);
    }

    //
    // convexHullIndices (float ant double overloads)
    //
    hkResult HK_CALL convexHullIndices(const hkArrayBase<hkVector2f>& originalPoints, hkArrayBase<int>& indicesOut)
    {
        return HK_UNITY_ANONYMOUS_NAMESPACE::convexHullIndicesT<hkVector2f, hkFloat32>(originalPoints, indicesOut);
    }

    hkResult HK_CALL convexHullIndices(const hkArrayBase<hkVector2d>& originalPoints, hkArrayBase<int>& indicesOut)
    {
        return HK_UNITY_ANONYMOUS_NAMESPACE::convexHullIndicesT<hkVector2d, hkDouble64>(originalPoints, indicesOut);
    }

    //
    // computeObb (float ant double overloads)
    // Based on code from Christer Ericson's "Real-time Collision Detection" book
    //
    hkFloat32 computeObb(const hkArrayBase<hkVector2f>& points, hkVector2f& centerOut, hkVector2f& e0Out, hkVector2f& e1Out, hkResult& resOut)
    {
        return HK_UNITY_ANONYMOUS_NAMESPACE::computeObbT<hkVector2f, hkFloat32>(points, HK_FLOAT_MAX, centerOut, e0Out, e1Out, resOut);
    }

    hkDouble64 computeObb(const hkArrayBase<hkVector2d>& points, hkVector2d& centerOut, hkVector2d& e0Out, hkVector2d& e1Out, hkResult& resOut)
    {
        return HK_UNITY_ANONYMOUS_NAMESPACE::computeObbT<hkVector2d, hkDouble64>(points, HK_DOUBLE_MAX, centerOut, e0Out, e1Out, resOut);
    }

} // hkVector2Util

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
