// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Math/Vector/hkVector4Util.h>

/// Miscellaneous operations involving symmetric 3x3 matrices
template <typename FT>
class hkSymmetricMatrixUtilImpl
{
public:

    typedef hkSymmetricMatrixUtilImpl<FT> ThisType;
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MATH, ThisType);

    /// Sets mtxOut = outer product of the given vector, i.e [xx, yy, zz], [xy, yz, zx]
    static HK_INLINE void HK_CALL _computeOuterProduct(typename hkFloatTypes<FT>::Vec4_ v, typename hkFloatTypes<FT>::SymmetricMatrix3& mtxOut);

    /// Projects the tensor on the given direction, i.e. computes Transpose(d) * mtx * d
    static HK_INLINE const typename hkFloatTypes<FT>::Scalar HK_CALL _computeProjection(const typename hkFloatTypes<FT>::SymmetricMatrix3& mtx, typename hkFloatTypes<FT>::Vec4_ vD);

    /// Sets mtxOut = Transpose(R) * Diag(a) * R, where Diag(a) is a diagonal matrix with diagonal elements a.
    static HK_INLINE void HK_CALL _computeInverseBasisTransform(const typename hkFloatTypes<FT>::Matrix3& R, typename hkFloatTypes<FT>::Vec4_ diagA, typename hkFloatTypes<FT>::SymmetricMatrix3& mtxOut);

    /// Solves the linear system: mtx * v = b, for v.
    static HK_INLINE void HK_CALL _solve(const typename hkFloatTypes<FT>::SymmetricMatrix3& mtx, typename hkFloatTypes<FT>::Vec4_ b, typename hkFloatTypes<FT>::Vector& solutionOut);

    /// Sets inverseOut = Inverse(t).
    static HK_INLINE void HK_CALL _computeInverse(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::SymmetricMatrix3& inverseOut);

    /// Computes vOut = t * vIn
    static HK_INLINE void HK_CALL _multiply(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::Vec4_ vIn, typename hkFloatTypes<FT>::Vector& vOut);

    /// Computes tOut = rot * ta * Transpose[rot]
    static HK_INLINE void HK_CALL _computeBasisTransform(const typename hkFloatTypes<FT>::Rotation& rot, const typename hkFloatTypes<FT>::SymmetricMatrix3& ta, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);
    static HK_INLINE void HK_CALL _computeBasisTransform(const typename hkFloatTypes<FT>::Rotation& rot, typename hkFloatTypes<FT>::Vec4_ vDiag, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);
    static HK_INLINE void HK_CALL _computeBasisTransform(typename hkFloatTypes<FT>::Quaternion_ rot, const typename hkFloatTypes<FT>::SymmetricMatrix3& ta, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);
    static HK_INLINE void HK_CALL _computeBasisTransform(typename hkFloatTypes<FT>::Quaternion_ rot, typename hkFloatTypes<FT>::Vec4_ vDiag, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);

    /// Solves the linear system: mtx * v = b for v, where |v| < vMax per component.
    /// The problem is a LCP and the returned solution is an approximation computed via Jacobi iterations.
    /// The number of iterations is given as a template parameter
    template <int NIter>
    static HK_INLINE void HK_CALL _solve(const typename hkFloatTypes<FT>::SymmetricMatrix3& mtx, typename hkFloatTypes<FT>::Vec4_ b, typename hkFloatTypes<FT>::Vec4_ vMax, typename hkFloatTypes<FT>::Vector& solutionOut);

    /// Computes tOut = Hat[r] * ta * Hat[r]
    static HK_INLINE void HK_CALL _computeDisplacedTensor(typename hkFloatTypes<FT>::Vec4_ r, const typename hkFloatTypes<FT>::SymmetricMatrix3& ta, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);
    static HK_INLINE void HK_CALL _computeDisplacedTensor(typename hkFloatTypes<FT>::Vec4_ r, typename hkFloatTypes<FT>::Vec4_ diag, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);

    /// Computes tOut = Hat[r] * Hat[r]
    static HK_INLINE void HK_CALL _computeDisplacement(typename hkFloatTypes<FT>::Vec4_ r, typename hkFloatTypes<FT>::SymmetricMatrix3& tOut);

    /// Returns the element with maximum absolute value in the matrix
    static HK_INLINE const typename hkFloatTypes<FT>::Scalar HK_CALL _getMaxAbsElement(const typename hkFloatTypes<FT>::SymmetricMatrix3& t);

    /// Computes the SVD of the given matrix t = R * D * Transpose[R]
    static HK_INLINE void HK_CALL _computeSvd(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::Rotation& eigenVecsOut, typename hkFloatTypes<FT>::Vector& eigenValsOut);

    /// Computes the coefficients of the characteristic polynomial
    static HK_INLINE void HK_CALL _computeCharacteristicPolynomial(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::Vector& coeffsOut);

    /// Computes the eigenvalues using the characteristic polynomial. Note that the results are often affected by numerical imprecisions!
    static HK_INLINE hkResult HK_CALL _computeEigenValues(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::Vector& eigenValsOut);

    /// Computes the symmetric matrix obtained by computing the cross products between consecutive columns, i.e. {Cross[c1,c2], Cross[c2, c0], Cross[c0, c1]}
    static HK_INLINE void HK_CALL _computeColumnCrossProducts(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::SymmetricMatrix3& crossProductsOut);

    /// Computes an eigenvector for the given eigenvalue
    static HK_INLINE void HK_CALL _computeEigenVector(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::Scalar_ eigenValueIn, typename hkFloatTypes<FT>::Vector& eigenVectorOut);

    /// Computes the squared lengths of the given matrix's columns
    static HK_INLINE void HK_CALL _computeColumnLengthsSq(const typename hkFloatTypes<FT>::SymmetricMatrix3& t, typename hkFloatTypes<FT>::Vector& sqLengthsOut);
};

#include <Common/Base/Math/SymmetricMatrix/hkSymmetricMatrix3Util.inl>

typedef hkSymmetricMatrixUtilImpl<hkFloat32>  hkSymmetricMatrix3fUtil;
typedef hkSymmetricMatrixUtilImpl<hkDouble64> hkSymmetricMatrix3dUtil;

#if defined(HK_REAL_IS_DOUBLE)
typedef hkSymmetricMatrix3dUtil hkSymmetricMatrix3Util;
#else
typedef hkSymmetricMatrix3fUtil hkSymmetricMatrix3Util;
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
